##############################################################################
# Module : utility.py
# Author :
# Date   :
# Updated and modified for ABC_OKC : Old King Cole
#
# Description : ABC_OKC parameters and various utilities
#
##############################################################################
import wx, sys, socket, platform
from binascii import hexlify
from shutil import copyfile

from lang.lang import Lang, ABC_OKC_BUILD
from os import rename, remove, path, walk, system, mkdir, listdir
from distutils.dir_util import copy_tree
from threading import Thread, Timer
from time import strftime, time
from math import ceil
from wx.lib.buttons import GenBitmapButton, GenBitmapToggleButton
from win32api import MessageBox
from win32con import VER_PLATFORM_WIN32_NT, VER_PLATFORM_WIN32_WINDOWS
if sys.getwindowsversion()[3] == VER_PLATFORM_WIN32_NT:
    from win32api import GetCurrentProcess, ExitWindowsEx
    from win32security import OpenProcessToken, LookupPrivilegeValue, AdjustTokenPrivileges
    from win32con import TOKEN_ALL_ACCESS, SE_SHUTDOWN_NAME, SE_PRIVILEGE_ENABLED

from BitTornado.utility import sysencoding

from filemanager import TorrentConfigFileManager

def getUserDirectories():
    # Returns user and ABC_OKC data directories
    userdir = path.expanduser('~')
    if not path.isdir(userdir):
        if sys.getwindowsversion()[3] == VER_PLATFORM_WIN32_WINDOWS:
            # Windows 9x
            standarduserappdata = path.join(path.expandvars('${WINDIR}'), 'application data')
            if path.exists(standarduserappdata):
                if path.isdir(standarduserappdata):
                    userdir = standarduserappdata
                else:
                    userdir = path.abspath(path.dirname(sys.argv[0]))
            else:
                mkdir(standarduserappdata)
                userdir = standarduserappdata
        else:
            userdir = path.abspath(path.dirname(sys.argv[0]))
    for var in ['${APPDATA}', '${HOME}', '${HOMEPATH}', '${USERPROFILE}']:
        appdatadir = path.expandvars(var)
        if appdatadir != var and path.isdir(appdatadir):
            break
    else:
        appdatadir = userdir
    add = unicode(path.join(appdatadir, 'ABC_OKC'))
    if not path.exists(add):
        mkdir(add)
    return unicode(userdir), add

USERPATH, DATAPATH = getUserDirectories()


class Utility:
    def __init__(self, parent, abcpath):
        self.time0 = time()
        self.abcpath = abcpath
        self.parent = parent

        # Invalid characters for file names in Windows
        self.invalidwinfilenamechar = ''
        for i in xrange(32):
            self.invalidwinfilenamechar += chr(i)
        self.invalidwinfilenamechar += '"*/:<>?\\|'

        # System support for IPv6 ?
        self.hasipv6 = sys.version_info >= (2, 3) and socket.has_ipv6

        # List to keep formated data from the column of a listctrl being sorted
        self.sortcol = []

        # Torrent advanced details columns
        try:
            fw = wx.SystemSettings_GetFont(wx.SYS_DEFAULT_GUI_FONT).GetPointSize() + 1
        except:
            fw = wx.SystemSettings_GetFont(wx.SYS_SYSTEM_FONT).GetPointSize() + 1
        self.advdetcolinfo = [('advdetoptunch', 'advdetwidthoptunch', wx.LIST_FORMAT_CENTER, fw * 2 + 4),
                              ('advdetip', 'advdetwidthip', wx.LIST_FORMAT_LEFT, fw * 11),
                              ('advdetlocrem', 'advdetwidthlocrem', wx.LIST_FORMAT_CENTER, fw * 3),
                              ('advdetdnrate', 'advdetwidthdnrate', wx.LIST_FORMAT_RIGHT, fw * 7),
                              ('advdetinteresting', 'advdetwidthinteresting', wx.LIST_FORMAT_CENTER, fw * 2 + 2),
                              ('advdetchoked', 'advdetwidthchoked', wx.LIST_FORMAT_CENTER, fw * 2 + 2),
                              ('advdetsnubbed', 'advdetwidthsnubbed', wx.LIST_FORMAT_CENTER, fw * 2 + 2),
                              ('advdetuprate', 'advdetwidthuprate', wx.LIST_FORMAT_RIGHT, fw * 7),
                              ('advdetinterested', 'advdetwidthinterested', wx.LIST_FORMAT_CENTER, fw * 2 + 2),
                              ('advdetchoking', 'advdetwidthchoking', wx.LIST_FORMAT_CENTER, fw * 2 + 2),
                              ('advdetreceived', 'advdetwidthreceived', wx.LIST_FORMAT_RIGHT, fw * 8),
                              ('advdetsent', 'advdetwidthsent', wx.LIST_FORMAT_RIGHT, fw * 8),
                              ('advdetprogress', 'advdetwidthprogress', wx.LIST_FORMAT_RIGHT, fw * 6),
                              ('advdetpeerdnspeed', 'advdetwidthpeerdnspeed', wx.LIST_FORMAT_RIGHT, fw * 7)]

        # RSS panel columns
        self.rsscolinfo = [('rsstitle', 'rsswidthtitle', wx.LIST_FORMAT_LEFT),
                           ('rssurl', 'rsswidthurl', wx.LIST_FORMAT_LEFT),
                           ('rsstime', 'rsswidthtime', wx.LIST_FORMAT_LEFT)]

        # User and data directories
        self.userpath = USERPATH
        self.datapath = DATAPATH

        # Old data directory
        if sys.getwindowsversion()[3] == VER_PLATFORM_WIN32_WINDOWS:
            # Windows 9x
            self.olddatapath = path.join(path.expandvars('${WINDIR}'), 'all users', 'application data', 'ABC_OKC', 'olddata')
        else:
            if sys.getwindowsversion()[0] >= 6:
                # Vista and above
                applicationdata = ''
            else:
                applicationdata = 'application data'
            self.olddatapath = path.join(path.expandvars('${ALLUSERSPROFILE}'), applicationdata, 'ABC_OKC', 'olddata')

        # Init configuration files
        self.initConfigFiles()

        # Read ABC_OKC parameters
        self.readParams()

        # Read language file
        self.lang = Lang(self.parent, self.abcpath, self.abcparams['language_file'])
        self.localize = self.lang.get 

        # Windows >=7 ?
        if 'win7fix' not in self.abcparams:
            try:
                self.abcparams['win7fix'] = str(int(int(platform.release()) >= 7))
            except:
                self.abcparams['win7fix'] = '0'

        # IPv6 enabled by user
        self.ipv6enabled = self.hasipv6 and self.abcparams['ipv6'] == '1'

        # Default main port
        self.mainport = 56666

        # Default remote ip
        self.remoteip = None

        # DHT port
        try:
            self.dhtport = int(self.abcparams['dhtport'])
        except:
            self.dhtport = 56667
        else:
            if self.dhtport < 0  or self.dhtport > 65535:
                self.dhtport = 56667

        # Peer Listener port
        try:
            self.plport = int(self.abcparams['plport'])
        except:
            self.plport = 56667
        else:
            if self.plport < 0  or self.plport > 65535:
                self.plport = 56667

        # GUI update rates
        try:
            self.GUIupdaterate_fast = float(self.abcparams['guiupdateratefast'])
            self.GUIupdaterate_slow = float(self.abcparams['guiupdaterateslow'])
        except:
            self.GUIupdaterate_fast = 1.0
            self.GUIupdaterate_slow = 2.0
        else:
            if self.GUIupdaterate_fast < 0.5:
                self.GUIupdaterate_fast = 0.5
            if self.GUIupdaterate_slow < 0.5:
                self.GUIupdaterate_slow = 0.5
        self.GUIupdaterate = self.GUIupdaterate_fast

        # Colours
        rgb = map(int, self.abcparams['colnocon'].split(','))
        self.colnocon = wx.Colour(rgb[0], rgb[1], rgb[2])
        rgb = map(int, self.abcparams['colok'].split(','))
        self.colok = wx.Colour(rgb[0], rgb[1], rgb[2])
        rgb = map(int, self.abcparams['colnocom'].split(','))
        self.colnocom = wx.Colour(rgb[0], rgb[1], rgb[2])
        rgb = map(int, self.abcparams['coldisc'].split(','))
        self.coldisc = wx.Colour(rgb[0], rgb[1], rgb[2])
        rgb = map(int, self.abcparams['colnoinc'].split(','))
        self.colnoinc = wx.Colour(rgb[0], rgb[1], rgb[2])
        rgb = map(int, self.abcparams['colstripes'].split(','))
        self.colstripes = wx.Colour(rgb[0], rgb[1], rgb[2])
        self.colsyswin = wx.SystemSettings.GetColour(wx.SYS_COLOUR_WINDOW)
        self.colnoeng = wx.Colour(0, 0, 0)
        rgb = map(int, self.abcparams['colprivate'].split(','))
        self.colprivate = wx.Colour(rgb[0], rgb[1], rgb[2])

        # Seeding time counter behaviour
        # Count
        # 0 : Always count seeding time whatever seeding option is set ;
        # 1 : Only count seeding time when seeding option is time limit ;
        self.seedingtimecount = self.abcparams['seedingtimecount']
        # Reset time count on time limit value change
        # 0 : Don't reset ;
        # 1 : Reset ;
        self.seedingtimeresetonvalue = self.abcparams['seedingtimeresetonvalue']
        # Reset time count on switch to time limit option
        # 0 : Don't reset ;
        # 1 : Reset ;
        self.seedingtimeresetonswitch = self.abcparams['seedingtimeresetonswitch']

        # Scheduler cyclical tasks rate
        if self.abcparams['tweaksysparams'] == '0':
            self.initDefaultCyclicalTasksRate()
        else:
            try:
                self.cyclicaltasksrate = float(self.abcparams['cyclicaltasksrate'])
            except:
                self.initDefaultCyclicalTasksRate()
            else:
                if self.cyclicaltasksrate < 0.5:
                    self.cyclicaltasksrate = 0.5

        # Data cache expiration (try once an hour)
        self.expirecacherate = 3600 / self.cyclicaltasksrate
        self.expirecachedata = int(self.abcparams['expire_cache_data']) * 86400

        # Torrent list auto save rate
        try:
            self.torlistautosaverate = float(self.abcparams['torlistautosaverate'])
        except:
            self.torlistautosaverate = 600.0
        else:
            if self.torlistautosaverate < 60:
                self.torlistautosaverate = 60.0

        # Expire unused buffers rate (0=disabled)
        try:
            self.expirebuffersrate = float(self.abcparams['expirebuffersrate'])
        except:
            self.expirebuffersrate = 600.0
        else:
            if self.expirebuffersrate and self.expirebuffersrate < 60:
                self.expirebuffersrate = 60.0

        # Expire unused buffers age
        try:
            self.expirebuffersage = float(self.abcparams['expirebuffersage'])
        except:
            self.expirebuffersage = 1800.0
        else:
            if self.expirebuffersage < 300:
                self.expirebuffersage = 300.0

        # PEX message sending rate
        try:
            self.utpexsendingrate = int(self.abcparams['utpexsendingrate'])
        except:
            self.utpexsendingrate = 90
        else:
            if self.utpexsendingrate < 60:
                self.utpexsendingrate = 60

        # Delay before URM starts after starting of ABC_OKC
        try:
            self.urmstartdelay = int(self.abcparams['urmstartdelay'])
        except:
            self.urmstartdelay = 180

        # Max number of extra torrents URM may start
        try:
            self.urmmaxtorrent = int(self.abcparams['urmmaxtorrent'])
        except:
            self.urmmaxtorrent = 3

        # URM threshold to start a torrent
        try:
            self.urmthresholdstart = float(self.abcparams['urmthresholdstart'])
        except:
            self.urmthresholdstart = 20.0

        # URM threshold to stop a torrent
        self.urmthresholdstop = self.abcparams['urmthresholdstop']
        if self.urmthresholdstop != '-':
            try:
                self.urmthresholdstop = float(self.urmthresholdstop)
            except:
                self.urmthresholdstop = 5.0

        # Delay over threshold for URM to start a torrent
        try:
            self.urmdelay = int(self.abcparams['urmdelay'])
        except:
            self.urmdelay = 10

        # Delay over threshold for URM to stop a torrent
        try:
            self.urmdelaystop = int(self.abcparams['urmdelaystop'])
        except:
            self.urmdelaystop = 5

        # Delay before URM starts/stops a torrent after a torrent starts
        try:
            self.urmtorrentstartdelay = int(self.abcparams['urmtorrentstartdelay'])
        except:
            self.urmtorrentstartdelay = 60

        # Delay before URM starts a torrent after a torrent stops
        try:
            self.urmtorrentstopdelay = int(self.abcparams['urmtorrentstopdelay'])
        except:
            self.urmtorrentstopdelay = 30

        # Delay before DRM starts after starting of ABC_OKC
        try:
            self.drmstartdelay = int(self.abcparams['drmstartdelay'])
        except:
            self.drmstartdelay = 180

        # Max number of extra torrents DRM may start
        try:
            self.drmmaxtorrent = int(self.abcparams['drmmaxtorrent'])
        except:
            self.drmmaxtorrent = 2

        # DRM threshold to start a torrent
        try:
            self.drmthresholdstart = float(self.abcparams['drmthresholdstart'])
        except:
            self.drmthresholdstart = 1000.0

        # DRM threshold to stop a torrent
        self.drmthresholdstop = self.abcparams['drmthresholdstop']
        if self.drmthresholdstop != '-':
            try:
                self.drmthresholdstop = float(self.drmthresholdstop)
            except:
                self.drmthresholdstop = 300.0

        # Delay over threshold for DRM to start a torrent
        try:
            self.drmdelay = int(self.abcparams['drmdelay'])
        except:
            self.drmdelay = 10

        # Delay over threshold for DRM to stop a torrent
        try:
            self.drmdelaystop = int(self.abcparams['drmdelaystop'])
        except:
            self.drmdelaystop = 5

        # Delay before DRM starts/stops a torrent after a torrent starts
        try:
            self.drmtorrentstartdelay = int(self.abcparams['drmtorrentstartdelay'])
        except:
            self.drmtorrentstartdelay = 60

        # Delay before DRM starts a torrent after a torrent stops
        try:
            self.drmtorrentstopdelay = int(self.abcparams['drmtorrentstopdelay'])
        except:
            self.drmtorrentstopdelay = 30

        # Minimum number of uploads
        try:
            self.minuploads = int(self.abcparams['minuploads'])
        except:
            self.minuploads = 0
        else:
            if self.minuploads < 0:
                self.minuploads = 0
            elif self.minuploads > 100:
                self.minuploads = 100

        # Minimum up rate
        try:
            self.minuprate = float(self.abcparams['minuprate'])
        except:
            self.minuprate = 3.
        else:
            if self.minuprate < 1:
                self.minuprate = 1.

        # Minimum down rate
        try:
            self.mindownrate = float(self.abcparams['mindownrate'])
        except:
            self.mindownrate = 5.
        else:
            if self.mindownrate < 1:
                self.mindownrate = 1.

        # Maximum upload rate
        try:
            self.maxup = float(self.abcparams['maxuploadrate'])
            self.maxseedup = float(self.abcparams['maxseeduploadrate'])
            self.maxsysup = float(self.abcparams['urmmaxsysuprate'])
        except:
            self.maxup = self.maxseedup = self.maxsysup = 0.
        else:
            if self.maxup and self.maxup < self.minuprate:
                self.maxup = self.minuprate
            if self.maxseedup and self.maxseedup < self.minuprate:
                self.maxseedup = self.minuprate
            if self.maxsysup and self.maxsysup < self.minuprate:
                self.maxsysup = self.minuprate

        # Maximum download rate
        try:
            self.maxdown = float(self.abcparams['maxdownloadrate'])
        except:
            self.maxdown = 0.
        else:
            if self.maxdown and self.maxdown < self.mindownrate:
                self.maxdown = self.mindownrate

        # Auto scrape rate
        try:
            self.scrapeauto = int(self.abcparams['scrapeauto'])
        except:
            self.scrapeauto = 1200
        else:
            if self.scrapeauto < 60:
                self.scrapeauto = 60

        # Standby minimum scrape period
        try:
            self.sbscrape = int(self.abcparams['sbscrape'])
        except:
            self.sbscrape = 1200
        else:
            if self.sbscrape < 60:
                self.sbscrape = 60

        # Standby scrape period per tracker
        try:
            self.sbscrapeperiod = 3600 / float(self.abcparams['sbscrapeph'])
        except:
            self.sbscrapeperiod = 120.

        # Delay to wait for leechers to connect after a standby uncomplete torrent with leechers was resumed
        try:
            self.sbwaitleechers = int(self.abcparams['sbwaitleechers'])
        except:
            self.sbwaitleechers = 120
        else:
            if self.sbwaitleechers < 30:
                self.sbwaitleechers = 30

        # Torrent file list height in torrent details window
        try:
            self.torrentfilelistheight = int(self.abcparams['torrentfilelistheight'])
        except:
            self.torrentfilelistheight = 106

        # Torrent file list widths in torrent details window
        try:
            self.torrentfilelistwidth0 = int(self.abcparams['torrentfilelistwidth0'])
        except:
            self.torrentfilelistwidth0 = 22
        try:
            self.torrentfilelistwidth3 = int(self.abcparams['torrentfilelistwidth3'])
        except:
            self.torrentfilelistwidth3 = 50
        try:
            self.torrentfilelistwidth4 = int(self.abcparams['torrentfilelistwidth4'])
        except:
            self.torrentfilelistwidth4 = 220
        try:
            self.torrentfilelistwidth5 = int(self.abcparams['torrentfilelistwidth5'])
        except:
            self.torrentfilelistwidth5 = 220

        # Torrent tracker list height in torrent details window
        try:
            self.torrenttrackerlistheight = int(self.abcparams['torrenttrackerlistheight'])
        except:
            self.torrenttrackerlistheight = 60

        # Torrent tracker list widths in torrent details window
        try:
            self.torrenttrackerlistwidth0 = int(self.abcparams['torrenttrackerlistwidth0'])
        except:
            self.torrenttrackerlistwidth0 = 47
        try:
            self.torrenttrackerlistwidth1 = int(self.abcparams['torrenttrackerlistwidth1'])
        except:
            self.torrenttrackerlistwidth1 = 0

        # Torrent find result list widths
        try:
            self.abcfindlistwidth0 = int(self.abcparams['abcfindlistwidth0'])
        except:
            self.abcfindlistwidth0 = 40

        # Unsatisfied download rate threshold that will trigger the stopping of an extra torrent
        # in the scheduler download rate distribution PHASE 3, low rate priority drm algo
        try:
            self.missingdownratetostopdrm = float(self.abcparams['missingdownratetostopdrm'])
        except:
            self.missingdownratetostopdrm = 10.0

        # Unsatisfied upload rate threshold that will trigger the stopping of an extra torrent
        # in the scheduler upload rate distribution PHASE 3, low priority urm algo
        try:
            self.missingupratetostopurm = float(self.abcparams['missingupratetostopurm'])
        except:
            self.missingupratetostopurm = 1.0

        # Algo to choose the extra torrent to be queued/on-hold
        # 0 : Choose the torrent with the lower upload rate influence
        # 1 : Choose the torrent which when stopped will bring the upload rate closer to the max upload rate
        #     from below while being in the upload rate dead band, or closer to the max upload rate from above,
        #     or closer to the max upload rate minus the dead band from below.
        try:
            self.haltextraalgo = int(self.abcparams['haltextraalgo'])
        except:
            self.haltextraalgo = 1
        else:
            if self.haltextraalgo not in (0, 1):
                self.haltextraalgo = 1

        # How to center found row in torrent list window when scrolling below complete
        # 0.0 : Top of list
        # 0.5 : Middle of list
        # 1.0 : Bottom of list
        try:
            self.scrollmiddlecentering = float(self.abcparams['scrollmiddlecentering'])
        except:
            self.scrollmiddlecentering = 0.5
        else:
            if self.scrollmiddlecentering < 0:
                self.scrollmiddlecentering = 0.
            elif self.scrollmiddlecentering > 1:
                self.scrollmiddlecentering = 1.

        # Min peer connection timeout
        try:
            self.minpeerconnectiontimeout = int(self.abcparams['minpeerconnectiontimeout'])
        except:
            self.minpeerconnectiontimeout = 5
        else:
            if self.minpeerconnectiontimeout < 1:
                self.minpeerconnectiontimeout = 1

        # Max peer connection timeout
        try:
            self.maxpeerconnectiontimeout = int(self.abcparams['maxpeerconnectiontimeout'])
        except:
            self.maxpeerconnectiontimeout = 15
        else:
            if self.maxpeerconnectiontimeout < 1:
                self.maxpeerconnectiontimeout = 1

        # Parameters for the formula to estimate the real upload rate from the download rate and the upload rate
        # and the number of connections
        if self.abcparams['tweaksysparams'] == '0':
            self.initDefaultDynMaxUpRate()
        else:
            try:
                self.tweakdynmaxuprate = int(self.abcparams['tweakdynmaxuprate'])
                self.urmupfromdownA = float(self.abcparams['urmupfromdownA'])
                self.urmupfromdownB = float(self.abcparams['urmupfromdownB'])
                self.urmupfromdownC = float(self.abcparams['urmupfromdownC'])
                self.urmupfromdownD = float(self.abcparams['urmupfromdownD'])
                self.urmupfromdownE = float(self.abcparams['urmupfromdownE'])
                self.urmupfromdownF = float(self.abcparams['urmupfromdownF'])
                self.urmupfromdownG = float(self.abcparams['urmupfromdownG'])
            except:
                self.initDefaultDynMaxUpRate()

        # Parameters to tweak the bandwidth distribution
        try:
            # Show/hide bandwidth distribution parameters in Preferences
            self.tweakbandwidthdistribution = int(self.abcparams['tweakbandwidthdistribution'])
            # Down half deadband
            self.bdcalcdownth1 = float(self.abcparams['bdcalcdownth1'])
            # Factor for down half dead band to pull up the down rate of torrents
            self.bdcalcdownth2 = float(self.abcparams['bdcalcdownth2'])
            # Up half deadband
            self.bdcalcupth1 = float(self.abcparams['bdcalcupth1'])
            # Factor for up half dead band to pull up the up rate of torrents
            self.bdcalcupth2 = float(self.abcparams['bdcalcupth2'])
        except:
            self.initDefaultBandwidthDistribution()

        # Garbage collector
        # cyclicalgcrate = 0 : Full cyclical gc disabled and default gc enabled
        # cyclicalgcrate > 0 : Full cyclical gc enabled and running every -cyclicalgcrate seconds, and default gc enabled
        # cyclicalgcrate < 0 : Full cyclical gc enabled and running every cyclicalgcrate seconds, and default gc disabled
        try:
            self.cyclicalgcrate = int(self.abcparams['cyclicalgcrate'])
        except:
            self.cyclicalgcrate = 0
        else:
            if 0 < self.cyclicalgcrate < 60:
                self.cyclicalgcrate = 60
            elif -60 < self.cyclicalgcrate < 0:
                self.cyclicalgcrate = -60
        # Automatic garbage collector thresholds
        # Default for python 2.7 is (700, 10, 10)
        try:
            self.gcth1 = int(self.abcparams['gcth1'])
            self.gcth2 = int(self.abcparams['gcth2'])
            self.gcth3 = int(self.abcparams['gcth3'])
        except:
            self.gcth1 = 700
            self.gcth2 = 10
            self.gcth3 = 10

        # Unit size regulation
        if self.abcparams['tweaksysparams'] == '0':
            self.initDefaultUnitSize()
        else:
            try:
                self.unitsizestep = float(self.abcparams['unitsizestep'])
                self.unitsizeinctrig = int(self.abcparams['unitsizeinctrig'])
            except:
                self.initDefaultUnitSize()

        # Time stamp
        self.timestamp = self.abcparams['timestamp']
        self.timestamplen = len(strftime(self.timestamp))

        # Message header
        self.messageheader = self.localize('messageheader')
        self.messageheaderlen = len(self.messageheader)

        # Show Prioritize marker or not in down and up rate fields
        self.showprioritizemarker = (self.abcparams['showprioritizemarker'] == '1')

        # Show or hide exit window
        # 0 : Hide window
        # 1 : Always show window
        # 2 : Show window if pending moving data
        try:
            self.exitwindow = int(self.abcparams['exitwindow'])
        except:
            self.exitwindow = 2
        else:
            if self.exitwindow not in (0, 1, 2):
                self.exitwindow = 2

        # Auto save peers
        self.autosavepeers = (self.abcparams['autosavepeers'] == '1')

        # Check internal tracker when resuming
        try:
            self.checkinttrack = int(self.abcparams['checkinttrack'])
        except:
            self.checkinttrack = 2
        else:
            if self.checkinttrack not in (0, 1, 2):
                self.checkinttrack = 2

        # Key mapping for shortcuts for internal tracker checking when resuming
        # 0 : Torrent tracker with Preferences mode, 1 : INT, 2 : EXT mode 2,
        # 3 : EXT mode 0, 4 : EXT mode 1, 5 : EXT with Preferences mode
        # CTRL : default to 1
        try:
            self.restrackctrl = int(self.abcparams['restrackctrl'])
        except:
            self.restrackctrl = 0
        # SHIFT : default to 4
        try:
            self.restrackshift = int(self.abcparams['restrackshift'])
        except:
            self.restrackshift = 0
        # CTRL + SHIFT : default to 2
        try:
            self.restrackctrlshift = int(self.abcparams['restrackctrlshift'])
        except:
            self.restrackctrlshift = 0
        # DEL : default to 3
        try:
            self.restrackdel = int(self.abcparams['restrackdel'])
        except:
            self.restrackdel = 0
        # INS : default to 2
        try:
            self.restrackins = int(self.abcparams['restrackins'])
        except:
            self.restrackins = 0

        # Direction for mouse wheel to change the current tab
        # 1 or -1
        try:
            self.mousewheelontab = int(self.abcparams['mousewheelontab'])
        except:
            self.mousewheelontab = 1
        else:
            if self.mousewheelontab not in (-1, 1):
                self.mousewheelontab = 1

        # Whether to force internal tracker check for public torrents or not
        self.checkinttrackautopublic = (self.abcparams['checkinttrackautopublic'] == '1')

        # Delay to auto check internal tracker for torrents with external tracker and no download
        try:
            self.checkinttrackautodelay = int(self.abcparams['checkinttrackautodelay'])
        except:
            self.checkinttrackautodelay = 300
        else:
            if self.checkinttrackautodelay < 120:
                self.checkinttrackautodelay = 120

        # Digit and small button UI field height
        f = wx.Frame(None, -1)
        defaultdigitheight = wx.TextCtrl(f, -1).GetBestSize()[1]
        defaultbuttonheight = wx.Button(f, -1).GetBestSize()[1]
        f.Destroy()
        try: 
            self.digitheight = int(self.abcparams['digitheight'])
        except:
            self.digitheight = 0
        self.digitheight += defaultdigitheight
        try: 
            self.buttonheight = int(self.abcparams['buttonheight'])
        except:
            self.buttonheight = 0
        self.buttonheight += defaultbuttonheight

        # Time duration UI field style
        try:
            self.timedurationstyle = [(self.digitheight, wx.TE_RIGHT), \
                                      (-1, wx.TE_RIGHT | wx.BORDER_RAISED)][int(self.abcparams['timedurationstyle'])]
        except:
            self.timedurationstyle = (self.digitheight, wx.TE_RIGHT)

        # Max number of last peers to remember
        try:
            self.maxlastpeers = int(self.abcparams['maxlastpeers'])
        except:
            self.maxlastpeers = 10

        # DHT security extension (BEP 42)
        try:
            self.dhtsecurityextension = int(self.abcparams['dhtsecurityextension'])
        except:
            self.dhtsecurityextension = 0
        else:
            if self.dhtsecurityextension not in (0, 1):
                self.dhtsecurityextension = 0

        # DHT max up rate used for responses
        try:
            self.dhtmaxresprate = int(self.abcparams['dhtmaxresprate'])
        except:
            self.dhtmaxresprate = 5120
        else:
            if self.dhtmaxresprate < 0:
                self.dhtmaxresprate = 5120

        # DHT max received request rate
        try:
            self.dhtmaxreqrate = int(self.abcparams['dhtmaxreqrate'])
        except:
            self.dhtmaxreqrate = 100
        else:
            if self.dhtmaxreqrate < 0:
                self.dhtmaxreqrate = 100

        # Skip HTTPS certificate verification
        try:
            self.skipcertifverif = int(self.abcparams['skipcertifverif'])
        except:
            self.skipcertifverif = 1
        else:
            if self.skipcertifverif not in (0, 1):
                self.skipcertifverif = 1

        # Whether to reset timeouts on torrent start or not
        try:
            self.resettimeoutsonstart = int(self.abcparams['resettimeoutsonstart'])
        except:
            self.resettimeoutsonstart = 0
        else:
            if self.resettimeoutsonstart not in (0, 1):
                self.resettimeoutsonstart = 0

        # Max torrent down rate to lower speed for big torrents with too small pieces
        try:
            self.nbpiecesfixmaxdownrate = int(self.abcparams['nbpiecesfixmaxdownrate'])
        except:
            self.nbpiecesfixmaxdownrate = 500
        else:
            if self.nbpiecesfixmaxdownrate < 0:
                self.nbpiecesfixmaxdownrate = 0

        # Number of pieces that triggers lowering speed for big torrents with too many small pieces
        try:
            self.nbpiecesfixtrigger = int(self.abcparams['nbpiecesfixtrigger'])
        except:
            self.nbpiecesfixtrigger = 10000
        else:
            if self.nbpiecesfixtrigger < 0:
                self.nbpiecesfixtrigger = 0

        # Piece size that triggers lowering speed for big torrents with too many small pieces (in kB)
        try:
            self.piecesizefixtrigger = int(self.abcparams['piecesizefixtrigger']) * 1024
        except:
            self.piecesizefixtrigger = 524288
        else:
            if self.piecesizefixtrigger < 0:
                self.piecesizefixtrigger = 0

        # Delay to keep the torrent loaded popup displayed each time a torrent is loaded
        try:
            self.popuploadeddelay = int(self.abcparams['popuploadeddelay'])
        except:
            self.popuploadeddelay = 3
        else:
            if self.popuploadeddelay < 1:
                self.popuploadeddelay = 1

        # Max global number of simultaneously opened files in ABC_OKC
        try:
            self.maxstdio = int(self.abcparams['maxstdio'])
        except:
            self.maxstdio = -1
        else:
            if self.maxstdio > 2048 or 0 <= self.maxstdio < 512:
                self.maxstdio = -1

        # Tracker categories
        self.trackercat = [self.localize('trackerinternal'),
                           self.localize('trackerextra')]
        self.trackercatlen = len(self.trackercat[0])

        # Tracker connection error messages
        self.trackercnxerrors = [self.localize('cantreachtracker'),
                                 self.localize('rejectedtracker'),
                                 self.localize('baddatatracker'),
                                 self.localize('nodatatracker'),
                                 self.localize('missingexttracker'),
                                 self.localize('missinginttracker')]

        # Status for file names inside torrent
        self.filenamestatus = [self.localize('filenamecharnotfixed'),
                               self.localize('filenametoolongname'),
                               self.localize('filenametoolongpath'),
                               self.localize('filenametoolongdestpath'),
                               self.localize('filenameduplicatenotfixed')]

        # Count seeding time also for standby torrents
        try:
            self.countsbasseed = int(self.abcparams['countsbasseed'])
        except:
            self.countsbasseed = 1
        else:
            if self.countsbasseed not in (0, 1):
                self.countsbasseed = 1

        # Number of selected torrent field width
        try:
            self.selectedwidth = int(self.abcparams['selectedwidth'])
        except:
            self.selectedwidth = 25
        else:
            if self.selectedwidth < 25:
                self.selectedwidth = 25

        # Max file and path name length
        self.initPathNameMaxLength(int(self.abcparams['longpath']))

        # Max uploading time without downloading
        try:
            self.upwithoutdown = int(self.abcparams['upwithoutdown'])
            self.upwithoutdowntime = int(self.abcparams['upwithoutdowntime']) * 3600 + int(self.abcparams['upwithoutdowntimemn']) * 60
        except:
            self.upwithoutdown = 0
            self.upwithoutdowntime = 3600
        else:
            if self.upwithoutdowntime < 0:
                self.upwithoutdowntime = 0
        try:
            self.upwithoutdown2 = int(self.abcparams['upwithoutdown2'])
            self.upwithoutdownrate = int(self.abcparams['upwithoutdownrate'])
        except:
            self.upwithoutdown2 = 0
            self.upwithoutdownrate = 50
        else:
            if self.upwithoutdownrate < 3:
                self.upwithoutdownrate = 3

        # Double buffering for ListCtrl in Advanced Details window
        try:
            self.advdetdbuf = int(self.abcparams['advdetdbuf'])
        except:
            self.advdetdbuf = 0

        # List scrolling delay
        try:
            self.scrolldelay = float(self.abcparams['scrolldelay'])
        except:
            self.scrolldelay = 0.2
        else:
            if self.scrolldelay and self.scrolldelay < 0.01:
                self.scrolldelay = 0.01

        # Inverse Ctrl+x and Ctrl+b to inverse selection and select block in list
        if self.abcparams['invertxb'] == '0':
            self.selinvert1=ord('x')
            self.selinvert2=ord('X')
            self.selblock1=ord('b')
            self.selblock2=ord('B')
        else:
            self.selinvert1=ord('b')
            self.selinvert2=ord('B')
            self.selblock1=ord('x')
            self.selblock2=ord('X')

        # Maximum peer queue per priority level
        try:
            self.maxpeerqueue = int(self.abcparams['maxpeerqueue'])
        except:
            self.maxpeerqueue = 50

        # Units
        self.mbps = self.localize('MBps')
        self.kbps = self.localize('kBps')
        self.b = self.localize('Byte')
        self.kb = self.localize('kB')
        self.mb = self.localize('MB')
        self.gb = self.localize('GB')
        self.tb = self.localize('TB')
        self.d = self.localize('days')

        self.priorities_s = [self.localize('highest_s'),
                             self.localize('high_s'),
                             self.localize('normal_s'),
                             self.localize('low_s'),
                             self.localize('lowest_s')]
        self.partialmarker = self.localize('partialmarker')
        self.activitymarker = self.localize('activitymarker')
        self.inactivitymarker = self.localize('inactivitymarker')
        self.standbymarker = self.localize('standbymarker')
        self.notrackermarker = self.localize('notrackermarker')
        self.notrackeranswermarker = self.localize('notrackeranswermarker')
        self.scrapingmarker = self.localize('scrapingmarker')
        self.prioritizemarker = self.localize('prioritizemarker')
        self.starflag = [' ', '*']
        self.trackerbutintlabel = self.localize('trackerbutint')
        self.trackerbutextlabel = self.localize('trackerbutext')
        self.trackerbutwaitlabel = self.localize('trackerbutwait')

        # To speed up lists construction :
        self.advdetcolumns = []
        for colinfo in self.advdetcolinfo:
            self.advdetcolumns.append([self.localize(colinfo[0]), int(self.abcparams[colinfo[1]]), colinfo[2]])
        self.rsscolumns = []
        for colinfo in self.rsscolinfo:
            self.rsscolumns.append([self.localize(colinfo[0]), int(self.abcparams[colinfo[1]]), colinfo[2]])

        if self.abcparams['win7fix'] == '1':
            self.win7fix = True
            self.win7fix_kbps = len([c for c in self.kbps if c.isalpha()])
        else:
            self.win7fix = False
            self.win7fix_kbps = 0

        self.allocbuf = memoryview(bytearray(2097152))

    def readParams(self):
        self.abcparams = {}

        # Read ABC config file
        tcfm = TorrentConfigFileManager(path.join(self.datapath, "abc.conf"), self.abcparams)
        tcfm.readAllConfig()

        abcparamsdefaults = [
            ('scrape', '0'),
            ('defaultpriority', '2'),
            ('language_file', 'english.lang'),
            ('urm', '0'),
            ('urmmaxtorrent', '3'),
            ('urmthresholdstart', '20.0'),
            ('urmthresholdstop', '5.0'),
            ('urmlowpriority', '0'),
            ('urmonhold', '1'),
            ('urmstopsman', '0'),
            ('urmdelay', '4'),
            ('urmdelaystop', '5'),
            ('urmdynmaxuprate', '0'),
            ('urmmaxsysuprate', '0'),
            ('urmupfromdownA', '0.027'),
            ('urmupfromdownB', '0.16'),
            ('urmupfromdownC', '0.08'),
            ('urmupfromdownD', '0.04'),
            ('urmupfromdownE', '0.'),
            ('urmupfromdownF', '0.1'),
            ('urmupfromdownG', '140.8'),
            ('urmstartdelay', '180'),
            ('urmtorrentstartdelay', '60'),
            ('urmtorrentstopdelay', '30'),
            ('stripedlist', '1'),
            ('mode', '1'),
            ('prioritizelocal', '1'),
            ('prioritizelocaldown', '1'),
            ('askduplicatedest', '0'),
            ('fastresume', '1'),
            ('fastresumefirst', '0'),
            ('keepeta', '1'),
            ('quickmetainfo', '1'),
            ('displayunitsize', '1'),
            ('displayunitrate', '1'),
            ('torrentwhencompleted', '0'),
            ('torrentwhenfinished', '0'),
            ('centerquickdetails', '0'),
            ('mainwindowx', '0'),
            ('mainwindowy', '0'),
            ('showschedcounters', '1'),
            ('templatedefault', 'default'),
            ('lastopentemptab', '0'),
            ('lastopentortab', '0'),
            ('templatedlgx', '-1'),
            ('templatedlgy', '-1'),
            ('conlimiterdlgx', '-1'),
            ('conlimiterdlgy', '-1'),
            ('torrentparamsdlgx', '-1'),
            ('torrentparamsdlgy', '-1'),
            ('abcoptiondlgx', '-1'),
            ('abcoptiondlgy', '-1'),
            ('abcfinddlgx', '-1'),
            ('abcfinddlgy', '-1'),
            ('abctweakdlgx', '-1'),
            ('abctweakdlgy', '-1'),
            ('createtorrentdlgx', '-1'),
            ('createtorrentdlgy', '-1'),
            ('webservicedlgx', '-1'),
            ('webservicedlgy', '-1'),
            ('aboutmedlgx', '-1'),
            ('aboutmedlgy', '-1'),
            ('checklatestdlgx', '-1'),
            ('checklatestdlgy', '-1'),
            ('abcrentordlgw', '500'),
            ('abcrentordlgx', '-1'),
            ('abcrentordlgy', '-1'),
            ('abcsetdestdlgw', '500'),
            ('abcsetdestdlgx', '-1'),
            ('abcsetdestdlgy', '-1'),
            ('schedulerdlgx', '-1'),
            ('schedulerdlgy', '-1'),
            ('scannerdlgx', '-1'),
            ('scannerdlgy', '-1'),
            ('rssscanfreqdlgx', '-1'),
            ('rssscanfreqdlgy', '-1'),
            ('scrolllistud', '0'),
            ('scrolllisttb', '0'),
            ('findtorrentnamebtn', '0'),
            ('findtorrentfilenamebtn', '0'),
            ('findlabelbtn', '0'),
            ('findtrackernamebtn', '0'),
            ('maxdownloadrate', '0'),
            ('activitythdown', '0.3'),
            ('activitythup', '0.3'),
            ('activitydelay', '30'),
            ('schedonhold', '0'),
            ('schedstopsman', '0'),
            ('lastopenabcoptab', '0'),
            ('activitymax', '200'),
            ('diskfullth', '50'),
            ('diskfullman', '1'),
            ('diskfullmode', '0'),
            ('savesizesonexit', '0'),
            ('countonlyalive', '1'),
            ('countonhold', '0'),
            ('urmstoponpause', '0'),
            ('bdcalcdownth1', '2.0'),
            ('bdcalcdownth2', '2.0'),
            ('bdcalcupth1', '2.0'),
            ('bdcalcupth2', '2.0'),
            ('tweakdynmaxuprate', '0'),
            ('msgindetails', '0'),
            ('defrentorwithdest', '0'),
            ('lastdestloc', ''),
            ('useragentid', '0'),
            ('useragent1', ''),
            ('useragent2', ''),
            ('useragent3', ''),
            ('userextended1', ''),
            ('userextended2', ''),
            ('userextended3', ''),
            ('userid1', ''),
            ('userid2', ''),
            ('userid3', ''),
            ('savetorrent', '2'),
            ('savetordeffolder', ''),
            ('listfont', ''),
            ('listfontsize', ''),
            ('listfontstyle', ''),
            ('listfontweight', ''),
            ('sorttoronclose', '1'),
            ('maketorforeach', '0'),
            ('movemidbtn', '2'),
            ('stoptrafficbtn', '1'),
            ('stopresallbtn', '0'),
            ('schedulerbtn', '1'),
            ('scannerbtn', '1'),
            ('webservbtn', '1'),
            ('statusbarshift', '4'),
            ('sortbytime', '0'),
            ('upnp', '0'),
            ('mousemode', '1'),
            ('rubmessages', '1'),
            ('rubmsgbutton', '0'),
            ('stoprejected', '1'),
            ('sortinglock', '3'),
            ('erasetrackal', '1'),
            ('keeptermevtmsg', '1'),
            ('rssscanfreq', '10'),
            ('rsssash', '0'),
            ('downvolcheck', '0'),
            ('upvolcheck', '0'),
            ('downvolmax', '0'),
            ('upvolmax', '0'),
            ('downvolcountdur', '1'),
            ('upvolcountdur', '1'),
            ('downvolchecktime', '0'),
            ('upvolchecktime', '0'),
            ('downvol', '0'),
            ('upvol', '0'),
            ('displaydownvol', '0'),
            ('displayupvol', '0'),
            ('rejectexceptions', ''),
            ('colnocon', '250,0,0'),
            ('colok', '0,170,0'),
            ('colnocom', '0,0,240'),
            ('coldisc', '140,140,140'),
            ('colnoinc', '160,160,0'),
            ('colstripes', '245,245,245'),
            ('colcust01', '0,170,0'),
            ('colcust02', '0,0,240'),
            ('colcust03', '140,140,140'),
            ('colcust04', '255,255,255'),
            ('colcust05', '255,255,255'),
            ('colcust06', '255,255,255'),
            ('colcust07', '255,255,255'),
            ('colcust08', '255,255,255'),
            ('colcust09', '160,160,0'),
            ('colcust10', '250,0,0'),
            ('colcust11', '255,255,255'),
            ('colcust12', '255,255,255'),
            ('colcust13', '255,255,255'),
            ('colcust14', '255,255,255'),
            ('colcust15', '255,255,255'),
            ('colcust16', '255,255,255'),
            ('flattopbtn', '1'),
            ('flatbotbtn', '1'),
            ('rssautograb', '0'),
            ('rsswidthtitle', '400'),
            ('rsswidthurl', '500'),
            ('rsswidthtime', '400'),
            ('initlang', '0'),
            ('scrollbelowfinished', '0'),
            ('guiupdateratefast', '1'),
            ('guiupdaterateslow', '2'),
            ('cyclicaltasksrate', '2.0'),
            ('timestamp', '%m/%d-%H:%M'),
            ('canceltimeouts', '0'),
            ('canceltimeoutsmsg', ''),
            ('torlistautosaverate', '600'),
            ('crypto', '1'),
            ('showcheckversion', '1'),
            ('expire_cache_data', '15'),
            ('displayscrapingmsg', '0'),
            ('displayscrapestatus', '0'),
            ('displaycheckstatus', '1'),
            ('sizeinmb', '1'),
            ('lockedcompletedcheck', '0'),
            ('lockedcompletedmax', '5'),
            ('gcth1', '700'),
            ('gcth2', '10'),
            ('gcth3', '10'),
            ('tweaksysparams', '0'),
            ('useridmask1', ''),
            ('useridmask2', ''),
            ('useridmask3', ''),
            ('completedresumedelay', '2'),
            ('seedingtimecount', '1'),
            ('seedingtimeresetonvalue', '0'),
            ('seedingtimeresetonswitch', '0'),
            ('minuploads', '0'),
            ('minuprate', '3'),
            ('activityextra', '1'),
            ('extratobestopped', '2'),
            ('findshortlabelbtn', '0'),
            ('scrapeauto', '1200'),
            ('sbscrape', '1200'),
            ('sbscrapeph', '30'),
            ('sbwaitleechers', '120'),
            ('sbtimeout', '1'),
            ('sbtimeoutvalue', '7'),
            ('sbtimeoutaction', '0'),
            ('compshutseedingtime', '1'),
            ('sbqueued', '0'),
            ('filenamescheckresdlgx', '-1'),
            ('filenamescheckresdlgy', '-1'),
            ('centerdetoneachother', '1'),
            ('displaytimeout', '1'),
            ('scheddonthalt', '0'),
            ('scheddonthaltuprategap', '15'),
            ('torrentfilelistheight', '106'),
            ('torrentfilelistwidth0', '22'),
            ('torrentfilelistwidth3', '50'),
            ('torrentfilelistwidth4', '220'),
            ('torrentfilelistwidth5', '220'),
            ('torrenttrackerlistwidth0', '47'),
            ('torrenttrackerlistwidth1', '0'),
            ('abcfindlistwidth0', '40'),
            ('missingdownratetostopdrm', '10.0'),
            ('missingupratetostopurm', '1.0'),
            ('haltextraalgo', '1'),
            ('tweakbandwidthdistribution', '0'),
            ('timeoutwhenextra', '1'),
            ('dht', '0'),
            ('dhtnodeid', ''),
            ('dhtrerequestinterval', '300'),
            ('rerequestinterval', '300'),
            ('torrenttrackerlistheight', '60'),
            ('dhtaddnodedlgx', '-1'),
            ('dhtaddnodedlgy', '-1'),
            ('addpeerdlgx', '-1'),
            ('addpeerdlgy', '-1'),
            ('dhtport', '56666'),
            ('cyclicalgcrate', '900'),
            ('multiparamdefault', '1'),
            ('ctrlscraperescanstrackers', '1'),
            ('keepmagnetname', '0'),
            ('metadatatimeout', '5'),
            ('utpex', '1'),
            ('utpexsendingrate', '90'),
            ('writebuffersize', '4.0'),
            ('readbuffersize', '0.5'),
            ('readbuffermaxglobalsize', '100'),
            ('readbufferexpiration', '60'),
            ('writebufferexpiration', '60'),
            ('readbuffermemorysaving', '1'),
            ('checkbuffersize', '0.25'),
            ('home', 'http://merapi.hd.free.fr/abc/'),
            ('multihomeipv4', ''),
            ('multihomeipv6', ''),
            ('scrollmiddlecentering', '0.5'),
            ('showprioritizemarker', '1'),
            ('exitwindow', '2'),
            ('minpeerconnectiontimeout', '5'),
            ('maxpeerconnectiontimeout', '15'),
            ('retryconnection', '1'),
            ('downvolwindowtype', '0'),
            ('upvolwindowtype', '0'),
            ('downvolrate', '0'),
            ('upvolrate', '0'),
            ('downvolstart', '0'),
            ('upvolstart', '0'),
            ('downvollast', '0'),
            ('upvollast', '0'),
            ('downvolmargin', '5.0'),
            ('upvolmargin', '5.0'),
            ('mindownrate', '5'),
            ('scrollpos0', '-1'),
            ('scrollpos1', '-1'),
            ('scrollpos2', '-1'),
            ('scrollpos3', '-1'),
            ('scrollpos4', '-1'),
            ('scrollpos5', '-1'),
            ('scrollpos6', '-1'),
            ('scrollpos7', '-1'),
            ('scrollpos8', '-1'),
            ('scrollpos9', '-1'),
            ('showpeermenu', '1'),
            ('colprivate', '255,255,255'),
            ('displaytrackcnxmsg', '1'),
            ('displaytrackcnxwng', '0'),
            ('inttrackwhencompleted', '0'),
            ('emptymsgindetails', '0'),
            ('altfilemanager', ''),
            ('autosavepeers', '1'),
            ('checkinttrack', '2'),
            ('resetdeftrackonstop', '0'),
            ('restrackctrl', '1'),
            ('restrackshift', '4'),
            ('restrackctrlshift', '2'),
            ('restrackdel', '3'),
            ('restrackins', '2'),
            ('mousewheelontab', '1'),
            ('checkinttrackautopublic', '1'),
            ('checkinttrackautodelay', '300'),
            ('digitheight', '-3'),
            ('buttonheight', '-3'),
            ('timedurationstyle', '0'),
            ('maxlastpeers', '10'),
            ('dhtsecurityextension', '1'),
            ('dhtip', ''),
            ('dhtmaxresprate', '5120'),
            ('resettimeoutsonstart', '0'),
            ('readbuffermaxglobalnumber', '150'),
            ('dhtmaxreqrate', '100'),
            ('nbpiecesfix', '0'),
            ('nbpiecesfixmaxdownrate', '500'),
            ('nbpiecesfixtrigger', '10000'),
            ('piecesizefixtrigger', '512'),
            ('popuploadeddelay', '3'),
            ('popuploaded', '1'),
            ('pl', '1'),
            ('plport', '56667'),
            ('upload_unit_size', '4200'),
            ('tornamevb', '-'),
            ('renfilew', '300'),
            ('findinfilenamesw', '300'),
            ('conlimiternamew', '300'),
            ('templatenamew', '300'),
            ('addtorrentlinkw', '300'),
            ('expirebuffersrate', '600'),
            ('expirebuffersage', '1800'),
            ('expirereadbufferssize', '524288'),
            ('expiregenbufferssize', '524288'),
            ('expirebuffersmsg', '0'),
            ('clearstatusbutton', '2'),
            ('unitsizestep', '0.03125'),
            ('unitsizeinctrig', '5'),
            ('checkpiecebeforeupload', '0'),
            ('skipcertifverif', '1'),
            ('strictprivate', '1'),
            ('countsbasseed', '1'),
            ('longpath', '1'),
            ('originalfilenameslist', '.originalfilenames.txt'),
            ('originalfilenameslistbom', '1'),
            ('incompletemark', '!'),
            ('skiplengthautofixonreset', '1'),
            ('kickerrperpiece', '5'),
            ('kickbadpieces', '5'),
            ('banerrperpiece', '10'),
            ('banbadpieces', '10'),
            ('downslicesize', '16384'),
            ('numsimseed', '10'),
            ('numsimseednodown', '15'),
            ('activitymaxforseed', '0.5'),
            ('urmonunlimiteduprate', '0'),
            ('nodownisinactive', '1'),
            ('progressongoing', '!'),
            ('progressongoing2', ''),
            ('selectedwidth', '25'),
            ('extratobestoppedfail', '1'),
            ('upwithoutdown', '0'),
            ('upwithoutdowntime', '1'),
            ('upwithoutdowntimemn', '0'),
            ('upwithoutdown2', '0'),
            ('upwithoutdownrate', '50'),
            ('abcproc', '1'),
            ('findselectfound', '1'),
            ('breakupseedbitfield', '0'),
            ('drm', '0'),
            ('drmdelay', '10'),
            ('drmdelaystop', '5'),
            ('drmlowpriority', '1'),
            ('drmmaxtorrent', '2'),
            ('drmonhold', '1'),
            ('drmonunlimiteddownrate', '0'),
            ('drmstartdelay', '180'),
            ('drmstoponpause', '0'),
            ('drmstopsman', '0'),
            ('drmthresholdstart', '1000.0'),
            ('drmthresholdstop', '300.0'),
            ('drmtorrentstartdelay', '60'),
            ('drmtorrentstopdelay', '30'),
            ('maxstdio', '-1'),
            ('lastrunversion', '-1'),
            ('rawserverslowdown', '0.01'),
            ('mincompforrequest', '0.75'),
            ('prevschedmode', '-1'),
            ('schedonlyonhold', '0'),
            ('advdetdbuf', '1'),
            ('drmtrigger', '0'),
            ('drmtriggerval', '0'),
            ('urmtrigger', '0'),
            ('urmtriggerval', '0'),
            ('abcrentorappendnb', '0'),
            ('rawserversockrcvbuf', '524288'),
            ('rawserversockrcvbufs', '65536'),
            ('rawserversocksndbuf', '524288'),
            ('priofactup', '2.0'),
            ('priofactdn', '2.0'),
            ('scrolldelay', '0.2'),
            ('findtorrentnamesim', '1'),
            ('findtorrentfilenamesim', '1'),
            ('invertxb', '0'),
            ('maxpeerqueue', '50')
            ]

        # First use of nbpiecesfix : set if nbpiecesfixmaxdownrate was not null
        if not self.abcparams.has_key('nbpiecesfix') and self.abcparams.has_key('nbpiecesfixmaxdownrate'):
            if self.abcparams['nbpiecesfixmaxdownrate'] == '0':
                self.abcparams['nbpiecesfix'] = '0'
            else:
                self.abcparams['nbpiecesfix'] = '1'

        for t in abcparamsdefaults:
            if not self.abcparams.has_key(t[0]):
                self.abcparams[t[0]] = t[1]

        # Init torrent advanced details window columns width
        for colinfo in self.advdetcolinfo:
            if not self.abcparams.has_key(colinfo[1]):
                self.abcparams[colinfo[1]] = str(colinfo[3])

        # Limit min_peers and max_connections to 255
        if int(self.abcparams['min_peers']) > 255:
            self.abcparams['min_peers'] = '255'
        if int(self.abcparams['max_connections']) > 255:
            self.abcparams['max_connections'] = '255'

        # Old format for allocation type
        oldalloctype = ['normal', 'background', 'pre-allocate', 'sparse']
        if not self.abcparams['alloc_type'].isdigit():
            self.abcparams['alloc_type'] = str(oldalloctype.index(self.abcparams['alloc_type']))

        # Old format for move folder option
        if not self.abcparams.has_key('movedatastatus'):
            if self.abcparams.has_key('movefoldercompleted') \
               and self.abcparams.has_key('movefolderfinished'):
                if self.abcparams['movefoldercompleted'] == "2":
                    self.abcparams['movedataonstatus'] = "1"
                    self.abcparams['movedatastatus'] = "0"
                elif self.abcparams['movefolderfinished'] == "2":
                    self.abcparams['movedataonstatus'] = "1"
                    self.abcparams['movedatastatus'] = "1"
                else:
                    self.abcparams['movedataonstatus'] = "0"
                    self.abcparams['movedatastatus'] = "1"
                if self.abcparams['movefoldercompleted'] == "1" or self.abcparams['movefolderfinished'] == "1":
                    self.abcparams['movedataondelete'] = "1"
                else:
                    self.abcparams['movedataondelete'] = "0"
                del self.abcparams['movefoldercompleted']
                del self.abcparams['movefolderfinished']
            else:
                self.abcparams['movedataonstatus'] = "0"
                self.abcparams['movedatastatus'] = "1"
                self.abcparams['movedataondelete'] = "0"
        else:
            if self.abcparams.has_key('movefoldercompleted'):
                del self.abcparams['movefoldercompleted']
            if self.abcparams.has_key('movefolderfinished'):
                del self.abcparams['movefolderfinished']

        # Old format for URM deadband
        if self.abcparams.has_key('urmupthreshold'):
            self.abcparams['urmthresholdstart'] = self.abcparams['urmupthreshold']
            del self.abcparams['urmupthreshold']

        # Old format for extratobestopped
        extratobestopped = int(self.abcparams['extratobestopped'])
        if extratobestopped >= 10:
            extratobestopped %= 10
            self.abcparams['extratobestopped'] = str(extratobestopped)

        # Old unused keys
        if self.abcparams.has_key('bdavth1'):
            del self.abcparams['bdavth1']
        if self.abcparams.has_key('bdgreedyth'):
            del self.abcparams['bdgreedyth']
        if self.abcparams.has_key('dhtaddpeerdlgx'):
            del self.abcparams['dhtaddpeerdlgx']
        if self.abcparams.has_key('dhtaddpeerdlgy'):
            del self.abcparams['dhtaddpeerdlgy']
        if self.abcparams.has_key('downvolcheckcycle'):
            del self.abcparams['downvolcheckcycle']
        if self.abcparams.has_key('upvolcheckcycle'):
            del self.abcparams['upvolcheckcycle']
        if self.abcparams.has_key('readbufferbase'):
            del self.abcparams['readbufferbase']
        if self.abcparams.has_key('trackautoswitch'):
            del self.abcparams['trackautoswitch']
        if self.abcparams.has_key('maxmetadatarequest'):
            del self.abcparams['maxmetadatarequest']
        if self.abcparams.has_key('maxrateperiod'):
            del self.abcparams['maxrateperiod']
        if self.abcparams.has_key('meandynmaxuprate'):
            del self.abcparams['meandynmaxuprate']
        if self.abcparams.has_key('meantotalrates'):
            del self.abcparams['meantotalrates']
        if self.abcparams.has_key('renfilewidth'):
            del self.abcparams['renfilewidth']
        if self.abcparams.has_key('dontcheckseedactivity'):
            del self.abcparams['dontcheckseedactivity']
        if self.abcparams.has_key('maxupload'):
            del self.abcparams['maxupload']
        if self.abcparams.has_key('searchpanel'):
            del self.abcparams['searchpanel']
        if self.abcparams.has_key('homepage'):
            del self.abcparams['homepage']
        if self.abcparams.has_key('starthome'):
            del self.abcparams['starthome']
        if self.abcparams.has_key('rawserverslowdown0'):
            del self.abcparams['rawserverslowdown0']
        if self.abcparams.has_key('rawserverslowdown1'):
            del self.abcparams['rawserverslowdown1']
        if self.abcparams.has_key('rawserverslowdown2'):
            del self.abcparams['rawserverslowdown2']
        if self.abcparams.has_key('rawserverslowdownstep'):
            del self.abcparams['rawserverslowdownstep']
        if self.abcparams.has_key('showactivitymax'):
            del self.abcparams['showactivitymax']
        if self.abcparams.has_key('uploadmessagequeue'):
            del self.abcparams['uploadmessagequeue']
        if self.abcparams.has_key('drmstopsonlyactive'):
            del self.abcparams['drmstopsonlyactive']
        if self.abcparams.has_key('drmstopsonlylowprio'):
            del self.abcparams['drmstopsonlylowprio']
        if self.abcparams.has_key('urmstopsonlyactive'):
            del self.abcparams['urmstopsonlyactive']
        if self.abcparams.has_key('urmstopsonlylowprio'):
            del self.abcparams['urmstopsonlylowprio']
        if self.abcparams.has_key('expirebufferssize'):
            del self.abcparams['expirebufferssize']

        if self.abcparams.has_key('bdcalcdownth3'):
            del self.abcparams['bdcalcdownth3']
        if self.abcparams.has_key('bdcalcdownth4'):
            del self.abcparams['bdcalcdownth4']
        if self.abcparams.has_key('bdcalcupth3'):
            del self.abcparams['bdcalcupth3']
        if self.abcparams.has_key('bdcalcupth4'):
            del self.abcparams['bdcalcupth4']
        if self.abcparams.has_key('bdminstepdownrate'):
            del self.abcparams['bdminstepdownrate']
        if self.abcparams.has_key('bdminstepuprate'):
            del self.abcparams['bdminstepuprate']
        if self.abcparams.has_key('bdstep1'):
            del self.abcparams['bdstep1']
        if self.abcparams.has_key('bdstep2'):
            del self.abcparams['bdstep2']
        if self.abcparams.has_key('bdavth2'):
            del self.abcparams['bdavth2']

        # No more "Keep duplicate destination" in Preferences/Misc.
        if self.abcparams['askduplicatedest'] == 2:
            self.abcparams['askduplicatedest'] = 0

        # setextdefaultfolder setting for using default download folder from external source
        if not self.abcparams.has_key('setextdefaultfolder'):
            self.abcparams['setextdefaultfolder'] = self.abcparams['setdefaultfolder']

        # max_connections may no longer be null
        if int(self.abcparams['max_connections']) == 0:
            self.abcparams['max_connections'] = '15'

        # Web service parameters
        self.webparams = {}

        # Read web service config file
        tcfm = TorrentConfigFileManager(path.join(self.datapath, "webservice.conf"), self.webparams)
        tcfm.readAllConfig()

        webparamsdefaults = [
            ('allow_onhold', '1'),
            ('allow_getparam', '1'),
            ('allow_setparam', '1'),
            ('webcoder', '0'),
            ('allow_move', '1'),
            ('allow_gettorparam', '1'),
            ('allow_settorparam', '1'),
            ('allow_superseed', '1'),
            ('allow_scheduler', '1')
            ]

        for parameter, value in webparamsdefaults:
            if not self.webparams.has_key(parameter):
                self.webparams[parameter] = value

        # Old format for web service IP
        if self.webparams['webIP'] == "Automatic" or self.webparams['webIP'] == "Automatics":
            self.webparams['webIP'] = '0'
        elif self.webparams['webIP'] == "Loop Back":
            self.webparams['webIP'] = '1'

        # Old format for web service encoding
        if self.webparams['webcoder'] not in ('0', '1'):
            if self.webparams['webcoder'] == 'utf_8':
                self.webparams['webcoder'] = '0'
            else:
                self.webparams['webcoder'] = '1'

        # Command scheduler parameters
        self.schedparams = {}

        # Read command scheduler config file
        tcfm = TorrentConfigFileManager(path.join(self.datapath, "scheduler.conf"), self.schedparams)
        tcfm.readAllConfig()

        schedparamsdefaults = [
            ('schedrunpast', '0'),
            ('schedautostart', '0')
            ]

        for parameter, value in schedparamsdefaults:
            if not self.schedparams.has_key(parameter):
                self.schedparams[parameter] = value

        # Scanner parameters
        self.scanparams = {}

        # Read scanner config file
        tcfm = TorrentConfigFileManager(path.join(self.datapath, "scanner.conf"), self.scanparams)
        tcfm.readAllConfig()

        scanparamsdefaults = [
            ('scandir', ''),
            ('scanfreqh', '1'),
            ('scanfreqm', '0'),
            ('scanstartpos', '1'),
            ('scanprio', '2'),
            ('scanstatus', '1'),
            ('scanmovetor', '0'),
            ('scanautostart', '0')
            ]

        for parameter, value in scanparamsdefaults:
            if not self.scanparams.has_key(parameter):
                self.scanparams[parameter] = value

        # Existing parameters to be set at first run of current version
        # according to last version run
        # (param, value, current version)
        tobesetatfirstrun = [
                            ('maxstdio', '-1', 614),
                            ('expirebuffersrate', '600', 616),
                            ('writebuffersize', '4.0', 616),
                            ('mincompforrequest', '0.0', 617),
                            ('readbuffersize', '0.5', 623),
                            ('readbufferexpiration', '60', 624),
                            ('writebufferexpiration', '60', 624),
                            ('rawserverslowdown', '0.01', 625),
                            ('cyclicalgcrate', '900', 628),
                            ('gcth1', '700', 629),
                            ('gcth2', '10', 629),
                            ('gcth3', '10', 629),
                            ('expirereadbufferssize', '524288', 631),
                            ('max_files_open', '75', 635),
                            ('tweakbandwidthdistribution', '0', 640),
                            ('bdcalcdownth1', '2.0', 640),
                            ('bdcalcdownth2', '2.0', 640),
                            ('bdcalcupth1', '2.0', 640),
                            ('bdcalcupth2', '2.0', 640),
                            ('metadatatimeout', '5', 651),
                            ('readbuffermaxglobalnumber', '150', 651),
                            ('checkinttrackautodelay', '300', 655),
                            ('rawserversockrcvbuf', '524288', 660),
                            ('rawserversocksndbuf', '524288', 661)
                            ]

        lastrunversion = int(self.abcparams['lastrunversion'])
        for parameter, value, version in tobesetatfirstrun:
            if lastrunversion < version:
                self.abcparams[parameter] = value
        self.abcparams['lastrunversion'] = ABC_OKC_BUILD

    def initDefaultDynMaxUpRate(self):
        self.tweakdynmaxuprate = 0
        self.urmupfromdownA = 0.027
        self.urmupfromdownB = 0.16
        self.urmupfromdownC = 0.08
        self.urmupfromdownD = 0.04
        self.urmupfromdownE = 0.
        self.urmupfromdownF = 0.1
        self.urmupfromdownG = 140.8

    def initDefaultBandwidthDistribution(self):
        self.tweakbandwidthdistribution = 0
        self.bdcalcdownth1 = 2.0
        self.bdcalcdownth2 = 2.0
        self.bdcalcupth1 = 2.0
        self.bdcalcupth2 = 2.0
        
    def initDefaultCyclicalTasksRate(self):
        self.cyclicaltasksrate = 2.0

    def initDefaultUnitSize(self):
        self.unitsizestep = .03125
        self.unitsizeinctrig = 5

    def initPathNameMaxLength(self, longpathoption):
        if longpathoption < 2:
            self.filenamemaxlength = 255
            self.filepathmaxlength = 259
            self.pathmaxlength = 247
        else:
            self.filenamemaxlength = 32767
            self.filepathmaxlength = 32767
            self.pathmaxlength = 32767

    def time_value(self, n, longformat = True, seconds = True, maxvalue = True):
        if n == -1:
            return '?'
        if maxvalue and n >= 8640000:
            return '>100' + self.lang.get('l_day')

        if longformat:
            day, r1 = divmod(n, 86400)
            hour, r2 = divmod(r1, 3600)
            if day:
                return ('%02d' % day) + self.localize('l_day') + ':' + ('%02d' % hour) + self.localize('l_hour')
            min, sec = divmod(r2, 60)
            if hour or not seconds:
                return ('%02d' % hour) + self.localize('l_hour') + ':' + ('%02d' % min) + self.localize('l_minute')
            return ('%02d' % min) + self.localize('l_minute') + ':' + ('%02d' % sec) + self.localize('l_second')

        hour, r2 = divmod(n, 3600)
        min, sec = divmod(r2, 60)
        return ('%02d' % hour) + self.localize('l_hour') + ':' + ('%02d' % min) + self.localize('l_minute')

    def eta_value(self, n, longformat = True, seconds = True, maxvalue = True):
        # to display countdowns
        if n == -1:
            return '?'
        if maxvalue and n >= 8640000:
            return '>100' + self.lang.get('l_day')

        if longformat:
            day, r1 = divmod(ceil(n), 86400)
            hour, r2 = divmod(r1, 3600)
            min, sec = divmod(r2, 60)
            if day:
                if sec or min:
                    # If seconds or minutes are not displayed and hours are displayed
                    if hour == 23:
                        hour = 0
                        day += 1
                    else:
                        hour += 1
            elif hour or not seconds:
                if sec:
                    # If seconds are not displayed and minutes are displayed
                    if min == 59:
                        min = 0
                        if hour == 23:
                            hour = 0
                            day += 1
                        else:
                            hour += 1
                    else:
                        min += 1
            if day:
                return ('%02d' % day) + self.localize('l_day') + ':' + ('%02d' % hour) + self.localize('l_hour')
            if hour or not seconds:
                return ('%02d' % hour) + self.localize('l_hour') + ':' + ('%02d' % min) + self.localize('l_minute')
            return ('%02d' % min) + self.localize('l_minute') + ':' + ('%02d' % ceil(sec)) + self.localize('l_second')

        hour, r2 = divmod(ceil(n), 3600)
        min, sec = divmod(r2, 60)
        if sec:
            if min == 59:
                min = 0
                hour += 1
            else:
                min += 1
        return ('%02d' % hour) + self.localize('l_hour') + ':' + ('%02d' % min) + self.localize('l_minute')

    def onWriteFilteredText(self, event):
        # Filter out the characters '|' and '\x01' from inputs because these are used as separators in the ABC conf files
        keycode = event.GetKeyCode()
        if keycode != 124 and keycode != 1:
            if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
               or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3 \
               or keycode == 22:
                # Keycode 22 (Ctrl-V) is added here because a text field may be filtered
                # also through EVT_TEXT_PASTE by a call to onPasteText
                event.Skip()
                return False
            event.Skip()
            return True
        return False

    def onWriteFilteredText2(self, event):
        # Filter out the characters '\x01' and '\x02' from inputs
        keycode = event.GetKeyCode()
        if keycode != 1 and keycode != 2:
            if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
               or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3 \
               or keycode == 22:
                # Keycode 22 (Ctrl-V) is added here because a text field may be filtered
                # also through EVT_TEXT_PASTE by a call to onPasteText2
                event.Skip()
                return False
            event.Skip()
            return True
        return False

    def onPasteText(self, event):
        # Filter out the characters '|' and '\x01' when pasting from the clipboard
        # to a TextCtrl or a ComboBox
        if not wx.TheClipboard.IsOpened():
            wx.TheClipboard.Open()
            tdo = wx.TextDataObject()
            if wx.TheClipboard.GetData(tdo):
                filteredtext = tdo.GetText().translate({124: None, 1: None})
                control = event.GetEventObject()
                if type(control) is wx.TextCtrl:
                    selectedfrom, selectedto = control.GetSelection()
                    control.Replace(selectedfrom, selectedto, filteredtext)
                else:
                    # For ComboBox
                    selectedfrom, selectedto = control.GetMark()
                    # workaround for ComboBox.Replace which is buggy (tries to access the clipboard ??)
                    text = control.GetValue()
                    control.SetValue(''.join([text[:selectedfrom], filteredtext, text[selectedto:]]))
                    control.SetInsertionPoint(selectedfrom + len(filteredtext))
            wx.TheClipboard.Close()
        ###########################################
        # Works for all but heavier
        # if not wx.TheClipboard.IsOpened():
            # wx.TheClipboard.Open()
            # tdo = wx.TextDataObject()
            # if wx.TheClipboard.GetData(tdo):
                # wx.TheClipboard.SetData(wx.TextDataObject(tdo.GetText().translate({124: None, 1: None})))
                # control = event.GetEventObject()
                # control.Unbind(wx.EVT_TEXT_PASTE)
                # control.Paste()
                # control.Bind(wx.EVT_TEXT_PASTE, self.onPasteText)
                # wx.TheClipboard.SetData(tdo)
            # wx.TheClipboard.Close()

    def onPasteText2(self, event):
        # Filter out the characters '\x01'  and '\x02' when pasting from the clipboard
        # to a TextCtrl or a ComboBox
        if not wx.TheClipboard.IsOpened():
            wx.TheClipboard.Open()
            tdo = wx.TextDataObject()
            if wx.TheClipboard.GetData(tdo):
                filteredtext = tdo.GetText().translate({1: None, 2: None})
                control = event.GetEventObject()
                if type(control) is wx.TextCtrl:
                    selectedfrom, selectedto = control.GetSelection()
                    control.Replace(selectedfrom, selectedto, filteredtext)
                else:
                    # For ComboBox
                    selectedfrom, selectedto = control.GetMark()
                    # workaround for ComboBox.Replace which is buggy (tries to access the clipboard ??)
                    text = control.GetValue()
                    control.SetValue(''.join([text[:selectedfrom], filteredtext, text[selectedto:]]))
                    control.SetInsertionPoint(selectedfrom + len(filteredtext))
            wx.TheClipboard.Close()

    def onWriteDigits(self, event):
        # Filter out not digit chars
        keycode = event.GetKeyCode()
        if 47 < keycode < 58 or keycode == wx.WXK_DELETE or keycode == wx.WXK_BACK \
           or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def onWriteDigitsDot(self, event):
        # Filter out not digit or dot chars
        keycode = event.GetKeyCode()
        if keycode == 46 or 47 < keycode < 58 or keycode == wx.WXK_DELETE \
           or keycode == wx.WXK_BACK or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def onWriteDigitsMinus(self, event):
        # Filter out not digit or dot chars
        keycode = event.GetKeyCode()
        if keycode == 45 or 47 < keycode < 58 or keycode == wx.WXK_DELETE \
           or keycode == wx.WXK_BACK or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def onWriteDigitsDotMinus(self, event):
        # Filter out not digit or dot or minus chars
        keycode = event.GetKeyCode()
        if keycode == 45 or keycode == 46 or 47 < keycode < 58 or keycode == wx.WXK_DELETE \
            or keycode == wx.WXK_BACK or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def onWritePrintable(self, event):
        # Filter out chars > 128 and not printable chars
        keycode = event.GetKeyCode()
        if 31 < keycode < 127 or keycode == wx.WXK_DELETE or keycode == wx.WXK_BACK \
           or keycode == 22 or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def getPathSize(self, pth):
        if path.isdir(pth):
            size = 0
            for root, dirs, files in walk(pth):
                size += sum(path.getsize(path.join(root, name)) for name in files)
        else:
            size = path.getsize(pth)
        return size

    def sortList(self, item1, item2):
        # Generic function to sort listctrl
        if self.sortcol[item1] < self.sortcol[item2]:
            return -1
        if self.sortcol[item1] > self.sortcol[item2]:
            return 1
        return 0

    def sortListReverse(self, item1, item2):
        # Generic function to sort listctrl in reverse order
        if self.sortcol[item1] < self.sortcol[item2]:
            return 1
        if self.sortcol[item1] > self.sortcol[item2]:
            return -1
        return 0

    def fixWindowsName(self, str, unit = False):
        # Check if str is a valid Windows file name (or unit name if unit is true)
        # If not, returns a fixed name
        strlen = len(str)
        if unit and (strlen != 2 or str[1] != ':'):
            return 'c:'
        if not str or str == '.' or str == '..':
            return '_'
        if unit:
            str = str[0]
            strlen = 1
        if strlen > self.filenamemaxlength:
            str = str[:self.filenamemaxlength]
            fixed = True
            strlen = self.filenamemaxlength
        else:
            fixed = False
        fixedname = ''
        i = 0
        while i < strlen:
            c = str[i]
            if c in self.invalidwinfilenamechar:
                fixedname += '_'
                fixed = True
                i += 1
            elif c == ' ' or c == '.':
                spaceordot = c
                i += 1
                while i < strlen:
                    c = str[i]
                    if c != ' ' and c != '.':
                        break
                    spaceordot += c
                    i += 1
                if i == strlen:
                    # Name ends with spaces and dots
                    fixedname += len(spaceordot) * '_'
                    fixed = True
                else:
                    fixedname += spaceordot
            else:
                fixedname += c
                i += 1
        if fixed:
            return fixedname
        return ''

    def checkWinPath(self, parent, pathtocheck, checklength = True):
        if pathtocheck and pathtocheck[-1] == '\\' and pathtocheck != '\\\\':
            pathitems = pathtocheck[:-1].split('\\')
        else:
            pathitems = pathtocheck.split('\\')
        nexttotest = 1
        if self.isPathRelative(pathtocheck):
            # Relative path
            if checklength and len(path.join(self.datapath, pathtocheck)) > self.filepathmaxlength:
                if parent is not None:
                    dlg = wx.MessageDialog(parent, '"' + pathtocheck + '"\n' + self.localize('errortoolongwinpath'),
                                           self.localize('abcokcerror'), wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                return False
            # Empty relative path is allowed
            if pathtocheck == '':
                return True
            fixedname = self.fixWindowsName(pathitems[0])
            if fixedname:
                if parent is not None:
                    dlg = wx.MessageDialog(parent, '"' + pathitems[0] + '"\n' + self.localize('errorinvalidwinname') + fixedname,
                                           self.localize('abcokcerror'), wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                return False
        else:
            # Absolute path
            if checklength and len(pathtocheck) > self.filepathmaxlength:
                if parent is not None:
                    dlg = wx.MessageDialog(parent, '"' + pathtocheck + '"\n' + self.localize('errortoolongwinpath'),
                                           self.localize('abcokcerror'), wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                return False
            # An absolute path must have at least one '\'
            if not '\\' in pathtocheck:
                if parent is not None:
                    dlg = wx.MessageDialog(parent, '"' + pathitems[0] + '"\n' + self.localize('errorinvalidpath'),
                                           self.localize('abcokcerror'), wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                return False
            if pathtocheck[:2] != '\\\\':
                # Not a network path
                fixedname = self.fixWindowsName(pathitems[0], unit = True)
                if fixedname:
                    if parent is not None:
                        dlg = wx.MessageDialog(parent, '"' + pathitems[0] + '"\n' + self.localize('errorinvalidwinunitname') + fixedname,
                                               self.localize('abcokcerror'), wx.ICON_ERROR)
                        dlg.ShowModal()
                        dlg.Destroy()
                    return False
            else:
                # Network path
                nexttotest = 2

        for name in pathitems[nexttotest:]:
            fixedname = self.fixWindowsName(name)
            if fixedname:
                if parent is not None:
                    dlg = wx.MessageDialog(parent, '"' + name + '"\n' + self.localize('errorinvalidwinname') + fixedname,
                                           self.localize('abcokcerror'), wx.ICON_ERROR)
                    dlg.ShowModal()
                    dlg.Destroy()
                return False

        return True

    def frameWindow(self, win, xl = 0, xr = 0, xt = 0, xb = 0, border = 15, midline = 0):
        parent = win.GetParent()
        winpos = win.GetPositionTuple()
        winsize = win.GetSizeTuple()
        xframe = winpos[0] - xl - 2
        yframe = winpos[1] - xt - 3
        wframe = winsize[0] + xl + xr + 5
        hframe = winsize[1] + xt + xb + 6
        if border & 1:
            wx.StaticLine(parent, -1, (xframe, yframe), (-1, hframe), wx.LI_VERTICAL)
        if border & 2:
            wx.StaticLine(parent, -1, (xframe + 1, yframe), (wframe - 1, -1), wx.LI_HORIZONTAL)
        if border & 4:
            wx.StaticLine(parent, -1, (xframe, yframe + hframe - 2), (wframe, -1), wx.LI_HORIZONTAL)
        if border & 8:
            wx.StaticLine(parent, -1, (xframe + wframe - 2, yframe), (-1, hframe), wx.LI_VERTICAL)
        if midline & 1:
            wx.StaticLine(parent, -1, (xframe + 1, yframe + hframe / 2 - 1), (xl, -1), wx.LI_HORIZONTAL)
        if midline & 2:
            wx.StaticLine(parent, -1, (xframe + wframe / 2, yframe - xb), (-1, xb), wx.LI_VERTICAL)
        if midline & 4:
            wx.StaticLine(parent, -1, (xframe + wframe / 2, yframe + hframe - 2), (-1, xt), wx.LI_VERTICAL)
        if midline & 8:
            wx.StaticLine(parent, -1, (xframe + wframe - 2 - xr, yframe + hframe / 2 - 1), (xr, -1), wx.LI_HORIZONTAL)

    def isPathRelative(self, pth):
        if pth and pth[0] == '\\' or len(pth) > 1 and pth[1] == ':':
            return False
        return True

    def completePath(self, pth):
        if not pth:
            return self.datapath
        if self.isPathRelative(pth):
            return path.join(self.datapath, pth)
        return pth

    def findUniqueFileName(self, dest, src = '', extraext = True):
        # If path dest exists or is too long, returns a new path with '_' and 3 digits added
        # before file extension and after truncated file name
        # If it can't be renamed this way, asks user for a new name
        # Returns None if a new name could not be set
        # If src is not empty ('web', 'scan', 'rss', 'qcl', 'queue', 'dum'), user is not asked if automatic renaming fails
        # Max file path length : self.filepathmaxlength
        # Max file name length : self.filenamemaxlength

        pathname, filename = path.split(dest)
        if len(dest) <= self.filepathmaxlength \
           and len(filename) <= self.filenamemaxlength \
           and not path.exists(dest):
            return dest

        extraantislash = pathname.endswith('\\') is False
        dot = filename.rfind('.')
        if dot == -1:
            shortfilename = filename
            ext = ''
        else:
            shortfilename = filename[:dot]
            ext = filename[dot:]

        # if extraext :
            # cut = min(251 - len(pathname) - extraantislash - len(ext), 247 - len(ext))
            # # 251 = 259 - <len(_123) = 4> - <len(extra extension .prs', '.pro',...) = 4>
            # # 247 = 255 - <len(_123) = 4> - <len(extra extension .prs', '.pro',...) = 4>
        # else:
            # cut = min(255 - len(pathname) - extraantislash - len(ext), 251 - len(ext))
            # # 255 = 259 - <len(_123) = 4>
            # # 251 = 255 - <len(_123) = 4>
        # All above can be simplified with :
        # cut = 251 - len(ext) - extraext * 4 + min(4 - len(pathname) - extraantislash, 0)
        cut = min(self.filepathmaxlength - len(pathname) - extraantislash, self.filenamemaxlength) - 4 - len(ext) - extraext * 4

        if cut >= 0:
            cutshortfilename = shortfilename[:cut]
            # Renaming will add '_' + 3 digit to the original name
            i = 1
            while i < 1000:
                newdest = path.join(pathname, cutshortfilename + ('_%03u' % i) + ext)
                if not path.exists(newdest):
                    break
                i += 1
            if i != 1000:
                return newdest

        if src:
            return None

        while True:
            # Impossible to find an automatic new name ; ask user for one
            dl = wx.FileDialog(self.window, self.localize('chooseuniquefilename'),\
                               pathname, filename, '*.*', wx.SAVE)
            result = dl.ShowModal()
            if result != wx.ID_OK:
                dl.Destroy()
                return None
            newdest = dl.GetPath()
            dl.Destroy()
            if not path.exists(newdest):
                return newdest

    def makeBitmap(self, bitmap, trans_color = wx.Colour(200, 200, 200)):
        button_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmap), wx.BITMAP_TYPE_BMP)
        button_mask = wx.Mask(button_bmp, trans_color)
        button_bmp.SetMask(button_mask)
        return button_bmp

    def makeBitmapButton(self, panel, bitmap, tooltip, event, trans_color = wx.Colour(200, 200, 200),
                         toggle = False, bitmapselected = None, bitmapselectedflat = None,
                         bitmapdetected = None, bitmapdetectedflat = None, flat = False):
        button_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmap), wx.BITMAP_TYPE_BMP)
        button_bmp.SetMask(wx.Mask(button_bmp, trans_color))
        if bitmap == 'modemanual.bmp':
            self.modebtnmanual = button_bmp
        if bitmapselected:
            buttonselected_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmapselected), wx.BITMAP_TYPE_BMP)
            buttonselected_bmp.SetMask(wx.Mask(buttonselected_bmp, trans_color))
        if bitmapselectedflat:
            buttonselectedflat_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmapselectedflat), wx.BITMAP_TYPE_BMP)
            buttonselectedflat_bmp.SetMask(wx.Mask(buttonselectedflat_bmp, trans_color))
        if bitmapdetected:
            buttondetected_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmapdetected), wx.BITMAP_TYPE_BMP)
            buttondetected_bmp.SetMask(wx.Mask(buttondetected_bmp, trans_color))
        if bitmapdetectedflat:
            buttondetectedflat_bmp = wx.Bitmap(path.join(self.abcpath, 'icons', bitmapdetectedflat), wx.BITMAP_TYPE_BMP)
            buttondetectedflat_bmp.SetMask(wx.Mask(buttondetectedflat_bmp, trans_color))
        ID_BUTTON = wx.NewId()
        btnsize = wx.Size(button_bmp.GetWidth() + 12, button_bmp.GetHeight() + 4)
        if toggle:
            button_btn = GenBitmapToggleButtonDual(panel, ID_BUTTON, button_bmp, size = btnsize, style = wx.SIMPLE_BORDER)
            if bitmapselected:
                button_btn.setBitmapSelectedNotFlat(buttonselected_bmp)
            else:
                button_btn.setBitmapSelectedNotFlat(button_bmp)
            if bitmapselectedflat:
                button_btn.setBitmapSelectedFlat(buttonselectedflat_bmp)
            else:
                button_btn.setBitmapSelectedFlat(button_bmp)
            if bitmapdetected:
                button_btn.setBitmapDetectedNotFlat(buttondetected_bmp)
            else:
                button_btn.setBitmapDetectedNotFlat(button_bmp)
            if bitmapdetectedflat:
                button_btn.setBitmapDetectedFlat(buttondetectedflat_bmp)
            else:
                button_btn.setBitmapDetectedFlat(button_bmp)
        else:
            button_btn = GenBitmapButtonDual(panel, ID_BUTTON, button_bmp, size = btnsize, style = wx.SIMPLE_BORDER)
            if bitmapselected:
                button_btn.SetBitmapSelected(buttonselected_bmp)
            else:
                button_btn.SetBitmapSelected(button_bmp)
        button_btn.SetUseFocusIndicator(False)
        button_btn.setFlat(flat)

        button_btn.SetToolTipString(tooltip)
        panel.Bind(wx.EVT_BUTTON, event, button_btn)
        return button_btn

    def shutDown(self):
        winver = sys.getwindowsversion()[3]
        if winver == VER_PLATFORM_WIN32_NT:
            shutdownthread = Thread(target = self.shutDown2k)
            shutdownthread.daemon = True
            shutdownthread.start()
        elif winver == VER_PLATFORM_WIN32_WINDOWS:
            system('rundll32 shell32.dll,SHExitWindowsEx 9')

    def shutDown2k(self):
        AdjustTokenPrivileges(OpenProcessToken(GetCurrentProcess(), TOKEN_ALL_ACCESS), False,
                              [(LookupPrivilegeValue(None, SE_SHUTDOWN_NAME), SE_PRIVILEGE_ENABLED)])
        ExitWindowsEx(9)

    def addToolbarIcon(self, parent, toolbar, event, short, long, bitmap1, bitmap2 = None, type = 'simple', firsttime = False):
        bmp1 = self.makeBitmap(bitmap1)
        if bitmap2 is not None:
            bmp2 = self.makeBitmap(bitmap2)
        if firsttime:
            # Find size of images so it will be dynamics
            toolbar.SetToolBitmapSize(wx.Size(bmp1.GetWidth(), bmp1.GetHeight()))
        newid = wx.NewId()
        parent.Bind(wx.EVT_TOOL, event, id = newid)
        if type == 'simple':
            toolbar.AddSimpleTool(newid, bmp1, short, long)
        elif type == 'toggle':
            toolbar.AddCheckTool(newid, bmp1, bmp2, short, long)
        elif type == 'radio':
            toolbar.AddRadioTool(newid, bmp1, wx.NullBitmap, short, long)
        return newid

    def countPathItems(self, filepath):
        # Count the number of items in a path (not ending with a path separator)
        item = filepath
        nbitem = 0
        while item:
            item = path.split(item)[0]
            nbitem += 1
        return nbitem

    def sizeFormat(self, s):
        if s < 1024:
            return '%i ' % s + self.b
        if s < 1048576:
            return '%.2f ' % (s / 1024.) + self.kb
        if s < 1073741824L:
            return '%.2f ' % (s / 1048576.) + self.mb
        if s < 1099511627776L:
            return '%.2f ' % (s / 1073741824.) + self.gb
        return '%.2f ' % (s / 1099511627776.) + self.tb

    def formatedSize(self, size):
        if self.abcparams['sizeinmb'] == '1':
            if self.abcparams['displayunitsize'] == '1':
                return (('%.2f ' % (size / 1048576.)) + self.mb, 1)
            return (('%.2f' % (size / 1048576.)), 0)
        return (self.sizeFormat(size), 1)

    def sizeWithUnit(self, formatedsize):
        if formatedsize[1]:
            return formatedsize[0]
        return formatedsize[0] + ' ' + self.mb

    def commaFormat(self, s):
        # Format the string s of an integer by grouping digits 3 by 3 separated by a space from the right to the left
        r = str(s)
        for i in xrange(len(r) - 3, 0, -3):
            r = r[:i] + ' ' + r[i:]
        return r

    def formatedRate(self, rate, prioritize = None):
        if self.showprioritizemarker and prioritize:
            pm = self.prioritizemarker
        else:
            pm = ''
        if rate is None:
            return (pm, 2)
        if self.abcparams['displayunitrate'] == '1':
            return (pm + ('%.1f ' % rate) + self.kbps, 1)
        return (pm + ('%.1f' % rate), 0)

    def rateWithUnit(self, formatedrate, prioritize = None):
        if formatedrate:
            if formatedrate[1] == 0:
                return formatedrate[0] + ' ' + self.kbps
            if formatedrate[1] == 1:
                return formatedrate[0]
        if self.showprioritizemarker and prioritize:
            return self.prioritizemarker + '0.0 ' + self.kbps
        return '0.0 ' + self.kbps

    def encodeWebServiceData(self, data):
        if self.webparams['webcoder'] == '0':
            return data.encode('utf_8')
        encoded = ''
        for c in data:
            try:
                encoded += c.encode(sysencoding)
            except:
                rs = []
                q = ord(c)
                while True:
                    q, r = divmod(q, 256)
                    rs.insert(0, chr(r))
                    if q == 0:
                        break
                encoded += ''.join(rs)
        return encoded

    def getListClickedColumn(self, list, x):
        # Returns the clicked column number of the list control from the x coordinate of a point inside the list
        currentwidth = -list.GetScrollPos(wx.HORIZONTAL)
        for col in xrange(list.GetColumnCount()):
            currentwidth += list.GetColumnWidth(col)
            if x < currentwidth:
                return col
        return -1

    def getPosCenteredOnWin(self, sizetobecentered, window, aligntop = False):
        # Returns position for a size to be centered on a window, in the extent of the screen display
        x, y, w, h = window.GetRect()
        wtobecentered, htobecentered = sizetobecentered
        wmax = 0
        for d in xrange(wx.Display.GetCount()):
            wmax += wx.Display(d).GetClientArea().GetSize()[0]
        hmax = wx.Display(max(wx.Display.GetFromWindow(window), 0)).GetClientArea().GetSize()[1]
        xcentered = x + int((w - wtobecentered) / 2.)
        if aligntop:
            ycentered = y
        else:
            ycentered = y + int((h - htobecentered) / 2.)
        constraint = False
        if xcentered < 0:
            xcentered = 0
            constraint = True
        elif xcentered > wmax - wtobecentered:
            xcentered = wmax - wtobecentered
            constraint = True
        if ycentered < 0:
            ycentered = 0
            constraint = True
        elif ycentered > hmax - htobecentered:
            ycentered = hmax - htobecentered
            constraint = True
        return (xcentered, ycentered), constraint

    def initConfigFiles(self):
        torrentpath = path.join(self.datapath, 'torrent')
        if not path.exists(torrentpath):
            mkdir(torrentpath)
        torrentbackuppath = path.join(self.datapath, 'torrentbackup')
        if not path.exists(torrentbackuppath):
            mkdir(torrentbackuppath)
        datacachepath = path.join(self.datapath, 'datacache')
        if not path.exists(datacachepath):
            mkdir(datacachepath)
        piececachepath = path.join(self.datapath, 'piececache')
        if not path.exists(piececachepath):
            mkdir(piececachepath)
        if path.exists(path.join(self.datapath, 'abc.conf')):
            # User conf already exists, just delete old log files
            abcokcbuild = int(ABC_OKC_BUILD)
            logfiles = [f for f in listdir(self.datapath) if f.endswith('.log') and path.isfile(path.join(self.datapath, f))]
            for logfile in logfiles:
                try:
                    logfilebuild = int(logfile.split('.')[-2])
                except:
                    logfilebuild = -1
                if logfilebuild < abcokcbuild:
                    try:
                        remove(path.join(self.datapath, logfile))
                    except:
                        pass
        else:
            # Copy old data or default config files to data path
            if path.exists(self.olddatapath):
                copy_tree(self.olddatapath, self.datapath)
            else:
                copy_tree(path.join(self.abcpath, 'defaultconf'), self.datapath)
            # Copy old cache data if any to data path
            appdatadir = path.dirname(self.datapath)
            datacache = path.join(appdatadir, '.ABC', 'datacache')
            if path.exists(datacache):
                copy_tree(datacache, path.join(self.datapath, 'datacache'))
            piececache = path.join(appdatadir, '.ABC', 'piececache')
            if path.exists(piececache):
                copy_tree(piececache, path.join(self.datapath, 'piececache'))
        # Delete old conf file for internal browser
        searchfile = path.join(self.datapath, 'search.lst')
        if path.exists(searchfile):
            try:
                remove(searchfile)
            except:
                pass

    def expandIPv6(self, ip):
        if ip == '::':
            ip = ''
        elif ip[:2] == '::':
            ip = ip[1:]
        elif ip[0] == ':':
            return None
        elif ip[-2:] == '::':
            ip = ip[:-1]
        elif ip[-1] == ':':
            return None
        blocs = ip.split(':')
        dc = None
        blocslen = 0
        i = 0
        for b in blocs:
            if b == '':
                # Double colon
                if dc is not None:
                    return None
                dc = i
            elif '.' in b:
                # IPv4 format
                blocs4 = b.split('.')
                if len(blocs4) != 4:
                    return None
                try:
                    blocs[i] = ':'.join([''.join(['%02x' % int(b4) for b4 in blocs4[j:j + 2]]) for j in xrange(0, 4, 2)])
                except:
                    return None
                blocslen += 2
            elif len(b) < 4:
                blocs[i] = '0' * (4 - len(b)) + b
                blocslen += 1
            else:
                blocslen += 1
            i += 1
        if dc is not None:
            blocs[dc] = ':'.join(['0000'] * (8 - blocslen))
        return ':'.join(blocs)

    def getHostByName(self, name):
        if self.ipv6enabled:
            try:
                host = socket.getaddrinfo(name, None, socket.AF_INET6)[0][4][0]
            except:
                pass
            else:
                return self.expandIPv6(host)
        try:
            host = socket.getaddrinfo(name, None, socket.AF_INET)[0][4][0]
        except:
            return None
        return host

    def getIpsByName(self, name):
        try:
            ai = socket.getaddrinfo(name, None)
        except:
            return []
        return [info[4][0] for info in ai]

    def decodeString(self, s):
        for coding in [('utf_8', 'strict'), (sysencoding, 'ignore')]:
            try:
                ds = s.decode(coding[0], coding[1])
            except:
                continue
            return ds
        return ''

    def encodeString(self, s):
        for coding in [('utf_8', 'strict'), (sysencoding, 'ignore')]:
            try:
                es = s.encode(coding[0], coding[1])
            except:
                continue
            return es
        return ''

    def messageIsTrackerCnxError(self, msg):
        for m in self.trackercnxerrors:
            if msg.startswith(m):
                return True
        return False

    def binToString(self, binary):
        # Build a string from a binary string in input with characters > 128 and non printable characters displayed in hexa as '\XX'
        # '\' is displayed as '\\'
        outstring = ''
        for c in binary:
            if c == '\\':
                outstring += '\\\\'
            elif 31 < ord(c) < 127:
                outstring += c
            else:
                outstring += '\\' + hexlify(c)
        return outstring


class GenBitmapToggleButtonDual(GenBitmapToggleButton):
    def setBitmapSelectedFlat(self, bitmap):
        self.bitmapselectedflat = bitmap

    def setBitmapSelectedNotFlat(self, bitmap):
        self.bitmapselectednotflat = bitmap

    def setBitmapDetectedFlat(self, bitmap):
        self.bitmapdetectedflat = bitmap

    def setBitmapDetectedNotFlat(self, bitmap):
        self.bitmapdetectednotflat = bitmap

    def setFlat(self, flat = True):
        if flat:
            self.SetBezelWidth(0)
            self.bitmapselected = self.bitmapselectedflat
            self.bitmapdetected = self.bitmapdetectedflat
        else:
            self.SetBezelWidth(2)
            self.bitmapselected = self.bitmapselectednotflat
            self.bitmapdetected = self.bitmapdetectednotflat
        self.SetBitmapSelected(self.bitmapselected)


class GenBitmapButtonDual(GenBitmapButton):
    def setFlat(self, flat = True):
        if flat:
            self.SetBezelWidth(0)
        else:
            self.SetBezelWidth(2)


class FileTree:
    # To store temporarily the torrent file/folder structure
    def __init__(self, filesname, dest, utility):
        self.utility = utility
        self.dest = dest
        self.tree = ({}, [])
        self.nbfiles = len(filesname)
        i = 0
        for file in filesname:
            items = file.split('\\')
            subitems = self.tree
            for dir in items[:-1]:
                if not dir in subitems[0]:
                    subitems[0][dir] = ({}, [])
                subitems = subitems[0][dir]
            subitems[1].append((i, items[-1]))
            i += 1

    def subItems(self, dirpath):
        if dirpath:
            items = dirpath.split('\\')
        else:
            items = []
        subitems = self.tree
        for dir in items:
            subitems = subitems[0].get(dir)
            if subitems is None:
                return ({}, [])
        return subitems

    def files(self, dirpath):
        return [filetuple[1] for filetuple in self.subItems(dirpath)[1]]

    def dirs(self, dirpath):
        return self.subItems(dirpath)[0]

    def hasFile(self, filepath):
        dir, file = path.split(filepath)
        return (file.lower() in map(unicode.lower, self.files(dir)))

    def hasDir(self, dirpath):
        updir, dir = path.split(dirpath)
        return (dir.lower() in map(unicode.lower, self.dirs(updir).keys()))

    def findUniqueWinName(self, name, namelist, pth):
        namelist = map(unicode.lower, namelist)
        if name.lower() not in namelist and not path.exists(path.join(pth, name)):
            return name
        if len(name) > 251:
            # Unable to rename to make unique without truncating
            return None

        namelen = len(name)
        # Renaming will add '_' + 3 digit to the original name
        i = 1
        dot = name.rfind('.')
        if dot == -1:
            dot = namelen
        while i < 1000:
            newname = name[:dot] + ('_%03u' % i) + name[dot:]
            if newname.lower() not in namelist and not path.exists(path.join(pth, newname)):
                break
            i += 1
        if i != 1000:
            return newname
        return None

    def digForFixingWinNames(self, dir, status, pth):
        # Updates status[X, [Y1, Y2,...]] :
        # X : True if at least one name was changed in the tree
        #     False if all names are already OK and none were changed
        # Yi : bit 1 : set if automatically finding unique file name failed while fixing bad char in file i
        #      bit 2 : set if file name is too long in file i
        #      bit 5 : set if automatically finding unique file name failed while fixing duplicate in file i
        # Check directory names
        dirkeylist = dir[0].keys()
        filelist = dir[1]
        for d in dirkeylist[:]:
            if len(d) > self.utility.filenamemaxlength:
                indexes = []
                self.digForFileIndexes(dir[0][d], indexes)
                for i in indexes:
                    status[1][i] |= 2
            else:
                fixedname = self.utility.fixWindowsName(d)
                if fixedname:
                    status[0] = True
                    uniquefixedname = self.findUniqueWinName(fixedname, dirkeylist + [file[1] for file in filelist], pth)
                    if uniquefixedname:
                        dir[0][uniquefixedname] = dir[0][d]
                        del dir[0][d]
                        dirkeylist.append(uniquefixedname)
                        dirkeylist.remove(d)
                    else:
                        indexes = []
                        self.digForFileIndexes(dir[0][d], indexes)
                        for k in indexes:
                            status[1][k] |= 2
            self.digForFixingWinNames(dir[0][d], status, path.join(pth, d))
        # Check file names
        for f in filelist[:]:
            if len(f[1]) > self.utility.filenamemaxlength:
                status[1][f[0]] |= 2
            else:
                fixedname = self.utility.fixWindowsName(f[1])
                if fixedname:
                    status[0] = True
                    uniquefixedname = self.findUniqueWinName(fixedname, [file[1] for file in filelist] + dirkeylist, pth)
                    if uniquefixedname:
                        filelist.append((f[0], uniquefixedname))
                        filelist.remove(f)
                    else:
                        status[1][f[0]] |= 1
        # Check duplicate directory names
        # At loading time directory names that differ only by case, are detected as duplicate and renamed to cope with
        # torrents generated by systems considering case
        # But a directory renaming by user matching an existing directory with the Windows case rules is not detected
        # as a duplicate
        dirs_lower = [file.lower() for file in dirkeylist]
        k = 0L
        for i in xrange(len(dirs_lower)):
            for j in xrange(i):
                k += 1
                if k % 100000 == 0:
                    wx.SafeYield(onlyIfNeeded = True)
                if dirs_lower[j] == dirs_lower[i]:
                    status[0] = True
                    uniquefixedname = self.findUniqueWinName(dirkeylist[i], dirkeylist + [file[1] for file in filelist], pth)
                    if uniquefixedname:
                        dir[0][uniquefixedname] = dir[0][dirkeylist[i]]
                        del dir[0][dirkeylist[i]]
                        dirkeylist[i] = uniquefixedname
                        dirs_lower[i] = uniquefixedname.lower()
                    else:
                        indexes = []
                        self.digForFileIndexes(dir[0][dirkeylist[i]], indexes)
                        for k in indexes:
                            status[1][k] |= 16
                    break
        # Check duplicate file names
        files_lower = [file[1].lower() for file in filelist]
        for i in xrange(len(files_lower)):
            # File and directories
            for j in xrange(len(dirs_lower)):
                k += 1
                if k % 100000 == 0:
                    wx.SafeYield(onlyIfNeeded = True)
                if dirs_lower[j] == files_lower[i]:
                    status[0] = True
                    uniquefixedname = self.findUniqueWinName(filelist[i][1], [file[1] for file in filelist] + dirkeylist, pth)
                    if uniquefixedname:
                        filelist[i] = (filelist[i][0], uniquefixedname)
                        files_lower[i] = uniquefixedname.lower()
                    else:
                        status[1][filelist[i][0]] |= 16
                        status[1][filelist[j][0]] |= 16
                    break
            # File and files
            for j in xrange(i):
                k += 1
                if k % 100000 == 0:
                    wx.SafeYield(onlyIfNeeded = True)
                if files_lower[j] == files_lower[i]:
                    status[0] = True
                    uniquefixedname = self.findUniqueWinName(filelist[i][1], [file[1] for file in filelist] + dirkeylist, pth)
                    if uniquefixedname:
                        filelist[i] = (filelist[i][0], uniquefixedname)
                        files_lower[i] = uniquefixedname.lower()
                    else:
                        status[1][filelist[i][0]] |= 16
                        status[1][filelist[j][0]] |= 16
                    break

    def fixWinNames(self):
        # Check validity of Windows names, and rename if necessary
        status = [False, self.nbfiles * [0]]
        self.digForFixingWinNames(self.tree, status, self.utility.completePath(self.dest))
        return status

    def digForFileIndexes(self, dir, indexes):
        for d in dir[0]:
            self.digForFileIndexes(dir[0][d], indexes)
        indexes.extend([f[0] for f in dir[1]])

    def digForFiles(self, dir, files, curpath = ''):
        for d in dir[0]:
            self.digForFiles(dir[0][d], files, path.join(curpath, d))
        files.extend([(f[0], path.join(curpath, f[1])) for f in dir[1]])
        curpath = path.split(curpath)[0]

    def getFileNames(self):
        # Rebuild the file list from the tree and returns it
        allfiletuples = []
        self.digForFiles(self.tree, allfiletuples)
        allfiletuples.sort()
        return [f[1] for f in allfiletuples]


class SpinBox():
    def __init__(self, parent, header = None, spacing = None, width = 25, height = -1, value = "0",
                 defvalue = None, min = 0, max = 99, raiseexcept = True, filterfunc = None, rightbutfunc = None,
                 tooltip = None, style = wx.TE_RIGHT):
        if defvalue is None:
            self.defvalue = value
        else:
            self.defvalue = defvalue
        self.min = min
        self.max = max
        self.raiseexcept = raiseexcept
        self.rightbutfunc = rightbutfunc
        if not filterfunc:
            filterfunc = self.onWriteDigits
        if spacing is None:
            if header:
                spacing = 5
            else:
                spacing = 0
        self.box = wx.TextCtrl(parent, -1, value, (-1, -1), (width, height), style)
        self.box.Bind(wx.EVT_CHAR, filterfunc)
        self.box.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        if height == -1 :
            height = self.box.GetSize().height
        self.spin = wx.SpinButton(parent, -1, (-1, -1), (height * 2 / 3, height), wx.SP_ARROW_KEYS | wx.SP_VERTICAL)
        self.spin.SetRange(min, max)
        self.spin.SetValue(int(value))
        self.spin.Bind(wx.EVT_SPIN_UP, self.onSpinUp)
        self.spin.Bind(wx.EVT_SPIN_DOWN, self.onSpinDown)
        if tooltip:
            self.box.setToolTipString(tooltip)
        self.sizer = wx.BoxSizer(wx.HORIZONTAL)
        if header:
            self.sizer.Add(wx.StaticText(parent, -1, header), 0, wx.ALIGN_CENTER_VERTICAL)
        self.sizer.Add(self.box, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, spacing)
        self.sizer.Add(self.spin, 0, wx.ALIGN_CENTER_VERTICAL)

    def onSpinUp(self, event):
        mousestate = wx.GetMouseState()
        try:
            val = int(self.box.GetValue())
        except:
            # Not integer
            val = self.min
        else:
            if val == self.max:
                return
            if val > self.max:
                val = self.max
            elif val < self.min:
                val = self.min
            else:
                val += 1
        self.spin.SetValue(val)
        self.box.SetValue(str(val))
        if self.rightbutfunc and (mousestate.RightDown() or mousestate.ControlDown()):
            self.rightbutfunc()

    def onSpinDown(self, event):
        mousestate = wx.GetMouseState()
        try:
            val = int(self.box.GetValue())
        except:
            # Not integer
            val = self.min
        else:
            if val == self.min:
                return
            if val < self.min:
                val = self.min
            elif val > self.max:
                val = self.max
            else:
                val -= 1
        self.spin.SetValue(val)
        self.box.SetValue(str(val))
        if self.rightbutfunc and (mousestate.RightDown() or mousestate.ControlDown()):
            self.rightbutfunc()

    def onWriteDigits(self, event):
        # Filter out not digit chars
        keycode = event.GetKeyCode()
        if 47 < keycode < 58 or keycode == wx.WXK_DELETE or keycode == wx.WXK_BACK \
           or keycode == 24:
            event.Skip()
            return True
        if keycode == wx.WXK_LEFT or keycode == wx.WXK_RIGHT or keycode == wx.WXK_HOME \
           or keycode == wx.WXK_END or keycode == wx.WXK_INSERT or keycode == 3:
            event.Skip()
            return False
        return False

    def setToolTipString(self, tooltip):
        self.box.SetToolTipString(tooltip)

    def setValue(self, value):
        try:
            val = int(value)
        except:
            # Not integer
            val = int(self.defvalue)
        else:
            if value < self.min:
                val = self.min
            if val > self.max:
                val = self.max
        self.spin.SetValue(val)
        self.box.SetValue(str(val))

    def getValue(self):
        try:
            val = int(self.box.GetValue())
        except:
            if self.raiseexcept:
                raise ValueError
            val = int(self.defvalue)
        else:
            if val < self.min:
                if self.raiseexcept:
                    raise ValueError
                val = self.min
            elif val > self.max:
                if self.raiseexcept:
                    raise ValueError
                val = self.max
        self.spin.SetValue(val)
        self.box.SetValue(str(val))
        return val

    def getSizer(self):
        return self.sizer

    def getTextWindow(self):
        return self.box

    def getSpinWindow(self):
        return self.spin

    def onRightButtonText(self, event):
        pass


# Redirect py2exe error logging to Windows user ABC_OKC app data directory
class Stderr(object):
    softspace = 0
    _file = None
    _error = None
    def write(self, text, alert = MessageBox, fname = sys.executable + '.' + ABC_OKC_BUILD + '.log'):
        if self._file is None and self._error is None:
            if not path.exists(DATAPATH):
                mkdir(DATAPATH)
            fname = path.join(DATAPATH, path.basename(fname))
            try:
                self._file = open(fname, 'a')
            except Exception, details:
                self._error = details
                import atexit
                atexit.register(alert, 0,
                                "The logfile '%s' could not be opened:\n %s" % \
                                (fname, details),
                                "Errors occurred")
            else:
                import atexit
                atexit.register(alert, 0,
                                "See the logfile '%s' for details" % fname,
                                "Errors occurred")
        if self._file is not None:
            for coding in [('utf_8', 'strict'), (sysencoding, 'ignore')]:
                try:
                    err = text.encode(coding[0], coding[1])
                except:
                    continue
                break
            else:
                err = "can't encode error message"
            self._file.write(err + '\n')
            self._file.flush()

    def flush(self):
        if self._file is not None:
            self._file.flush()


class ScrolledMessageDialog(wx.Dialog):
    def __init__(self, parent, title, message, scrolledmessage, size = None, cancel = True, hscroll = True, centeronparent = True, localize = None):
        if localize is None:
            localize = parent.localize

        pre = wx.PreDialog()
        pre.Create(parent, -1, title)
        self.this = pre.this
        
        if size is None:
            size = (600, 300)
        style = wx.TE_MULTILINE | wx.TE_READONLY
        if hscroll:
            style |= wx.HSCROLL

        outerbox = wx.BoxSizer(wx.VERTICAL)

        if message:
            self.message = wx.StaticText(self, -1, message)

        self.scrolledmessage = wx.TextCtrl(self, -1, scrolledmessage, size = size, style = style)

        okbtn = wx.Button(self, -1, localize('ok'))
        self.Bind(wx.EVT_BUTTON, self.onOK, okbtn)

        if cancel:
            cancelbtn = wx.Button(self, wx.ID_CANCEL, localize('cancel'))
            self.Bind(wx.EVT_BUTTON, self.onCancel, cancelbtn)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        buttonbox.Add(okbtn, 0, wx.ALL, 5)
        if cancel:
            buttonbox.Add(cancelbtn, 0, wx.ALL, 5)

        if message:
            outerbox.Add(self.message, 0, wx.ALIGN_CENTER_VERTICAL | wx.TOP | wx.LEFT, 10)
        outerbox.Add(self.scrolledmessage, 0, wx.EXPAND | wx.ALL, 5)
        outerbox.Add(buttonbox, 0, wx.ALIGN_CENTER)

        self.SetAutoLayout(True)
        self.SetSizer(outerbox)
        self.Fit()

        if centeronparent:
            self.CentreOnParent()
        else:
            self.CentreOnScreen()

    def onOK(self, event):
        self.EndModal(wx.ID_OK)

    def onCancel(self, event):
        self.EndModal(wx.ID_CANCEL)
