##############################################################################
# Module : scheduler.py
# Author :
# Date   :
# Updated and modified for ABC_OKC : Old King Cole
#
# Description : Scheduler (Torrent queue manager + bandwidth distribution)
#
##############################################################################
import sys, wx, gc

from hashlib import sha1
from threading import Event, Thread, Timer
from os import path, remove, rmdir, getcwd, listdir, renames, rename, walk
from time import clock, time, strftime, sleep
from shutil import move
from socket import error as socketerror
from binascii import hexlify, unhexlify
from traceback import print_exc
from string import whitespace
from StringIO import StringIO

from BitTornado.bencode import bencode, bdecode
from BitTornado.utility import exceptionArgsToString
from BitTornado.natpunch import UPnP_test
from BitTornado.RawServer import RawServer
from BitTornado.buffer import PIECEREADBUFFERPOOL, GENBUFFERPOOL
from BitTornado.CurrentRateMeasure import Measure
from BitTornado.PeerListener import PeerListener

from khashmir.khashmir import Khashmir

from abctorrent import ABCTorrent
from abcengine import ABCEngine, FILESEM, CHECKWAITING
from filemanager import TorrentListFileManager
from scrape import ScrapeThread
from abcmetainfoframe import ABCMetaInfoFrame
from Dialogs.setdestdlg import SetDestDialog
from Dialogs.addduplicatedlg import AddDuplicateDialog
from Dialogs.addsamedestdlg import AddSameDestDialog
from Dialogs.scrapedlg import ScrapeDialog

ints = (long, int)
wxEVT_SCH = wx.NewEventType()


def EVT_SCH(win, func):
    win.Connect(-1, -1, wxEVT_SCH, func)


class SchEvent(wx.PyEvent):
    def __init__(self, func, args, kwargs):
        wx.PyEvent.__init__(self)
        self.SetEventType(wxEVT_SCH)
        self.func = func
        self.args = args
        self.kwargs = kwargs


class PopUpLoaded(wx.PopupWindow):
    def __init__(self, parent):
        wx.PopupWindow.__init__(self, parent, wx.SIMPLE_BORDER)
        self.SetBackgroundColour("CADET BLUE")
        self.counter = 1
        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        popupbox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.VERTICAL)
        self.title = wx.StaticText(self, -1, "ABC_OKC", style = wx.ALIGN_CENTRE_HORIZONTAL | wx.ST_NO_AUTORESIZE)
        self.title.SetFont(wx.Font(20, wx.DEFAULT, wx.NORMAL, wx.NORMAL, False))
        popupbox.Add(self.title, 0, wx.EXPAND | wx.TOP, 5)
        self.textcounter = wx.StaticText(self, -1, "+1", style = wx.ALIGN_CENTRE_HORIZONTAL | wx.ST_NO_AUTORESIZE)
        self.textcounter.SetFont(wx.Font(40, wx.DEFAULT, wx.NORMAL, wx.NORMAL, False))
        popupbox.Add(self.textcounter, 0, wx.EXPAND | wx.TOP, 5)
        outerbox.Add(popupbox, 0, wx.ALIGN_CENTER | wx.TOP, -6)
        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())
        self.SetPosition(map(lambda x, y, z: x + y - z,
                             wx.Display(max(wx.Display.GetFromWindow(parent), 0)).GetClientArea().GetBottomRight(),
                             (1, 1),
                             self.GetSize()))

    def incCounter(self):
        self.counter += 1
        self.textcounter.SetLabel("+" + str(self.counter))


class ABCScheduler(wx.EvtHandler):
    def __init__(self, parent, abcparams, guiman, tempmanager, conlimitermanager):
        wx.EvtHandler.__init__(self)

        self.guiman = guiman
        self.abcparams = abcparams

        # Counter for torrents considered as downloading by the main scheduler
        # (without torrentwaitingforleechers, with pause/on-hold)
        self.currentproc = 0
        # Counter for torrents considered as seeding by the main scheduler
        # (without torrentwaitingforleechers, with pause/on-hold)
        self.currentprocseed = 0
        self.parent = parent
        self.list = self.parent.list
        self.frame = self.parent.parent
        self.tempmanager = tempmanager
        self.templates = tempmanager.templates
        self.conlimitermanager = conlimitermanager
        self.conlimiters = self.conlimitermanager.conlimiters
        self.sb = self.frame.abc_sb
        self.popuploaded = None
        self.popuploadedtimer = None

        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.getRankfromID = self.guiman.getRankfromID

        self.expirecachecounter = 0
        if self.utility.expirecachedata:
            self.cleanOldCache()
        self.cleanTorrentBackup()
        self.maxnumsim = int(abcparams['numsimdownload'])
        self.maxnumsimseed = int(abcparams['numsimseed'])
        self.maxnumsimseednodown = int(abcparams['numsimseednodown'])
        # No downloading status, used to detect transitions to queue back torrents if needed
        self.nodownloading = True
        self.activitymax = int(abcparams['activitymax'])
        self.activitymaxseed = int(round(self.activitymax * float(abcparams['activitymaxforseed'])))
        self.activitymaxdown = self.activitymax - self.activitymaxseed
        self.activitythdown = float(abcparams['activitythdown'])
        self.activitythup = float(abcparams['activitythup'])
        self.activitydelay = int(abcparams['activitydelay'])

        self.proctab = []

        # Counter for loaded torrents
        self.torrentloaded = 0
        # Counter for finished torrents
        self.torrentfinished = 0
        # Counter for all running torrents (with torrentwaitingforleechers, with pause/on-hold)
        self.torrentrunning = 0
        # Counter for total downloading torrents (with torrentwaitingforleechers, without pause/on-hold)
        self.torrentdownloading = 0
        # Counter for seeding torrents (with torrentwaitingforleechers, without pause/on-hold)
        self.torrentseeding = 0
        # Counter for inactive torrents downloading (torrents with activity unset)
        self.torrentinactive = 0
        # Counter for inactive torrents seeding (torrents with activity unset)
        self.torrentinactiveseed = 0
        # Counter for paused and on-hold torrents downloading
        self.torrentpaused = 0
        # Counter for paused and on-hold torrents seeding
        self.torrentpausedseed = 0
        # Counter for on-hold torrents downloading
        self.torrentonhold = 0
        # Counter for on-hold torrents seeding
        self.torrentonholdseed = 0
        # Counter for queued torrents
        self.torrentqueued = 0
        # Counter for torrents with 100% progress
        self.torrentavailable = 0
        self.updateRunningTorrentCounters()

        # Alive torrents (downloading or seeding, without torrentwaitingforleechers, without pause/on-hold)
        # It doesn't include standby torrents that are connected just to check for sharing opportunities
        # Order in this list reflects order in the queue
        self.torrentalive = []

        # For Download Rate Maximizer
        # Time when the download rate is lower than the DRM threshold for the first time
        # This is used to trigger the start of a torrent by the DRM only after a certain amount
        # of time of exceeding of the threshold
        self.drm_time1 = 0.
        # Time when the upload rate is greater than the global max upload rate for the first time
        # This is used to trigger the stop of a torrent by the DRM only after a certain amount of
        # time of exceeding of the threshold
        self.drm_time2 = 0.
        # For Upload Rate Maximizer
        # Time when the upload rate is lower than the URM threshold for the first time
        # This is used to trigger the start of a torrent by the URM only after a certain amount
        # of time of exceeding of the threshold
        self.urm_time1 = 0.
        # Time when the upload rate is greater than the global max upload rate for the first time
        # This is used to trigger the stop of a torrent by the URM only after a certain amount of
        # time of exceeding of the threshold
        self.urm_time2 = 0.
        # Max upload rate available for ABC to upload torrents, dynamically computed from ABC
        # download rate, number of connections and max upload rate available for ABC in the system
        self.dynmaxuprate = 0.
        # Total measured upload and download rate
        self.totalupload = 0.
        self.totaldownload = 0.
        # Time counter used to delay the starting of a torrent by the DRM just after ABC starts
        # or mode is switched to automatic or all torrents are unpaused or the max download rate
        # is increased
        self.drminitialstartingtime = 0.
        # Time counter used to delay the starting of a torrent by the URM just after ABC starts
        # or mode is switched to automatic or all torrents are unpaused or the max upload rate
        # is increased or the max upload rate is set to dynamic or static
        self.urminitialstartingtime = 0.
        # Time counter used to delay the next possible torrent start/stop by DRM after a torrent start
        self.drmstartingtime = 0.
        # Time counter used to delay the next possible torrent start by DRM after a torrent stop
        self.drmstoppingtime = 0.
        # Time counter used to delay the next possible torrent start/stop by URM after a torrent start
        self.urmstartingtime = 0.
        # Time counter used to delay the next possible torrent start by URM after a torrent stop
        self.urmstoppingtime = 0.

        # True is some torrents checking or allocating or waiting have been found
        self.torrentcheckingwaiting = False

        self.extraglob = self.extradown = self.extraseed = -1
        self.extratobestopped = 0
        self.extratobestoppedseed = 0
        self.freshactivetorrents = []
        self.freshactivetorrentsseed = []

        # To store current sorting direction (0 ; direct ; 1 : reverse ; 2 : unsorted)
        self.sortdir = 2

        # Last sorting column id, left/right part of column shifter, selected block
        self.lastsortingid = -1
        self.lastsortingshift = -1
        self.lastsortingblock = None

        # 1 if must close ABC when transfers are over
        # 2 if must shut down when transfers are over
        self.whenidle = 0

        # Time when progress dialog for canceling closing is opened
        # 0 is dialog is not opened
        self.abouttoclose = 0

        # True when an auto-shutdown of the computer was requested
        self.abouttoshutdown = False

        # Volume counters and volume limiter data
        self.downvolmax = long(float(self.abcparams['downvolmax']) * 1073741824)
        self.upvolmax = long(float(self.abcparams['upvolmax']) * 1073741824)
        self.downvolcountdur = int(self.abcparams['downvolcountdur']) * 3600
        self.upvolcountdur = int(self.abcparams['upvolcountdur']) * 3600
        # Parameters for fixed window measuring
        self.downvolchecktime = int(self.abcparams['downvolchecktime'])
        self.upvolchecktime = int(self.abcparams['upvolchecktime'])
        self.downvolfixedwin = float(self.abcparams['downvol'])
        self.upvolfixedwin = float(self.abcparams['upvol'])
        # Parameters for sliding window measuring
        self.downvolthreshold = (100 - float(self.abcparams['downvolmargin'])) / 100
        self.upvolthreshold = (100 - float(self.abcparams['upvolmargin'])) / 100
        self.downvolmeasure = Measure(self.downvolcountdur,
                                      start = float(self.abcparams['downvolstart']) - self.utility.time0,
                                      last = float(self.abcparams['downvollast']) - self.utility.time0,
                                      rate = float(self.abcparams['downvolrate']))
        self.upvolmeasure = Measure(self.upvolcountdur,
                                    start = float(self.abcparams['upvolstart']) - self.utility.time0,
                                    last = float(self.abcparams['upvollast']) - self.utility.time0,
                                    rate = float(self.abcparams['upvolrate']))
        # Flags for volume limiter ; if both are true mode can be switched back to auto
        self.canswitchautodown = False
        self.canswitchautoup = False

        # If all rejection messages from the trackers of a torrent contain at least one string from
        # the list rejectexceptions, the torrent is not considered as definitely rejected and will
        # not be stopped if rejection immediately occurs after starting.
        # This is useful for trackers that periodically reject all their torrents for a short lapse of time
        # for maintenance purposes.
        # This list is set from Preferences/Queue
        self.rejectexceptions = self.abcparams['rejectexceptions'].split()

        # List of strings that will cause canceling of timeouts if found in tracker error messages
        self.canceltimeoutsmsg = self.abcparams['canceltimeoutsmsg'].split()

        # Which extra torrents may be stopped by the main scheduler when a torrent turns active
        # 0 : don't stop any active torrent
        # 1 : stop an active torrent, and allow stopping of torrents with a priority higher than the priority of the torrent that just turned active
        # 2 : stop an active torrent, but don't stop any active torrent with a priority higher than the priority of the torrent that just turned active
        # Note : stopped torrents are always the lowest in priority
        self.extratobestoppedmode = int(self.abcparams['extratobestopped'])

        # "Add duplicate", "add to same dest" dialog position for the current session
        # Note : A wx.MessageDialog can't be positioned
        self.addtorrentwarningpos = wx.DefaultPosition

        # Flag unset as long as there is at least one torrent moving its data
        self.enddatamovingflag = Event()
        self.enddatamovingflag.set()

        # Number of threads running to move torrent data
        self.movingthreads = 0

        # Timer to update torrent list
        self.timerupdatetorrentlist = None
        # Timer for cyclical tasks
        self.timercyclicaltasks = None
        # Timer for cyclical garbage collecting
        self.timercyclicalgc = None
        # Timer for expired buffers
        self.timerexpirebuffers = None
        # Timer for standby torrent seeding time
        self.timerstandbyseedingtime = None

        # List of torrents to be completed before auto-shutdown of computer
        self.compshut = []
        # Time when all complete and shut down torrents are completed and start seeding
        # 0 if these torrents are not yet all completed
        self.compshutseedingstart = 0

        # While a local torrent adding is being processed, it may be pending because
        # a dialog window is opened and waiting for an answer from the user. So this
        # adding may be partially validated. These 2 lists are used to store the hash
        # and the destination for such torrents, for they must be taken into account
        # in the detection of duplicate torrents or destinations
        self.localtorrentaddingpendinghash = []
        self.localtorrentaddingpendingdest = []

        # True while a standby torrent is connected only to check # copies and peer progress
        # Used to allow only one standby checking at a time
        self.standbychecking = False

        # Torrents with standby status
        self.torrentstandby = []

        # Counters of standby torrents per tracker
        self.trackerstandbycount = {}

        # Standby torrent currently checking for sharing opportunities
        self.torrentwaitingforleechers = None

        # DHT feeder
        self.dht = None
        self.dhtdoneflag = Event()
        self.dhtstartflag = Event()
        self.dhtshutdownflag = Event()
        self.dhtuprate = 0.
        self.dhtlastupdate = clock()

        # Peer listener (single port mode)
        self.pl = None
        self.pldoneflag = Event()
        self.plstartflag = Event()
        self.plshutdownflag = Event()

        # Start queue
        self.startqueue = []

        # Lists to dispatch torrents before distributing upload and download bandwidth
        self.allactive = []                # Active torrents not magnet
        self.allactivenotseeding = []      # Active torrents not magnet and not seeding
        self.toberaisedbelowmin_u = []     # Running torrents for which upload rate can be raised, with up rate < minuprate
                                           # These will always be raised even there's no available up bandwidth, to fulfill the min minuprate up rate rule
        self.toberaisedbelowmin_d = []     # Running torrents for which download rate can be raised, with down rate < mindownrate
        self.priotoberaised_u = []         # Torrents with prioritized upload, or local upload rate settings when prioritizelocal is set (see prioritizelocal)
        self.priotoberaised_d = []         # Torrents with prioritized download, or local download rate settings when prioritizelocal is set (see prioritizelocal)
        self.toberaised_u = []             # Not prioritized running torrents for which upload rate can be raised
        self.toberaised_d = []             # Not prioritized running torrents for which download rate can be raised
        self.tobeloweredbelowmin_u = []    # Running torrents for which upload rate can be lowered, with up rate < minuprate
        self.tobeloweredbelowmin_d = []    # Running torrents for which download rate can be lowered, with down rate < mindownrate
        self.priotobelowered_u = []        # Prioritized running torrents for which upload rate can be lowered
        self.priotobelowered_d = []        # Prioritized running torrents for which download rate can be lowered
        self.tobelowered_u = []            # Not prioritized running torrents for which up rate can be lowered
        self.tobelowered_d = []            # Not prioritized running torrents for which down rate can be lowered
        self.nottobechangedbelowmin_u = [] # Running torrents for which up rate is not to be changed, with up rate < minuprate
        self.nottobechangedbelowmin_d = [] # Running torrents for which down rate is not to be changed, with down rate < mindownrate
        self.prionottobechanged_u = []     # Prioritized running torrents for which up rate is not to be changed
        self.prionottobechanged_d = []     # Prioritized running torrents for which down rate is not to be changed
        self.nottobechanged_u = []         # Not prioritized running torrents for which up rate is not to be changed
        self.nottobechanged_d = []         # Not prioritized running torrents for which down rate is not to be changed
        self.nonextratoberaised_u = []     # Non extra torrents and with up rate >= minuprate
        self.nonextratoberaised_d = []     # Non extra torrents
                                           # If not set, torrents with upload rate local settings are treated like other torrents, except they have their
                                           # own max upload rate they can't cross over. The consequence is a behaviour slightly different from the behavior of
                                           # ABC releases prior to 2.7.0.
                                           # If set, this gives the algorithm the behaviour of ABC releases prior to 2.7.0 : the torrents with an upload rate
                                           # local setting will be granted upload bandwidth in priority to fulfill their local setting, even is this one
                                           # is higher than the global max upload rate setting, and even if this bandwidth must be taken from other active
                                           # torrents wihout a local upload rate setting. These torrents will not take part in the upload rate exchange
                                           # between active torrents when all bandwidth has been distributed, since they will have been served in priority.
        self.allextra = []                 # All extra torrents

        # Prioritized torrents factors
        self.priofactup = float(self.abcparams['priofactup'])
        self.priofactdn = float(self.abcparams['priofactdn'])

        EVT_SCH(self, self.onInvoke)

    def initScheduler(self):
        # Read old list from torrent.lst
        self.sb.SetStatusText(self.localize('loading'))
        torrentlistfile = TorrentListFileManager(self)
        torrentlistfile.open()
        oldproc_array = []
        while True:
            config = torrentlistfile.readList()
            if config is None:
                break
            if len(config) != 54:
                continue
            # name, src, dest, status, prio, downsize, upsize, progress,
            # maxupload, maxlocaluploadrate, uploadopt, uploadtimeh, uploadtimem, uploadratio
            # seedingtime
            # label, shortlabel,
            # timeoutswitch, timeouttracker, timeoutdownload, timeoutupload,
            # movefolder, maxlocaldownloadrate, exttracker, extrackerurl
            # timeoutwork, timeoutworkdown, timeoutworktrack, timeoutseed, timeoutseedup, timeoutseedtrack,
            # termevt, totalseedingtime, timeoutaction, magnetname, prioritizedown, prioritizeup, checkinttrackwait
            # extrainttracker, inttrackerurl, inactivitytime, trackererrortime, sbtime, restoreindex[10], magnetprop
            oldproc_array.append(config)
        torrentlistfile.close()
        self.addOldProc(oldproc_array)

        self.sb.SetStatusText('')
        self.frame.SetStatusBarPane(0)

        # Start DHT
        if self.abcparams['dht'] == '1':
            self.startDHT()

        # Start peer listener (single port mode)
        if self.abcparams['pl'] == '1':
            self.startPL()
        else:
            self.nbreservedports = int(self.abcparams['maxport']) - int(self.abcparams['minport']) + 1

        self.parent.torrentlistloading = False
        self.frame.menusettings.Enable(self.frame.menutweakid, True)
        self.frame.menusettings.Enable(self.frame.menuparamtemplateid, True)
        self.frame.menusettings.Enable(self.frame.menuconlimiterid, True)
        self.frame.menusettings.Enable(self.frame.menuabcoptionid, True)
        self.frame.menusettings.Enable(self.frame.menudhtaddnodeid, True)
        self.frame.menusettings.Enable(self.frame.idlesubmenuid, True)

        self.invokeLater(self.initScheduler2)

    def initScheduler2(self):
        # If there's a torrent in the command line, add it to the torrent list
        if len(self.parent.params) == 1:
            # Accept only 1 non empty parameter as the torrent to be added
            self.parent.params.insert(0, '-a')
        try:
            a = self.parent.params.index('-a')
        except:
            pass
        else:
            self.invokeLater(self.parent.commandLineAddTorrent, [self.parent.params, a])

        # Autostart command scheduler
        if self.utility.schedparams['schedautostart'] == '1':
            self.parent.schedulerbutton.SetValue(True)
            self.parent.commandscheduler.start()

        # Autostart scanner
        if self.utility.scanparams['scanautostart'] == '1' and self.parent.scanner.canScan():
            self.parent.scannerbutton.SetValue(True)
            self.parent.scanner.start()

        # Autostart webservice
        if self.utility.webparams["webautostart"] == "1":
            self.parent.webservicebutton.SetValue(True)
            self.parent.startWebService()

        # Init RSS timer
        if self.abcparams['rssautograb'] == '1':
            self.parent.rsspanel.autograbbutt.SetValue(True)
            self.parent.rsspanel.startAllTimers()

        # Tweak garbage collector
        #gc.set_debug(gc.DEBUG_COLLECTABLE | gc.DEBUG_UNCOLLECTABLE | gc.DEBUG_INSTANCES)
        #gc.set_debug(gc.DEBUG_SAVEALL | gc.DEBUG_STATS | gc.DEBUG_UNCOLLECTABLE | gc.DEBUG_INSTANCES)
        if self.utility.cyclicalgcrate >= 0:
            # Enable automatic  garbage collecting
            gc.set_threshold(self.utility.gcth1, self.utility.gcth2, self.utility.gcth3)
            gc.enable()
        if self.utility.cyclicalgcrate != 0:
            # Start full cyclical garbage collecting
            self.invokeLater(self.cyclicalGarbageCollect)

        # Start timer to periodically delete expired buffers
        if self.utility.expirebuffersrate:
            GENBUFFERPOOL.setExpire(self.utility.expirebuffersage)
            PIECEREADBUFFERPOOL.setExpire(self.utility.expirebuffersage)
            #On 1st time it's useles to start it before self.utility.expirebuffersage
            self.timerexpirebuffers = Timer(self.utility.expirebuffersage, self.invokeLater, [self.expireBuffers])
            self.timerexpirebuffers.start()

        # Start timer to periodically update the torrent.lst file
        self.timerupdatetorrentlist = Timer(self.utility.torlistautosaverate, self.invokeLater, [self.updateTorrentList])
        self.timerupdatetorrentlist.start()

        self.parent.initializing = False
        self.frame.menutools.Enable(self.frame.menurestartid, True)
        self.frame.menutools.Enable(self.frame.menuexitid, True)
        self.frame.EnableCloseButton(True)
        self.frame.menutray.Enable(self.frame.TBMENU_CLOSE, True)

        # Last time there was no active torrent and last time of DRM and URM start
        self.lastinactivitytime = self.drminitialstartingtime = self.urminitialstartingtime = clock()

        # Counter to detect possible network disconnection
        # > 0 : time of last disconnection ; < 0 : time of last reconnection ; 0 while disconnection once it has been detected
        self.networkdisconnection = -self.lastinactivitytime

        self.cyclicalTasks()

        if self.utility.countsbasseed:
            self.startStandbySeeding()

        self.scheduler()

    def cyclicalGarbageCollect(self):
        try:
            if self.parent.exiting:
                return
        except:
            return
        self.timercyclicalgc = Timer(abs(self.utility.cyclicalgcrate), self._cyclicalGarbageCollect)
        self.timercyclicalgc.start()

    def _cyclicalGarbageCollect(self):
        # Run full collection
        gc.collect()
        self.invokeLater(self.cyclicalGarbageCollect)

    def expireBuffers(self):
        try:
            if self.parent.exiting:
                return
        except:
            return
        expirereadbufferssize = long(self.abcparams['expirereadbufferssize'])
        if expirereadbufferssize == -1:
            # Never expire except if readbuffersize is changed to lower values and buffers were previously created with size over current value
            expirereadbufferssize = long(float(self.abcparams['readbuffersize']) * 1048576) + 1
        elif expirereadbufferssize == -2:
            # Only expire buffers with size over or equal to readbuffersize
            expirereadbufferssize = long(float(self.abcparams['readbuffersize']) * 1048576)
        freed = int((GENBUFFERPOOL.cleanUp(long(self.abcparams['expiregenbufferssize'])) + PIECEREADBUFFERPOOL.cleanUp(expirereadbufferssize)) / 1024)
        if self.abcparams['expirebuffersmsg'] == '1' and freed:
            self.sb.SetStatusText(strftime(self.utility.timestamp) + self.localize('bufferscleanup')
                                  + '{:,}'.format(freed).replace(',', ' ') + ' ' + self.localize('kB'))
        if not self.parent.exiting:
            self.timerexpirebuffers = Timer(self.utility.expirebuffersrate, self.invokeLater, [self.expireBuffers])
            self.timerexpirebuffers.start()

    def startStandbySeeding(self):
        self.initStandbySeedingTime(clock())
        self.timerstandbyseedingtime = Timer(self.utility.GUIupdaterate_fast, self.invokeLater, [self.standbySeedingTime])
        self.timerstandbyseedingtime.start()

    def stopStandbySeeding(self):
        # Restore ETA display for standby torrents seeding until another complete copy exists
        for torrent in self.torrentstandby[:]:
            if torrent.complete and torrent.forceseeding:
                torrent.forceseeding = False
                index = torrent.index
                etarank = self.guiman.getRankfromID(8)
                # Display null ETA without turning FINISHED because we don't know yet
                # if the condition "Another copy complete" is fulfilled
                if etarank != -1:
                    field = self.localize('etaS') + " " + self.utility.eta_value(0.)
                    if self.list.GetItem(index, etarank).GetText() != field:
                        self.list.SetStringItem(index, etarank, field)
        try:
            self.timerstandbyseedingtime.cancel()
        except:
            pass
        self.timerstandbyseedingtime = None

    def initStandbySeedingTime(self, t):
        for torrent in self.torrentstandby[:]:
            if torrent.complete:
                torrent.initStandbySeedingTime(t)

    def standbySeedingTime(self):
        try:
            if self.parent.exiting:
                return
        except:
            return
        if self.abcparams['mode'] == '1':
            now = clock()
            for torrent in self.torrentstandby[:]:
                # Compute and refresh seeding times in torrent list and check finished 
                if torrent.complete:
                    index = torrent.index
                    torrent.totalseedingtime = torrent.sblasttotalseedingtime + now - torrent.sbseedstart
                    if torrent.uploadoption == 1 or torrent.uploadoption == 3:
                        torrent.seedingtime = torrent.sblastseedingtime + now - torrent.sbseedstart
                        # Correct seeding times
                        seedingtimeleft = torrent.uploadtimemax - torrent.seedingtime
                        if seedingtimeleft > 0:
                            torrent.olduploadtimemax = torrent.uploadtimemax
                        elif torrent.uploadtimemax >= torrent.olduploadtimemax:
                            torrent.seedingtime = torrent.uploadtimemax
                            torrent.totalseedingtime += seedingtimeleft
                        torrent.seedingtimeleft = max(seedingtimeleft, 0)
                        # Display ETA
                        etarank = self.guiman.getRankfromID(8)
                        if etarank != -1:
                            field = self.localize('etaS') + " " + self.utility.eta_value(torrent.seedingtimeleft)
                            if self.list.GetItem(index, etarank).GetText() != field:
                                self.list.SetStringItem(index, etarank, field)
                        # Display seeding time
                        rank = self.guiman.getRankfromID(27)
                        if rank != -1:
                            field = self.utility.time_value(torrent.seedingtime, seconds = False, maxvalue = False)
                            if self.list.GetItem(index, rank).GetText() != field:
                                self.list.SetStringItem(index, rank, field)
                    # Display total seeding time
                    rank = self.guiman.getRankfromID(28)
                    if rank != -1:
                        field = self.utility.time_value(torrent.totalseedingtime, seconds = False, maxvalue = False)
                        if self.list.GetItem(index, rank).GetText() != field:
                            self.list.SetStringItem(index, rank, field)
                    # Check if standby is finished or must go on till there's another complete copy
                    if torrent.seedingtimeleft == 0:
                        if (not torrent.uploadtillcopycomplete \
                            or (not torrent.scraping and torrent.scrapeisfresh \
                               and torrent.seed != self.utility.notrackeranswermarker \
                               and torrent.seed != self.utility.notrackermarker \
                               and int(torrent.seed) > 1) \
                            or (torrent.numcopy != '?'
                               and float(torrent.numcopy) >= 1)):
                            torrent.forceseeding = False
                            self.procFINISHED(index, False)
                            self.removeTorrentStandby(torrent)
                        elif etarank != -1:
                            torrent.forceseeding = True
                            field = self.localize('etaS') + " " + self.localize('tillcc')
                            if self.list.GetItem(index, etarank).GetText() != field:
                                self.list.SetStringItem(index, etarank, field)
        else:
            # Restore ETA display for standby torrents seeding until another complete copy exists
            for torrent in self.torrentstandby[:]:
                if torrent.complete and torrent.forceseeding:
                    torrent.forceseeding = False
                    index = torrent.index
                    etarank = self.guiman.getRankfromID(8)
                    # Display null ETA without turning FINISHED because we don't know yet
                    # if the condition "Another copy complete" is fulfilled
                    if etarank != -1:
                        field = self.localize('etaS') + " " + self.utility.eta_value(0.)
                        if self.list.GetItem(index, etarank).GetText() != field:
                            self.list.SetStringItem(index, etarank, field)
        if not self.parent.exiting:
            self.timerstandbyseedingtime = Timer(self.utility.GUIupdaterate_fast, self.invokeLater, [self.standbySeedingTime])
            self.timerstandbyseedingtime.start()

    def startDHT(self):
        self.sb.SetStatusText(self.localize('startingdht'))
        Thread(target = self.startDHTFeeder).start()
        self.dhtstartflag.wait()
        self.sb.SetStatusText('')

    def startDHTFeeder(self):
        self.dhtstoppedonrequest = False
        try:
            rawserver = RawServer(self.dhtdoneflag, 60.0, 300.0,
                                  ipv6_enable = self.utility.ipv6enabled,
                                  failfunc = self.DHTError, udp = True)
            try:
                sock = rawserver.bind(self.utility.dhtport, self.abcparams['bind'],
                                      ipv6_socket_style = int(self.abcparams['ipv6_binds_v4']),
                                      upnp = UPnP_test(int(self.abcparams['upnp'])))
            except socketerror, e:
                self.dht = False
                self.DHTError(self.localize('cantlisten') + " - " + exceptionArgsToString(e))
            else:
                self.dht = Khashmir(self.utility, rawserver, sock, self.abcparams['bind'], self.utility.dhtport,
                                    self.updateDHTStatus)
                self.dhtstartflag.set()
                rawserver.listen_forever(self.dht.udp)
        except:
            self.dht = False
            data = StringIO()
            print_exc(file = data)
            sys.stderr.write(data.getvalue())
        try:
            rawserver.shutdown()
        except:
            pass
        if not self.dhtstoppedonrequest:
            self.DHTFailed()

        self.dhtstartflag.set()
        self.dhtshutdownflag.set()

    def stopDHT(self):
        self.sb.SetStatusText(self.localize('stoppingdht'))
        self.dhtstoppedonrequest = True
        self.dhtdoneflag.set()
        self.dhtshutdownflag.wait()
        self.dht.saveNodes()
        self.dht.savePeers()
        self.sb.SetStatusText('')

    def updateDHTStatus(self, dpflag, uprate):
        if self.dhtlastupdate + self.utility.GUIupdaterate > clock():
            dpflag.set()
            return
        self.invokeLater(self.onUpdateDHTStatus, [dpflag, uprate])

    def onUpdateDHTStatus(self, dpflag, uprate):
        self.dhtlastupdate = clock()
        self.dhtuprate = uprate
        dpflag.set()

    def DHTError(self, errormsg):
        sys.stderr.write(errormsg)

    def DHTFailed(self):
        self.invokeLater(self.onDHTFailEvent)

    def onDHTFailEvent(self):
        self.dhtdoneflag.clear()
        self.dhtstartflag.clear()
        self.dhtshutdownflag.clear()
        if self.dht:
            self.dht.saveNodes()
            self.dht.savePeers()
            self.dht = False
        dlg = wx.MessageDialog(self.frame,
                               self.localize('dhthadtoclose'),
                               self.localize('abcokcerror'), wx.OK)
        dlg.ShowModal()
        dlg.Destroy()

    def startPL(self):
        self.sb.SetStatusText(self.localize('startingpl'))
        Thread(target = self.startPLServer).start()
        self.plstartflag.wait()
        self.sb.SetStatusText('')

    def startPLServer(self):
        self.plstoppedonrequest = False
        try:
            rawserver = RawServer(self.pldoneflag, 60.0, 300.0,
                                  ipv6_enable = self.utility.ipv6enabled,
                                  failfunc = self.PLError)
            try:
                rawserver.bind(self.utility.plport, self.abcparams['bind'],
                               ipv6_socket_style = int(self.abcparams['ipv6_binds_v4']),
                               upnp = UPnP_test(int(self.abcparams['upnp'])))
            except socketerror, e:
                self.pl = False
                self.PLError(self.localize('cantlisten') + " - " + exceptionArgsToString(e))
            else:
                self.pl = PeerListener(rawserver, self.dht, self.utility)
                self.plstartflag.set()
                rawserver.listen_forever(self.pl)
        except:
            self.pl = False
            data = StringIO()
            print_exc(file = data)
            sys.stderr.write(data.getvalue())
        try:
            rawserver.shutdown()
        except:
            pass
        if not self.plstoppedonrequest:
            self.PLFailed()

        self.plstartflag.set()
        self.plshutdownflag.set()

    def stopPL(self):
        self.sb.SetStatusText(self.localize('stoppingpl'))
        self.plstoppedonrequest = True
        self.pldoneflag.set()
        self.plshutdownflag.wait()
        self.sb.SetStatusText('')

    def PLError(self, errormsg):
        sys.stderr.write(errormsg)

    def PLFailed(self):
        self.invokeLater(self.onPLFailEvent)

    def onPLFailEvent(self):
        self.pldoneflag.clear()
        self.plstartflag.clear()
        self.plshutdownflag.clear()
        if self.pl:
            self.pl = False
        dlg = wx.MessageDialog(self.frame,
                               self.localize('plhadtoclose'),
                               self.localize('abcokcerror'), wx.OK)
        dlg.ShowModal()
        dlg.Destroy()

    def diskFullWatchdog(self, torrent, nexttobeallocated):
        canwrite = torrent.canWriteToDisk(nexttobeallocated)
        if canwrite[0]:
            return True
        self.invokeLater(self.onDiskFull, [torrent, canwrite[1]])
        return False

    def onDiskFull(self, torrent, message):
        if torrent not in self.proctab:
            # The torrent is no longer in the list
            return
        if self.abcparams['diskfullman'] == '1':
            if self.abcparams['diskfullmode'] == '0':
                if self.abcparams['mode'] != '0':
                    self.parent.onModeToggle(mode = 0)
            elif self.abcparams['trigwhenfinishseed'] != '2':
                self.toggleSchedulerMode()
        if self.abcparams['mode'] != '1' or self.abcparams['trigwhenfinishseed'] == '2':
            self.procQUEUE([torrent.index])
        else:
            self.procSTOP([torrent.index])
        torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + message)

    def arePortsAvailable(self):
        return self.pl or (self.pl is None and self.torrentrunning < self.nbreservedports)

    def adjustActiveSeeding(self, glob = False):
        if self.nodownloading:
            oldmaxactive = self.maxnumsimseed
            newmaxactive = self.maxnumsimseednodown
        else:
            oldmaxactive = self.maxnumsimseednodown
            if glob:
                newmaxactive = max(self.maxnumsim + self.maxnumsimseed - self.currentproc + self.torrentinactive + self.torrentpaused, 0)
            else:
                newmaxactive = self.maxnumsimseed
        extraactive = min(self.currentprocseed - self.torrentinactiveseed - self.torrentpausedseed, oldmaxactive) - newmaxactive
        if extraactive > 0:
            for t in self.findTorrentToBeQueuedBack(nbtorrent = extraactive):
                if t is not None:
                    self.procQUEUE([t.index], False)
            if self.extratobestoppedseed:
                self.freshactivetorrentsseed[:] = []
                self.extratobestoppedseed = 0

    def getMaxNumSimGlobal(self):
        # Always consider no downloading as false
        if self.nodownloading:
            self.nodownloading = False
            self.adjustActiveSeeding(True)
        return self.maxnumsim + self.maxnumsimseed

    def getMaxNumSimSeed(self):
        # Consider no downloading if the only downloading one is torrentwaitingforleechers 
        # Switch only if no torrent is paused
        if self.abcparams['nodownisinactive'] == "1" and self.currentproc - self.torrentinactive - self.torrentpaused > 0 \
           or self.abcparams['nodownisinactive'] == "0" and self.torrentdownloading and (self.torrentdownloading != 1 or not self.torrentwaitingforleechers or self.torrentwaitingforleechers.complete):
            # Downloading
            if self.nodownloading:
                if self.torrentpaused - self.torrentonhold > 0:
                    return self.maxnumsimseednodown
                self.nodownloading = False
                self.adjustActiveSeeding()
            return self.maxnumsimseed
        # No downloading
        if not self.nodownloading:
            if self.torrentpaused - self.torrentonhold > 0:
                return self.maxnumsimseed
            self.nodownloading = True
            self.adjustActiveSeeding()
        return self.maxnumsimseednodown

    def getActivityMax(self):
        if self.abcparams['activityextra'] == '1':
            return self.activitymax
        return self.getMaxNumSimGlobal()
    
    def getActivityMaxSeed(self):
        if self.abcparams['activityextra'] == '1':
            return self.activitymaxseed
        return self.getMaxNumSimSeed()

    def getActivityMaxDown(self):
        if self.abcparams['activityextra'] == '1':
            return self.activitymaxdown
        return self.maxnumsim

    def setSchedulerModeToSeeding(self, oldtrig):
        self.abcparams['prevschedmode'] = oldtrig
        tobequeued = []
        tobestandby = []
        for t in self.proctab:
            if t.abcengine and not t.complete:
                if t.abcengine.sbwaitingforleechers:
                    tobestandby.append(t.index)
                else:
                    tobequeued.append(t.index)
        self.invokeLater(self.procSTANDBY, [tobestandby, False])
        self.invokeLater(self.procQUEUE, [tobequeued, False])

    def toggleSchedulerMode(self):
        # Toggle scheduler mode between "Only seed without queuing incomplete torrents" and previous mode
        curmode = self.abcparams['trigwhenfinishseed']
        if curmode != '2':
            self.abcparams['prevschedmode'] = curmode
            self.abcparams['trigwhenfinishseed'] = '2'
            self.abcparams['schedonlyonhold'] = '1'
            trignexttorrentonlyseedind = self.localize('schedmodeseedcap')
            if trignexttorrentonlyseedind:
                self.utility.frame.SetTitle(self.localize('mainwindowtitle') + trignexttorrentonlyseedind)
        elif self.abcparams['prevschedmode'] != '-1':
            self.abcparams['trigwhenfinishseed'] = self.abcparams['prevschedmode']
            self.abcparams['schedonlyonhold'] = '0'
            self.utility.frame.SetTitle(self.localize('mainwindowtitle'))
            self.invokeLater(self.scheduler)

    def canBeScheduled(self, torrent):
        return (self.abcparams['trigwhenfinishseed'] == "1" and self.getMaxNumSimGlobal() > 0
                or self.abcparams['trigwhenfinishseed'] == "0" and (torrent.complete and self.getMaxNumSimSeed() > 0 or not torrent.complete and self.maxnumsim > 0)
                or self.abcparams['trigwhenfinishseed'] == "2" and torrent.complete and self.getMaxNumSimSeed() > 0)

    def isSchedulerOverrun(self, torrent):
        if self.abcparams['trigwhenfinishseed'] == "1":
            return (self.currentproc + self.currentprocseed - self.torrentonhold - self.torrentonholdseed 
                    >= min(self.getMaxNumSimGlobal() + self.torrentinactive + self.torrentinactiveseed, self.getActivityMax()))
        if torrent.complete:
            return (self.currentprocseed - self.torrentonholdseed 
                    >= min(self.getMaxNumSimSeed() + self.torrentinactiveseed, self.getActivityMaxSeed()))
        return (self.currentproc - self.torrentonhold 
                >= min(self.maxnumsim + self.torrentinactive, self.getActivityMaxDown()))

    def updateRunningTorrentCounters(self):
        if self.abcparams['trigwhenfinishseed'] == "1":
            current = self.currentproc + self.currentprocseed
        else:
            current = self.currentproc
            currentseed = self.currentprocseed

        runningcounter = self.torrentrunning
        pausedcounter = self.torrentpaused + self.torrentpausedseed
        if self.abcparams['countonhold'] == "0":
            runningcounter -= self.torrentpaused + self.torrentpausedseed
            pausedcounter -= self.torrentonhold + self.torrentonholdseed
            if self.abcparams['countonlyalive'] == "1":
                if self.abcparams['trigwhenfinishseed'] == "1":
                    current -= self.torrentpaused + self.torrentpausedseed
                else:
                    current -= self.torrentpaused
                    currentseed -= self.torrentpausedseed
            else:
                if self.abcparams['trigwhenfinishseed'] == "1":
                    current -= self.torrentonhold + self.torrentonholdseed
                else:
                    current -= self.torrentonhold
                    currentseed -= self.torrentonholdseed
        elif self.abcparams['countonlyalive'] == "1":
            if self.abcparams['trigwhenfinishseed'] == "1":
                current -= self.torrentpaused + self.torrentpausedseed
            else:
                current -= self.torrentpaused
                currentseed -= self.torrentpausedseed

        if self.abcparams['drm'] == "1" or self.abcparams['urm'] == "1":
            self.computeExtraTorrents()

        win7fix_sc = 0
        if self.abcparams['trigwhenfinishseed'] == '1':
            mns = mnss = self.getMaxNumSimGlobal()
            if (self.abcparams['drm'] == '1' or self.abcparams['urm'] == '1'):
                torrentrm = max(0, self.extraglob)
                torrentscheduler = current - torrentrm
            else:
                torrentrm = 0
                torrentscheduler = current
            if self.abcparams['showschedcounters'] == '1':
                schedcounters = "[ %u/%u ] " % (torrentscheduler, self.getActivityMax())
            else:
                schedcounters = ""
        else:
            mns = self.maxnumsim
            mnss = self.getMaxNumSimSeed()
            if (self.abcparams['drm'] == '1' or self.abcparams['urm'] == '1'):
                torrentrm = max(0, self.extradown) + max(0, self.extraseed)
                torrentscheduler = current - max(0, self.extradown)
                torrentschedulerseed = currentseed - max(0, self.extraseed)
            else:
                torrentrm = 0
                torrentscheduler = current
                torrentschedulerseed = currentseed
            if self.abcparams['showschedcounters'] == '1':
                schedcounters = "[ %u/%u ] [ %u/%u ] " % (torrentscheduler, self.getActivityMaxDown(), torrentschedulerseed, self.getActivityMaxSeed())
                if self.utility.win7fix:
                    win7fix_sc += 1
            else:
                schedcounters = ""

        if self.abcparams['showschedcounters'] != '0' :
            maxrm = 0
            showrmcounters = False
            if self.abcparams['drm'] == '1':
                maxrm += self.utility.drmmaxtorrent
                showrmcounters = True
            if self.abcparams['urm'] == '1':
                maxrm += self.utility.urmmaxtorrent
                showrmcounters = True
            if showrmcounters:
                schedcounters += "[ +%u/%u ] " % (torrentrm, maxrm)
                if self.utility.win7fix:
                    win7fix_sc += 1

        widthhaschanged = self.sb.setText(self.localize('sb_R') + '%u ' % runningcounter, 2)
        widthhaschanged |= self.sb.setText(self.localize('sb_W') + '%u~%u/%u '
                           % (self.torrentdownloading, self.currentproc - self.torrentinactive - self.torrentpaused, mns), 3)
        widthhaschanged |= self.sb.setText(self.localize('sb_S') + '%u~%u/%u '
                           % (self.torrentseeding, self.currentprocseed - self.torrentinactiveseed - self.torrentpausedseed, mnss), 4)
        widthhaschanged |= self.sb.setText(self.localize('sb_P') + '%u ' % pausedcounter, 5)
        widthhaschanged |= self.sb.setText(self.localize('sb_F') + '%u/%u ' % (self.torrentfinished, self.torrentavailable), 6)
        if self.abcparams['showschedcounters'] == '1' \
           or self.abcparams['showschedcounters'] == '2' and (self.abcparams['drm'] == '1' or self.abcparams['urm'] == '1'):
            widthhaschanged |= self.sb.setText(schedcounters, 7, False, win7fix_sc)

        if widthhaschanged:
            self.sb.setWidths()

    def updateDownVolWindow(self, window):
        self.downvolcountdur = window
        self.downvolmeasure.setWindow(window)

    def updateUpVolWindow(self, window):
        self.upvolcountdur = window
        self.upvolmeasure.setWindow(window)

    def maxUploadRate(self):
        if self.abcparams['urmdynmaxuprate'] == "0":
            # No downloading if the only one is torrentwaitingforleechers 
            if self.abcparams['nodownisinactive'] == "1" and self.currentproc - self.torrentinactive - self.torrentpaused > 0 \
               or self.abcparams['nodownisinactive'] == "0" and self.torrentdownloading and (self.torrentdownloading != 1 or not self.torrentwaitingforleechers or self.torrentwaitingforleechers.complete):
                # Static overall maximum up rate when downloading
                return self.utility.maxup        
            # Static overall maximum up rate when seeding
            return self.utility.maxseedup
        # Dynamic maximum download rate, computed from ABC download and upload rates so that ABC stays
        # below the max system download rate it's been granted
        return self.dynmaxuprate

    def computeExtraTorrents(self):
        if self.abcparams['activityextra'] == '1':
            if self.abcparams['trigwhenfinishseed'] == "1":
                self.extraglob = self.currentproc + self.currentprocseed \
                                 - self.torrentonhold - self.torrentonholdseed \
                                 - self.getMaxNumSimGlobal() - self.torrentinactive - self.torrentinactiveseed
                self.extraseed = self.currentprocseed - self.torrentonholdseed - self.maxnumsimseed - self.torrentinactiveseed
            else:
                self.extraseed = self.currentprocseed - self.torrentonholdseed - self.getMaxNumSimSeed() - self.torrentinactiveseed
            self.extradown = self.currentproc - self.torrentonhold - self.maxnumsim - self.torrentinactive
        else:
            if self.abcparams['trigwhenfinishseed'] == "1":
                self.extraglob = self.currentproc + self.currentprocseed \
                                 - self.torrentonhold - self.torrentonholdseed \
                                 - self.getMaxNumSimGlobal()
                self.extraseed = self.currentprocseed - self.torrentonholdseed - self.maxnumsimseed
            else:
                self.extraseed = self.currentprocseed - self.torrentonholdseed - self.getMaxNumSimSeed()
            self.extradown = self.currentproc - self.torrentonhold - self.maxnumsim

    def resetDRMTimes(self):
        self.drm_time1 = 0.
        self.drm_time2 = 0.

    def downloadRateMaximizer(self, now):
        if self.abcparams['trigwhenfinishseed'] == "1":
            if self.extraglob < 0:
                self.resetDRMTimes()
                return
        elif self.extradown < 0:
            self.resetDRMTimes()
            return
        # Check if PauseAll toggle button is pushed
        if self.parent.pauseall:
            self.resetDRMTimes()
            return
        # Check to see if a new torrent must be started
        maxdownloadrate = self.utility.maxdown
        if self.abcparams['drmtrigger'] == '1':
            # if maxdownloadrate:
                # drmtrigger = min(maxdownloadrate, int(self.abcparams['drmtriggerval']))
            # else:
            drmtrigger = int(self.abcparams['drmtriggerval'])
        else:
            drmtrigger = maxdownloadrate
        if self.abcparams['drmonunlimiteddownrate'] == '1' and maxdownloadrate == 0 and self.abcparams['drmtrigger'] == '0' \
           or (self.abcparams['drmtrigger'] == '1' or maxdownloadrate) and self.totaldownload < drmtrigger - self.utility.drmthresholdstart:
            self.drm_time2 = 0.
            # If a torrent is currently checking or waiting or allocating, no torrent will be started except if it's on-hold
            if self.torrentcheckingwaiting and not self.torrentonhold:
                self.drm_time1 = 0.
                return
            # If option drmstoponpause is set and at least one torrent is paused, no torrent will be started.
            if self.abcparams['drmstoponpause'] == '1' and self.torrentpaused + self.torrentpausedseed != self.torrentonhold + self.torrentonholdseed:
                self.drm_time1 = 0.
                return
            if self.drm_time1 == 0.:
                self.drm_time1 = now
            # If scheduler mode is "seed only" (from Preferences or toggled without on-hold torrents), no torrent will be started.
            if self.abcparams['trigwhenfinishseed'] == "2" and (self.abcparams['schedonlyonhold'] == '0' or not self.torrentonhold):
                return
            # Check if automatic mode was switched on or ABC started or all torrents were unpaused more
            # than drmstartdelay seconds ago
            if now - self.drminitialstartingtime < self.utility.drmstartdelay:
                return
            # Check if the last started torrent was started less than drmtorrentstartdelay seconds ago
            # and if the last stopped torrent was stopped less than drmtorrentstopdelay seconds ago
            if self.drmstartingtime and now - self.drmstartingtime < self.utility.drmtorrentstartdelay \
               or self.drmstoppingtime and now - self.drmstoppingtime < self.utility.drmtorrentstopdelay:
                return
            # Below max that can be started and threshold exceeded for more than drmdelay s ?
            if self.abcparams['trigwhenfinishseed'] == "1":
                belowmaxstarted = self.extradown < self.utility.drmmaxtorrent \
                                  and self.extraglob < self.utility.drmmaxtorrent + self.utility.urmmaxtorrent * (self.abcparams['urm'] == "1") \
                                  and self.currentproc + self.currentprocseed - self.torrentonhold - self.torrentonholdseed \
                                      < self.getActivityMax() + self.utility.drmmaxtorrent + self.utility.urmmaxtorrent * (self.abcparams['urm'] == "1")
            else:
                belowmaxstarted = self.extradown < self.utility.drmmaxtorrent \
                                  and self.currentproc - self.torrentonhold < self.getActivityMaxDown() + self.utility.drmmaxtorrent
            if belowmaxstarted \
               and now - self.drm_time1 > self.utility.drmdelay:
                self.drm_time1 = 0.
                # Search for the torrent to be started
                drmtorrent = None
                priority = 5
                scannedtorrentonhold = 0
                fastresume = (self.abcparams['fastresume'] == '1')
                if self.torrentonhold:
                    # On-hold torrents have the highest priority level
                    for torrent in [t for t in self.proctab if not t.complete]:
                        if torrent.status == 'onhold':
                            scannedtorrentonhold += 1
                            if torrent.priority < priority:
                                drmtorrent = torrent
                                priority = torrent.priority
                            if scannedtorrentonhold == self.torrentonhold:
                                break
                elif self.abcparams['mode'] == '1':
                    sbqueued = (self.abcparams['sbqueued'] == '1')
                    drmonhold = (self.abcparams['drmonhold'] == '1')
                    for torrent in [t for t in self.proctab if not t.complete]:
                        if (torrent.status == 'queue' or sbqueued and torrent.status == 'standby') and torrent.priority < priority \
                           and torrent.isNotLimited() and (drmonhold or fastresume or torrent.downsize < 400000000):
                            # Torrents that are not in on-hold status will be handled only in automatic mode
                            # If stopped torrent are re-enqueued (so not on-hold) and fast resume is not enabled, check
                            # if at least 400 MB have already been downloaded to skip this torrent
                            drmtorrent = torrent
                            priority = torrent.priority
                # A torrent will be started only if there's an available port or the torrent is in the on-hold state
                if drmtorrent is not None and (self.torrentonhold or self.arePortsAvailable()):
                    self.procRESUMEsingle(drmtorrent.index, fastresume)
        # Check to see if a running torrent must be queued (if the DRM has already started some)
        elif self.utility.drmthresholdstop != '-' and self.totaldownload > drmtrigger - self.utility.drmthresholdstop:
            self.drm_time1 = 0.
            # If maxdownloadrate is unlimited and the trigger is maxdownloadrate, the DRM will never stop a torrent
            if maxdownloadrate or self.abcparams['drmtrigger'] == '1':
                # If there is no extra torrent, no torrent can be queued/on-hold
                if self.abcparams['trigwhenfinishseed'] == "1":
                    if self.extraglob == 0:
                        self.drm_time2 = 0.
                        return
                elif self.extradown == 0:
                    self.drm_time2 = 0.
                    return
                if self.drm_time2 == 0.:
                    self.drm_time2 = now
                # Check if the last started torrent was started less than drmtorrentstartdelay seconds ago
                if self.drmstartingtime and now - self.drmstartingtime < self.utility.drmtorrentstartdelay:
                    return
                # Threshold exceeded for more than drmdelaystop s ?
                if now - self.drm_time2 > self.utility.drmdelaystop:
                    self.drm_time2 = 0.
                    self.haltExtraTorrentForDownload(self.getExtraTorrentsForRate())
        else:
            # The download rate is in the download rate dead band
            self.resetDRMTimes()

    def resetURMTimes(self):
        self.urm_time1 = 0.
        self.urm_time2 = 0.

    def uploadRateMaximizer(self, now):
        if self.abcparams['trigwhenfinishseed'] == "1":
            if self.extraglob < 0:
                self.resetURMTimes()
                return
        elif self.extraseed < 0:
            self.resetURMTimes()
            return
        # Check if PauseAll toggle button is pushed
        if self.parent.pauseall:
            self.resetURMTimes()
            return
        # Check to see if a new torrent must be started
        maxuploadrate = self.maxUploadRate()
        if self.abcparams['urmtrigger'] == '1':
            # if maxuploadrate:
                # urmtrigger = min(maxuploadrate, int(self.abcparams['urmtriggerval']))
            # else:
            urmtrigger = int(self.abcparams['urmtriggerval'])
        else:
            urmtrigger = maxuploadrate
        if self.abcparams['urmonunlimiteduprate'] == '1' and maxuploadrate == 0 and self.abcparams['urmtrigger'] == '0' \
           or (self.abcparams['urmtrigger'] == '1' or maxuploadrate) and self.totalupload < urmtrigger - self.utility.urmthresholdstart:
            self.urm_time2 = 0.
            # If a torrent is currently checking or waiting or allocating, no torrent will be started except if it's on-hold
            if self.torrentcheckingwaiting and not self.torrentonholdseed:
                self.urm_time1 = 0.
                return
            # If option urmstoponpause is set and at least one torrent is paused, no torrent will be started.
            if self.abcparams['urmstoponpause'] == '1' and self.torrentpaused + self.torrentpausedseed != self.torrentonhold + self.torrentonholdseed:
                self.urm_time1 = 0.
                return
            if self.urm_time1 == 0.:
                self.urm_time1 = now
            # Check if automatic mode was switched on or ABC started or all torrents were unpaused more
            # than urmstartdelay seconds ago
            if now - self.urminitialstartingtime < self.utility.urmstartdelay:
                return
            # Check if the last started torrent was started less than urmtorrentstartdelay seconds ago
            # and if the last stopped torrent was stopped less than urmtorrentstopdelay seconds ago
            if self.urmstartingtime and now - self.urmstartingtime < self.utility.urmtorrentstartdelay \
               or self.urmstoppingtime and now - self.urmstoppingtime < self.utility.urmtorrentstopdelay:
                return
            # Below max that can be started and threshold exceeded for more than urmdelay s ?
            if self.abcparams['trigwhenfinishseed'] == "1":
                belowmaxstarted = self.extraseed < self.utility.urmmaxtorrent \
                                  and self.extraglob < self.utility.urmmaxtorrent + self.utility.drmmaxtorrent * (self.abcparams['drm'] == "1") \
                                  and self.currentproc + self.currentprocseed - self.torrentonhold - self.torrentonholdseed \
                                      < self.getActivityMax() + self.utility.urmmaxtorrent + self.utility.drmmaxtorrent * (self.abcparams['drm'] == "1")
            else:
                belowmaxstarted = self.extraseed < self.utility.urmmaxtorrent \
                                  and self.currentprocseed - self.torrentonholdseed < self.getActivityMaxSeed() + self.utility.urmmaxtorrent
            if belowmaxstarted \
               and now - self.urm_time1 > self.utility.urmdelay:
                self.urm_time1 = 0.
                # Search for the torrent to be started
                urmtorrent = None
                priority = 5
                scannedtorrentonhold = 0
                fastresume = (self.abcparams['fastresume'] == '1')
                if self.torrentonholdseed:
                    # On-hold torrents have the highest priority level
                    for torrent in [t for t in self.proctab if t.complete]:
                        if torrent.status == 'onhold':
                            scannedtorrentonhold += 1
                            if torrent.priority < priority:
                                urmtorrent = torrent
                                priority = torrent.priority
                            if scannedtorrentonhold == self.torrentonholdseed:
                                break
                elif self.abcparams['mode'] == '1':
                    sbqueued = (self.abcparams['sbqueued'] == '1')
                    urmonhold = (self.abcparams['urmonhold'] == '1')
                    for torrent in [t for t in self.proctab if t.complete]:
                        if (torrent.status == 'queue' or sbqueued and torrent.status == 'standby') and torrent.priority < priority \
                           and torrent.isNotLimited() and (urmonhold or fastresume or torrent.downsize < 400000000):
                            # Torrents that are not in on-hold status will be handled only in automatic mode
                            # If stopped torrent are re-enqueued (so not on-hold) and fast resume is not enabled, check
                            # if at least 400 MB have already been downloaded to skip this torrent
                            urmtorrent = torrent
                            priority = torrent.priority
                # A torrent will be started only if there's an available port or the torrent is in the on-hold state
                if urmtorrent is not None and (self.torrentonholdseed or self.arePortsAvailable()):
                    self.procRESUMEsingle(urmtorrent.index, fastresume)
        # Check to see if a running torrent must be queued (if the URM has already started some)
        elif self.utility.urmthresholdstop != '-' and self.totalupload > urmtrigger - self.utility.urmthresholdstop:
            self.urm_time1 = 0.
            # If maxuploadrate is unlimited and the trigger is maxuploadrate, the URM will never stop a torrent
            if maxuploadrate or self.abcparams['urmtrigger'] == '1':
                # If there is no extra torrent, no torrent can be queued/on-hold
                if self.abcparams['trigwhenfinishseed'] == "1":
                    if self.extraglob == 0:
                        self.urm_time2 = 0.
                        return
                elif self.extraseed == 0:
                    self.urm_time2 = 0.
                    return
                if self.urm_time2 == 0.:
                    self.urm_time2 = now
                # Check if the last started torrent was started less than urmtorrentstartdelay seconds ago
                if self.urmstartingtime and now - self.urmstartingtime < self.utility.urmtorrentstartdelay:
                    return
                # Threshold exceeded for more than urmdelaystop s ?
                if now - self.urm_time2 > self.utility.urmdelaystop:
                    self.urm_time2 = 0.
                    self.haltExtraTorrentForUpload(self.getExtraTorrentsForRate())
        else:
            # The upload rate is in the upload rate dead band
            self.resetURMTimes()

    def haltExtraTorrentForDownload(self, extratorrents):
        # On-hold (or queue) an extra torrent selected based on the lower drop in download rate
        # or the drop matching the current overhead in download rate once set on-hold
        drmstopsman = (self.abcparams['drmstopsman'] == "1")
        extratorrentsdown = [t for t in extratorrents if not t.complete and (t.abcengine.auto or drmstopsman)]
        if not extratorrentsdown:
            return
        if self.utility.haltextraalgo == 0:
            rmtorrent = None
            rmtorrentdowngap = 1.e9
            for torrent in extratorrentsdown:
                torrentdowngap = torrent.abcengine.downrate
                if torrentdowngap < rmtorrentdowngap:
                    rmtorrent = torrent
                    rmtorrentdowngap = torrentdowngap
        else:
            downgapbelowmax = downgapbelowdead = 1.e9
            downgapabovemax = -1.e9
            torbelowmax = torbelowdead = torabovemax = None
            currentdowngap = self.utility.maxdown - self.totaldownload
            for torrent in extratorrentsdown:
                torrentdowngap = torrent.abcengine.downrate
                newdowngap = currentdowngap + torrentdowngap
                if newdowngap >= 0:
                    if newdowngap <= self.utility.drmthresholdstart:
                        if newdowngap < downgapbelowmax:
                            torbelowmax = torrent
                            downgapbelowmax = newdowngap
                    elif newdowngap < downgapbelowdead:
                        torbelowdead = torrent
                        downgapbelowdead = newdowngap
                elif newdowngap > downgapabovemax:
                    torabovemax = torrent
                    downgapabovemax = newdowngap
            if torbelowmax:
                rmtorrent = torbelowmax
            elif torabovemax:
                rmtorrent = torabovemax
            elif torbelowdead:
                rmtorrent = torbelowdead
            else:
                rmtorrent = None
        if rmtorrent is not None:
            self.totalupload -= rmtorrent.abcengine.uprate
            self.totaldownload -= rmtorrent.abcengine.downrate
            if self.totalupload < 0:
                self.totalupload = 0.
            if self.totaldownload < 0:
                self.totaldownload = 0.
            if self.abcparams['drmonhold'] == "1":
                self.procONHOLD(rmtorrent.index)
            else:
                self.procQUEUE([rmtorrent.index], schedule = False)
        return rmtorrent

    def haltExtraTorrentForUpload(self, extratorrents):
        # On-hold (or queue) an extra torrent selected based on the lower drop in upload rate
        # or the drop matching the current overhead in upload rate once set on-hold
        urmstopsman = (self.abcparams['urmstopsman'] == "1")
        extratorrentsup = [t for t in extratorrents if t.complete and (t.abcengine.auto or urmstopsman)]
        if not extratorrentsup:
            return
        if self.utility.haltextraalgo == 0:
            rmtorrent = None
            rmtorrentupgap = 1.e9
            schedalwayshalt = (self.abcparams['scheddonthalt'] == '0')
            scheddonthaltuprategap = int(self.abcparams['scheddonthaltuprategap'])
            for torrent in extratorrentsup:
                torrentupgap = self.getTorrentUprateGap(torrent)
                if (schedalwayshalt or torrentupgap <= scheddonthaltuprategap) and torrentupgap < rmtorrentupgap:
                    rmtorrent = torrent
                    rmtorrentupgap = torrentupgap
        else:
            upgapbelowmax = upgapbelowdead = 1.e9
            upgapabovemax = -1.e9
            torbelowmax = torbelowdead = torabovemax = None
            currentupgap = self.maxUploadRate() - self.totalupload
            scheddonthaltupgap = int(self.abcparams['scheddonthaltuprategap'])
            schedalwayshalt = (self.abcparams['scheddonthalt'] == '0')
            for torrent in extratorrentsup:
                torrentupgap = self.getTorrentUprateGap(torrent)
                if schedalwayshalt or torrentupgap <= scheddonthaltupgap :
                    newupgap = currentupgap + torrentupgap
                    if newupgap >= 0:
                        if newupgap <= self.utility.urmthresholdstart:
                            if newupgap < upgapbelowmax:
                                torbelowmax = torrent
                                upgapbelowmax = newupgap
                        elif newupgap < upgapbelowdead:
                            torbelowdead = torrent
                            upgapbelowdead = newupgap
                    elif newupgap > upgapabovemax:
                        torabovemax = torrent
                        upgapabovemax = newupgap
            if torbelowmax:
                rmtorrent = torbelowmax
            elif torabovemax:
                rmtorrent = torabovemax
            elif torbelowdead:
                rmtorrent = torbelowdead
            else:
                rmtorrent = None
        if rmtorrent is not None:
            self.totalupload -= rmtorrent.abcengine.uprate
            self.totaldownload -= rmtorrent.abcengine.downrate
            if self.totalupload < 0:
                self.totalupload = 0.
            if self.totaldownload < 0:
                self.totaldownload = 0.
            if self.abcparams['urmonhold'] == "1":
                self.procONHOLD(rmtorrent.index)
            else:
                self.procQUEUE([rmtorrent.index], schedule = False)
        return rmtorrent

    def getExtraTorrentsForRate(self):
        # Returns extra torrents with the lowest priority that may be queued/on-hold by DRM or URM
        if self.abcparams['trigwhenfinishseed'] == "1":
            if self.extraglob <= 0:
                return []
            torrentranges = [(self.extraglob, [t for t in self.torrentalive if t.activity])]
        else:
            if self.extraseed <= 0 and self.extradown <= 0:
                return []
            torrentranges = []
            if self.extradown > 0:
                torrentranges.append((self.extradown, [t for t in self.torrentalive if not t.complete
                                                       and t.activity]))
            if self.extraseed > 0:
                torrentranges.append((self.extraseed, [t for t in self.torrentalive if t.complete
                                                       and t.activity]))

        extratorrents = []
        # Keep only the extra torrents with lower priority from each torrent range
        for tr in torrentranges:
            for i in xrange(tr[0]):
                extratorrent = None
                extrapriority = -1
                for torrent in tr[1]:
                    if torrent.priority >= extrapriority:
                        extratorrent = torrent
                        extrapriority = torrent.priority
                if extratorrent is None:
                    break
                extratorrents.append(extratorrent)
                tr[1].remove(extratorrent)
        return extratorrents

    def getTorrentUprateGap(self, torrent):
        return self.utility.urmupfromdownA * torrent.abcengine.downrate \
               + (self.utility.urmupfromdownB + 1) * torrent.abcengine.uprate \
               - self.utility.urmupfromdownF * torrent.abcengine.uprate**2 / (torrent.abcengine.uprate + self.utility.urmupfromdownG) \
               + self.utility.urmupfromdownC * torrent.abcengine.numdownloads \
               + self.utility.urmupfromdownD * torrent.abcengine.numuploads \
               + self.utility.urmupfromdownE * torrent.abcengine.numdeaddown

    def findTorrentToBeQueuedBack(self, curtorrent = None, nbtorrent = None, torrentpool = None):
        # curtorrent (if not None) has always an activity 0 or 2
        # and so will never be returned as to be queued back
        if nbtorrent is None:
            nbt = 1
        else:
            nbt = nbtorrent
        schedstopsman = (self.abcparams['schedstopsman'] == "1")
        schedalwayshalt = (self.abcparams['scheddonthalt'] == '0')
        scheddonthaltuprategap = int(self.abcparams['scheddonthaltuprategap'])
        if self.abcparams['trigwhenfinishseed'] == "1":
            if torrentpool is None:
                torrentrange = [t for t in self.proctab if t.activity == 1 and not t.isWaiting() and not t.isCheckingOrAllocating()]
            else:
                torrentrange = [t for t in torrentpool if t.activity == 1]
        elif curtorrent is None or curtorrent.complete:
            if torrentpool is None:
                torrentrange = [t for t in self.proctab if t.activity == 1 and t.status == 'completed' and not t.isWaiting() and not t.isCheckingOrAllocating()]
            else:
                torrentrange = [t for t in torrentpool if t.activity == 1 and t.status == 'completed']
        else:
            if torrentpool is None:
                torrentrange = [t for t in self.proctab if t.activity == 1 and t.status != 'completed' and not t.isWaiting() and not t.isCheckingOrAllocating()]
            else:
                torrentrange = [t for t in torrentpool if t.activity == 1 and t.status != 'completed']
        queuedtorrents = []
        for i in xrange(nbt):
            qt = None
            qtpriority = -1
            for t in torrentrange:
                if not t.abcengine.awaken \
                   and t.priority >= qtpriority \
                   and (t.abcengine.auto or schedstopsman) \
                   and (schedalwayshalt or self.getTorrentUprateGap(t) <= scheddonthaltuprategap):
                    qt = t
                    qtpriority = t.priority
            queuedtorrents.append(qt)
            if qt is None:
                break
            torrentrange.remove(qt)
        if nbtorrent is None:
            return queuedtorrents[0]
        return queuedtorrents

    def addTorrentStandby(self, torrent):
        self.torrentstandby.append(torrent)
        tracker = torrent.getTracker()
        if tracker in self.trackerstandbycount:
            self.trackerstandbycount[tracker] += 1
        else:
            self.trackerstandbycount[tracker] = 1

    def removeTorrentStandby(self, torrent):
        self.torrentstandby.remove(torrent)
        tracker = torrent.getTracker()
        self.trackerstandbycount[tracker] -= 1
        if self.trackerstandbycount[tracker] == 0:
            del self.trackerstandbycount[tracker]

    def initRateDistrib(self, torrent, deadbandup, deadbanddn, running = True):
        if torrent.maxlocaluploadrate:
            torrent.maxuploadrate = min(deadbandup, torrent.maxlocaluploadrate)
        elif running and self.maxUploadRate():
            # Global upload is limited
            torrent.maxuploadrate = deadbandup
        else:
            # Global upload is unlimited
            torrent.maxuploadrate = 0.
        torrent.bdcalcupbase = 0.
        if torrent.maxlocaldownloadrate:
            torrent.maxdownloadrate = min(deadbanddn, torrent.maxlocaldownloadrate)
        elif self.utility.maxdown:
            # Global download is limited
            torrent.maxdownloadrate = deadbanddn
        else:
            # Global download is unlimited
            torrent.maxdownloadrate = 0.
        torrent.bdcalcdownbase = 0.

    def dispatchTorrents(self, deadbandup, deadbanddn):
        # Dispatch torrents according to download and upload rates before distributing upload and download bandwidth
        # Empty lists
        self.allactive[:] = []
        self.allactivenotseeding[:] = []
        self.toberaisedbelowmin_u[:] = []
        self.toberaisedbelowmin_d[:] = []
        self.priotoberaised_u[:] = []
        self.priotoberaised_d[:] = []
        self.toberaised_u[:] = []
        self.toberaised_d[:] = []
        self.tobeloweredbelowmin_u[:] = []
        self.tobeloweredbelowmin_d[:] = []
        self.priotobelowered_d[:] = []
        self.priotobelowered_u[:] = []
        self.tobelowered_u[:] = []
        self.tobelowered_d[:] = []
        self.nottobechangedbelowmin_u[:] = []
        self.nottobechangedbelowmin_d[:] = []
        self.prionottobechanged_u[:] = []
        self.prionottobechanged_d[:] = []
        self.nottobechanged_u[:] = []
        self.nottobechanged_d[:] = []
        self.nonextratoberaised_u[:] = []
        self.nonextratoberaised_d[:] = []

        prioritizelocalup = (self.abcparams['prioritizelocal'] == '1')
        prioritizelocaldown = (self.abcparams['prioritizelocaldown'] == '1')
        extrawillgetuploadrate = (self.abcparams['urmlowpriority'] == "0")
        extrawillgetdownloadrate = (self.abcparams['drmlowpriority'] == "0")

        # All extra torrents
        if self.abcparams['drm'] == "1" or self.abcparams['urm'] == "1":
            self.allextra[:] = self.getExtraTorrentsForRate()
        else:
            self.allextra[:] = []

        # Torrents dispatch
        for torrent in self.torrentalive:
            if not torrent.magnet:
                if torrent.abcengine.spew:
                    self.allactive.append(torrent)                        
                    # Upload
                    if torrent.maxlocaluploadrate and torrent.abcengine.uprate > torrent.maxlocaluploadrate:
                        torrent.maxuploadrate = torrent.maxlocaluploadrate
                        torrent.bdcalcupbase = torrent.abcengine.uprate
                    else:
                        torrent.maxuploadrate = -1
                        if torrent.abcengine.uprate - torrent.bdcalcupbase >= self.utility.bdcalcupth1:
                            torrent.bdcalcupbase = torrent.abcengine.uprate
                            if torrent.abcengine.uprate < self.utility.minuprate:
                                self.toberaisedbelowmin_u.append(torrent)
                            elif torrent.prioritizeup \
                                 or prioritizelocalup and torrent.maxlocaluploadrate and (extrawillgetuploadrate or not torrent in self.allextra):
                                self.priotoberaised_u.append(torrent)
                            else:
                                self.toberaised_u.append(torrent)
                        elif torrent.bdcalcupbase - torrent.abcengine.uprate >= self.utility.bdcalcupth1:
                            torrent.bdcalcupbase = torrent.abcengine.uprate
                            if torrent.abcengine.uprate < self.utility.minuprate:
                                self.tobeloweredbelowmin_u.append(torrent)
                            elif torrent.prioritizeup \
                                 or prioritizelocalup and torrent.maxlocaluploadrate and (extrawillgetuploadrate or not torrent in self.allextra):
                                self.priotobelowered_u.append(torrent)
                            else:
                                self.tobelowered_u.append(torrent)
                        else:
                            if torrent.abcengine.uprate < self.utility.minuprate:
                                self.nottobechangedbelowmin_u.append(torrent)
                            elif torrent.prioritizeup \
                                 or prioritizelocalup and torrent.maxlocaluploadrate and (extrawillgetuploadrate or not torrent in self.allextra):
                                self.prionottobechanged_u.append(torrent)
                            else:
                                self.nottobechanged_u.append(torrent)
                    # Download
                    if torrent.status != 'completed':
                        self.allactivenotseeding.append(torrent)
                        if torrent.maxlocaldownloadrate and torrent.abcengine.downrate > torrent.maxlocaldownloadrate:
                            torrent.maxdownloadrate = torrent.maxlocaldownloadrate
                            torrent.bdcalcdownbase = torrent.abcengine.downrate
                        else:
                            torrent.maxdownloadrate = -1
                            if torrent.abcengine.downrate - torrent.bdcalcdownbase >= self.utility.bdcalcdownth1:
                                torrent.bdcalcdownbase = torrent.abcengine.downrate
                                if torrent.abcengine.downrate < self.utility.mindownrate:
                                    self.toberaisedbelowmin_d.append(torrent)
                                elif torrent.prioritizedown \
                                     or prioritizelocaldown and torrent.maxlocaldownloadrate and (extrawillgetdownloadrate or not torrent in self.allextra):
                                    self.priotoberaised_d.append(torrent)
                                else:
                                    self.toberaised_d.append(torrent)
                            elif torrent.bdcalcdownbase - torrent.abcengine.downrate >= self.utility.bdcalcdownth1:
                                torrent.bdcalcdownbase = torrent.abcengine.downrate
                                if torrent.abcengine.downrate < self.utility.mindownrate:
                                    self.tobeloweredbelowmin_d.append(torrent)
                                elif torrent.prioritizedown \
                                     or prioritizelocaldown and torrent.maxlocaldownloadrate and (extrawillgetdownloadrate or not torrent in self.allextra):
                                    self.priotobelowered_d.append(torrent)
                                else:
                                    self.tobelowered_d.append(torrent)
                            else:
                                if torrent.abcengine.downrate < self.utility.mindownrate:
                                    self.nottobechangedbelowmin_d.append(torrent)
                                elif torrent.prioritizedown \
                                   or prioritizelocaldown and torrent.maxlocaldownloadrate and (extrawillgetdownloadrate or not torrent in self.allextra):
                                    self.prionottobechanged_d.append(torrent)
                                else:
                                    self.nottobechanged_d.append(torrent)
                else:
                    self.initRateDistrib(torrent, deadbandup, deadbanddn)

        # Find not extra torrents
        if self.abcparams['drm'] == "1" or self.abcparams['urm'] == "1":
            self.nonextratoberaised_u = [t for t in self.toberaised_u if t not in self.allextra]
            self.nonextratoberaised_d = [t for t in self.toberaised_d if t not in self.allextra]

            #if self.allextra and (self.nonextratoberaised_u or self.nonextratoberaised_d):
            #    print '=====================\n' + strftime('%d-%H:%M:%S')
            #    print 'ALLEXTRA :'
            #    for t in self.allextra:
            #        print t.name
            #    print 'NONEXTRATOBERAISED_U :'
            #    for t in self.nonextratoberaised_u:
            #        print t.name
            #    print 'NONEXTRATOBERAISED_D :'
            #    for t in self.nonextratoberaised_d:
            #        print t.name
            #
            #print "torrent:", torrent.index, " ; meanuprate: %.1f ; reserved: %.1f ; maxlocal: %.1f" % (uploadrate,
            #      torrent.maxuploadrate, torrent.maxlocaluploadrate)
            #print "torrent:", torrent.index, " ; meandownrate: %.1f ; reserved: %.1f ; maxlocal: %.1f" % (downloadrate,
            #      torrent.maxdownloadrate, torrent.maxlocaldownloadrate)

        #print "tobelowered_u =", [t.index for t in self.tobelowered_u]
        #print "toberaised_u =", [t.index for t in self.toberaised_u]
        #print "nottobechanged_u =", [t.index for t in self.nottobechanged_u]
        #print "tobelowered_d =", [t.index for t in self.tobelowered_d]
        #print "toberaised_d =", [t.index for t in self.toberaised_d]
        #print "nottobechanged_d =", [t.index for t in self.nottobechanged_d]

    def cyclicalTasks(self):
        try:
            if self.parent.exiting:
                return
        except:
            return

        now = clock()

        # Max volume limiter
        self.downvolcheckcycle = max(-1, (int(time()) - self.downvolchecktime) / self.downvolcountdur)
        if self.downvolcheckcycle > 0:
            self.downvolfixedwin = 0.
            self.downvolchecktime += self.downvolcheckcycle * self.downvolcountdur
            self.abcparams['downvolchecktime'] = str(self.downvolchecktime)
            self.downvolcheckcycle = 0
        elif self.downvolcheckcycle < 0:
            self.downvolfixedwin = 0.
        self.upvolcheckcycle = max(-1, (int(time()) - self.upvolchecktime) / self.upvolcountdur)
        if self.upvolcheckcycle > 0:
            self.upvolfixedwin = 0.
            self.upvolchecktime += self.upvolcheckcycle * self.upvolcountdur
            self.abcparams['upvolchecktime'] = str(self.upvolchecktime)
            self.upvolcheckcycle = 0
        elif self.upvolcheckcycle < 0:
            self.upvolfixedwin = 0.
        # Return to normal mode test
        if self.abcparams['mode'] == '2':
            self.canswitchautodown = False
            self.canswitchautoup = False
            if self.abcparams['downvolwindowtype'] == "0" \
               and self.downvolmeasure.get_rate() * self.downvolcountdur < self.downvolmax * self.downvolthreshold \
               or self.abcparams['downvolwindowtype'] == "1" \
               and (self.downvolcheckcycle == -1 or self.downvolfixedwin < self.downvolmax):
                self.canswitchautodown = True
            if self.abcparams['upvolwindowtype'] == "0" \
               and self.upvolmeasure.get_rate() * self.upvolcountdur < self.upvolmax * self.upvolthreshold \
               or self.abcparams['upvolwindowtype'] == "1" \
               and (self.upvolcheckcycle == -1 or self.upvolfixedwin < self.upvolmax):
                self.canswitchautoup = True
            if (self.abcparams['downvolcheck'] == '0' or self.canswitchautodown) \
               and (self.abcparams['upvolcheck'] == '0' or self.canswitchautoup):
                # Switch to auto mode
                self.parent.onModeToggle(mode = 1)

        self.torrentalive[:] = []
        self.torrentcheckingwaiting = False
        sbtimeout = (self.abcparams['sbtimeout'] == '1')
        sbtimeoutvalue = int(self.abcparams['sbtimeoutvalue']) * 86400
        sbtimeoutaction = (self.abcparams['sbtimeoutaction'] == '1')
        torwithmaxdownvolreached = []
        torwithmaxupvolreached = []
        maxdownvolreached = False
        if self.abcparams['downvolcheck'] == '1':
            if self.abcparams['downvolwindowtype'] == "0":
                downvol = self.downvolmeasure.get_rate() * self.downvolcountdur
            elif self.downvolcheckcycle >= 0:
                downvol = self.downvolfixedwin
            else:
                downvol = None
            if downvol is not None and downvol >= self.downvolmax:
                maxdownvolreached = True
        maxupvolreached = False
        if self.abcparams['upvolcheck'] == '1':
            if self.abcparams['upvolwindowtype'] == "0":
                upvol = self.upvolmeasure.get_rate() * self.upvolcountdur
            elif self.upvolcheckcycle >= 0:
                upvol = self.upvolfixedwin
            else:
                upvol = None
            if upvol is not None and upvol >= self.upvolmax:
                maxupvolreached = True

        for torrent in self.proctab:
            index = torrent.index
            if torrent.abcengine:
                if torrent.status == 'pause' or torrent.status == 'onhold':
                    continue
                if torrent.isWaiting() or torrent.isCheckingOrAllocating():
                    if not torrent.abcengine.onlycheck:
                        self.torrentcheckingwaiting = True
                        # Reset last starting and stopping time for DRM and URM to delay new starts
                        if torrent.complete:
                            self.urmstartingtime = now
                        else:
                            self.drmstartingtime = now
                        # Keep activity on
                        torrent.setActivity()
                    continue
                # Check max volumes
                if maxdownvolreached and torrent.abcengine.downrate > 0:
                    torwithmaxdownvolreached.append(index)
                if maxupvolreached and torrent.abcengine.uprate > 0:
                    torwithmaxupvolreached.append(index)
                # Set last activity time since ABC_OKC startup
                if torrent.abcengine.downrate > 0 or torrent.abcengine.uprate > 0:
                    torrent.lastabsactivity = now
                self.torrentalive.append(torrent)

        # Standby timeout
        for torrent in self.torrentstandby[:]:
            if now - torrent.sbstart >= sbtimeoutvalue and not torrent.isbeingdeleted and sbtimeout:
                index = torrent.index
                if self.abcparams['displaytimeout'] == '1':
                    torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('timeoutstandbymsg'))
                if torrent.priority == 4 or sbtimeoutaction:
                    self.procSTOP([index])
                else:
                    torrent.incDecPriority(-1)
                    self.procQUEUE([index])

        # Check resumed standby torrent waiting for peers to estimate #copy and peers progress
        if self.torrentwaitingforleechers and self.torrentwaitingforleechers in self.torrentalive:
            if self.torrentwaitingforleechers.abcengine.sbwaitingforleechers > 1:
                if not self.torrentwaitingforleechers.magnet:
                    sharingopportunity = False
                    if self.torrentwaitingforleechers.abcengine.downrate or self.torrentwaitingforleechers.abcengine.uprate:
                        sharingopportunity = True
                    elif self.torrentwaitingforleechers.complete:
                        if self.torrentwaitingforleechers.peer != self.utility.notrackeranswermarker \
                           and self.torrentwaitingforleechers.peer != self.utility.notrackermarker \
                           and int(self.torrentwaitingforleechers.peer) > 0:
                            sharingopportunity = True
                    elif self.torrentwaitingforleechers.abcengine.spew \
                         and self.torrentwaitingforleechers.progress != -1 and self.torrentwaitingforleechers.numcopy != '?':
                        allpeershavegotbitfield = True
                        for peer in self.torrentwaitingforleechers.abcengine.spew:
                            if peer['completed'] is None:
                                allpeershavegotbitfield = False
                                break
                        if allpeershavegotbitfield:
                            numcopy = float(self.torrentwaitingforleechers.numcopy)
                            if abs(numcopy - self.torrentwaitingforleechers.progress / 100) > 0.001:
                                sharingopportunity = True
                            else:
                                for peer in self.torrentwaitingforleechers.abcengine.spew:
                                    if abs(numcopy - peer['completed']) > 0.001:
                                        sharingopportunity = True
                                        break
                        else:
                            for peer in self.torrentwaitingforleechers.abcengine.spew:
                                if peer['completed'] is not None and abs(self.torrentwaitingforleechers.progress / 100 - peer['completed']) > 0.001:
                                    sharingopportunity = True
                                    break
                if not self.torrentwaitingforleechers.magnet and sharingopportunity:
                    # If another torrent can be queued back, stay resumed to share and leave standby status
                    # Search for another torrent to be queued
                    if (self.abcparams['mode'] != '1' or self.isSchedulerOverrun(self.torrentwaitingforleechers)) \
                       and self.torrentwaitingforleechers.abcengine.auto:
                        if self.abcparams['mode'] != '1':
                            queuedtorrent = None
                        else:
                            queuedtorrent = self.findTorrentToBeQueuedBack(self.torrentwaitingforleechers, torrentpool = self.torrentalive)
                        if queuedtorrent is None:
                            # Update sbstart because there was potential activity that could not be started
                            self.torrentwaitingforleechers.sbstart = now
                            if now - self.torrentwaitingforleechers.abcengine.sbwaitingforleechers > self.utility.sbwaitleechers:
                                # No torrent can be queued or on-hold, return to standby
                                self.procSTANDBY([self.torrentwaitingforleechers.index])
                            # The torrent that is checking for sharing opportunities is not considered as an alive torrent
                            else:
                                self.torrentalive.remove(self.torrentwaitingforleechers)
                        else:
                            self.procQUEUE([queuedtorrent.index], schedule = False)
                            self.torrentwaitingforleechers.abcengine.sbwaitingforleechers = 0
                            self.torrentwaitingforleechers.abcengine.awaken = True
                            if self.torrentwaitingforleechers.complete:
                                self.urmstartingtime = now
                            else:
                                self.drmstartingtime = now
                            self.torrentwaitingforleechers.setActivity(1)
                            if self.torrentwaitingforleechers.complete:
                                self.currentprocseed += 1
                            else:
                                self.currentproc += 1
                            self.updateRunningTorrentCounters()
                            self.torrentwaitingforleechers = None
                            self.standbychecking = False
                    else:
                        self.torrentwaitingforleechers.abcengine.sbwaitingforleechers = 0
                        self.torrentwaitingforleechers.abcengine.awaken = True
                        if self.torrentwaitingforleechers.complete:
                            self.urmstartingtime = now
                        else:
                            self.drmstartingtime = now
                        self.torrentwaitingforleechers.setActivity(1)
                        if self.torrentwaitingforleechers.complete:
                            self.currentprocseed += 1
                        else:
                            self.currentproc += 1
                        self.updateRunningTorrentCounters()
                        self.torrentwaitingforleechers = None
                        self.standbychecking = False
                elif now - self.torrentwaitingforleechers.abcengine.sbwaitingforleechers > self.utility.sbwaitleechers:
                    # Return to standby status
                    self.procSTANDBY([self.torrentwaitingforleechers.index])
                # The torrent that is checking for sharing opportunities is not considered as an alive torrent
                else:
                    self.torrentalive.remove(self.torrentwaitingforleechers)
            else:
                self.torrentalive.remove(self.torrentwaitingforleechers)

        # Max volume limiter
        # Queue back torrents if needed
        if torwithmaxdownvolreached:
            if self.abcparams['mode'] == '1':
                self.parent.onModeToggle(mode = 2)
            self.procQUEUE(torwithmaxdownvolreached)
        if torwithmaxupvolreached:
            if self.abcparams['mode'] == '1':
                self.parent.onModeToggle(mode = 2)
            self.procQUEUE(torwithmaxupvolreached)

        # Check complete and shut down torrents
        self.checkCompleteAndShutDownForClosing(now)

        # Manage progress dialog to wait for closing
        if self.abouttoclose:
            dialogprogress = min(60, now - self.abouttoclose)
            keepgoingforclosing = self.delacdlg.Update(dialogprogress)[0]
            if dialogprogress < 60:
                if not keepgoingforclosing:
                    self.delacdlg.Destroy()
                    if self.abcparams['rssautograb'] == '1':
                        self.parent.rsspanel.startAllTimers()
                    self.abouttoclose = 0
            else:
                self.delacdlg.Destroy()
                self.frame.onMenuExit(silent = True, shutdown = self.abouttoshutdown)
                return

        # Check standby torrents
        if self.torrentstandby and not self.standbychecking and self.abcparams['mode'] == '1':
            torrentrange = None
            if self.abcparams['trigwhenfinishseed'] == "1":
                if self.getMaxNumSimGlobal() > 0:
                    if self.maxnumsim == 0:
                        torrentrange = [t for t in self.torrentstandby if t.complete]
                    elif self.maxnumsimseed == 0:
                        torrentrange = [t for t in self.torrentstandby if not t.complete]
                    else:
                        torrentrange = self.torrentstandby[:]
            elif self.abcparams['trigwhenfinishseed'] == "0":
                if self.maxnumsim + self.getMaxNumSimSeed() > 0:
                    if self.maxnumsim == 0:
                        torrentrange = [t for t in self.torrentstandby if t.complete]
                    elif self.getMaxNumSimSeed() == 0:
                        torrentrange = [t for t in self.torrentstandby if not t.complete]
                    else:
                        torrentrange = self.torrentstandby[:]
            elif self.getMaxNumSimSeed() > 0:
                torrentrange = [t for t in self.torrentstandby if t.complete]

            if torrentrange:
                sbscrapeperiod = {}
                for tr in self.trackerstandbycount:
                    if tr:
                        sbscrapeperiod[tr] = max(self.utility.sbscrape, self.trackerstandbycount[tr] * self.utility.sbscrapeperiod)
                    else:
                        sbscrapeperiod[tr] = self.utility.sbscrape
                for t in torrentrange:
                    if not t.isbeingdeleted and not t.scraping \
                       and now - t.standbylastscrape >= sbscrapeperiod[t.getTracker()] \
                       and t.isNotLimited():
                        self.standbychecking = True
                        t.standbylastscrape = now
                        self.getScrapeData(t, autoscraping = True, standby = True)
                        # Move torrent to end of list to make sure every standby torrents will scrape
                        self.torrentstandby.remove(t)
                        self.torrentstandby.append(t)
                        break

        # Check activity
        # (Torrents started by checkActivity will be processed in next cycle)
        self.checkActivity(now)

        # Compute counters for all alive torrents
        self.totalupload = 0.
        self.totaldownload = 0.
        totalextradatauprate = 0.
        totalnumuploads = 0
        totalnumdownloads = 0
        totalconnections = 0
        totalnumdeaddown = 0
        for torrent in self.torrentalive:
            self.totalupload += torrent.abcengine.uprate
            self.totaldownload += torrent.abcengine.downrate
            totalextradatauprate += torrent.abcengine.extradatauprate
            totalnumuploads += torrent.abcengine.numuploads
            totalnumdownloads += torrent.abcengine.numdownloads
            totalconnections += torrent.abcengine.numconnections
            totalnumdeaddown += torrent.abcengine.numdeaddown

        #print "Total up rate :", self.totalupload, "; Total down rate :", self.totaldownload

        # URM (Compute dynamic ABC max upload rate)
        if self.abcparams['urmdynmaxuprate'] == "1":
            # Computing of the last dynamic upload rate (meaned over the 5 last values) based on max system upload rate
            # granted to ABC and the down and up rates and the number of connections
            if self.utility.maxsysup:
                self.dynmaxuprate = self.utility.maxsysup \
                                    - self.utility.urmupfromdownA * self.totaldownload \
                                    - self.utility.urmupfromdownB * self.totalupload \
                                    + self.utility.urmupfromdownF * self.totalupload**2 / (self.totalupload + self.utility.urmupfromdownG) \
                                    - self.utility.urmupfromdownC * totalnumdownloads \
                                    - self.utility.urmupfromdownD * totalnumuploads \
                                    - self.utility.urmupfromdownE * totalnumdeaddown \
                                    - totalextradatauprate
                if self.dht:
                    self.dynmaxuprate -= self.dhtuprate / 1024
                if self.dynmaxuprate < self.utility.minuprate:
                    self.dynmaxuprate = self.utility.minuprate
            else:
                self.dynmaxuprate = 0.

        # DRM (start or stop extra torrents)
        # (Torrents started by DRM will be processed in next cycle)
        if self.abcparams['drm'] == "1":
            self.downloadRateMaximizer(now)

        # URM (start or stop extra torrents)
        # (Torrents started by URM will be processed in next cycle)
        if self.abcparams['urm'] == "1":
            self.uploadRateMaximizer(now)

        # Distribute available upload and download bandwidth to alive torrents
        self.distributeBandwidth(now)

        # Build strings to show status bar and minimize icon counters 
        maxdownloadrate = self.utility.maxdown
        if maxdownloadrate != 0:
            downloadspeed = ('%u/%u ' % (self.totaldownload, maxdownloadrate))
        else:
            downloadspeed = ('%u ' % self.totaldownload)
        maxuploadrate = self.maxUploadRate()
        if maxuploadrate != 0:
            uploadspeed = ('%u/%u ' % (self.totalupload, maxuploadrate))
        else:
            uploadspeed = ('%u ' % self.totalupload)
        win7fix_d = self.utility.win7fix_kbps
        dlstring = self.localize('sb_DL') + downloadspeed + self.utility.kbps + ' '
        if self.abcparams['displaydownvol'] == '1':
            if self.downvolmax:
                if self.abcparams['downvolwindowtype'] == "0":
                    dlstring += '%.0f%% ' % (self.downvolmeasure.get_rate() * self.downvolcountdur / self.downvolmax * 100)
                else:
                    dlstring += '%.0f%% ' % (self.downvolfixedwin / self.downvolmax * 100)
            else:
                dlstring += '100% '
            if self.utility.win7fix:
                win7fix_d -= 1
        win7fix_u = self.utility.win7fix_kbps
        ulstring = self.localize('sb_UL') + uploadspeed + self.utility.kbps + ' '
        if self.abcparams['displayupvol'] == '1':
            if self.upvolmax:
                if self.abcparams['upvolwindowtype'] == "0":
                    ulstring += '%.0f%% ' % (self.upvolmeasure.get_rate() * self.upvolcountdur / self.upvolmax * 100)
                else:
                    ulstring += '%.0f%% ' % (self.upvolfixedwin / self.upvolmax * 100)
            else:
                ulstring += '100% '
            if self.utility.win7fix:
                win7fix_u -= 1

        # Update value in minimize icon
        if self.frame.taskbariconized:
            tt = self.localize('schedmodeseedtray')
            if tt and self.abcparams['trigwhenfinishseed'] == '2':
                tt += '\n'
            else:
                tt =''
            self.frame.tbicon.SetIcon(self.frame.icon,
                                      tt + self.localize('totaldlspeed') + downloadspeed
                                      + self.localize('totalulspeed') + uploadspeed)

        # Update values in status bar
        widthhaschanged = self.sb.setText(self.localize('sb_CX') + str(totalconnections) + ' ', 8)
        widthhaschanged |= self.sb.setText(dlstring, 9, False, win7fix_d)
        widthhaschanged |= self.sb.setText(ulstring, 10, True, win7fix_u)
        if widthhaschanged:
            self.sb.setWidths()

        # Update values in advanced settings
        if self.frame.abcoptiondlg is not None and self.frame.abcoptiondlg.advancedMenuBox is not None:
            rbgs = self.frame.abcoptiondlg.advancedMenuBox.readbufferglobalsize
            if PIECEREADBUFFERPOOL.overloaded:
                rbgs.SetForegroundColour('Red')
            rbgs.SetLabel('%.1f' % (PIECEREADBUFFERPOOL.totalSize() / 1048576.))

        # Clear old cache data
        if self.utility.expirecachedata:
            if self.expirecachecounter >= self.utility.expirecacherate:
                self.deleteOldCacheData()
                self.expirecachecounter = 0
            else:
                self.expirecachecounter += 1

        # Warn active torrents of possible network disconnection
        if not self.torrentalive:
            self.lastinactivitytime = now
            self.networkdisconnection = -now
        elif now - self.lastinactivitytime > 300:
            if self.totaldownload == self.totalupload == 0:
                if self.networkdisconnection < 0:
                    self.networkdisconnection = now
                if self.networkdisconnection > 0 and now - self.networkdisconnection > 30:
                    for torrent in self.torrentalive:
                        if not torrent.abcengine.networkdisconnected:
                            torrent.abcengine.networkdisconnected = True
                    self.networkdisconnection = 0
            elif self.networkdisconnection >= 0:
                self.networkdisconnection = -now

        # Start timer for next run
        if not self.parent.exiting:
            self.timercyclicaltasks = Timer(self.utility.cyclicaltasksrate, self.invokeLater, [self.cyclicalTasks])
            self.timercyclicaltasks.start()

    def onInvoke(self, event):
        event.func(*event.args, **event.kwargs)

    def invokeLater(self, func, args = [], kwargs = {}):
        wx.PostEvent(self, SchEvent(func, args, kwargs))

    def updateBackgroundColour(self, index):
        # Update list background for index
        for i in xrange(index, self.list.GetItemCount()):
            if i % 2:
                self.list.SetItemBackgroundColour(i, self.utility.colstripes)
            else:
                self.list.SetItemBackgroundColour(i, self.utility.colsyswin)

    def updateStripesColour(self, clear = False):
        if clear:
            for i in xrange(1, self.list.GetItemCount(), 2):
                self.list.SetItemBackgroundColour(i, self.utility.colsyswin)
        else:
            for i in xrange(1, self.list.GetItemCount(), 2):
                self.list.SetItemBackgroundColour(i, self.utility.colstripes)

    def deleteOldCacheData(self):
        exptime = time() - self.utility.expirecachedata
        dir_datacache = path.join(self.utility.datapath, 'datacache')
        dir_piececache = path.join(self.utility.datapath, 'piececache')
        for f in listdir(dir_datacache):
            p = path.join(dir_datacache, f)
            try:
                if path.getmtime(p) < exptime:
                    remove(p)
                    # Remove piece cache
                    p = path.join(dir_piececache, f)
                    if path.isdir(p):
                        for f2 in listdir(p):
                            remove(path.join(p, f2))
                        rmdir(p)
            except:
                pass

    def cleanOldCache(self):
        exptime = time() - self.utility.expirecachedata
        dir_datacache = path.join(self.utility.datapath, 'datacache')
        dir_piececache = path.join(self.utility.datapath, 'piececache')
        for f in listdir(dir_datacache):
            p = path.join(dir_datacache, f)
            try:
                if path.getmtime(p) < exptime:
                    remove(p)
            except:
                pass
        for f in listdir(dir_piececache):
            p = path.join(dir_piececache, f)
            try:
                if path.getmtime(p) < exptime:
                    if path.isdir(p):
                        for f2 in listdir(p):
                            remove(path.join(p, f2))
                        rmdir(p)
            except:
                pass

    def cleanTorrentBackup(self):
        dir_torrentbackup = path.join(self.utility.datapath, 'torrentbackup')
        for f in listdir(dir_torrentbackup):
            try:
                remove(path.join(dir_torrentbackup, f))
            except:
                pass

    def procREMOVEsingle(self, index, movedata = True, caller = ''):
        torrent = self.proctab[index]

        # Move files if option set
        if movedata and torrent.magnet != 1 \
           and self.abcparams['movedataondelete'] == "1" \
           and torrent.movefolder \
           and not torrent.destIsMove():
            onlyfileisnamelist = False
            completedest = self.utility.completePath(torrent.dest)
            if path.exists(completedest):
                if not torrent.isSingleFile(completedest):
                    if listdir(completedest) == [self.abcparams['originalfilenameslist']]:
                        onlyfileisnamelist = True
                if not onlyfileisnamelist:
                    self.dataMovingThread(torrent, self.procREMOVEsingle2, caller)
                    return
            # If the downloaded files or folders for this torrent don't exist, ask for confirmation of deletion
            if torrent.progress > 0 and caller != 'web' and caller != 'queue':
                self.parent.moveremove = False
                self.parent.nbselected.SetLabel(str(self.list.GetSelectedItemCount()))
                dlg = wx.MessageDialog(self.parent.windowparent,
                                       self.localize('torrentname') + ' ' + torrent.name + '\n' + self.localize('movefilesnotfound'),
                                       self.localize('abcokcwarning'), wx.OK | wx.CANCEL)
                result = dlg.ShowModal()
                dlg.Destroy()
                self.parent.moveremove = True
                if result != wx.ID_OK:
                    torrent.isbeingdeleted = False
                    return
            if onlyfileisnamelist:
                try:
                    remove(path.join(completedest, self.abcparams['originalfilenameslist']))
                    rmdir(completedest)
                except:
                    pass

        self.procREMOVEsingle2(torrent, caller)

    def procREMOVEsingle2(self, torrent, caller, movedataerror = None):
        # Code run after torrent data were moved or when moving is not needed, when a torrent is deleted
        index = torrent.index
        if movedataerror is not None:
            self.parent.moveremove = True
        
        self.sb.SetStatusText(self.localize('deleting'))
        self.list.DeleteItem(index)

        del self.proctab[index]

        for i in xrange(index, len(self.proctab)):
            self.updateItemIndex(i)

        if torrent.status == 'queue':
            self.torrentqueued -= 1
        elif torrent.status == 'finished':
            self.torrentfinished -= 1
        elif torrent.status == 'standby':
            self.removeTorrentStandby(torrent)
        if torrent.complete:
            self.torrentavailable -= 1
        self.updateRunningTorrentCounters()

        if torrent in self.compshut:
            self.compshut.remove(torrent)

        self.torrentloaded -= 1
        if self.sb.setText(self.localize('sb_L') + '%u ' % self.torrentloaded, 1):
            self.sb.setWidths()

        if self.abcparams['stripedlist'] == '1':
            self.updateBackgroundColour(index)
        if movedataerror is not None:
            self.parent.moveremove = False
            self.parent.nbselected.SetLabel(str(self.list.GetSelectedItemCount()))

        # Update bookmarks
        self.parent.updateBookmarks(index)

        if self.abcparams['removetorrent'] == "1":
            try:
                remove(torrent.src)
            except:
                pass
        else:
            backup = self.utility.findUniqueFileName(path.join(self.utility.datapath, 'torrentbackup', path.split(torrent.src)[1]), src = caller)
            if backup is not None:
                try:
                    renames(torrent.src, backup)
                except:
                    pass
        if not torrent.singlefile:
            try:
                remove(torrent.src + '.nam')
            except:
                pass
            try:
                remove(torrent.src + '.fns')
            except:
                pass
            try:
                remove(torrent.src + '.pri')
            except:
                pass
            try:
                remove(torrent.src + '.pro')
            except:
                pass
        try:
            remove(torrent.src + '.prs')
        except:
            pass
        # Remove BitTornado cache data
        try:
            remove(path.join(self.utility.datapath, 'datacache', hexlify(torrent.infohash)))
        except:
            pass
        piececachedir = path.join(self.utility.datapath, 'piececache', hexlify(torrent.infohash))
        try:
            if path.isdir(piececachedir):
                for f in listdir(piececachedir):
                    remove(path.join(piececachedir, f))
                rmdir(piececachedir)
        except:
            pass

        self.sb.SetStatusText("")

        if movedataerror:
            dlg = wx.MessageDialog(self.parent, self.localize('failedmovingdatawhendeleting') + '\n' + torrent.name + '\n'
                                   + self.localize('checkdata') + '\n'
                                   + self.utility.completePath(torrent.dest) + '\n' + torrent.getFinalDest(),
                                   self.localize('failedmovingdata'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()

    def procREMOVE(self, indexlist, caller = '', movedata = True):
        torrentselected = []
        for i in indexlist:
            torrentselected.append(self.proctab[i])
        scheduleneeded = False
        for torrent in torrentselected:
            index = torrent.index
            # Stop torrent if needed
            if torrent.abcengine or torrent.status == 'queue':
                if torrent.abcengine:
                    scheduleneeded = True
                self.procSTOP([index], schedule = False)
            if torrent.ongoing is not None:
                dlg = wx.MessageDialog(self.frame,
                                       self.localize('faileddeletebgallocnotover'),
                                       self.localize('abcokcwarning'), wx.OK)
                dlg.ShowModal()
                dlg.Destroy()
                continue
            torrent.isbeingdeleted = True
            self.procREMOVEsingle(index, movedata = movedata, caller = caller)
        if scheduleneeded:
            self.scheduler()

    def procREMOVEFILE(self, indexlist):
        torrentselected = []
        for i in indexlist:
            torrentselected.append(self.proctab[i])
        scheduleneeded = False
        for torrent in torrentselected:
            index = torrent.index
            # Stop torrent if needed
            if torrent.abcengine or torrent.status == 'queue':
                if torrent.abcengine:
                    scheduleneeded = True
                self.procSTOP([index], schedule = False)
            if torrent.ongoing is not None:
                dlg = wx.MessageDialog(self.frame,
                                       self.localize('faileddeletebgallocnotover'),
                                       self.localize('abcokcwarning'), wx.OK)
                dlg.ShowModal()
                dlg.Destroy()
                continue
            torrent.isbeingdeleted = True
            if torrent.magnet != 1:
                # Remove dest (only for not magnet torrents or magnets with already known dest)
                removedest = self.utility.completePath(torrent.dest)
                if path.exists(removedest):
                    self.sb.SetStatusText(self.localize('deletingdata'))
                    if torrent.isSingleFile(removedest):
                        # Single file
                        try:
                            remove(removedest)
                        except:
                            pass
                    else:
                        # Folder
                        filestoberemoved = torrent.getFileNames()
                        filestoberemoved.append(self.abcparams['originalfilenameslist'])
                        i = 0
                        for filename in filestoberemoved:
                            i += 1
                            if i % 100 == 0:
                                wx.SafeYield(onlyIfNeeded = True)
                            try:
                                remove(path.join(removedest, filename))
                            except:
                                pass
                        for p, ds, fs in walk(removedest, topdown = False):
                            for d in ds:
                                pd = path.join(p, d)
                                if not listdir(pd):
                                    try:
                                        rmdir(pd)
                                    except:
                                        pass
                        try:
                            rmdir(removedest)
                        except:
                            pass
                        
            # Remove torrent from list
            self.procREMOVEsingle(index, movedata = False)
            self.updateRunningTorrentCounters()
        if scheduleneeded:
            self.scheduler()

    def updateSingleItemStatus(self, id, disconnected = False):
        torrent = self.proctab[id]
        status = self.localize(torrent.status)
        rank = self.getRankfromID(6)
        if rank != -1:
            self.list.SetStringItem(id, rank, status)
        if disconnected:
            rank = self.getRankfromID(10)
            if rank != -1:
                self.list.SetStringItem(id, rank, self.utility.formatedRate(None, torrent.prioritizedown)[0])
            rank = self.getRankfromID(11)
            if rank != -1:
                self.list.SetStringItem(id, rank, self.utility.formatedRate(None, torrent.prioritizeup)[0])
        ranks = []
        if self.abcparams['keepeta'] == '0' \
           and torrent.status != 'standby' and torrent.status != 'onhold' and torrent.status != 'pause':
            ranks.append(self.getRankfromID(8))
        if disconnected:
            ranks.extend([self.getRankfromID(17), self.getRankfromID(20)])
        for rank in ranks:
            if rank != -1:
                self.list.SetStringItem(id, rank, "")
        if disconnected:
            for rank in [self.getRankfromID(14), self.getRankfromID(15)]:
                if rank != -1:
                    # Reset #Seed and #Peer (but keep scrape result)
                    text = self.list.GetItem(id, rank).GetText()
                    if text:
                        self.list.SetStringItem(id, rank, '0 ' + text[text.find('('):])

        self.list.SetItemTextColour(id, self.utility.colnoeng)

        if torrent.infowin is not None:
            torrent.infowin.status.SetLabel(status)
            torrent.infowin.colour.SetBackgroundColour(self.utility.colnoeng)
            torrent.infowin.colour.Refresh()
        if torrent.detailwin is not None:
            torrent.detailwin.status.SetLabel(status)
            torrent.detailwin.colour.SetBackgroundColour(self.utility.colnoeng)
            torrent.detailwin.colour.Refresh()
            if disconnected:
                torrent.detailwin.trackercolour.SetBackgroundColour(torrent.coltrackernotrunning)
                torrent.detailwin.trackercolour.Refresh()

    def delayedProcRESUMEsingle(self, torrent, standbywaitingforleechers):
        try:
            if self.parent.exiting:
                return
        except:
            return
        torrent.delayedresumetimer = None
        if torrent not in self.proctab:
            return
        index = torrent.index
        if torrent.status == 'completed':
            if standbywaitingforleechers:
                # Torrent was standby checking for leechers when it turned completed and was stopped to move data
                # Check seeders and leechers for restarting torrent
                self.procSTANDBY([index])
                if self.abcparams['mode'] == '1' and self.canBeScheduled(torrent) > 0 \
                   and torrent.peer != self.utility.notrackeranswermarker \
                   and torrent.peer != self.utility.notrackermarker and int(torrent.peer) > 0:
                    torrent.sbstart = clock()
                    self.procRESUMEsingle(index, True, fromstandby = True)
            else:
                # Search for another torrent to be queued
                if self.abcparams['mode'] != '1' or self.isSchedulerOverrun(torrent):
                    if self.abcparams['mode'] != '1':
                        queuedtorrent = None
                    else:
                        queuedtorrent = self.findTorrentToBeQueuedBack(torrent)
                    if queuedtorrent is None:
                        # No other torrent can be queued ; queue back torrent
                        self.procQUEUE([index], schedule = False)
                        return
                else:
                    queuedtorrent = None
                self.procRESUMEsingle(index, True)
                if queuedtorrent is not None:
                    self.procQUEUE([queuedtorrent.index], schedule = False)

    def dataMovingThread(self, torrent, callback = None, callbackinfo = None):
        # Thread to avoid freezing the GUI while moving data when a torrent turns completed or finished or is deleted
        self.enddatamovingflag.clear()
        self.movingthreads += 1
        torrent.displayStatus('waitingmoving')
        torrent.ismovingdata = True
        self.startqueue.append(torrent.infohash)
        thread = Thread(target = self.moveDownloadedFiles, args = [torrent, callback, callbackinfo])
        thread.daemon = False
        thread.start()

    def moveDownloadedFiles(self, torrent, callback, callbackinfo):
        # Move data to final move directory
        destfolder = self.utility.completePath(torrent.dest)
        movefolder = self.utility.completePath(torrent.movefolder)
        usesem = path.splitdrive(destfolder)[0].lower() != path.splitdrive(movefolder)[0].lower()

        if usesem:
            while torrent.infohash != self.startqueue[0] or not FILESEM.acquire(False):
                sleep(.1)
            del self.startqueue[0]
        else:
            self.startqueue.remove(torrent.infohash)
        self.invokeLater(torrent.displayStatus, ['moving'])

        error = False
        if torrent.isSingleFile(destfolder):
            # Single file
            # To prevent renames from deleting destination root if empty
            dummyname = self.utility.findUniqueFileName(path.join(path.split(destfolder)[0], 'dummy'), src = 'dum')
            if dummyname is None:
                error = True
            else:
                try:
                    file(dummyname, 'w').close()
                except:
                    error = True
                else:
                    movename = path.join(movefolder, path.split(destfolder)[1])
                    if not torrent.complete:
                        im = self.abcparams['incompletemark']
                        if im:
                            movename = movename[:-1] + im
                    try:
                        renames(destfolder, movename)
                    except:
                        error = True
        else:
            # Folder
            parentdir = path.split(destfolder)[0]
            # To prevent renames from deleting destination root if empty
            dummyname = self.utility.findUniqueFileName(path.join(parentdir, 'dummy'), src = 'dum')
            if dummyname is None:
                error = True
            else:
                try:
                    file(dummyname, 'w').close()
                except:
                    error = True
                else:
                    i = 0
                    filestobemoved = torrent.getFileNames()
                    filestobemoved.append(self.abcparams['originalfilenameslist'])
                    for filename in filestobemoved:
                        destname = path.join(destfolder, filename)
                        movename = path.join(path.join(movefolder, path.split(destfolder)[1]), filename)
                        if i != torrent.filesnumber and torrent.filesprogress is not None and torrent.filesprogress[i] != 'Done':
                            im = self.abcparams['incompletemark']
                            if im:
                                movename = movename[:-1] + im
                        if path.exists(destname):
                            if i == torrent.filesnumber or torrent.filespriority is None or torrent.filespriority[i] != -1:
                                try:
                                    renames(destname, movename)
                                except:
                                    error = True
                                    break
                            else:
                                try:
                                    remove(destname)
                                except:
                                    error = True
                                    break
                        i += 1
        try:
            remove(dummyname)
        except:
            pass

        if usesem:
            FILESEM.release()

        self.invokeLater(self.endMoveDownloadedFiles, [torrent, error, callback, callbackinfo])

    def endMoveDownloadedFiles(self, torrent, movedataerror, callback, callbackinfo):
        torrent.ismovingdata = False
        if not movedataerror:
            torrent.updateDestination(path.join(torrent.movefolder, path.split(torrent.dest)[1]))
        if callback:
            callback(torrent, callbackinfo, movedataerror)
        else:
            torrent.displayStatus()
        if movedataerror and torrent in self.proctab:
            torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('movefileserror'))
        self.movingthreads -= 1
        if not self.movingthreads:
            self.enddatamovingflag.set()

    def dataMovingToNewDestThread(self, torrent, olddestfolder, callback = None, callbackinfo = None):
        # Thread to avoid freezing the GUI while moving data to a new destination from the set destination dialog or from web
        self.enddatamovingflag.clear()
        self.movingthreads += 1
        torrent.displayStatus('waitingmoving')
        torrent.ismovingdata = True
        self.startqueue.append(torrent.infohash)
        thread = Thread(target = self.moveDownloadedFilesToNewDest, args = [torrent, olddestfolder, callback, callbackinfo])
        thread.daemon = False
        thread.start()

    def moveDownloadedFilesToNewDest(self, torrent, olddestfolder, callback, callbackinfo):
        # Move already received data to new destination from changing destination dialog
        error = False
        if path.exists(olddestfolder):
            destfolder = self.utility.completePath(torrent.dest)
            usesem = path.splitdrive(destfolder)[0].lower() != path.splitdrive(olddestfolder)[0].lower()

            if usesem:
                while torrent.infohash != self.startqueue[0] or not FILESEM.acquire(False):
                    sleep(.1)
                del self.startqueue[0]
            else:
                self.startqueue.remove(torrent.infohash)
            self.invokeLater(torrent.displayStatus, ['moving'])

            if destfolder.lower() != olddestfolder.lower():
                if torrent.isSingleFile(olddestfolder):
                    # Single file
                    # To prevent renames from deleting destination root if empty
                    dummyname = self.utility.findUniqueFileName(path.join(path.split(olddestfolder)[0], 'dummy'), src = 'dum')
                    if dummyname is None:
                        error = True
                    else:
                        try:
                            file(dummyname, 'w').close()
                        except:
                            error = True
                        else:
                            try:
                                renames(olddestfolder, destfolder)
                            except:
                                error = True
                else:
                    # Folder
                    parentdir = path.split(olddestfolder)[0]
                    # To prevent renames from deleting destination root if empty
                    dummyname = self.utility.findUniqueFileName(path.join(parentdir, 'dummy'), src = 'dum')
                    if dummyname is None:
                        error = True
                    else:
                        try:
                            file(dummyname, 'w').close()
                        except:
                            error = True
                        else:
                            filestobemoved = torrent.getFileNames()
                            filestobemoved.append(self.abcparams['originalfilenameslist'])
                            for filename in filestobemoved:
                                olddestname = path.join(olddestfolder, filename)
                                destname = path.join(destfolder, filename)
                                if path.exists(olddestname):
                                    try:
                                        renames(olddestname, destname)
                                    except:
                                        error = True
                                        break
                                else:
                                    # Delete empty dir structure
                                    dir = path.split(olddestname)[0]
                                    try:
                                        while not listdir(dir) and dir != parentdir:
                                            rmdir(dir)
                                            dir = path.split(dir)[0]
                                    except:
                                        pass
                try:
                    remove(dummyname)
                except:
                    pass

            if usesem:
                FILESEM.release()

        self.invokeLater(self.endMoveDownloadedFilesToNewDest, [torrent, error, callback, callbackinfo])

    def endMoveDownloadedFilesToNewDest(self, torrent, movedataerror, callback, callbackinfo):
        torrent.ismovingdata = False
        if callback:
            callback(torrent, callbackinfo, movedataerror)
        else:
            torrent.displayStatus()
        if movedataerror and torrent in self.proctab:
            torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('movefilesnewdesterror'))
        self.movingthreads -= 1
        if not self.movingthreads:
            self.enddatamovingflag.set()

    def procRESUMEALL(self):
        self.procQUEUE([t.index for t in self.proctab if t.status == 'stop'], schedule = False)
        self.scheduler()
    
    def procSTOPALL(self):
        self.procSTOP(xrange(len(self.proctab)), schedule = False)

    def procQUEUEALL(self):
        self.procQUEUE(xrange(len(self.proctab)), schedule = False)
        self.scheduler()

    def procQUEUERUNNING(self):
        tobequeued = []
        tobestandby = []
        for torrent in self.proctab:
            if torrent.abcengine:
                if torrent.abcengine.sbwaitingforleechers:
                    tobestandby.append(torrent.index)
                else:
                    tobequeued.append(torrent.index)
        self.procSTANDBY(tobestandby, schedule = False)
        self.procQUEUE(tobequeued, schedule = False)
        self.checkIdleForClosing()

    def procSTOP(self, indexlist, schedule = True, keepunwanted = False, torrentfailed = False):
        try:
            if self.parent.exiting:
                return
        except:
            return
        if not indexlist:
            return

        if len(indexlist) > 1:
            self.sb.SetStatusText(self.localize('stopping'))
        now = clock()

        # Start shutting down in advance all raw servers to speed up stopping of the list of torrents
        for index in indexlist:
            torrent = self.proctab[index]
            if torrent.abcengine:
                torrent.abcengine.stopBT()
        
        for index in indexlist:
            torrent = self.proctab[index]
            oldstatus = torrent.status
            if oldstatus == 'stop' \
               or oldstatus == 'completed' and not torrent.abcengine \
               or oldstatus == 'finished':
                continue

            torrent.oldstatus = oldstatus
            torrent.onlygetmetadata = 0

            if torrent.abcengine:
                onlycheck = torrent.abcengine.onlycheck
                sbwaitingforleechers = torrent.abcengine.sbwaitingforleechers
                inttrackswitched = (torrent.abcengine.networkdisconnected == 2 or torrent.abcengine.checkinginttrack)

                if torrent.isCheckingOrAllocating() or torrent.isWaiting():
                    # The file checking didn't terminate : the display in the progress column of the list must be overwritten
                    # with the last progress value
                    if torrentfailed:
                        # Progress is unknown, complete status is left unset
                        torrent.progress = -1
                        # Reset terminate event
                        torrent.termevt = torrent.prevtermevt = ""
                    else:
                        # Restore terminate event
                        torrent.termevt = torrent.prevtermevt
                    progressrank = self.getRankfromID(5)
                    if progressrank != -1:
                        self.list.SetStringItem(index, progressrank, torrent.formatedProgress())
                    if torrent.infowin:
                        torrent.infowin.totalprogress.SetLabel(torrent.formatedProgress())
                    if torrent.detailwin:
                        torrent.detailwin.totalprogress.SetLabel(torrent.formatedProgress())
                else:
                    torrent.progress = torrent.abcengine.progress
                torrent.inactivitytime, torrent.trackererrortime = torrent.abcengine.getTimeoutTimes(now)

                torrent.abcengine.stop()

                if torrent.detailwin is not None:
                    torrent.detailwin.init()
                if torrent.infowin is not None:
                    torrent.infowin.init()

                if not onlycheck:
                    if sbwaitingforleechers:
                        self.standbychecking = False
                        self.torrentwaitingforleechers = None
                    else:
                        if torrent.complete:
                            self.urmstoppingtime = now
                            self.currentprocseed -= 1
                        else:
                            self.drmstoppingtime = now
                            self.currentproc -= 1
                    if oldstatus == 'pause':
                        if torrent.complete:
                            self.torrentpausedseed -= 1
                        else:
                            self.torrentpaused -= 1
                    elif oldstatus == 'onhold':
                        if torrent.complete:
                            self.torrentpausedseed -= 1
                            self.torrentonholdseed -= 1
                        else:
                            self.torrentpaused -= 1
                            self.torrentonhold -= 1
                    else:
                        if torrent.complete:
                            self.torrentseeding -= 1
                        else:
                            self.torrentdownloading -= 1
                        if torrent in self.torrentalive:
                            self.torrentalive.remove(torrent)

                    # If in reconnection phase with tracker switching
                    if inttrackswitched:
                        torrent.switchTracker(1)

                    # Reset default tracker
                    if not torrent.exttracker and torrent.multitracker and self.abcparams['resetdeftrackonstop'] == '1':
                        torrent.setDefaultTracker('')

                    # Update counter for running torrents
                    if oldstatus != 'onhold' and oldstatus != 'pause' and not torrent.activity:
                        if torrent.complete:
                            self.torrentinactiveseed -= 1
                        else:
                            self.torrentinactive -= 1
                    torrent.updateLimiters(-1)
                    self.torrentrunning -= 1

                    torrent.setActivity(0)

                    self.updateRunningTorrentCounters()

            elif oldstatus == 'queue':
                self.torrentqueued -= 1

            elif oldstatus == 'standby':
                torrent.sbstart = 0
                self.removeTorrentStandby(torrent)

            if torrent.complete:
                if not torrent.singlefile and not keepunwanted:
                    torrent.deleteUnwanted()
                if torrent.uploadoption == 0 or torrent.seedingtimeleft > 0:
                    torrent.status = 'completed'
                else:
                    torrent.status = 'finished'
                    self.torrentfinished += 1
                    self.updateRunningTorrentCounters()
            else:
                torrent.status = 'stop'
            self.updateSingleItemStatus(index, disconnected = True)

            # Enable checkbox to rename destination in torrent rename dialog
            if self.parent.torrentrendlg and torrent == self.parent.torrentrendlg.torrent and not torrent.magnet:
                self.parent.torrentrendlg.rendestwithtor.SetValue(self.abcparams['defrentorwithdest'] == "1")
                self.parent.torrentrendlg.rendestwithtor.Enable()

        if len(indexlist) > 1:
            self.sb.SetStatusText("")
        if schedule:
            self.scheduler()

    def clearMessage(self, indexlist, rank = None):
        self.displayMessage('', indexlist, rank = rank)

    def displayMessage(self, msg, indexlist, rank = None):
        if rank is None:
            rank = self.getRankfromID(13)
        keeptermevtmsg = (self.abcparams['keeptermevtmsg'] == '1')
        msgindetails = (self.abcparams['msgindetails'] == "1")
        for index in indexlist:
            self.proctab[index].displayMessage(msg, rank = rank, keeptermevtmsg = keeptermevtmsg,
                                               msgindetails = msgindetails)

    def moveFinishedTorrentToTop(self, torrent, ensurevisible = False):
        if torrent not in self.proctab:
            return -1
        index = torrent.index
        row = 0
        nbtor = len(self.proctab)
        while row < nbtor:
            tor = self.proctab[row]
            if tor.status != 'finished' and row != index:
                break
            row += 1
        if index > row:
            if self.abcparams['sortbytime'] == '1':
                self.parent.itemMoveRow([index], row, scroll = False)
            else:
                self.parent.itemMoveRow([index], 0, scroll = False)
            if ensurevisible:
                self.list.EnsureVisible(row)
        return row

    def moveCompletedTorrentToTop(self, torrent, ensurevisible = False):
        # Slightly different algo from findMiddleRow in abc_okc.py
        # Here if the torrent is already inside the finished/completed block
        # It's not moved to the end of the block (It would reorder the
        # completed and finished torrent each time they are resumed)
        if torrent not in self.proctab:
            return -1
        index = torrent.index
        row = 0
        nbtor = len(self.proctab)
        if self.abcparams['torrentwhenfinished'] == "1":
            while row < nbtor:
                if self.proctab[row].status != 'finished' and row != index:
                    break
                row += 1
        firstcompletedrow = row
        while row < nbtor:
            torrent = self.proctab[row]
            if row != index and not torrent.complete:
                break
            row += 1
        newrow = index
        if index > row or index < firstcompletedrow - 1:
            if self.abcparams['sortbytime'] == '1':
                newrow = row
            else:
                newrow = firstcompletedrow
            self.parent.itemMoveRow([index], newrow, scroll = False)
            if ensurevisible:
                self.list.EnsureVisible(newrow)
        return newrow

    def deleteTorrent(self, torrent):
        if torrent not in self.proctab:
            return
        self.parent.commandfromlist = False
        self.parent.windowparent = self.parent
        self.parent.onRemove(None, [torrent.index], caller = 'queue')

    def procCOMPLETED(self, index, onlycheck, standbywaitingforleechers, auto):
        # Returns True if data were moved
        torrent = self.proctab[index]
        oldstatus = torrent.status
        torrent.status = 'completed'
        torrent.progress = 100.
        oldcomplete = torrent.complete
        torrent.complete = True
        if not oldcomplete:
            self.torrentavailable += 1

        if not oldcomplete or torrent.oldstatus == 'finished' or not torrent.prevtermevt:
            torrent.prevtermevt = time()
        torrent.termevt = torrent.prevtermevt
        torrent.displayMessage(torrent.termEvtMsg())

        torrent.oldstatus = oldstatus

        if not torrent.singlefile and not torrent.unwantedfiles:
            # All files for a multifile torrent are Done
            try:
                remove(torrent.src + '.pro')
            except:
                pass
            torrent.filesprogress = None

        queueback = False
        if onlycheck:
            # Wait for the checking thread to stop
            torrent.abcengine.shutdownflag.wait()
        else:
            torrent.downsize, torrent.upsize = torrent.abcengine.getDownUpSize()
            if not oldcomplete:
                self.torrentdownloading -= 1
                self.torrentseeding += 1
                if not standbywaitingforleechers:
                    self.currentproc -= 1
                    self.currentprocseed += 1
                    if not torrent.activity:
                        self.torrentinactive -= 1
                        self.torrentinactiveseed += 1
                    self.drmstoppingtime = clock()
                torrent.updateLimitersWhenSeeding()
                # The torrent will be queued back if switching to seeding caused the torrent to be limited by connections limiters
                queueback = torrent.isOverSeedingLimit()

        self.updateRunningTorrentCounters()

        if not oldcomplete \
           and self.abcparams['movedataonstatus'] == "1" and self.abcparams['movedatastatus'] == "0" \
           and torrent.movefolder \
           and path.exists(self.utility.completePath(torrent.dest)) and not torrent.destIsMove():
            # Downloaded files from completed torrent will be moved : the torrent is stopped
            # if it was not checking
            if not onlycheck:
                self.procSTOP([index], schedule = False, keepunwanted = True)
            # Move downloaded files from completed torrents
            self.dataMovingThread(torrent, self.procCOMPLETED2,
                                  (onlycheck, queueback, oldcomplete, standbywaitingforleechers, auto))
            return False

        self.procCOMPLETED2(torrent, (onlycheck, queueback, oldcomplete, standbywaitingforleechers, auto))
        return True
    
    def procCOMPLETED2(self, torrent, callbackinfo, movedataerror = None):
        # Code run after torrent data were moved, when a torrent turns completed
        onlycheck, queueback, oldcomplete, standbywaitingforleechers, auto = callbackinfo
        index = torrent.index
        seedschedule = False

        if movedataerror:
            if onlycheck:
                self.displayCheckResult(torrent, 'completed', True)
                torrent.abcengine.stopEngine()
            else:
                torrent.displayStatus()
        elif (torrent.uploadoption == 1 or torrent.uploadoption == 3) and torrent.seedingtimeleft == 0 \
             or (torrent.uploadoption == 2 or torrent.uploadoption == 3) and torrent.ratioIsReached():
            # Test if torrent is finished
            if onlycheck:
                torrent.abcengine.terminateUpload()
            elif movedataerror is not None:
                # No more abcengine
                if torrent.uploadtillcopycomplete:
                    self.checkToResumeCompletedAfterMoving(torrent, queueback, standbywaitingforleechers)
                else:
                    # No need to restart to check if at least one complete copy exists
                    # before turning finished
                    self.procFINISHED(index, False)
            else:
                torrent.abcengine.checkForTerminating(queueback = queueback)

        elif movedataerror is not None:
            self.checkToResumeCompletedAfterMoving(torrent, queueback, standbywaitingforleechers)

        elif queueback:
            # Torrent is limited and must not seed
            if standbywaitingforleechers:
                self.procSTANDBY([index])
            else:
                self.procQUEUE([index])

        elif onlycheck:
            # Display check result and stop engine
            self.displayCheckResult(torrent, 'completed')
            torrent.abcengine.stopEngine()

        elif not oldcomplete:
            if self.abcparams['trigwhenfinishseed'] == '1':
                self.urmstartingtime = clock()
            else:
                seedschedule = True

        # Switch tracker
        if torrent.exttracker:
            if self.abcparams['inttrackwhencompleted'] == "1":
                # Switch to internal tracker
                if torrent.abcengine and torrent.abcengine.checkinginttrack:
                    torrent.toggleCheckIntTrack()
                else:
                    torrent.switchTracker(0)
            elif torrent.abcengine and torrent.abcengine.checkinginttrack:
                # Switch to external tracker if checking internal tracker
                torrent.switchTracker(1)

        # Move completed torrent
        if not oldcomplete and (torrent.status != 'finished' or self.abcparams['torrentwhenfinished'] == "0") \
           and (self.abcparams['torrentwhencompleted'] == "1" or self.abcparams['torrentwhencompleted'] == "2"):
            if self.abcparams['torrentwhencompleted'] == "2":
                self.parent.onChangePrio(index = index, prio = 0)
            index = self.moveCompletedTorrentToTop(torrent, ensurevisible = onlycheck)

        if seedschedule:
            # Distinct schedulers for complete and incomplete torrents
            if torrent.isNotLimited() and (not auto or self.isNextToBeResumed(index, torrent.priority)):
                # Keep torrent running as complete and trigger the scheduler for incomplete torrents
                self.urmstartingtime = clock()
                self.scheduler()
            else:
                # Halt torrent for later scheduling with other seeding torrents
                # procSTANDBY and procQUEUE will trigger the scheduler for incomplete torrents
                if standbywaitingforleechers:
                    self.procSTANDBY([index])
                else:
                    self.procQUEUE([index])

    def checkToResumeCompletedAfterMoving(self, torrent, queueback, standbywaitingforleechers):
        if self.parent.exiting:
            # Force status because torrent is still stopped
            if standbywaitingforleechers:
                torrent.status = 'standby'
            else:
                torrent.status = 'queue'
        elif queueback:
            # Torrent is limited and must not seed
            if standbywaitingforleechers:
                self.procSTANDBY([torrent.index])
            else:
                self.procQUEUE([torrent.index])
        else:
            torrent.displayStatus()
            if self.abcparams['trigwhenfinishseed'] == "1":
                # Data were moved
                # Delayed resume to prevent some trackers from thinking they are hammered with multiple announcement
                torrent.delayedresumetimer = Timer(int(self.abcparams['completedresumedelay']), self.invokeLater,
                                                   [self.delayedProcRESUMEsingle, [torrent, standbywaitingforleechers]])
                torrent.delayedresumetimer.start()
            elif standbywaitingforleechers:
                self.procSTANDBY([torrent.index])
            else:
                self.procQUEUE([torrent.index])

    def procUNCOMPLETED(self, index, standbywaitingforleechers, onlycheck, auto):
        # When a declared complete at start torrent finally checks incomplete
        torrent = self.proctab[index]
        torrent.complete = False
        self.torrentavailable -= 1
        torrent.termevt = torrent.prevtermevt = ''
        if onlycheck:
            return
        # -1- Fix counters
        self.torrentdownloading += 1
        self.torrentseeding -= 1
        if not standbywaitingforleechers:
            self.currentproc += 1
            self.currentprocseed -= 1
            self.drmstartingtime = clock()
            if not torrent.activity:
                self.torrentinactive += 1
                self.torrentinactiveseed -= 1
        # -2- Queue or stand-by torrent when scheduler is set to only seed
        if self.abcparams['trigwhenfinishseed'] == "2":
            if standbywaitingforleechers:
                self.procSTANDBY([index], schedule = False)
            elif auto:
                self.procQUEUE([index], schedule = False)
        self.scheduler(True)

    def procFINISHED(self, index, onlycheck):
        torrent = self.proctab[index]
        oldstatus = torrent.status
        torrent.status = 'finished'
        self.torrentfinished += 1

        torrent.prevtermevt = torrent.termevt = time()
        torrent.displayMessage(torrent.termEvtMsg())

        torrent.oldstatus = oldstatus

        if not onlycheck and torrent.abcengine:
            torrent.inactivitytime, torrent.trackererrortime = torrent.abcengine.getTimeoutTimes(clock())
            if torrent.abcengine.sbwaitingforleechers:
                # In case a complete torrent runs finished while resuming from standby status
                # to estimate #copy and peers progress
                torrent.abcengine.sbwaitingforleechers = 0
                self.standbychecking = False
                self.torrentwaitingforleechers = None
            else:
                self.currentprocseed -= 1

            torrent.abcengine.stop()

            if torrent in self.torrentalive:
                self.torrentalive.remove(torrent)

            # Update counter for running torrents
            if not torrent.activity:
                self.torrentinactiveseed -= 1
            torrent.updateLimiters(-1)
            self.torrentrunning -= 1
            self.torrentseeding -= 1

            torrent.setActivity(0)

        self.updateRunningTorrentCounters()

        if not torrent.singlefile and torrent.complete:
            torrent.deleteUnwanted()

        if self.abcparams['movedataonstatus'] == "1" and self.abcparams['movedatastatus'] == "1" \
           and torrent.movefolder \
           and path.exists(self.utility.completePath(torrent.dest)) and not torrent.destIsMove():
            # Move downloaded files from finished torrent
            torrent.displayFinished()
            self.dataMovingThread(torrent, self.procFINISHED2, onlycheck)
        else:
            self.procFINISHED2(torrent, onlycheck)
        if not onlycheck:
            self.scheduler()

    def procFINISHED2(self, torrent, onlycheck, movedataerror = None):
        # Code run after torrent data were moved or when no move is needed, when a torrent turns finished
        if onlycheck:
            self.displayCheckResult(torrent, 'finished')
            torrent.abcengine.stopEngine()
        else:
            torrent.displayStatus()
            if movedataerror is None:
                torrent.displayFinished()

        if self.abcparams['torrentwhenfinished'] == "1":
            # Move torrent
            self.moveFinishedTorrentToTop(torrent, ensurevisible = onlycheck)
        elif self.abcparams['torrentwhenfinished'] == "2" and not movedataerror and not onlycheck:
            # Delete torrent except when only checking or there were errors when moving data
            self.deleteTorrent(torrent)

    def procSUPERSEED(self, indexlist, caller = ''):
        status = ''
        i = 0
        for index in indexlist:
            if index != -1:
                msg = ''
                torrent = self.proctab[index]
                if torrent.abcengine:
                    if torrent.status != 'completed':
                        msg = "can't superseed ; not completed"
                        msgfordisplay = 'errorsuperseednotcomplete'
                    elif torrent.abcengine.superseed:
                        msg = "can't superseed ; already superseeding"
                        msgfordisplay = 'errorsuperseedalreadysuperseeding'
                    else:
                        # Enter super-seed mode
                        torrent.abcengine.superseed = True
                        torrent.abcengine.dow.set_super_seed()
                        torrent.abcengine.btstatus = 'superseeding'
                        torrent.displayStatus('superseeding')
                else:
                    msg = "can't superseed ; not running"
                    msgfordisplay = 'errorsuperseednotrunning'
                if msg:
                    status += str(i) + ':' + msg + ','
                    torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                           + self.localize('error') + ' : ' + self.localize(msgfordisplay))
            i += 1
        return status

    def procRELEASEPAUSEALL(self):
        releaselist = []
        for torrent in self.proctab:
            if torrent.status == 'pause':
                releaselist.append(torrent.index)
        self.procRESUME(releaselist, True)

    def procPAUSEALL(self):
        self.procPAUSE(xrange(len(self.proctab)))

    def procPAUSE(self, indexlist):
        try:
            if self.parent.exiting:
                return
        except:
            return
        for index in indexlist:
            torrent = self.proctab[index]
            if not torrent.abcengine:
                continue
            oldstatus = torrent.status
            if oldstatus == 'pause' or oldstatus == 'onhold':
                continue

            if not torrent.abcengine.pause():
                continue

            torrent.oldstatus = oldstatus
            torrent.status = 'pause'

            if torrent.abcengine.onlycheck:
                self.updateSingleItemStatus(index)
                continue

            if not torrent.abcengine.sbwaitingforleechers:
                if torrent.complete:
                    self.urmstoppingtime = clock()
                else:
                    self.drmstoppingtime = clock()
    
            # Stop seeding timer if needed
            if oldstatus == 'completed':
                torrent.abcengine.stopSeedingTimers()

            if torrent in self.torrentalive:
                self.torrentalive.remove(torrent)

            # Update counter for running torrents
            if torrent.complete:
                self.torrentpausedseed += 1
                self.torrentseeding -= 1
                if not torrent.activity:
                    self.torrentinactiveseed -= 1
            else:
                self.torrentpaused += 1
                self.torrentdownloading -= 1
                if not torrent.activity:
                    self.torrentinactive -= 1

            torrent.setActivity(0)

            self.updateRunningTorrentCounters()

            self.updateSingleItemStatus(index)

    def procCHECK(self, indexlist):
        for index in indexlist:
            torrent = self.proctab[index]
            if torrent.abcengine or torrent.ismovingdata:
                continue
            if torrent.magnet:
                self.displayCheckResult(torrent)
                continue
            if torrent.delayedresumetimer:
                torrent.delayedresumetimer.cancel()
                torrent.delayedresumetimer = None
            torrent.oldstatus = torrent.oldstatusbeforecheck = torrent.status
            # Disable checkbox to rename destination in torrent rename dialog
            if self.parent.torrentrendlg and torrent == self.parent.torrentrendlg.torrent:
                self.parent.torrentrendlg.rendestwithtor.Disable()
                self.parent.torrentrendlg.rendestwithtor.SetValue(False)
            if torrent.status == 'queue':
                self.torrentqueued -= 1
            elif torrent.status == 'finished':
                self.torrentfinished -= 1
            elif torrent.status == 'standby':
                self.removeTorrentStandby(torrent)
            self.updateRunningTorrentCounters()
            self.runProc(index, False, onlycheck = True)

    def procRESUMEsingle(self, index, fastresume, auto = True, reseed = False, tracker = 0, getmetadata = 0, fromstandby = False, sbwaitingforleechers = False):
        try:
            if self.parent.exiting:
                return False, ""
        except:
            return False, ""

        if index == -1:
            # Default to the last torrent in the list
            index = len(self.proctab) - 1
        torrent = self.proctab[index]
        if getmetadata == 2 and not torrent.magnet:
            return True, ""

        if torrent.ismovingdata:
            return False, "torrent is moving data"

        oldstatus = torrent.status

        # If reseeding
        if reseed:
            if torrent.abcengine:
                torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                       + self.localize('error') + ' : ' + self.localize('errorreseedrunning'))
                return False, "can't reseed ; already running"
            if not torrent.checkForReseed():
                torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                       + self.localize('error') + ' : ' + self.localize('errorreseednotcomplete'))
                return False, "can't reseed ; torrent is not complete"

        # When resuming active process other than pause or on-hold
        # Get more peers with tracker configuration set with keyboard shortcuts
        elif torrent.abcengine and oldstatus != 'pause' and oldstatus != 'onhold':
            if torrent.abcengine.onlycheck:
                return False, "torrent is checking"

            if getmetadata and torrent.magnet:
                torrent.onlygetmetadata = 1
            torrent.abcengine.auto = False
            torrent.abcengine.loadPeers()

            if torrent.exttracker and tracker == 1:
                torrent.switchTracker(0)
            else:
                mustcheckinternal = (tracker == 0 and self.utility.checkinttrack == 1
                                     or tracker == 4
                                     or tracker == 6
                                     or tracker == 8
                                     or (tracker == 2 or tracker == 7 or tracker == 9
                                         or tracker == 0 and self.utility.checkinttrack == 2)
                                     and (torrent.private or not self.dht))
                if mustcheckinternal:
                    mustcheckinternalandwait = (tracker != 0 or not torrent.complete) \
                                               and (torrent.checkinttrackwait and tracker != 8 and tracker != 9
                                                    or tracker == 6 or tracker == 7)

                if torrent.exttracker:
                    if mustcheckinternal:
                        if mustcheckinternalandwait:
                            torrent.abcengine.checkinginttrack = True
                            torrent.switchTracker(0)
                        else:
                            torrent.switchTrackerExternal()
                    else:
                        torrent.reannounce(force = True)
                elif tracker >= 2:
                    if mustcheckinternal:
                        if mustcheckinternalandwait:
                            torrent.abcengine.checkinginttrack = True
                            if torrent.detailwin:
                                torrent.detailwin.toggleTrackerBut(0, False)
                            torrent.reannounce(force = True)
                        else:
                            torrent.reannounce(force = True)
                            torrent.switchTracker(1)
                    else:
                        torrent.switchTracker(1)
                else:
                    if tracker == 1 and torrent.abcengine.checkinginttrack:
                        torrent.toggleCheckIntTrack()
                    torrent.reannounce(force = True)

            return True, "already running"

        # If there are no more reserved ports available, only paused or on-hold torrents may start
        if oldstatus != 'pause' and oldstatus != 'onhold' and not self.arePortsAvailable():
            torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                   + self.localize('error') + ' : ' + self.localize('errorresumenoport'))
            if sbwaitingforleechers:
                self.standbychecking = False
                self.torrentwaitingforleechers = None
            return False, "can't resume ; no port available"

        if fromstandby and not sbwaitingforleechers:
            # Resume was initiated by a torrent that awakes from standby status following
            # a successful scrape that shows there are seeds and peers that can share data.
            # The torrent doesn't need to wait for sharing opportunities, it starts
            # immediately as a normal torrent, so another torrent must be halted.
            # Search for another torrent to be queued
            if self.abcparams['mode'] != '1' or self.isSchedulerOverrun(torrent):
                if self.abcparams['mode'] != '1':
                    queuedtorrent = None
                else:
                    queuedtorrent = self.findTorrentToBeQueuedBack(torrent)
                if queuedtorrent is None:
                    return False, "standby can't resume ; no torrent can be halted"
            else:
                queuedtorrent = None

        if torrent.delayedresumetimer:
            torrent.delayedresumetimer.cancel()
            torrent.delayedresumetimer = None

        if getmetadata and torrent.magnet:
            torrent.onlygetmetadata = 1

        now = clock()

        # Resume for paused or on-hold states (active processes)
        if oldstatus == 'pause' or oldstatus == 'onhold':
            torrent.abcengine.unpause()

            if torrent.abcengine.onlycheck:
                torrent.status = torrent.oldstatus
                torrent.oldstatus = oldstatus
                return True, ""

            # Reset last starting and stopping time for DRM and URM
            if not torrent.isCheckingOrAllocating() and not torrent.isWaiting() and not sbwaitingforleechers:
                if torrent.complete:
                    self.urmstartingtime = now
                else:
                    self.drmstartingtime = now
            if oldstatus == 'onhold':
                torrent.abcengine.auto = auto
            torrent.setActivity(1)
            torrent.status = torrent.oldstatus
            torrent.oldstatus = oldstatus
            if torrent.complete:
                self.torrentpausedseed -= 1
                if oldstatus == 'onhold':
                    self.torrentonholdseed -= 1
                self.torrentseeding += 1
                torrent.abcengine.startSeedingTimers()
            else:
                self.torrentpaused -= 1
                if oldstatus == 'onhold':
                    self.torrentonhold -= 1
                self.torrentdownloading += 1
            self.updateRunningTorrentCounters()
            return True, ""

        # Resume for other inactive states
        if sbwaitingforleechers:
            torrent.setActivity(2)
            self.torrentwaitingforleechers = torrent
        else:
            torrent.setActivity(1)
            if torrent.complete:
                self.currentprocseed += 1
                self.urmstartingtime = now
            else:
                self.currentproc += 1
                self.drmstartingtime = now
        if not fromstandby:
            torrent.sbstart = 0

        # Disable checkbox to rename destination in torrent rename dialog
        if self.parent.torrentrendlg and torrent == self.parent.torrentrendlg.torrent and not torrent.magnet:
            self.parent.torrentrendlg.rendestwithtor.Disable()
            self.parent.torrentrendlg.rendestwithtor.SetValue(False)

        # Reset terminate event persistent message
        torrent.termevt = ''

        # Update counter for running torrents
        self.torrentrunning += 1
        if torrent.complete:
            self.torrentseeding += 1
        else:
            self.torrentdownloading += 1
        torrent.updateLimiters(1)
        if oldstatus == 'queue':
            self.torrentqueued -= 1
        elif oldstatus == 'finished':
            self.torrentfinished -= 1
        elif oldstatus == 'standby':
            self.removeTorrentStandby(torrent)
        self.updateRunningTorrentCounters()

        torrent.oldstatus = oldstatus

        self.runProc(index, fastresume, auto = auto, reseed = reseed, tracker = tracker, sbwaitingforleechers = sbwaitingforleechers)

        if fromstandby and not sbwaitingforleechers and queuedtorrent is not None:
            torrent.abcengine.awaken = True
            self.procQUEUE([queuedtorrent.index], schedule = False)

        return True, ""

    def procRESUME(self, indexlist, fastresume, auto = True, reseed = False, tracker = 0, getmetadata = 0):
        # tracker :
        # 0 : Tracker set for torrent with mode set in Preferences, 1 : INT, 2 : EXT mode 2, 3 : EXT mode 0,
        # 4 : EXT mode 1, 6 : EXT mode 1 with wait in INT, 7 : EXT mode 2 with wait in INT,
        # 8 : EXT mode 1 without wait in INT, 9 : EXT mode 2 without wait in INT
        status = ''
        i = 0
        for index in indexlist:
            if index != -1:
                ret, msg = self.procRESUMEsingle(index, fastresume, auto, reseed, tracker, getmetadata)
                if not ret:
                    status += str(i) + ':' + msg + ','
            i += 1
        return status

    def procQUEUE(self, indexlist, schedule = True, standby = False):
        try:
            if self.parent.exiting:
                return
        except:
            return
        if not indexlist:
            return

        if standby:
            if len(indexlist) > 1:
                self.sb.SetStatusText(self.localize('standingby'))
            sbqueued = (self.abcparams['sbqueued'] == '1')
        elif len(indexlist) > 1:
            self.sb.SetStatusText(self.localize('queueing'))
        now = clock()

        # Start shutting down in advance all raw servers to speed up queuing of the list of torrents
        for index in indexlist:
            torrent = self.proctab[index]
            if torrent.abcengine:
                torrent.abcengine.stopBT()

        for index in indexlist:
            torrent = self.proctab[index]
            oldstatus = torrent.status
            if standby and oldstatus == 'standby' or not standby and oldstatus == 'queue' or torrent.ismovingdata:
                continue

            if standby:
                # Standby mode
                if torrent.complete and self.utility.countsbasseed and self.abcparams['mode'] == '1':
                    # if "Count seeding time also for standby torrents" is on
                    torrent.initStandbySeedingTime(now)
                if sbqueued and torrent.abcengine:
                    torrent.incDecPriority(-1)
                torrent.status = 'standby'
            else:
                torrent.status = 'queue'
                self.torrentqueued += 1

            torrent.oldstatus = oldstatus

            if torrent.abcengine:
                onlycheck = torrent.abcengine.onlycheck
                sbwaitingforleechers = torrent.abcengine.sbwaitingforleechers
                inttrackswitched = (torrent.abcengine.networkdisconnected == 2 or torrent.abcengine.checkinginttrack)

                if torrent.isCheckingOrAllocating() or torrent.isWaiting():
                    # The file checking didn't terminate : the display in the progress column of the list must be overwritten
                    # with the last progress value
                    # Restore terminate event
                    torrent.termevt = torrent.prevtermevt
                    progressrank = self.getRankfromID(5)
                    if progressrank != -1:
                        self.list.SetStringItem(index, progressrank, torrent.formatedProgress())
                    if torrent.infowin:
                        torrent.infowin.totalprogress.SetLabel(torrent.formatedProgress())
                    if torrent.detailwin:
                        torrent.detailwin.totalprogress.SetLabel(torrent.formatedProgress())
                else:
                    torrent.progress = torrent.abcengine.progress
                torrent.inactivitytime, torrent.trackererrortime = torrent.abcengine.getTimeoutTimes(now)

                torrent.abcengine.stop()

                if torrent.detailwin is not None:
                    torrent.detailwin.init()
                if torrent.infowin is not None:
                    torrent.infowin.init()

                if not onlycheck:
                    if sbwaitingforleechers:
                        self.standbychecking = False
                        self.torrentwaitingforleechers = None
                        if standby:
                            self.addTorrentStandby(torrent)
                    else:
                        if standby:
                            self.addTorrentStandby(torrent)
                            torrent.standbylastscrape = now
                            if torrent.lastabsactivity >= torrent.sbstart:
                                torrent.sbstart = now
                        if torrent.complete:
                            self.currentprocseed -= 1
                            self.urmstoppingtime = now
                        else:
                            self.currentproc -= 1
                            self.drmstoppingtime = now
                    if oldstatus == 'pause':
                        if torrent.complete:
                            self.torrentpausedseed -= 1
                        else:
                            self.torrentpaused -= 1
                    elif oldstatus == 'onhold':
                        if torrent.complete:
                            self.torrentpausedseed -= 1
                            self.torrentonholdseed -= 1
                        else:
                            self.torrentpaused -= 1
                            self.torrentonhold -= 1
                    else:
                        if torrent.complete:
                            self.torrentseeding -= 1
                        else:
                            self.torrentdownloading -= 1
                        if torrent in self.torrentalive:
                            self.torrentalive.remove(torrent)

                    # If in reconnection phase with tracker switching
                    if inttrackswitched:
                        torrent.switchTracker(1)

                    # Reset default tracker
                    if not torrent.exttracker and torrent.multitracker and self.abcparams['resetdeftrackonstop'] == '1':
                        torrent.setDefaultTracker('')

                    # Update counter for running torrents
                    if oldstatus != 'onhold' and oldstatus != 'pause' and not torrent.activity:
                        if torrent.complete:
                            self.torrentinactiveseed -= 1
                        else:
                            self.torrentinactive -= 1
                    torrent.updateLimiters(-1)
                    self.torrentrunning -= 1

                    torrent.setActivity(0)

                    self.updateRunningTorrentCounters()

            else:
                if oldstatus == 'standby':
                    torrent.sbstart = 0
                    self.removeTorrentStandby(torrent)
                else:
                    if standby:
                        self.addTorrentStandby(torrent)
                        torrent.sbstart = torrent.standbylastscrape = now
                        if oldstatus == 'queue':
                            self.torrentqueued -= 1
                    if oldstatus == 'finished':
                        self.torrentfinished -= 1
                        self.updateRunningTorrentCounters()

            self.updateSingleItemStatus(index, disconnected = True)

            # Enable checkbox to rename destination in torrent rename dialog
            if self.parent.torrentrendlg and torrent == self.parent.torrentrendlg.torrent:
                self.parent.torrentrendlg.rendestwithtor.SetValue(self.abcparams['defrentorwithdest'] == "1")
                self.parent.torrentrendlg.rendestwithtor.Enable()

        if len(indexlist) > 1:
            self.sb.SetStatusText("")
        if schedule:
            self.scheduler()

    def procSTANDBY(self, indexlist, schedule = True):
        try:
            if self.parent.exiting:
                return
        except:
            return
        self.procQUEUE(indexlist, schedule = schedule, standby = True)

    def procONHOLD(self, index):
        try:
            if self.parent.exiting:
                return
        except:
            return
        # procONHOLD will keep the torrent running with an engine (as in the Pause state)
        torrent = self.proctab[index]
        if not torrent.abcengine:
            return
        oldstatus = torrent.status
        if oldstatus == 'onhold' or oldstatus == 'pause':
            return

        if not torrent.abcengine.pause():
            return

        torrent.oldstatus = oldstatus
        torrent.status = 'onhold'

        # Stop seeding timer if needed
        if oldstatus == 'completed':
            torrent.abcengine.stopSeedingTimers()

        if torrent in self.torrentalive:
            self.torrentalive.remove(torrent)

        # Update counter for running torrents
        if torrent.complete:
            self.torrentpausedseed += 1
            self.torrentonholdseed += 1
            self.torrentseeding -= 1
            if not torrent.activity:
                self.torrentinactiveseed -= 1
            self.urmstoppingtime = clock()
        else:
            self.torrentpaused += 1
            self.torrentonhold += 1
            self.torrentdownloading -= 1
            if not torrent.activity:
                self.torrentinactive -= 1
            self.drmstoppingtime = clock()

        torrent.setActivity(0)

        self.updateRunningTorrentCounters()

        self.updateSingleItemStatus(index)

    def addNewProc(self, src, origfilename, forceasklocation, multifile = False, caller = '', prio = None, status = 'queue',
                   movetop = False, startnow = False, downloc = None, replace = -1):
        if (caller == "web" or caller == 'rss' or caller == 'qcl') and self.abcparams['setextdefaultfolder'] == '0' and downloc is None:
            # It's impossible to add a torrent from the web service or the RSS feeder in auto mode or the command line in quiet mode
            # if there's no default download folder
            # The directory scanner checks this by itself
            return False, 'No default download folder'

        self.sb.SetStatusText(self.localize('adding'))
        if prio == None:
            prio = int(self.abcparams['defaultpriority'])
        warning = ''
        invalidtorrent = False
        try:
            metainfo_file = open(src, 'rb')
            metainfo, metainfowarning = bdecode(metainfo_file.read(), sloppy = 2)
            metainfo_file.close()
            info = metainfo['info']
            if type(info) != dict:
                raise ValueError
            pieces = info.get('pieces')
            if type(pieces) != str or len(pieces) % 20 != 0:
                raise ValueError
            piecelength = info.get('piece length')
            if type(piecelength) not in ints or piecelength <= 0:
                raise ValueError
            torrentinfohash = metainfo.get('infohash')
            if torrentinfohash is None:
                torrentinfohash = sha1(bencode(info)).digest()
                magnet = 0
            else:
                magnet = 1
        except:
            invalidtorrent = True

        if not invalidtorrent:
            if replace >= 0:
                if self.abcparams['keepmagnetname'] == '1':
                    torrentname = self.proctab[replace].name
                else:
                    torrentname = origfilename
            else:
                # Check and get torrent name
                rawname = info.get('name.utf-8')
                if rawname is None:
                    rawname = info.get('name')
                if rawname is None:
                    invalidtorrent = True
                else:
                    torrentname = self.utility.decodeString(rawname).replace('|', self.abcparams['tornamevb'])

        if not invalidtorrent:
            # Search for duplicate torrent (= same hash)
            # This is done before checking further torrent validity to speed up the duplicate detection
            originaldest = None
            if replace < 0:
                rescanlist = True
                while rescanlist:
                    rescanlist = False
                    row = 0
                    for torrent in self.proctab:
                        if torrent.infohash == torrentinfohash:
                            # Found duplicate
                            if not caller:
                                result = -1
                                if not multifile or not self.duptorskipall and not self.duptoroverwriteall:
                                    dlg = AddDuplicateDialog(self.parent, -1, self.localize('duplicatetorrent'),
                                                             torrentname, origfilename, torrent, row, multifile,
                                                             pos = self.addtorrentwarningpos)
                                    self.frame.restore()
                                    result = dlg.ShowModal()
                                    if self.list.IsShown():
                                        # To force focus in list if duplicate torrent location had to switch
                                        # the main panels to show location
                                        self.list.SetFocus()
                                    self.addtorrentwarningpos = dlg.GetPosition()
                                    dlg.Destroy()
                                if result == 0 or result == 2 or multifile and self.duptoroverwriteall:
                                    if result == 2:
                                        self.duptoroverwriteall = True
                                    if torrent not in self.proctab:
                                        # Torrent is no longer in proctab
                                        # It was removed from list so rescan list for duplicate
                                        dlg = wx.MessageDialog(self.parent, self.localize('errortorrentnolongerinlist2'),
                                                               self.localize('abcokcerror'), wx.ICON_ERROR)
                                        dlg.ShowModal()
                                        dlg.Destroy()
                                        rescanlist = True
                                        break
                                    # Save destination for original torrent
                                    fileswererenamed = path.exists(torrent.src + '.nam')
                                    if path.exists(self.utility.completePath(torrent.dest)) and not fileswererenamed:
                                        originaldest = torrent.dest
                                        # to make use of originaldest when magnet will turn torrent
                                        if magnet == 1:
                                            magnet = 2
                                    # Remove original torrent
                                    if torrent.ismovingdata:
                                        dlg = wx.MessageDialog(self.parent, self.localize('failedoverwritetorrentprefix')
                                                               + origfilename + '\n' + self.localize('failedoverwritetorrent'),
                                                               self.localize('failedloadingduplicate'), wx.ICON_ERROR)
                                        dlg.ShowModal()
                                        dlg.Destroy()
                                    else:
                                        self.parent.commandfromlist = False
                                        self.parent.onRemove(indexlist = [torrent.index], movedata = fileswererenamed)
                                        break
                                elif result == 3:
                                    self.duptorskipall = True
                            try:
                                remove(src)
                            except:
                                pass
                            self.sb.SetStatusText("")
                            return False, "Duplicate torrent"
                        row += 1
                for h in self.localtorrentaddingpendinghash:
                    if h == torrentinfohash:
                        # Found duplicate with pending locally added
                        if not caller:
                            dlg = wx.MessageDialog(self.parent, self.localize('failedduptorwithpendinglocaddedprefix')
                                                   + torrentname + '\n' + self.localize('failedduptorwithpendinglocadded'),
                                                   self.localize('duplicatetorrent'), wx.ICON_ERROR)
                            self.frame.restore()
                            dlg.ShowModal()
                            dlg.Destroy()
                        try:
                            remove(src)
                        except:
                            pass
                        self.sb.SetStatusText("")
                        return False, "Duplicate torrent"

        # Go on checking for torrent validity and get main infos
        # Files
        if not invalidtorrent:
            if info.has_key('files') == info.has_key('length'):
                invalidtorrent = True
            elif info.has_key('length'):
                filelength = info['length']
                if type(filelength) not in ints or filelength < 0:
                    invalidtorrent = True
                else:
                    singlefile = True
                    filesnumber = 1
                    if magnet:
                        filesname = []
                    else:
                        filesname = [torrentname.lower()]
            else:
                filesinfo = info['files']
                if type(filesinfo) != list:
                    invalidtorrent = True
                else:
                    filelength = 0
                    singlefile = False
                    filesnumber = 0
                    filesname = []
                    for f in filesinfo:
                        if type(f) != dict:
                            invalidtorrent = True
                            break
                        if f.has_key('length'):
                            flength = f['length']
                            if type(flength) not in ints or flength < 0:
                                invalidtorrent = True
                                break
                            filelength += flength
                            filesnumber += 1
                        else:
                            invalidtorrent = True
                            break
                        rawpath = f.get('path.utf-8')
                        if rawpath is None:
                            rawpath = f.get('path')
                        if rawpath is None or type(rawpath) != list or rawpath == []:
                            invalidtorrent = True
                            break
                        filename = u''
                        for p in rawpath:
                            if str != type(p) != unicode:
                                invalidtorrent = True
                                break
                            filename = path.join(filename, self.utility.decodeString(p)).lower()
                        if invalidtorrent:
                            break
                        filesname.append(filename)

        # Trackers
        if not invalidtorrent:
            trackerlist = []
            announce_list = metainfo.get('announce-list')
            multitracker = 0
            if announce_list is None :
                announce = metainfo.get('announce')
                if announce is None:
                    # A torrent without tracker info may be loaded because it could query
                    # an extra internal or external tracker or get peers with DHT
                    defaultannounce = None
                else:
                    defaultannounce = self.utility.decodeString(announce)
                    trackerlist.append(defaultannounce.lower())
            else:
                try:
                    firstannounce = announce_list[0][0]
                except:
                    invalidtorrent = True
                else:
                    for t in announce_list:
                        multitracker += len(t)
                        for announce in t:
                            trackerlist.append(self.utility.decodeString(announce).lower())
                    defaultannounce = ''

        # Nodes
        if not invalidtorrent:
            nodes = metainfo.get('nodes')
            if nodes is not None and type(nodes) != list:
                invalidtorrent = True

        if invalidtorrent:
            # Torrent is invalid and will not be loaded
            self.sb.SetStatusText("")
            try:
                metainfo_file.close()
            except:
                pass
            if not caller and replace < 0:
                dlg = wx.MessageDialog(self.parent, self.localize('failedinvalidtorrentprefix') + origfilename + '\n'
                                       + self.localize('failedinvalidtorrent'), self.localize('invalidtorrent'),
                                       wx.ICON_ERROR)
                self.frame.restore()
                dlg.ShowModal()
                dlg.Destroy()
            try:
                remove(src)
            except:
                pass
            if replace >= 0:
                self.proctab[replace].displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                                     + self.localize('error') + ' : ' +
                                                     self.localize('failedinvalidtorrent_short'))
            return False, "Invalid torrent file"

        # Torrent is OK and will be loaded
        if not caller and replace < 0:
            self.localtorrentaddingpendinghash.append(torrentinfohash)

        if magnet and metainfo['validmagnetname']:
            magnetname = torrentname
        else:
            magnetname = ''

        # Triggered template and its default download location
        sticky = False
        if replace >= 0:
            curtemp = self.templates.get(self.proctab[replace].label)
            if curtemp:
                sticky = curtemp.sticky
        if sticky:
            triggeredtemplates = None
        else:
            triggeredtemplates = self.tempmanager.getTriggeredTemplates(magnet, torrentname.lower(), path.split(src)[1].lower(),
                                                                        trackerlist, filesname, filelength)
        defaultdownloc = None
        if triggeredtemplates:
            defaultdownloc = self.tempmanager.getTemplateDefaultDownLoc(triggeredtemplates)
        if replace >= 0:
            if defaultdownloc:
                self.proctab[replace].dest = path.join(defaultdownloc, path.split(self.proctab[replace].dest)[1])
        elif not defaultdownloc:
            defaultdownloc = self.abcparams['defaultfolder']

        # Set dest
        dest = None
        if replace >= 0:
            # Metadata are complete ; A magnet is turning torrent
            if self.proctab[replace].magnet == 1 and self.abcparams['keepmagnetname'] == '1':
                # Set to magnet name
                destname = self.proctab[replace].name.rstrip(whitespace + '.')
                fixeddest = self.utility.fixWindowsName(destname)
                if fixeddest:
                    destname = fixeddest
                dest = path.join(path.split(self.proctab[replace].dest)[0], destname)
            elif self.proctab[replace].magnet == 2:
                # Set to dir or file entered by user (set dest dialog) or by previous torrent or magnet overwrite
                dest = self.proctab[replace].dest
        elif originaldest is not None:
            # If there was a torrent overwrite the torrent dest defaults to the overwritten torrent one
            dest = originaldest

        if dest is None:
            # Set to torrent name (taken from metadata) and build dest
            # Check for valid Windows file name
            # In Windows you can't create a file or folder ending with spaces and dots
            # If you try it Windows silently creates the file or folder with these spaces and dots stripped
            destname = torrentname.rstrip(whitespace + '.')
            fixeddest = self.utility.fixWindowsName(destname)
            if fixeddest:
                destname = fixeddest
            # If replacing torrent, get the old torrent location
            if replace >= 0:
                dest = path.join(path.split(self.proctab[replace].dest)[0], destname)
            elif not caller:
                # Ask for dest folder if there's no default download folder or if the "open to non default" button is clicked
                if self.abcparams['setdefaultfolder'] == '0' or forceasklocation:
                    # Set destination location that will be used in next set destination dialog
                    if self.abcparams['setdefaultfolder'] == '0':
                        if self.abcparams['lastdestloc'] == '':
                            destloc = self.utility.userpath
                        else:
                            destloc = self.abcparams['lastdestloc']
                    else:
                        destloc = defaultdownloc
                    if magnet:
                        # Unknown file length
                        filelength_str = ''
                    else:
                        filelength_str = ' (' + self.localize('totalsize') + (' %.1f ' % (filelength / 1048576)) + self.utility.mb + ')'
                    message = self.localize('choosedirtosaveto') + filelength_str + ' :'
                    dl = wx.DirDialog(self.parent, message, destloc, style = wx.DD_NEW_DIR_BUTTON)
                    self.frame.restore()
                    result = dl.ShowModal()
                    if result != wx.ID_OK:
                        dl.Destroy()
                        try:
                            remove(src)
                        except:
                            pass
                        self.localtorrentaddingpendinghash.remove(torrentinfohash)
                        self.sb.SetStatusText("")
                        return False, ''
                    dest = dl.GetPath()
                    dl.Destroy()
                    self.abcparams['lastdestloc'] = dest
                    dest = path.join(dest, destname)
                # No asking for a dest folder necessary
                else:
                    dest = path.join(defaultdownloc, destname)
            else:
                if downloc is None:
                    dest = path.join(defaultdownloc, destname)
                else:
                    dest = path.join(downloc, destname)

        # Search for distinct torrents with the same destination
        # This is really a necessary check to avoid having several torrents writing to the same file
        dest = self.utility.completePath(dest)
        destlowercase = dest.lower()
        duplicatedestfound = False
        autorenamingfailed = False
        if replace < 0 and magnet != 1 or replace >= 0 and self.proctab[replace].magnet == 1:
            searchingforduplicatedest = True
            if self.abcparams['askduplicatedest'] == '0' or caller == 'qcl':
                destlen = len(dest)
                if singlefile:
                    if path.split(dest)[1].find('.') == -1:
                        dot = destlen
                    else:
                        dot = dest.rfind('.')
                else:
                    dot = destlen
                autorenameindex = 0
            if len(dest) > self.utility.filepathmaxlength - 4:
                # Keep at least 4 characters from original file or dir name (259 - 4 - 250 - '\')
                if len(path.split(dest)[0]) > self.utility.filepathmaxlength - 9:
                    autorenamingfailed = True
                else:
                    dest = dest[:self.utility.filepathmaxlength - 4]
                    destlowercase = dest.lower()
            newdest = dest
            while searchingforduplicatedest:
                i = 0
                duplicatedestfound = False
                for d in [self.utility.completePath(torrent.dest).lower() for torrent in self.proctab if torrent.magnet != 1] \
                    + self.localtorrentaddingpendingdest:
                    if i != replace and d == destlowercase:
                        duplicatedestfound = True
                        if (self.abcparams['askduplicatedest'] == '0' or caller == 'qcl') and not autorenamingfailed:
                            # Automatically rename destination
                            autorenameindex += 1
                            if autorenameindex == 1000:
                                autorenamingfailed = True
                            else:
                                newdest = dest[:dot] + ('_%03u' % autorenameindex) + dest[dot:]
                                destlowercase = newdest.lower()
                                break
                        if self.abcparams['askduplicatedest'] == '1' or autorenamingfailed:
                            # Ask for new destination if possible
                            if replace < 0:
                                if not caller:
                                    result = -1
                                    if not multifile or not self.dupdestskipall and not self.dupdestkeepall:
                                        dlg = AddSameDestDialog(self.parent, -1, self.localize('duplicatedestination'),
                                                                torrentname, origfilename, multifile, pos = self.addtorrentwarningpos)
                                        self.frame.restore()
                                        result = dlg.ShowModal()
                                        self.addtorrentwarningpos = dlg.GetPosition()
                                        dlg.Destroy()
                                    if result == 0:
                                        # Duplicate destination was found : Ask for new dest folder
                                        if singlefile:
                                            setdesdlgmessage2 = self.localize('duplicatechoosenewlocation')
                                        else:
                                            setdesdlgmessage2 = self.localize('duplicatechoosenewfolder')
                                        if magnet == 1 or magnet == 2 and not path.exists(newdest):
                                            desttype = None
                                        else:
                                            desttype = not magnet and singlefile or magnet == 2 and path.isfile(newdest)
                                        setdestdlg = SetDestDialog(self.parent, -1, None, self, self.parent, False, torrentname, setdesdlgmessage2,
                                                                   destloc = path.split(newdest)[0], destname = destname,
                                                                   torfilename = src, origtorname = torrentname,
                                                                   desttype = desttype,
                                                                   magnet = magnet)
                                        resultsetdestdlg = setdestdlg.ShowModal()
                                        x, y = setdestdlg.GetPositionTuple()
                                        self.abcparams['abcsetdestdlgx'], self.abcparams['abcsetdestdlgy'] = str(x), str(y)
                                        if resultsetdestdlg == wx.ID_OK:
                                            newdest, magnet = setdestdlg.getResult()
                                            if setdestdlg.getRenTorWithDest():
                                                # Update torrent name with dest name
                                                torrentname = path.split(newdest)[1]
                                            setdestdlg.Destroy()
                                            destlowercase = newdest.lower()
                                            if self.abcparams['setdefaultfolder'] == '0' or forceasklocation:
                                                self.abcparams['lastdestloc'] = path.split(newdest)[0]
                                            break
                                        setdestdlg.Destroy()
                                    elif result == 1 or result == 3 or multifile and self.dupdestkeepall:
                                        if result == 3:
                                            self.dupdestkeepall = True
                                        # A duplicate was found but we want to keep this destination nevertheless
                                        searchingforduplicatedest = False
                                        break
                                    elif result == 4:
                                        self.dupdestskipall = True
                                    self.localtorrentaddingpendinghash.remove(torrentinfohash)
                            else:
                                self.proctab[replace].displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                                                     + self.localize('error') + ' : '
                                                                     + self.localize('duplicatedestmsg_short'))
                                searchingforduplicatedest = False
                                break
                            # Cancel torrent loading
                            try:
                                remove(src)
                            except:
                                pass
                            self.sb.SetStatusText("")
                            return False, 'Duplicate destination'
                    i += 1
                if not duplicatedestfound:
                    break
            dest = newdest
        if not caller and replace < 0 and magnet != 1:
            self.localtorrentaddingpendingdest.append(destlowercase)

        # Private flag
        if info.has_key('private'):
            private = info['private']
        else:
            private = 0

        if filelength or magnet:
            progress = "0.0"
        else:
            progress = "100.0"

        deftemp = self.tempmanager.getDefaultTemplate()

        if replace < 0:
            torrent = ABCTorrent(
                                self,
                                src,
                                dest,
                                singlefile,
                                status,
                                prio,
                                torrentname,
                                deftemp.maxupload,
                                deftemp.maxuprate,
                                deftemp.maxdownrate,
                                deftemp.uploadoption,
                                deftemp.uploadtimeh,
                                deftemp.uploadtimem,
                                deftemp.uploadratio,
                                0.,
                                0.,
                                piecelength,
                                filelength,
                                torrentinfohash,
                                progress,
                                0.,
                                0.,
                                filesnumber,
                                self.abcparams['templatedefault'],
                                deftemp.shortlabel,
                                deftemp.timeoutswitch,
                                deftemp.timeoutaction,
                                deftemp.timeoutwork,
                                deftemp.timeoutworkdown,
                                deftemp.timeoutworktrack,
                                deftemp.timeoutseed,
                                deftemp.timeoutseedup,
                                deftemp.timeoutseedtrack,
                                deftemp.timeouttracker,
                                deftemp.timeoutdownload,
                                deftemp.timeoutupload,
                                deftemp.movefolder,
                                deftemp.exttracker,
                                deftemp.exttrackerurl,
                                '',
                                defaultannounce,
                                private,
                                multitracker,
                                magnet,
                                magnetname,
                                deftemp.prioritizedown,
                                deftemp.prioritizeup,
                                deftemp.checkinttrackwait,
                                deftemp.extrainttracker,
                                deftemp.inttrackerurl,
                                0.,
                                0.,
                                0.,
                                0,
                                10 * [-1],
                                0
                                )
        else:
            torrent = self.proctab[replace]
            torrent.updateData(src,
                               dest,
                               singlefile,
                               torrentname,
                               piecelength,
                               filelength,
                               filesnumber,
                               defaultannounce,
                               private,
                               multitracker,
                               magnet)

        # Check and fix names for directories and files inside the torrent
        # Compute results to be displayed after torrent initialization
        if singlefile:
            checkstatus = 0
            showdetails = False
        else:
            checkstatus, showdetails = torrent.checkFileNames(self.parent, filesinfo,
                                                              showmessage = (not caller and replace < 0))
            if checkstatus:
                if replace < 0:
                    torrent.status = 'stop'
                    startnow = False
                metainfowarning |= checkstatus << 3

        if torrent.status == 'queue':
            self.torrentqueued += 1

        # Init torrent parameters, enter torrent into list and apply actions on adding
        self.initNewTorrent(torrent, triggeredtemplates, movetop, startnow, metainfowarning, caller,
                            showdetails, replace)

        if not caller and replace < 0:
            self.localtorrentaddingpendinghash.remove(torrentinfohash)
            if magnet != 1:
                self.localtorrentaddingpendingdest.remove(destlowercase)

        self.sb.SetStatusText("")
        # Check cases when the metadata have been correctly loaded from the magnet but the torrent must be stopped
        if replace >= 0 and (duplicatedestfound and (self.abcparams['askduplicatedest'] == '1' or autorenamingfailed)
                             or checkstatus & 255):
            # A duplicate destination was found and user wants to be warned or auto-renaming failed
            # or there's a problem with file names inside the torrent
            return True, 'Stop torrent'

        return True, ''

    def initNewTorrent(self, torrent, triggeredtemplates, movetop, startnow, metainfowarning, caller,
                       showdetails, replace):
        # Add new torrent in list
        if replace < 0:
            numslot = len(self.proctab)
            self.proctab.append(torrent)
            self.list.InsertStringItem(numslot, "")
        else:
            numslot = replace
        torrent.index = numslot

        # Add info in list
        rank = self.getRankfromID(4)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.name)
        rank = self.getRankfromID(5)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.formatedProgress())
        rank = self.getRankfromID(7)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.utility.priorities_s[torrent.priority])
        rank = self.getRankfromID(8)
        if rank != -1:
            if self.abcparams['keepeta'] == "1":
                self.list.SetStringItem(numslot, rank, self.localize('etaD') + "      ?")
            else:
                self.list.SetStringItem(numslot, rank, '')
        rank = self.getRankfromID(9)
        if rank != -1:
            if torrent.magnet:
                self.list.SetStringItem(numslot, rank, '?')
            else:
                self.list.SetStringItem(numslot, rank, self.utility.formatedSize(torrent.requestedsize)[0])
        if replace < 0:
            rank = self.getRankfromID(10)
            if rank != -1:
                self.list.SetStringItem(numslot, rank, self.utility.formatedRate(None, torrent.prioritizedown)[0])
            rank = self.getRankfromID(11)
            if rank != -1:
                self.list.SetStringItem(numslot, rank, self.utility.formatedRate(None, torrent.prioritizeup)[0])
        rank = self.getRankfromID(12)
        if rank != -1:
            if torrent.requestedsize <= 0 and not torrent.magnet:
                self.list.SetStringItem(numslot, rank, '999.9%')
            else:
                self.list.SetStringItem(numslot, rank, '0.0%')                        
        if replace < 0:
            rank = self.getRankfromID(14)
            if rank != -1:
                self.list.SetStringItem(numslot, rank, '')
            rank = self.getRankfromID(15)
            if rank != -1:
                self.list.SetStringItem(numslot, rank, '')
            rank = self.getRankfromID(16)
            if rank != -1:
                self.list.SetStringItem(numslot, rank, '')
        rank = self.getRankfromID(17)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, '')
        rank = self.getRankfromID(18)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.utility.formatedSize(0)[0])
        rank = self.getRankfromID(19)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.utility.formatedSize(0)[0])
        rank = self.getRankfromID(20)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, '')
        rank = self.getRankfromID(21)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, path.split(torrent.src)[1])
        rank = self.getRankfromID(22)
        if rank != -1:
            if torrent.uploadoption == 0:
                self.list.SetStringItem(numslot, rank, "oo")
            elif torrent.uploadoption == 1:
                self.list.SetStringItem(numslot, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False))
            elif torrent.uploadoption == 2:
                self.list.SetStringItem(numslot, rank, str(torrent.uploadratio) + "%")
            else:
                self.list.SetStringItem(numslot, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False)
                                        + '/' + str(torrent.uploadratio) + "%")
        rank = self.getRankfromID(23)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.shortlabel)
        rank = self.getRankfromID(24)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.label)
        rank = self.getRankfromID(25)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.activityMarker())
        rank = self.getRankfromID(26)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, torrent.getRebuiltDest())
        rank = self.getRankfromID(27)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.utility.time_value(0, seconds = False, maxvalue = False))
        rank = self.getRankfromID(28)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.utility.time_value(0, seconds = False, maxvalue = False))

        if replace < 0:
            self.torrentloaded += 1
            if self.sb.setText(self.localize('sb_L') + '%u ' % self.torrentloaded, 1):
                self.sb.setWidths()

        # Set background colour
        if self.abcparams['stripedlist'] == '1' and numslot % 2:
            self.list.SetItemBackgroundColour(numslot, self.utility.colstripes)

        if triggeredtemplates:
            self.parent.onSetTemplate(None, [numslot], triggeredtemplates)

        # Check peer source
        if torrent.defaulttracker is None and not torrent.exttracker and not torrent.extrainttracker:
            if torrent.private or not self.dht:
                if replace < 0:
                    torrent.status = 'stop'
                    startnow = False
                if torrent.private:
                    # No tracker and private
                    metainfowarning |= 2
                if not self.dht:
                    # No tracker and no DHT
                    metainfowarning |= 4

        rank = self.getRankfromID(6)
        if rank != -1:
            self.list.SetStringItem(numslot, rank, self.localize(torrent.status))
        rank = self.getRankfromID(13)
        if rank != -1:
            if metainfowarning:
                self.list.SetStringItem(numslot, rank, torrent.decodeMetainfoWarnings(metainfowarning))
            elif replace < 0:
                self.list.SetStringItem(numslot, rank, '')

        # Set download speed limit if there are too many pieces of a too small size
        if self.abcparams['nbpiecesfix'] == '1':
            newmaxdownrate = self.utility.nbpiecesfixmaxdownrate
            if not torrent.magnet \
               and (torrent.maxlocaldownloadrate == 0 or newmaxdownrate < torrent.maxlocaldownloadrate) \
               and torrent.piecelength <= self.utility.piecesizefixtrigger \
               and int((torrent.totalsize + torrent.piecelength - 1) / torrent.piecelength) > self.utility.nbpiecesfixtrigger:
                torrent.maxlocaldownloadrate = newmaxdownrate

        # Show loaded popup (except when a torrent replaces a magnet because it's got its metadata)
        if self.abcparams['popuploaded'] == '1' and caller != 'magnet' :
            if self.popuploaded:
                self.popuploaded.incCounter()
                # Next cancel may fail so there's a key (popuploaded.counter) to disable action of an obsolete onEndPopUpLoadedTimer
                self.popuploadedtimer.cancel()
            else:
                self.popuploaded = PopUpLoaded(self.frame)
                self.popuploaded.Show()
            self.popuploadedtimer = Timer(self.utility.popuploadeddelay, self.invokeLater, [self.onEndPopUpLoadedTimer, [self.popuploaded.counter]])
            self.popuploadedtimer.start()

        if movetop:
            numslot = self.parent.onItemMoveMiddle(None, [numslot], scroll = False, forcebelowcompleted = True)

        if startnow:
            self.procRESUMEsingle(numslot, int(self.abcparams['fastresume']), auto = False)

        if showdetails:
            torrent.infowin = ABCMetaInfoFrame(self.parent, -1, '', self.utility,
                                               numslot, -1,
                                               (self.guiman.getMetaWidth(),
                                               self.guiman.getMetaHeight()),
                                               centeratpointer = False)

        if replace < 0:
            self.scheduler()
        else:
            if torrent.infowin is not None and not showdetails:
                # Refresh details window
                torrent.infowin.displayInfo()
            if torrent.detailwin is not None:
                # Refresh advanced details window
                torrent.detailwin.displayInfo()
            if self.parent.torrentrendlg and torrent == self.parent.torrentrendlg.torrent:
                # Refresh rename dialog
                self.parent.torrentrendlg.onTorrentReplace()

    def addOldProc(self, oldproc_array):
        notdht = (self.abcparams['dht'] == '0')
        # Counter for proc (can be different from i if some torrents can't be loaded)
        iproc = 0
        for i in xrange(len(oldproc_array)):
            torrentrecord = oldproc_array[i]
            src = path.join(self.utility.datapath, 'torrent', torrentrecord[1])
            warning = ''
            try:
                metainfo_file = open(src, 'rb')
                metainfo, metainfowarning = bdecode(metainfo_file.read(), sloppy = 2)
                metainfo_file.close()
            except:
                try:
                    metainfo_file.close()
                except:
                    pass
                dlg = wx.MessageDialog(self.parent, src + '\n' + self.localize('failedcantaccesstorrent'),
                                       self.localize('abcokcerror'),
                                       wx.OK | wx.ICON_ERROR)
                dlg.ShowModal()
                dlg.Destroy()
                continue

            info = metainfo['info']
            torrentinfohash = metainfo.get('infohash')
            onlygetmetadata, keepmagnetdest = divmod(int(torrentrecord[53]), 10)
            if torrentinfohash is None:
                torrentinfohash = sha1(bencode(info)).digest()
                magnet = 0
            elif keepmagnetdest == 0:
                magnet = 1
            else:
                magnet = 2

            # Tracker info
            announce_list = metainfo.get('announce-list')
            multitracker = 0
            if announce_list is None :
                announce = metainfo.get('announce')
                if announce is None:
                    defaultannounce = None
                else:
                    defaultannounce = self.utility.decodeString(announce)
            else:
                for t in announce_list:
                    multitracker += len(t)
                defaultannounce = ''

            piecelength = info['piece length']

            if info.has_key('length'):
                # 1 file for this torrent
                filelength = info['length']
                singlefile = True
                filesnumber = 1
            else:
                # Directory torrent
                filelength = 0
                singlefile = False
                filesnumber = 0
                for x in info['files']:
                    filelength += x['length']
                    filesnumber += 1

            # Private flag
            private = info.get('private', 0)

            # Store old message string or new float of message time
            termevt = torrentrecord[31]
            if termevt:
                try:
                    termevt = float(termevt)
                except:
                    pass

            torrent = ABCTorrent(
                                self,
                                src,
                                torrentrecord[2],
                                singlefile,
                                torrentrecord[3],
                                int(torrentrecord[4]),
                                torrentrecord[0],
                                int(torrentrecord[8]),
                                int(torrentrecord[9]),
                                int(torrentrecord[22]),
                                int(torrentrecord[10]),
                                int(torrentrecord[11]),
                                int(torrentrecord[12]),
                                int(torrentrecord[13]),
                                float(torrentrecord[5]),
                                float(torrentrecord[6]),
                                piecelength,
                                filelength,
                                torrentinfohash,
                                torrentrecord[7],
                                float(torrentrecord[14]),
                                float(torrentrecord[32]),
                                filesnumber,
                                torrentrecord[15],
                                torrentrecord[16],
                                int(torrentrecord[17]),
                                int(torrentrecord[33]),
                                int(torrentrecord[25]),
                                int(torrentrecord[26]),
                                int(torrentrecord[27]),
                                int(torrentrecord[28]),
                                int(torrentrecord[29]),
                                int(torrentrecord[30]),
                                int(torrentrecord[18]),
                                int(torrentrecord[19]),
                                int(torrentrecord[20]),
                                torrentrecord[21],
                                int(torrentrecord[23]),
                                torrentrecord[24],
                                termevt,
                                defaultannounce,
                                private,
                                multitracker,
                                magnet,
                                torrentrecord[34],
                                int(torrentrecord[35]),
                                int(torrentrecord[36]),
                                int(torrentrecord[37]),
                                int(torrentrecord[38]),
                                torrentrecord[39],
                                float(torrentrecord[40]),
                                float(torrentrecord[41]),
                                float(torrentrecord[42]),
                                onlygetmetadata,
                                [int(torrentrecord[j]) for j in xrange(43, 53)],
                                iproc
                                )
            if not torrent.singlefile:
                torrent.readFileNames()
                torrent.readFileNameStatus()
                torrent.readFilePriorities(info['files'])
                torrent.readFileProgress()
            torrent.readPeers()

            # File names check status
            if torrent.filenamestatus is not None:
                checkstatus = 0
                for status in torrent.filenamestatus:
                    checkstatus |= status
                if checkstatus:
                    metainfowarning |= checkstatus << 3

            # Check peer source
            if torrent.defaulttracker is None and not torrent.exttracker and not torrent.extrainttracker:
                if private or notdht:
                    if torrent.status != 'finished':
                        torrent.status = 'stop'
                    if private:
                        # No tracker and private
                        metainfowarning |= 2
                    if notdht:
                        # No tracker and no DHT
                        metainfowarning |= 4

            self.proctab.append(torrent)

            # Add info in list
            self.list.InsertStringItem(iproc, "")
            rank = self.getRankfromID(4)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.name)
            rank = self.getRankfromID(5)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.formatedProgress())
            rank = self.getRankfromID(6)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.localize(torrent.status))
            rank = self.getRankfromID(7)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.priorities_s[torrent.priority])
            rank = self.getRankfromID(8)
            if rank != -1:
                if self.abcparams['keepeta'] == "1" and torrent.status != 'finished':
                    if torrent.complete:
                        if torrent.uploadoption == 0:
                            self.list.SetStringItem(iproc, rank, self.localize('etaS') + '     oo')
                        elif torrent.uploadtillcopycomplete and \
                             ((torrent.uploadoption == 1 or torrent.uploadoption == 3) and torrent.seedingtimeleft == 0 \
                             or (torrent.uploadoption == 2 or torrent.uploadoption == 3) and torrent.ratioIsReached()):
                            self.list.SetStringItem(iproc, rank, self.localize('etaS') + " " + self.localize('tillcc'))
                        elif torrent.uploadoption == 2:
                            self.list.SetStringItem(iproc, rank, self.localize('etaS') + "      ?")
                        else:
                            self.list.SetStringItem(iproc, rank, self.localize('etaS') + " " + self.utility.eta_value(torrent.seedingtimeleft))
                    else:
                        self.list.SetStringItem(iproc, rank, self.localize('etaD') + "      ?")
                else:
                    self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(9)
            if rank != -1:
                if torrent.magnet:
                    self.list.SetStringItem(iproc, rank, '?')
                else:
                    self.list.SetStringItem(iproc, rank, self.utility.formatedSize(torrent.requestedsize)[0])
            rank = self.getRankfromID(10)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.formatedRate(None, torrent.prioritizedown)[0])
            rank = self.getRankfromID(11)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.formatedRate(None, torrent.prioritizeup)[0])
            rank = self.getRankfromID(12)
            if rank != -1:
                if torrent.requestedsize <= 0 and not torrent.magnet:
                    self.list.SetStringItem(iproc, rank, '999.9%')
                elif torrent.progress <= 0:
                    self.list.SetStringItem(iproc, rank, '0.0%')
                else:
                    self.list.SetStringItem(iproc, rank, '%.1f' % (torrent.upsize / torrent.requestedsize / torrent.progress * 10000) + "%")
            rank = self.getRankfromID(13)
            if rank != -1:
                if metainfowarning:
                    self.list.SetStringItem(iproc, rank, torrent.decodeMetainfoWarnings(metainfowarning))
                elif self.abcparams['keeptermevtmsg'] == '1':
                    torrent.message = torrent.termEvtMsg()
                    self.list.SetStringItem(iproc, rank, torrent.message)
                else:
                    self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(14)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(15)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(16)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(17)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(18)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.formatedSize(torrent.downsize)[0])
            rank = self.getRankfromID(19)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.formatedSize(torrent.upsize)[0])
            rank = self.getRankfromID(20)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, '')
            rank = self.getRankfromID(21)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, path.split(torrent.src)[1])
            rank = self.getRankfromID(22)
            if rank != -1:
                if torrent.uploadoption == 0:
                    self.list.SetStringItem(iproc, rank, "oo")
                elif torrent.uploadoption == 1:
                    self.list.SetStringItem(iproc, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False))
                elif torrent.uploadoption == 2:
                    self.list.SetStringItem(iproc, rank, str(torrent.uploadratio) + "%")
                else:
                    self.list.SetStringItem(iproc, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False)
                                            + '/' + str(torrent.uploadratio) + "%")
            rank = self.getRankfromID(23)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.shortlabel)
            rank = self.getRankfromID(24)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.label)
            rank = self.getRankfromID(25)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.activityMarker())
            rank = self.getRankfromID(26)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, torrent.getRebuiltDest())
            rank = self.getRankfromID(27)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.time_value(torrent.seedingtime, seconds = False, maxvalue = False))
            rank = self.getRankfromID(28)
            if rank != -1:
                self.list.SetStringItem(iproc, rank, self.utility.time_value(torrent.totalseedingtime, seconds = False, maxvalue = False))

            self.torrentloaded += 1
            updatesbF = False
            if self.sb.setText(self.localize('sb_L') + '%u ' % self.torrentloaded, 1):
                self.sb.setWidths()
            if torrent.status == 'queue':
                self.torrentqueued += 1
            elif torrent.status == 'finished':
                self.torrentfinished += 1
                updatesbF = True
            elif torrent.status == 'standby':
                self.addTorrentStandby(torrent)
            if torrent.complete:
                self.torrentavailable += 1
                updatesbF = True
            if updatesbF:
                if self.sb.setText(self.localize('sb_F') + "%u/%u " % (self.torrentfinished, self.torrentavailable), 6):
                    self.sb.setWidths()
            # Set background colour
            if self.abcparams['stripedlist'] == '1' and iproc % 2:
                self.list.SetItemBackgroundColour(iproc, self.utility.colstripes)
            iproc += 1
        # Init standbylastscrape for each standby torrent for scraping from top to bottom of list
        tsc = self.trackerstandbycount.copy()
        for t in self.torrentstandby:
            tracker = t.getTracker()
            t.standbylastscrape -= tsc[tracker] * self.utility.sbscrapeperiod
            tsc[tracker] -= 1

    def onEndPopUpLoadedTimer(self, key):
        if key == self.popuploaded.counter:
            self.popuploaded.Show(False)
            self.popuploaded.Destroy()
            self.popuploaded = None
            self.popuploadedtimer = None

    def runProc(self, index, fastresume, auto = True, onlycheck = False, reseed = False, tracker = 0, sbwaitingforleechers = False):
        torrent = self.proctab[index]
        torrent.status = 'working'
        # Init torrent rate distribution parameters
        self.initRateDistrib(torrent, self.utility.bdcalcupth1 * self.utility.bdcalcupth2,
                                      self.utility.bdcalcdownth1 * self.utility.bdcalcdownth2,
                                      running = False)
        torrent.abcengine = ABCEngine(self, self.guiman, torrent, tracker, auto, onlycheck, fastresume, reseed,
                                      self.getBTParams(index), sbwaitingforleechers, self.dht, self.pl, self.startqueue)
        torrent.abcengine.start()

    def getBTParams(self, index):
        torrent = self.proctab[index]
        alloctype = ['normal', 'background', 'pre-allocate', 'sparse1', 'sparse2']
        # Construct BT params
        btparams = []
        btparams.append("--responsefile")
        btparams.append(torrent.src)
        if self.pl is None:
            btparams.append("--minport")
            btparams.append(self.abcparams['minport'])
            btparams.append("--maxport")
            btparams.append(self.abcparams['maxport'])
        btparams.append("--random_port")
        btparams.append("0")
        btparams.append("--max_uploads")
        btparams.append(str(torrent.maxupload))
        btparams.append("--min_uploads")
        btparams.append(str(torrent.maxupload))

        if self.abcparams['ip'] != '':
            btparams.append("--ip")
            btparams.append(self.abcparams['ip'])
        if self.abcparams['bind'] != '':
            btparams.append("--bind")
            btparams.append(self.abcparams['bind'])
        if self.utility.ipv6enabled:
            btparams.append("--ipv6_enabled")
            btparams.append("1")
            btparams.append("--ipv6_binds_v4")
            btparams.append(self.abcparams['ipv6_binds_v4'])
        if self.abcparams['multihomeipv4'] != '':
            btparams.append("--multihome_ipv4")
            btparams.append(self.abcparams['multihomeipv4'])
        if self.abcparams['multihomeipv6'] != '':
            btparams.append("--multihome_ipv6")
            btparams.append(self.abcparams['multihomeipv6'])

        # Always set fileselector
        btparams.append("--selector_enabled")
        btparams.append("1")
        btparams.append("--auto_kick")
        btparams.append(self.abcparams['kickban'])
        btparams.append("--double_check")
        btparams.append(self.abcparams['double_check'])
        btparams.append("--triple_check")
        btparams.append(self.abcparams['triple_check'])
        btparams.append("--lock_files")
        btparams.append(self.abcparams['lock_files'])
        btparams.append("--lock_while_reading")
        btparams.append(self.abcparams['lock_while_reading'])
        btparams.append("--security")
        btparams.append(self.abcparams['notsameip'])
        btparams.append("--min_peers")
        btparams.append(self.abcparams['min_peers'])
        btparams.append("--alloc_type")
        btparams.append(alloctype[int(self.abcparams['alloc_type'])])
        btparams.append("--alloc_rate")
        btparams.append(self.abcparams['alloc_rate'])
        btparams.append("--max_files_open")
        btparams.append(self.abcparams['max_files_open'])
        btparams.append("--max_connections")
        btparams.append(self.abcparams['max_connections'])

        btparams.append("--max_upload_rate")
        btparams.append(str(torrent.maxuploadrate))
        btparams.append("--max_download_rate")
        btparams.append(str(torrent.maxdownloadrate))

        btparams.append("--max_initiate")
        btparams.append(self.abcparams['max_initiate'])
        #if torrent.maxupload < 4 :
        #    btparams.append(str(min(12, int(self.abcparams['max_initiate']))))
        #elif torrent.maxupload < 30:
        #    btparams.append(str(min(40, int(self.abcparams['max_initiate']))))
        #else:
        #    btparams.append(str(min(torrent.maxupload + 10, int(self.abcparams['max_initiate']))))

        btparams.append("--upload_unit_size")
        btparams.append(self.abcparams['upload_unit_size'])

        if torrent.dest != "":
            btparams.append("--saveas")
            btparams.append(torrent.dest)

        if not torrent.singlefile:
            btparams.append("--priority")
            if torrent.filespriority is None:
                btparams.append(','.join(torrent.filesnumber * ['1']))
            else:
                btparams.append(','.join(map(str, torrent.filespriority)))

        btparams.append("--upnp_nat_access")
        btparams.append(self.abcparams['upnp'])

        btparams.append("--crypto_allowed")
        btparams.append(str(int(int(self.abcparams['crypto']) >= 1)))
        btparams.append("--crypto_only")
        btparams.append(str(int(int(self.abcparams['crypto']) >= 2)))
        btparams.append("--crypto_stealth")
        btparams.append(str(int(int(self.abcparams['crypto']) == 3)))

        btparams.append("--dir_root")
        btparams.append(self.utility.datapath)

        btparams.append("--dhtport")
        btparams.append(str(self.utility.dhtport))

        btparams.append("--dhtrerequest_interval")
        btparams.append(self.abcparams['dhtrerequestinterval'])

        btparams.append("--rerequest_interval")
        btparams.append(self.abcparams['rerequestinterval'])

        btparams.append("--default_tracker")
        if torrent.defaulttracker is None:
            btparams.append('')
        else:
            btparams.append(torrent.defaulttracker)

        btparams.append("--magnet")
        btparams.append(torrent.magnet)

        btparams.append("--metadata_timeout")
        btparams.append(self.abcparams['metadatatimeout'])

        btparams.append("--min_peer_connection_timeout")
        btparams.append(str(self.utility.minpeerconnectiontimeout))

        btparams.append("--max_peer_connection_timeout")
        btparams.append(str(self.utility.maxpeerconnectiontimeout))

        btparams.append("--retry_connection")
        btparams.append(self.abcparams['retryconnection'])

        btparams.append("--ut_pex")
        btparams.append(self.abcparams['utpex'])

        btparams.append("--ut_pex_sending_rate")
        btparams.append(str(self.utility.utpexsendingrate))

        btparams.append("--write_buffer_size")
        btparams.append(self.abcparams['writebuffersize'])

        btparams.append("--read_buffer_expiration")
        btparams.append(self.abcparams['readbufferexpiration'])

        btparams.append("--write_buffer_expiration")
        btparams.append(self.abcparams['writebufferexpiration'])

        btparams.append("--read_buffer_memory_saving")
        btparams.append(self.abcparams['readbuffermemorysaving'])

        btparams.append("--floor_uploads")
        btparams.append(str(self.utility.minuploads))

        btparams.append("--display_interval")
        btparams.append(str(self.utility.GUIupdaterate_fast))

        btparams.append("--unit_size_step")
        btparams.append(str(self.utility.unitsizestep))

        btparams.append("--unit_size_inc_trig")
        btparams.append(str(self.utility.unitsizeinctrig))

        btparams.append("--read_buffer_size")
        btparams.append(self.abcparams['readbuffersize'])

        btparams.append("--check_piece_before_upload")
        btparams.append(self.abcparams['checkpiecebeforeupload'])

        btparams.append("--kick_err_per_piece")
        btparams.append(self.abcparams['kickerrperpiece'])

        btparams.append("--kick_bad_pieces")
        btparams.append(self.abcparams['kickbadpieces'])

        btparams.append("--ban_err_per_piece")
        btparams.append(self.abcparams['banerrperpiece'])

        btparams.append("--ban_bad_pieces")
        btparams.append(self.abcparams['banbadpieces'])

        btparams.append("--download_slice_size")
        btparams.append(self.abcparams['downslicesize'])

        btparams.append("--breakup_seed_bitfield")
        btparams.append(self.abcparams['breakupseedbitfield'])

        btparams.append("--raw_server_slowdown")
        btparams.append(self.abcparams['rawserverslowdown'])

        btparams.append("--min_comp_for_request")
        btparams.append(self.abcparams['mincompforrequest'])

        btparams.append("--check_buffer_size")
        btparams.append(self.abcparams['checkbuffersize'])

        return btparams

    def updateTorrentList(self, finalupdate = False):
        if not finalupdate:
            try:
                if self.parent.exiting:
                    return
            except:
                return

        now = clock()
        torrentlist = TorrentListFileManager(self)
        torrentlist.open("w")

        try:
            torrentlist.writeHeader()
            for torrent in self.proctab:
                status = torrent.status
                # Update up and down size, progress
                if torrent.abcengine:
                    torrent.downsize, torrent.upsize = torrent.abcengine.getDownUpSize()
                    if not torrent.isCheckingOrAllocating() and not torrent.isWaiting():
                        torrent.progress = torrent.abcengine.progress
                    if status != 'pause' and status != 'onhold' and not torrent.abcengine.onlycheck:
                        torrent.inactivitytime, torrent.trackererrortime = torrent.abcengine.getTimeoutTimes(now)
    
                if status == 'pause' or status == 'onhold' or status == 'working' \
                    or status == 'seeding' or torrent.abcengine and status == 'completed' \
                    or torrent.delayedresumetimer and torrent.status == 'completed':
                    status = 'queue'
                elif status == 'onlycheck' or torrent.abcengine and torrent.abcengine.onlycheck:
                    if torrent.oldstatus == 'finished' or torrent.oldstatus == 'completed':
                        status = 'stop'
                    else:
                        status = torrent.oldstatus
                elif status == 'sbwaitingforleechers' or torrent.abcengine and torrent.abcengine.sbwaitingforleechers:
                    status = 'standby'
    
                if torrent.complete:
                    progress = "100"
                else:
                    progress = str(torrent.progress)

                if torrent.sbstart:
                    sbtime = now - torrent.sbstart
                else:
                    sbtime = 0.
                
                torrentlist.appendList(torrent.name,
                                       path.split(torrent.src)[1],
                                       torrent.dest,
                                       status,
                                       torrent.priority,
                                       torrent.downsize,
                                       torrent.upsize,
                                       progress,
                                       torrent.maxupload,
                                       torrent.maxlocaluploadrate,
                                       torrent.maxlocaldownloadrate,
                                       10 * int(torrent.uploadtillcopycomplete) + torrent.uploadoption,
                                       torrent.uploadtimeh,
                                       torrent.uploadtimem,
                                       torrent.uploadratio,
                                       torrent.seedingtime,
                                       torrent.totalseedingtime,
                                       torrent.label,
                                       torrent.shortlabel,
                                       torrent.timeoutswitch,
                                       torrent.timeoutaction,
                                       torrent.timeoutwork,
                                       torrent.timeoutworkdown,
                                       torrent.timeoutworktrack,
                                       torrent.timeoutseed,
                                       torrent.timeoutseedup,
                                       torrent.timeoutseedtrack,
                                       torrent.timeouttracker,
                                       torrent.timeoutdownload,
                                       torrent.timeoutupload,
                                       torrent.movefolder,
                                       torrent.exttracker,
                                       torrent.exttrackerurl,
                                       torrent.termevt,
                                       torrent.magnetname,
                                       torrent.prioritizedown,
                                       torrent.prioritizeup,
                                       torrent.checkinttrackwait,
                                       torrent.extrainttracker,
                                       torrent.inttrackerurl,
                                       torrent.inactivitytime,
                                       torrent.trackererrortime,
                                       sbtime,
                                       torrent.restoreindex,
                                       10 * torrent.onlygetmetadata + (torrent.magnet == 2))
        except:
            torrentlist.close()
        else:
            torrentlist.close()
            try:
                remove(torrentlist.filename)
            except:
                pass
            rename(torrentlist.filename + '.part', torrentlist.filename)

        if not finalupdate:
            # Start timer
            if not self.parent.exiting:
                self.timerupdatetorrentlist = Timer(self.utility.torlistautosaverate, self.invokeLater, [self.updateTorrentList])
                self.timerupdatetorrentlist.start()

    def scheduleTorrents(self, schedulerange):
        # schedulerange :
        # 0 : global
        # 1 : download
        # 2 : seed

        modeauto = (self.abcparams['mode'] == '1')
        dontschedonlyonhold = (schedulerange != 1 or self.abcparams['schedonlyonhold'] == '0')
        sbqueued = (self.abcparams['sbqueued'] == '1')
        if schedulerange == 0:
            cp = self.currentproc + self.currentprocseed
            to = self.torrentonhold + self.torrentonholdseed
            mt = min(self.getMaxNumSimGlobal() + self.torrentinactive + self.torrentinactiveseed, self.getActivityMax())
            if cp - to < mt:
                torrentrange = self.proctab
            etbs = self.extratobestopped + self.extratobestoppedseed
            fat = self.freshactivetorrentsseed + self.freshactivetorrents
        elif schedulerange == 1:
            cp = self.currentproc
            to = self.torrentonhold
            mt = min(self.maxnumsim + self.torrentinactive, self.getActivityMaxDown())
            if cp - to < mt:
                torrentrange = [t for t in self.proctab if not t.complete]
            etbs = self.extratobestopped
            fat = self.freshactivetorrents
        else:
            cp = self.currentprocseed
            to = self.torrentonholdseed
            mt = min(self.getMaxNumSimSeed() + self.torrentinactiveseed, self.getActivityMaxSeed())
            if cp - to < mt:
                torrentrange = [t for t in self.proctab if t.complete]
            etbs = self.extratobestoppedseed
            fat = self.freshactivetorrentsseed

        while cp - to < mt:
            schedulertorrent = None
            priority = 5
            scannedtorrentonhold = 0
            if to:
                # On-hold torrents have the highest priority level
                for torrent in torrentrange:
                    if torrent.status == 'onhold':
                        scannedtorrentonhold += 1
                        if torrent.priority < priority:
                            schedulertorrent = torrent
                            priority = torrent.priority
                        if scannedtorrentonhold == to:
                            break
            elif modeauto and dontschedonlyonhold:
                for torrent in torrentrange:
                    if (torrent.status == 'queue' or sbqueued and torrent.status == 'standby') and torrent.priority < priority \
                       and torrent.isNotLimited():
                        # Torrents that are not in on-hold status will be handled only in automatic mode
                        schedulertorrent = torrent
                        priority = torrent.priority
            if schedulertorrent is not None and (to or self.arePortsAvailable()):
                self.procRESUMEsingle(schedulertorrent.index,
                                      int(self.abcparams['fastresume']) or (schedulertorrent.filespriority is not None))
            else:
                break
            if schedulerange == 0:
                cp = self.currentproc + self.currentprocseed
                to = self.torrentonhold + self.torrentonholdseed
            elif schedulerange == 1:
                cp = self.currentproc
                to = self.torrentonhold
            elif schedulerange == 2:
                cp = self.currentprocseed
                to = self.torrentonholdseed

        if etbs > 0 and cp - to > mt:
            # If some torrents turned active, on-hold or queue some extra torrents
            # Torrents will be queued back first from alive torrents, then if necessary from blocked waiting torrents
            # and finally if necessary from torrents that just turn active
            
            alivepool = []
            waitingpool = []
            CHECKWAITING.clear()
            for t in self.proctab:
                if t.abcengine and t.activity != 2 and t.status != 'pause' and t.status != 'onhold':
                    if not t.isWaiting() and not t.isCheckingOrAllocating():
                        alivepool.append(t)
                    elif not t.abcengine.onlycheck and t not in fat and t.isBlockedWaiting():
                        waitingpool.append(t)
            
            if schedulerange == 0:
                tobestoppedpool1 = [t for t in alivepool if t.activity and t not in fat]
                tobestoppedpool2 = waitingpool
            elif schedulerange == 1:
                tobestoppedpool1 = [t for t in alivepool if t.activity and t.status != 'completed' and not t.complete and t not in fat]
                tobestoppedpool2 = [t for t in waitingpool if not t.complete]
            else:
                tobestoppedpool1 = [t for t in alivepool if t.activity and t.status == 'completed' and t not in fat]
                tobestoppedpool2 = [t for t in waitingpool if t.complete]

            schedstopsman = (self.abcparams['schedstopsman'] == "1")
            scheddonthalt = (self.abcparams['scheddonthalt'] == '1')
            scheddonthaltuprategap = int(self.abcparams['scheddonthaltuprategap'])

            while etbs > 0:
                schedulertorrent = None
                newactivetorrent = fat[-etbs]
                if self.extratobestoppedmode == 2:
                    newactiveindex = newactivetorrent.index
                    newactivepriority = newactivetorrent.priority
                    queuepriority = newactivepriority
                else:
                    newactiveindex = -1
                    newactivepriority = -1
                    queuepriority = -1
                queueactivity = -1
                for tbsp in [tobestoppedpool1, tobestoppedpool2]:
                    for torrent in tbsp:
                        if torrent.priority >= queuepriority \
                           and (newactiveindex == -1 or torrent.priority > newactivepriority or torrent.priority == newactivepriority and torrent.index > newactiveindex) \
                           and torrent.activity >= queueactivity \
                           and (torrent.abcengine.auto or schedstopsman) \
                           and (not scheddonthalt or self.getTorrentUprateGap(torrent) <= scheddonthaltuprategap):
                            schedulertorrent = torrent
                            queuepriority = torrent.priority
                            queueactivity = torrent.activity
                    if schedulertorrent is not None:
                        tbsp.remove(schedulertorrent)
                        break

                if schedulertorrent is None:
                    if newactiveindex == -1:
                        # No need to continue for other newly active torrent
                        if self.abcparams['extratobestoppedfail'] == "1":
                            # Queue back all newly active torrents because option is set and no already active torrents
                            # could be queued back
                            while True:
                                if (newactivetorrent.abcengine.auto or schedstopsman) \
                                   and (not scheddonthalt or self.getTorrentUprateGap(newactivetorrent) <= scheddonthaltuprategap):
                                    if self.abcparams['schedonhold'] == "1":
                                        self.procONHOLD(newactivetorrent.index)
                                    else:
                                        self.procQUEUE([newactivetorrent.index], schedule = False)
                                etbs -= 1
                                if etbs == 0:
                                    break
                                newactivetorrent = fat[-etbs]
                        break
                    if self.abcparams['extratobestoppedfail'] == "1" \
                       and (newactivetorrent.abcengine.auto or schedstopsman) \
                       and (not scheddonthalt or self.getTorrentUprateGap(newactivetorrent) <= scheddonthaltuprategap):
                        # Queue back newly active torrent because option is set and no already active torrents
                        # could be queued back
                        if self.abcparams['schedonhold'] == "1":
                            self.procONHOLD(newactiveindex)
                        else:
                            self.procQUEUE([newactiveindex], schedule = False)
                else:
                    if self.abcparams['schedonhold'] == "1":
                        self.procONHOLD(schedulertorrent.index)
                    else:
                        self.procQUEUE([schedulertorrent.index], schedule = False)
                etbs -= 1

            CHECKWAITING.set()

    def scheduler(self, updatecount = False):
        if self.parent.torrentlistloading:
            return

        if self.abcparams['trigwhenfinishseed'] == "0" \
           or self.abcparams['schedonlyonhold'] == "1":
            # For downloading torrents
            self.scheduleTorrents(1)
            # For seeding torrents
            self.scheduleTorrents(2)
        elif self.abcparams['trigwhenfinishseed'] == "1":
            # Treat globally downloading and seeding torrents
            self.scheduleTorrents(0)
        else:
            # Only for seeding torrents
            self.scheduleTorrents(2)

        self.extratobestopped = 0
        self.extratobestoppedseed = 0
        self.freshactivetorrents[:] = []
        self.freshactivetorrentsseed[:] = []

        if updatecount:
            self.updateRunningTorrentCounters()

        self.checkIdleForClosing()

    def isNextToBeResumed(self, index, prio):
        # Check if the running torrent at index with priority prio would be the next to be resumed
        # by the scheduler at next run, if it were in queue status
        if self.abcparams['trigwhenfinishseed'] == "1":
            if self.abcparams['mode'] != '1' and self.torrentonhold + self.torrentonholdseed == 0 \
               or self.currentproc + self.currentprocseed - 1 - self.torrentonhold - self.torrentonholdseed \
               >= min(self.getMaxNumSimGlobal() + self.torrentinactive + self.torrentinactiveseed - 1 \
                      + self.proctab[index].activity, self.getActivityMax()):
                # In this case no torrent will be resumed by the scheduler
                return False
            sbqueued = (self.abcparams['sbqueued'] == '1')
            for t in self.proctab[:index]:
                if t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority <= prio:
                    return False
            for t in self.proctab[index + 1:]:
                if t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority < prio:
                    return False
            return True
        else:
            if self.proctab[index].complete:
                if self.abcparams['mode'] != '1' and self.torrentonholdseed == 0 \
                   or self.currentprocseed - 1 - self.torrentonholdseed >= min(self.getMaxNumSimSeed() + self.torrentinactiveseed - 1
                                                                               + self.proctab[index].activity, self.getActivityMaxSeed()):
                    # In this case no torrent will be resumed by the scheduler
                    return False
                sbqueued = (self.abcparams['sbqueued'] == '1')
                for t in self.proctab[:index]:
                    if t.complete and (t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority <= prio):
                        return False
                for t in self.proctab[index + 1:]:
                    if t.complete and (t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority < prio):
                        return False
                return True
            else:
                if self.abcparams['mode'] != '1' and self.torrentonhold == 0 \
                   or self.currentproc - 1 - self.torrentonhold >= min(self.maxnumsim + self.torrentinactive - 1
                                                                       + self.proctab[index].activity, self.getActivityMaxDown()):
                    # In this case no torrent will be resumed by the scheduler
                    return False
                sbqueued = (self.abcparams['sbqueued'] == '1')
                for t in self.proctab[:index]:
                    if not t.complete and (t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority <= prio):
                        return False
                for t in self.proctab[index + 1:]:
                    if not t.complete and (t.status == 'onhold' or (t.status == 'queue' or sbqueued and t.status == 'standby') and t.priority < prio):
                        return False
                return True
  
    def isTimeoutAllowed(self):
        # Checks if timeout is allowed
        # If timeoutwhenextra is set, timeout :
        # 1 - in manual mode if there are on-hold torrents
        # 2 - in auto mode if there are on-hold or queued torrents
        if self.abcparams['timeoutwhenextra'] == '0':
            return True
        if self.torrentonhold \
           or self.abcparams['mode'] == '1' and (self.torrentqueued or \
                                                (self.abcparams['sbqueued'] == '1' and self.torrentstandby)):
            return True
        return False

    def checkIdleForClosing(self):
        if not self.torrentrunning and self.whenidle:
            self.abouttoshutdown |= (self.whenidle == 2)
            if self.whenidle == 1:
                wintitle = self.localize('delcloseidletitle')
            else:
                wintitle = self.localize('delcloseshutidletitle')
            self.whenidle = 0
            self.frame.idlesubmenu.Check(954, True)
            if not self.abouttoclose:
                self.startAboutToClose('checkidle', wintitle)

    def checkCompleteAndShutDownForClosing(self, now):
        compshutallcomplete = bool(self.compshut)
        for torrent in self.compshut:
            if not torrent.complete:
                compshutallcomplete = False
                break
        if compshutallcomplete:
            if not self.compshutseedingstart:
                self.compshutseedingstart = now
            elif now - self.compshutseedingstart >= int(self.abcparams['compshutseedingtime']) * 3600:
                self.abouttoshutdown = True
                self.compshut[:] = []
                if not self.abouttoclose:
                    self.startAboutToClose('compshut', self.localize('delcloseshutcompshuttitle'))
        else:
            self.compshutseedingstart = 0

    def startAboutToClose(self, caller, title):
        self.abouttoclose = clock()
        self.parent.rsspanel.stopAllTimers()
        # Show progress dialog to wait 60 seconds before closing
        self.delacdlg = wx.ProgressDialog(title, self.localize('delclosing'),
                                          maximum = 60, parent = self.parent,
                                          style = wx.PD_CAN_ABORT | wx.PD_AUTO_HIDE | wx.PD_SMOOTH)
        self.delacdlg.SetFocus()

    def changeLocalInfo(self, indexlist, info):
        if self.parent.torrentlistloading:
            return
        now = clock()
        for index in indexlist:
            torrent = self.proctab[index]
            olduploadoption = torrent.uploadoption
            olduploadtimemax = torrent.uploadtimemax
            olduploadratio = torrent.uploadratio
            oldexttrackerurl = torrent.exttrackerurl
            oldextrainttracker = torrent.extrainttracker
            oldinttrackerurl = torrent.inttrackerurl
            mustswitchtracker = -1

            # Handle seeding option switching
            # Seeding parameters
            if info[6] is not None:
                uploadoption = info[6] % 10
            else:
                uploadoption = olduploadoption
            if info[7] is not None:
                uploadtimeh = info[7]
            else:
                uploadtimeh = torrent.uploadtimeh
            if info[8] is not None:
                uploadtimem = info[8]
            else:
                uploadtimem = torrent.uploadtimem
            newuploadtimemax = float(uploadtimeh * 3600 + uploadtimem * 60)
            if info[9] is not None:
                uploadratio = info[9]
            else:
                uploadratio = olduploadratio

            resetseedingtimers = False
            if (uploadoption != olduploadoption or newuploadtimemax != olduploadtimemax) \
               and torrent.complete and torrent.abcengine:
                # Logic for resetting seeding timers
                # Seeding options have been changed and are in use in a running torrent
                if uploadoption == 0:
                    if self.utility.seedingtimecount == "1" and \
                       (olduploadoption == 1 or olduploadoption == 3):
                        resetseedingtimers = True
                elif uploadoption == 1:
                    if self.utility.seedingtimecount == "1" and \
                       (olduploadoption == 0 or olduploadoption == 2):
                        resetseedingtimers = True
                    if self.utility.seedingtimeresetonswitch == "1" and \
                       (olduploadoption == 0 or olduploadoption == 2) \
                       or self.utility.seedingtimeresetonvalue == "1" and newuploadtimemax != olduploadtimemax:
                        torrent.seedingtime = 0.
                        resetseedingtimers = True
                elif uploadoption == 2:
                    if self.utility.seedingtimecount == "1" and \
                       (olduploadoption == 1 or olduploadoption == 3):
                        resetseedingtimers = True
                elif uploadoption == 3:
                    if self.utility.seedingtimecount == "1" and \
                       (olduploadoption == 0 or olduploadoption == 2):
                        resetseedingtimers = True
                    if self.utility.seedingtimeresetonswitch == "1" and \
                       (olduploadoption == 0 or olduploadoption == 2) \
                       or self.utility.seedingtimeresetonvalue == "1" and newuploadtimemax != olduploadtimemax:
                        torrent.seedingtime = 0.
                        resetseedingtimers = True
                if resetseedingtimers:
                    torrent.abcengine.stopSeedingTimers(resettotalseeding = False)

            # Update seeding options in torrent
            if info[6] is not None:
                torrent.uploadoption = uploadoption
                torrent.uploadtillcopycomplete = (info[6] / 10) == 1
                if torrent.abcengine:
                    torrent.abcengine.forceseeding = torrent.uploadtillcopycomplete
            if info[7] is not None:
                torrent.uploadtimeh = uploadtimeh
            if info[8] is not None:
                torrent.uploadtimem = uploadtimem
            if info[7] is not None or info[8] is not None:
                torrent.uploadtimemax = newuploadtimemax
            if info[9] is not None:
                torrent.uploadratio = uploadratio

            if torrent.complete:           
                updatetimeinfos = False
                if resetseedingtimers:
                    # Seeding timers were reset
                    # Compute again seeding times to be able to immediately display ETA, and restart seeding timers
                    updatetimeinfos = True
                    torrent.abcengine.countSeedingTimeLeft(now)
                    torrent.abcengine.startSeedingTimers(resettotalseeding = False)

                elif uploadoption != olduploadoption or newuploadtimemax != olduploadtimemax or uploadratio != olduploadratio:
                    updatetimeinfos = True
                    if not torrent.abcengine:
                        # For not running torrents that had their upload options modified
                        if uploadoption == 0:
                            if torrent.status == 'finished':
                                torrent.status = 'completed'
                                torrent.displayStatus()
                                torrent.termevt = torrent.prevtermevt = ''
                                self.torrentfinished -= 1
                                self.updateRunningTorrentCounters()
                        elif uploadoption == 1:
                            if self.utility.seedingtimeresetonswitch == "1" and (olduploadoption == 0 or olduploadoption == 2) \
                               or self.utility.seedingtimeresetonvalue == "1" and newuploadtimemax != olduploadtimemax:
                                torrent.seedingtime = 0
                            if torrent.seedingtime >= newuploadtimemax:
                                if torrent.status == 'completed':
                                    torrent.status = 'finished'
                                    torrent.displayStatus()
                                    torrent.displayFinished()
                                    torrent.termevt = torrent.prevtermevt = time()
                                    torrent.displayMessage(torrent.termEvtMsg())
                                    self.torrentfinished += 1
                                    self.updateRunningTorrentCounters()
                            elif torrent.status == 'finished':
                                torrent.status = 'completed'
                                torrent.displayStatus()
                                torrent.termevt = torrent.prevtermevt = ''
                                self.torrentfinished -= 1
                                self.updateRunningTorrentCounters()
                        elif uploadoption == 2:
                            if torrent.ratioIsReached():
                                if torrent.status == 'completed':
                                    torrent.status = 'finished'
                                    torrent.displayStatus()
                                    torrent.displayFinished()
                                    torrent.termevt = torrent.prevtermevt = time()
                                    torrent.displayMessage(torrent.termEvtMsg())
                                    self.torrentfinished += 1
                                    self.updateRunningTorrentCounters()
                            elif torrent.status == 'finished':
                                torrent.status = 'completed'
                                torrent.displayStatus()
                                torrent.termevt = torrent.prevtermevt = ''
                                self.torrentfinished -= 1
                                self.updateRunningTorrentCounters()
                        elif uploadoption == 3:
                            if self.utility.seedingtimeresetonswitch == "1" and (olduploadoption == 0 or olduploadoption == 2) \
                               or self.utility.seedingtimeresetonvalue == "1" and newuploadtimemax != olduploadtimemax:
                                torrent.seedingtime = 0
                            if torrent.seedingtime >= newuploadtimemax or torrent.ratioIsReached():
                                if torrent.status == 'completed':
                                    torrent.status = 'finished'
                                    torrent.displayStatus()
                                    torrent.displayFinished()
                                    torrent.termevt = torrent.prevtermevt = time()
                                    torrent.displayMessage(torrent.termEvtMsg())
                                    self.torrentfinished += 1
                                    self.updateRunningTorrentCounters()
                            elif torrent.status == 'finished':
                                torrent.status = 'completed'
                                torrent.displayStatus()
                                torrent.termevt = torrent.prevtermevt = ''
                                self.torrentfinished -= 1
                                self.updateRunningTorrentCounters()
                        torrent.initSeedingTimeLeft()
                    else:
                        torrent.abcengine.countSeedingTimeLeft(now)

                # Immediately update ETA in list
                if updatetimeinfos and torrent.status != 'finished' and (torrent.abcengine
                   or torrent.status == 'standby' and self.utility.countsbasseed \
                   or self.abcparams['keepeta'] == "1"):
                    rank = self.getRankfromID(8)
                    if rank != -1:
                        if uploadoption == 0:
                            if olduploadoption != 0:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + "     oo")
                        elif torrent.status == 'standby' and self.utility.countsbasseed and torrent.forceseeding\
                           or torrent.abcengine and torrent.abcengine.forceseeding and torrent.abcengine.normalseedingover:
                            self.list.SetStringItem(index, rank, self.localize('etaS') + " " + self.localize('tillcc'))
                        elif uploadoption == 1 or uploadoption == 3:
                            if (torrent.status == 'standby' and self.utility.countsbasseed or torrent.seedingtimeleft or \
                                not torrent.forceseeding) \
                                or \
                               (not torrent.abcengine or torrent.seedingtimeleft or \
                                 not (torrent.abcengine.forceseeding and torrent.abcengine.normalseedingover)):
                                self.list.SetStringItem(index, rank, self.localize('etaS') + " " +
                                                        self.utility.eta_value(torrent.seedingtimeleft))
                        else:
                            if torrent.abcengine:
                                if torrent.seedingtimeleft or \
                                   not (torrent.abcengine.forceseeding and torrent.abcengine.normalseedingover):
                                    self.list.SetStringItem(index, rank, self.localize('etaS') + " " +
                                                            self.utility.eta_value(torrent.seedingtimeleft))
                            elif olduploadoption != 2 or uploadratio != olduploadratio:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + "      ?")

                # Init standby torrents to count seeding time if necessary
                if torrent.status == 'standby' and self.abcparams['mode'] == '1' and self.utility.countsbasseed \
                   and (olduploadoption == 0 or olduploadoption == 2) and (uploadoption == 1 or uploadoption == 3):
                    torrent.initStandbySeedingTime(now)

                # Update seeding time columns
                rank = self.getRankfromID(27)
                if rank != -1:
                    self.list.SetStringItem(index, rank, self.utility.time_value(torrent.seedingtime, seconds = False, maxvalue = False))
                rank = self.getRankfromID(28)
                if rank != -1:
                    self.list.SetStringItem(index, rank, self.utility.time_value(torrent.totalseedingtime, seconds = False, maxvalue = False))
            else:
                torrent.initSeedingTimeLeft()

            # Update torrent with other new parameters
            if info[0] is not None:
                torrent.shortlabel = info[0]
            if info[1] is not None:
                torrent.maxlocaldownloadrate = info[1]
            if info[2] is not None:
                torrent.prioritizedown = info[2]
                if not torrent.abcengine or torrent.status == 'pause' or torrent.status == 'onhold':
                    rank = self.getRankfromID(10)
                    if rank != -1:
                        self.list.SetStringItem(index, rank, self.utility.formatedRate(None, torrent.prioritizedown)[0])
                    if torrent.infowin is not None:
                        torrent.infowin.downrate.SetLabel(self.utility.rateWithUnit('', torrent.prioritizedown))
                    if torrent.detailwin is not None:
                        torrent.detailwin.downrate.SetLabel(self.utility.rateWithUnit('', torrent.prioritizedown))
            if info[3] is not None:
                torrent.maxupload = info[3]
                if torrent.abcengine and not torrent.abcengine.overmaxupwithoutdown:
                    torrent.abcengine.dow.setConns(torrent.maxupload)
            if info[4] is not None:
                torrent.maxlocaluploadrate = info[4]
            if info[5] is not None:
                torrent.prioritizeup = info[5]
                if not torrent.abcengine or torrent.status == 'pause' or torrent.status == 'onhold':
                    rank = self.getRankfromID(11)
                    if rank != -1:
                        self.list.SetStringItem(index, rank, self.utility.formatedRate(None, torrent.prioritizeup)[0])
                    if torrent.infowin is not None:
                        torrent.infowin.uprate.SetLabel(self.utility.rateWithUnit('', torrent.prioritizeup))
                    if torrent.detailwin is not None:
                        torrent.detailwin.uprate.SetLabel(self.utility.rateWithUnit('', torrent.prioritizeup))
            if info[10] is not None:
                torrent.timeoutswitch = info[10]
            if info[11] is not None:
                torrent.timeoutaction = info[11]
            if info[12] is not None:
                torrent.timeoutwork = info[12]
            if info[13] is not None:
                torrent.timeoutworkdown = info[13]
            if info[14] is not None:
                torrent.timeoutworktrack = info[14]
            if info[15] is not None:
                torrent.timeoutseed = info[15]
            if info[16] is not None:
                torrent.timeoutseedup = info[16]
            if info[17] is not None:
                torrent.timeoutseedtrack = info[17]
            if info[18] is not None:
                torrent.timeouttracker = info[18]
            if info[19] is not None:
                torrent.timeoutdownload = info[19]
            if info[20] is not None:
                torrent.timeoutupload = info[20]
            if info[21] is not None:
                torrent.movefolder = info[21]
            if info[23] is not None:
                torrent.exttrackerurl = info[23]
            if info[24] is not None and info[24] != torrent.checkinttrackwait:
                torrent.checkinttrackwait = info[24]
                if not torrent.checkinttrackwait and torrent.abcengine and torrent.abcengine.checkinginttrack:
                    mustswitchtracker = 1
                elif torrent.checkinttrackwait and torrent.abcengine and info[22] == 1 and torrent.exttracker and not torrent.complete: 
                    mustswitchtracker = 2
            if info[25] is not None:
                torrent.extrainttracker = info[25]
            if info[26] is not None:
                torrent.inttrackerurl = info[26]
            if info[22] is not None and info[22] != torrent.exttracker:
                if mustswitchtracker < 0:
                    mustswitchtracker = info[22]
                elif mustswitchtracker == 1 and info[22] == 0:
                    # To avoid switching to ext and then back to int
                    mustswitchtracker = -1
                    torrent.toggleCheckIntTrack()

            intextswitch = False
            if mustswitchtracker == 0:
                if torrent.abcengine and torrent.abcengine.checkinginttrack:
                    torrent.toggleCheckIntTrack()
                else:
                    torrent.switchTracker(0)
                    intextswitch = True
            elif mustswitchtracker == 1:
                if torrent.abcengine and self.utility.checkinttrack and torrent.checkinttrackwait and not torrent.complete:
                    torrent.toggleCheckIntTrack()
                else:
                    torrent.switchTracker(1)
                    intextswitch = True
            elif mustswitchtracker == 2:
                torrent.abcengine.checkinginttrack = True
                torrent.switchTracker(0)
                intextswitch = True

            if not intextswitch:
                if torrent.exttracker and (not torrent.abcengine or not torrent.abcengine.checkinginttrack):
                    if torrent.exttrackerurl != oldexttrackerurl:
                        if torrent.infowin is not None:
                            torrent.infowin.showExtTrackerUrl(torrent.exttrackerurl)
                        if torrent.status == 'standby':
                            self.trackerstandbycount[oldexttrackerurl] -= 1
                            if self.trackerstandbycount[oldexttrackerurl] == 0:
                                del self.trackerstandbycount[oldexttrackerurl]
                            if torrent.exttrackerurl in self.trackerstandbycount:
                                self.trackerstandbycount[torrent.exttrackerurl] += 1
                            else:
                                self.trackerstandbycount[torrent.exttrackerurl] = 1
                        elif torrent.abcengine:
                            torrent.reannounce(tracker = 'ext', url = torrent.exttrackerurl, force = True)
                elif torrent.extrainttracker != oldextrainttracker:
                    if torrent.extrainttracker:
                        if torrent.infowin is not None:
                            torrent.infowin.showExtTrackerUrl(torrent.inttrackerurl)
                        if torrent.status == 'standby':
                            self.trackerstandbycount[torrent.defaulttracker] -= 1
                            if self.trackerstandbycount[torrent.defaulttracker] == 0:
                                del self.trackerstandbycount[torrent.defaulttracker]
                            if torrent.inttrackerurl in self.trackerstandbycount:
                                self.trackerstandbycount[torrent.inttrackerurl] += 1
                            else:
                                self.trackerstandbycount[torrent.inttrackerurl] = 1
                        elif torrent.abcengine:
                            torrent.reannounce(tracker = 'ext', url = torrent.inttrackerurl, force = True)
                    else:
                        if torrent.infowin is not None:
                            torrent.infowin.hideExtTrackerUrl()
                        if torrent.status == 'standby':
                            self.trackerstandbycount[torrent.inttrackerurl] -= 1
                            if self.trackerstandbycount[torrent.inttrackerurl] == 0:
                                del self.trackerstandbycount[torrent.inttrackerurl]
                            if torrent.defaulttracker in self.trackerstandbycount:
                                self.trackerstandbycount[torrent.defaulttracker] += 1
                            else:
                                self.trackerstandbycount[torrent.defaulttracker] = 1
                        elif torrent.abcengine:
                            torrent.reannounce(tracker = 'int', force = True)
                elif torrent.extrainttracker and torrent.inttrackerurl != oldinttrackerurl:
                    if torrent.infowin is not None:
                        #torrent.infowin.exttrackerurl.SetValue(torrent.inttrackerurl)
                        torrent.infowin.showExtTrackerUrl(torrent.inttrackerurl)
                    if torrent.status == 'standby':
                        self.trackerstandbycount[oldinttrackerurl] -= 1
                        if self.trackerstandbycount[oldinttrackerurl] == 0:
                            del self.trackerstandbycount[oldinttrackerurl]
                        if torrent.inttrackerurl in self.trackerstandbycount:
                            self.trackerstandbycount[torrent.inttrackerurl] += 1
                        else:
                            self.trackerstandbycount[torrent.inttrackerurl] = 1
                    elif torrent.abcengine:
                        torrent.reannounce(tracker = 'ext', url = torrent.inttrackerurl, force = True)

                torrent.lastgetscrape = 0
                torrent.lastgetscrapemanual = 0
                if torrent.scraping:
                    self.getScrapeData(torrent, autoscraping = torrent.scrapingcontext[0], standby = torrent.scrapingcontext[1])

            rank = self.getRankfromID(23)
            if rank != -1:
                self.list.SetStringItem(index, rank, torrent.shortlabel)

            # Update stop seeding conditions column
            rank = self.getRankfromID(22)
            if rank != -1:
                if uploadoption == 0:
                    self.list.SetStringItem(index, rank, "oo")
                elif uploadoption == 1:
                    self.list.SetStringItem(index, rank, self.utility.time_value(newuploadtimemax, seconds = False, maxvalue = False))
                elif uploadoption == 2:
                    self.list.SetStringItem(index, rank, str(torrent.uploadratio) + "%")
                else:
                    self.list.SetStringItem(index, rank, self.utility.time_value(newuploadtimemax, seconds = False, maxvalue = False)
                                            + '/' + str(torrent.uploadratio) + "%")

            #    if torrent.maxupload < 4:
            #        max_initiate = min(12, int(self.abcparams['max_initiate']))
            #    elif torrent.maxupload < 30:
            #        max_initiate = min(40, int(self.abcparams['max_initiate']))
            #    else:
            #        max_initiate = min(torrent.maxupload + 10, int(self.abcparams['max_initiate']))
            #    torrent.abcengine.dow.setInitiate(max_initiate)

    def deleteInfoWin(self, index):
        torrent = self.proctab[index]
        if self.abcparams['savesizesonexit'] == '1':
            self.guiman.setMetaWidth(torrent.infowin.widthfornext)
            self.guiman.setMetaHeight(torrent.infowin.heightfornext)
        torrent.infowin = None
        if torrent.abcengine:
            torrent.abcengine.infowin = None

    def deleteDetailWin(self, index):
        torrent = self.proctab[index]
        if self.abcparams['savesizesonexit'] == '1':
            self.guiman.setDetailWidth(torrent.detailwin.widthfornext)
            self.guiman.setDetailHeight(torrent.detailwin.heightfornext)
        torrent.detailwin = None
        if torrent.abcengine:
            torrent.abcengine.detailwin = None

    def bgalloc(self, index):
        torrent = self.proctab[index]
        if not torrent.magnet and torrent.abcengine and torrent.abcengine.dow.storagewrapper:
            torrent.abcengine.dow.storagewrapper.bgalloc()

    def moveListIndex(self, index1, index2):
        # Move a line of the list from index1 to index2
        numcolumn = xrange(self.guiman.getNumCol())
        tempdisplay = []
        if self.abcparams['stripedlist'] == '1':
            backgrey = self.utility.colstripes
        else:
            backgrey = self.utility.colsyswin
        if index1 < index2:
            direction = 1
        else:
            direction = -1
        # Keep safe all data from index1
        tempdata = self.proctab[index1]
        for rank in numcolumn:
            tempdisplay.append(self.list.GetItem(index1, rank).GetText())
        # Delete data at index1
        del self.proctab[index1]
        self.list.DeleteItem(index1)
        # Insert data from index1 at index2
        self.proctab.insert(index2, tempdata)
        # Update display list
        self.list.InsertStringItem(index2, tempdisplay[0])
        for rank in numcolumn:
            self.list.SetStringItem(index2, rank, tempdisplay[rank])
        # Update data between index1 and index2
        for index in xrange(index1, index2 + direction, direction):
            torrent = self.proctab[index]
            # Update colour in display
            item = self.list.GetItem(index)
            if index % 2:
                item.SetBackgroundColour(backgrey)
            else:
                item.SetBackgroundColour(self.utility.colsyswin)
            if torrent.abcengine \
               and torrent.status != 'pause' \
               and torrent.status != 'onhold':
                item.SetTextColour(torrent.abcengine.colour)
            self.list.SetItem(item)
            # Update data in engine
            self.updateItemIndex(index)
        # Update bookmarks
        self.parent.updateBookmarks(index1, index2)

    def upDownListIndex(self, index, direction):
        # Move a line of the list one line up (direction = -1) or down (direction = 1) 
        numcolumn = xrange(self.guiman.getNumCol())
        tempdisplay = []
        index2 = index + direction
        # Keep safe all data from index
        temptorrent = self.proctab[index]
        for rank in numcolumn:
            tempdisplay.append(self.list.GetItem(index, rank).GetText())
        # Move data
        torrent = self.proctab[index] = self.proctab[index2]
        # Update display list
        tempstate = self.list.IsSelected(index)
        tempstate2 = self.list.IsSelected(index2)
        for rank in numcolumn:
            item = self.list.GetItem(index2, rank)
            item.SetId(index)
            self.list.SetItem(item)
        self.list.Select(index, tempstate2)
        # Store all data for first line
        self.proctab[index2] = temptorrent
        # Update display list
        for rank in numcolumn:
            self.list.SetStringItem(index2, rank, tempdisplay[rank])
        self.list.Select(index2, tempstate)
        # Background and text colour
        item = self.list.GetItem(index)
        item2 = self.list.GetItem(index2)
        if self.abcparams['stripedlist'] == '1':
            if index2 % 2:
                item.SetBackgroundColour(self.utility.colsyswin)
                item2.SetBackgroundColour(self.utility.colstripes)
            else:
                item.SetBackgroundColour(self.utility.colstripes)
                item2.SetBackgroundColour(self.utility.colsyswin)
        if torrent.abcengine \
           and torrent.status != 'pause' \
           and torrent.status != 'onhold':
            item.SetTextColour(torrent.abcengine.colour)
        else:
            item.SetTextColour(self.utility.colnoeng)
        if temptorrent.abcengine \
           and temptorrent.status != 'pause' \
           and temptorrent.status != 'onhold':
            item2.SetTextColour(temptorrent.abcengine.colour)
        else:
            item2.SetTextColour(self.utility.colnoeng)
        self.list.SetItem(item)
        self.list.SetItem(item2)
        self.updateItemIndex(index)
        self.updateItemIndex(index2)

    def updateItemIndex(self, index):
        # Change Index in ABCEngine, ABCDetailWin, ABCInfoWin
        torrent = self.proctab[index]
        torrent.index = index
        if torrent.abcengine:                
            torrent.abcengine.listid = index
        if torrent.detailwin is not None:
            torrent.detailwin.index = index
        if torrent.infowin is not None:
            torrent.infowin.index = index

    def updateItemDisplay(self, index):
        item = self.list.GetItem(index)
        torrent = self.proctab[index]
        if self.abcparams['stripedlist'] == '1':
            if index % 2:
                item.SetBackgroundColour(self.utility.colstripes)
            else:
                item.SetBackgroundColour(self.utility.colsyswin)
        else:
            item.SetBackgroundColour(self.utility.colsyswin)
        if torrent.abcengine \
           and torrent.status != 'pause' \
           and torrent.status != 'onhold':
            item.SetTextColour(torrent.abcengine.colour)
        self.list.SetItem(item)

    def moveItemsUp(self, listtomove):
        newloc = []
        for index in listtomove:
            if index == 0:       # First Item can't move up anymore
                newloc.append(index)
            elif index - 1 in newloc: # Don't move if we've already tried moving the item above
                newloc.append(index)
            else:
                self.upDownListIndex(index, -1)
                newloc.append(index - 1)

    def moveItemsDown(self, listtomove):
        listtomove.reverse()
        newloc = []
        for index in listtomove:
            if index == len(self.proctab) - 1: # Last Item can't move down anymore
                newloc.append(index)
            elif index + 1 in newloc: # Don't move if we've already moved the item below
                newloc.append(index)
            else:
                self.upDownListIndex(index, 1)
                newloc.append(index + 1)

    def moveItemsRow(self, selected, row):
        self.list.Freeze()
        for index in selected:
            self.moveListIndex(index, row)
        self.list.Thaw()

    def sortColumn(self, col = None, shift = False, restore = None, block = None):
        if self.parent.torrentlistloading or not self.proctab:
            return
        nbitem = len(self.proctab)
        if col is None or block is None:
            sortingrange = xrange(nbitem)
        else:
            sortingrange = xrange(block[0], block[1])
        self.utility.sortcol = []

        def storeSortingStringVal(col, func, phase):
            if block is not None:
                for index in xrange(0, block[0]):
                    self.list.SetItemData(index, index)
                    self.utility.sortcol.append(pre0)
            for index in sortingrange:
                torrent = self.proctab[index]
                self.list.SetItemData(index, index)
                if phase == 1:
                    if torrent.status == 'finished':
                        self.utility.sortcol.append(pre1 + func(torrent, index, col).lower())
                        continue
                    if sortinglock == "1":
                        phase = 3
                    else:
                        phase = 2
                if phase == 2:
                    if torrent.complete:
                        self.utility.sortcol.append(pre2 + func(torrent, index, col).lower())
                        continue
                    phase = 3
                if phase == 3:
                    self.utility.sortcol.append(pre3 + func(torrent, index, col).lower())
            if block is not None:
                for index in xrange(block[1], nbitem):
                    self.list.SetItemData(index, index)
                    self.utility.sortcol.append(pre4)

        def storeSortingFloatVal(col, func, rng, phase):
            if block is not None:
                for index in xrange(0, block[0]):
                    self.list.SetItemData(index, index)
                    self.utility.sortcol.append(- 3 * dir * rng)
            for index in sortingrange:
                torrent = self.proctab[index]
                self.list.SetItemData(index, index)
                if phase == 1:
                    if torrent.status == 'finished':
                        self.utility.sortcol.append(func(torrent, index, col) - 2 * dir * rng)
                        continue
                    if sortinglock == "1":
                        phase = 3
                    else:
                        phase = 2
                if phase == 2:
                    if torrent.complete:
                        self.utility.sortcol.append(func(torrent, index, col) - dir * rng)
                        continue
                    phase = 3
                if phase == 3:
                    self.utility.sortcol.append(func(torrent, index, col))
            if block is not None:
                for index in xrange(block[1], nbitem):
                    self.list.SetItemData(index, index)
                    self.utility.sortcol.append(dir * rng)

        def storeSortingUndoVal(restore = None):
            if restore is None:
                # Undo sorting in sorting actions cycle
                for index in xrange(nbitem):
                    self.list.SetItemData(index, index)
                    oldindex = self.proctab[index].previndex
                    if oldindex == -1:
                        oldindex = 1000000000
                    self.utility.sortcol.append(oldindex)
                return True

            torestore = False
            if restore == 10:
                # Undo restoring list order 0-9
                for index in xrange(nbitem):
                    self.list.SetItemData(index, index)
                    oldindex = self.proctab[index].restoreindexundo
                    if oldindex == -1:
                        oldindex = 1000000000
                    else:
                        torestore = True
                    self.utility.sortcol.append(oldindex)
            else:
                # Restore saved ordering configuration 0-9
                for index in xrange(nbitem):
                    self.list.SetItemData(index, index)
                    oldindex = self.proctab[index].restoreindex[restore]
                    if oldindex == -1:
                        oldindex = 1000000000
                    else:
                        torestore = True
                    self.utility.sortcol.append(oldindex)
            return torestore

        def storeSortingFinishedCompleted(completed = True):
            sortbytime = 2 * (self.abcparams['sortbytime'] == '1') - 1
            for index in xrange(nbitem):
                self.list.SetItemData(index, index)
                torrent = self.proctab[index]
                if torrent.status == 'finished':
                    if type(torrent.termevt) is float:
                        self.utility.sortcol.append(sortbytime * torrent.termevt)
                    else:
                        self.utility.sortcol.append(0)
                elif completed and torrent.complete:
                    if type(torrent.termevt) is float:
                        self.utility.sortcol.append(1e15 + sortbytime * torrent.termevt)
                    else:
                        self.utility.sortcol.append(1e15)
                else:
                    self.utility.sortcol.append(2e15)

        def valName(torrent, index, col):
            return torrent.name
        def valProgress(torrent, index, col):
            return torrent.progress
        def valStatus(torrent, index, col):
            return self.list.GetItem(index, col).GetText()
        def valPriority(torrent, index, col):
            return torrent.priority
        def valETA(torrent, index, col):
            etastring = self.list.GetItem(index, col).GetText()
            if etastring == '':
                return -8999999 * dir
            if etastring == downloading + '      ?' or etastring == seeding + '      ?':
                eta = 8999999 * dir
            elif etastring == downloading + '     oo' or etastring == seeding + '     oo':
                eta = 8999998
            elif etastring == seeding + ' ' + tillcc:
                # Until copy complete
                eta = 8999997
            elif etastring == downloading + ' >100' + l_day or etastring == seeding + ' >100' + l_day:
                eta = 8999996
            else:
                if etastring.endswith(second):
                    eta = int(etastring[-lensecond - 2:-lensecond]) + int(etastring[-lensecond - 5 - lenminute:-lensecond - 3 - lenminute]) * 60
                elif etastring.endswith(minute):
                    eta = int(etastring[-lenminute - 2:-lenminute]) * 60 + int(etastring[-lenminute - 5 - lenhour:-lenminute - 3 - lenhour]) * 3600
                else:
                    eta = int(etastring[-lenhour - 2:-lenhour]) * 3600 + int(etastring[-lenhour - 5 - lenday:-lenhour - 3 - lenday]) * 86400
            if not torrent.complete:
                eta += 9000000
            return eta
        def valSize(torrent, index, col):
            return torrent.requestedsize
        def valDnRate(torrent, index, col):
            if torrent.abcengine and torrent.status != 'pause' and torrent.status != 'onhold':
                return torrent.abcengine.downrate
            return 0
        def valUpRate(torrent, index, col):
            if torrent.abcengine and torrent.status != 'pause' and torrent.status != 'onhold':
                return torrent.abcengine.uprate
            return 0
        def valShareRatio(torrent, index, col):
            if torrent.abcengine:
                return torrent.abcengine.shareratio
            if torrent.requestedsize <= 0 and not torrent.magnet:
                return 999.9
            if torrent.progress <= 0:
                return 0.0
            return torrent.upsize / torrent.requestedsize / torrent.progress * 10000
        def valMessageLabel(torrent, index, col):
            return torrent.message[self.utility.timestamplen + self.utility.messageheaderlen:]
        def valMessageTime(torrent, index, col):
            return torrent.message[:self.utility.timestamplen]
        def valSeenSeeds(torrent, index, col):
            seeds = torrent.seed
            if seeds == self.utility.notrackeranswermarker or seeds == self.utility.notrackermarker:
                return 99999 * dir
            return int(seeds)
        def valConnectedSeeds(torrent, index, col):
            if torrent.abcengine:
                return int(torrent.abcengine.numseed)
            return 0
        def valSeenPeers(torrent, index, col):
            peers = torrent.peer
            if peers == self.utility.notrackeranswermarker or peers == self.utility.notrackermarker:
                return 99999 * dir
            return int(peers)
        def valConnectedPeers(torrent, index, col):
            if torrent.abcengine:
                return int(torrent.abcengine.numpeer)
            return 0
        def valNbCopies(torrent, index, col):
            copies = torrent.numcopy
            if copies == '?':
                return 99999 * dir
            return float(copies)
        def valAvgProgress(torrent, index, col):
            avgprog = self.list.GetItem(index, col).GetText()[:-1]
            if avgprog:
                return float(avgprog)
            return 0
        def valDnSize(torrent, index, col):
            if torrent.abcengine:
                return torrent.abcengine.olddownsize + torrent.abcengine.newdownsize
            return torrent.downsize
        def valUpSize(torrent, index, col):
            if torrent.abcengine:
                return torrent.abcengine.oldupsize + torrent.abcengine.newupsize
            return torrent.upsize
        def valTotalSpeed(torrent, index, col):
            if torrent.abcengine and torrent.status != 'pause' and torrent.status != 'onhold':
                return torrent.abcengine.totalspeed
            return 0
        def valTorFileName(torrent, index, col):
            return path.split(torrent.src)[1]
        def valStopSeedingCond(torrent, index, col):
            # Sorting order is : first from time and %, time, %, infinite time
            if torrent.uploadoption == 3:
                return '\x00' + self.list.GetItem(index, col).GetText()
            if torrent.uploadoption == 1:
                return '\x01' + self.list.GetItem(index, col).GetText()
            if torrent.uploadoption == 2:
                return '\x02' + self.list.GetItem(index, col).GetText()
            return self.list.GetItem(index, col).GetText()
        def valShortLabel(torrent, index, col):
            return torrent.shortlabel
        def valLabel(torrent, index, col):
            return torrent.label
        def valActivity(torrent, index, col):
            return torrent.activity
        def valDestination(torrent, index, col):
            return torrent.dest
        def valSeedingTime(torrent, index, col):
            return torrent.seedingtime
        def valTotalSeedingTime(torrent, index, col):
            return torrent.totalseedingtime

        if restore is not None:
            if restore == 11:
                # Sort Finished and Completed at top of list
                storeSortingFinishedCompleted()
            elif restore == 12:
                # Sort Finished at top of list
                storeSortingFinishedCompleted(False)
            else:
                # Restore saved sorting order 0-9 or undo restoring
                if not storeSortingUndoVal(restore):
                    return False
                self.sortdir = 2
        elif col is not None:
            # Cycle through sorting actions
            id = self.guiman.getIDfromRank(col)
            if id == self.lastsortingid and shift == self.lastsortingshift \
               and block == self.lastsortingblock:
                if self.sortdir == 2:
                    # Save undo sorting configuration
                    for index in xrange(nbitem):
                        self.proctab[index].previndex = index
                if self.guiman.getSortingLock(id) == 0:
                    if self.sortdir == 2:
                        self.sortdir = self.guiman.getSortingOrder(id)
                    elif self.guiman.getSortingOrder(id):
                        if self.sortdir == 0:
                            self.sortdir = 2
                        else:
                            self.sortdir = 0
                    else:
                        self.sortdir += 1
                elif self.sortdir == 2:
                    self.sortdir = self.guiman.getSortingOrder(id)
                else:
                    self.sortdir = 2
            else:
                if self.lastsortingid == -1 or block != self.lastsortingblock:
                    # Save undo sorting configuration
                    for index in xrange(nbitem):
                        self.proctab[index].previndex = index
                self.sortdir = self.guiman.getSortingOrder(id)

            self.lastsortingid = id
            self.lastsortingshift = shift
            self.lastsortingblock = block

            if self.sortdir == 2:
                # Undo sorting
                storeSortingUndoVal()
            else:
                # Sorting direction (+1 or -1) and headers to force string sorting
                if self.sortdir == 0:
                    dir = 1
                    pre0 = u'\u0000'
                    pre1 = u'\u0001'
                    pre2 = u'\u0002'
                    pre3 = u'\u0003'
                    pre4 = u'\u0004'
                else:
                    dir = -1
                    pre0 = u'\uFFFF'
                    pre1 = u'\uFFFE'
                    pre2 = u'\uFFFD'
                    pre3 = u'\uFFFC'
                    pre4 = u'\uFFFB'
                sortinglock = self.abcparams['sortinglock']
                if sortinglock == "2":
                    # Do the same for completed torrents just below finished torrents
                    phase = 2
                elif sortinglock == "0":
                    phase = 3
                else:
                    # If finished torrent are moved to top, keep at top the ones that are already at top while sorting
                    phase = 1
        
                if id == 4:
                    # Torrent name
                    storeSortingStringVal(col, valName, phase)
                elif id == 5:
                    # Progress
                    storeSortingFloatVal(col, valProgress, 200, phase)
                elif id == 6:
                    # Status
                    storeSortingStringVal(col, valStatus, phase)
                elif id == 7:
                    # Priority
                    storeSortingFloatVal(col, valPriority, 5, phase)
                elif id == 8:
                    # ETA
                    lenday = len(self.localize('l_day'))
                    lenhour = len(self.localize('l_hour'))
                    minute = self.localize('l_minute')
                    lenminute = len(minute)
                    second = self.localize('l_second')
                    lensecond = len(second)
                    downloading = self.localize('etaD')
                    seeding = self.localize('etaS')
                    tillcc = self.localize('tillcc')
                    l_day = self.localize('l_day')
                    storeSortingFloatVal(col, valETA, 27000000, phase)
                elif id == 9:
                    # Size
                    storeSortingFloatVal(col, valSize, 1000000000000, phase)
                elif id == 10:
                    # Down rate
                    storeSortingFloatVal(col, valDnRate, 1000000000, phase)
                elif id == 11:
                    # Up rate
                    storeSortingFloatVal(col, valUpRate, 1000000000, phase)
                elif id == 12:
                    # % Up/Down
                    storeSortingFloatVal(col, valShareRatio, 1000000, phase)
                elif id == 13:
                    # Message
                    if shift:
                        # Message label
                        storeSortingStringVal(col, valMessageLabel, phase)
                    else:
                        # Message time
                        storeSortingStringVal(col, valMessageTime, phase)
                elif id == 14:
                    # # seeds
                    if shift:
                        # Total seen # seeds
                        storeSortingFloatVal(col, valSeenSeeds, 100000, phase)
                    else:
                        # # connected seeds
                        storeSortingFloatVal(col, valConnectedSeeds, 10000, phase)
                elif id == 15:
                    # # peers
                    if shift:
                        # Total seen # peers
                        storeSortingFloatVal(col, valSeenPeers, 100000, phase)
                    else:
                        # # connected peers
                        storeSortingFloatVal(col, valConnectedPeers, 10000, phase)
                elif id == 16:
                    # # copies
                        storeSortingFloatVal(col, valNbCopies, 100000, phase)
                elif id == 17:
                    # Peer average progress
                    storeSortingFloatVal(col, valAvgProgress, 200, phase)
                elif id == 18:
                    # Download size
                    storeSortingFloatVal(col, valDnSize, 1000000000000, phase)
                elif id == 19:
                    # Upload size
                    storeSortingFloatVal(col, valUpSize, 1000000000000, phase)
                elif id == 20:
                    # Total speed
                    storeSortingFloatVal(col, valTotalSpeed, 1000000000, phase)
                elif id == 21:
                    # Torrent file name
                    storeSortingStringVal(col, valTorFileName, phase)
                elif id == 22:
                    # Stop seeding conditions (Sorting order is : first from time and %, time, %, infinite time) 
                    storeSortingStringVal(col, valStopSeedingCond, phase)
                elif id == 23:
                    # Short label
                    storeSortingStringVal(col, valShortLabel, phase)
                elif id == 24:
                    # Label
                    storeSortingStringVal(col, valLabel, phase)
                elif id == 25:
                    # Activity
                    storeSortingFloatVal(col, valActivity, 2, phase)
                elif id == 26:
                    # Destination
                    storeSortingStringVal(col, valDestination, phase)
                elif id == 27:
                    # Seeding time
                    storeSortingFloatVal(col, valSeedingTime, 500000000, phase)
                elif id == 28:
                    # Total seeding time
                    storeSortingFloatVal(col, valTotalSeedingTime, 500000000, phase)
        else:
            return False

        if restore is not None or self.sortdir == 0 or self.sortdir == 2:
            self.list.SortItems(self.utility.sortList)
        else:
            self.list.SortItems(self.utility.sortListReverse)

        # Update proctab
        proctab2 = self.proctab[:]
        self.proctab[:] = []
        for index in xrange(nbitem):
            self.proctab.append(proctab2[self.list.GetItemData(index)])
            self.updateItemIndex(index)
        # Update background and text colour
        for index in xrange(nbitem):
            torrent = self.proctab[index]
            item = self.list.GetItem(index)
            if self.abcparams['stripedlist'] == '1':
                if index % 2:
                    item.SetBackgroundColour(self.utility.colstripes)
                else:
                    item.SetBackgroundColour(self.utility.colsyswin)
            if torrent.abcengine \
               and torrent.status != 'pause' \
               and torrent.status != 'onhold':
                item.SetTextColour(torrent.abcengine.colour)
            else:
                item.SetTextColour(self.utility.colnoeng)
            self.list.SetItem(item)
        return True

    def updateColumns(self, old_colid, tobeupdated):
        for index in xrange(len(self.proctab)):
            torrent = self.proctab[index]
            if not torrent.abcengine \
               or torrent.status == 'pause' \
               or torrent.status == 'onhold':
                for rank in tobeupdated:
                    ID = old_colid[rank]
                    if ID == 4:     # Name
                        self.list.SetStringItem(index, rank, torrent.name)
                    elif ID == 5:   # Progress
                        self.list.SetStringItem(index, rank, torrent.formatedProgress())
                    elif ID == 6:   # Status
                        if torrent.ismovingdata:
                            self.list.SetStringItem(index, rank, self.localize('moving'))
                        else:
                            self.list.SetStringItem(index, rank, self.localize(torrent.status))
                    elif ID == 7:   # Priority
                        self.list.SetStringItem(index, rank, self.utility.priorities_s[torrent.priority])
                    elif ID == 8:   # ETA
                        if self.abcparams['keepeta'] == "1" and torrent.status != 'finished':
                            if torrent.complete:
                                if torrent.uploadoption == 0:
                                    self.list.SetStringItem(index, rank, self.localize('etaS') + '     oo')
                                elif torrent.uploadtillcopycomplete and \
                                     ((torrent.uploadoption == 1 or torrent.uploadoption == 3) and torrent.seedingtimeleft == 0 \
                                     or (torrent.uploadoption == 2 or torrent.uploadoption == 3) and torrent.ratioIsReached()):
                                    self.list.SetStringItem(index, rank, self.localize('etaS') + " "
                                                            + self.localize('tillcc'))
                                elif torrent.uploadoption == 2:
                                    self.list.SetStringItem(index, rank, self.localize('etaS') + "      ?")
                                else:
                                    self.list.SetStringItem(index, rank, self.localize('etaS') + " "
                                                            + self.utility.eta_value(torrent.seedingtimeleft))
                            else:
                                self.list.SetStringItem(index, rank, self.localize('etaD') + "      ?")
                    elif ID == 9:   # Size
                        if torrent.magnet:
                            self.list.SetStringItem(index, rank, '?')
                        else:
                            self.list.SetStringItem(index, rank, self.utility.formatedSize(torrent.requestedsize)[0])
                    elif ID == 10:   # Down rate
                        self.list.SetStringItem(index, rank, self.utility.formatedRate(None, torrent.prioritizedown)[0])
                    elif ID == 11:   # Up rate
                        self.list.SetStringItem(index, rank, self.utility.formatedRate(None, torrent.prioritizeup)[0])
                    elif ID == 12:  # %U/D Size
                        if torrent.requestedsize <= 0 and not torrent.magnet:
                            self.list.SetStringItem(index, rank, '999.9%')
                        elif torrent.progress <= 0:
                            self.list.SetStringItem(index, rank, '0.0%')
                        else:
                            self.list.SetStringItem(index, rank, str(.1 * int(torrent.upsize / torrent.requestedsize
                                                                              / torrent.progress * 100000)) + "%")
                    elif ID == 13:  # Message
                        self.list.SetStringItem(index, rank, torrent.message)
                    elif ID == 14:  # Seeds
                        if torrent.abcengine:
                            self.list.SetStringItem(index, rank, torrent.abcengine.numseedtext)
                        elif torrent.numcopy != '?' or torrent.scrapedonce:
                            self.list.SetStringItem(index, rank, '0 (' + torrent.seed + ')')
                    elif ID == 15:  # Peers
                        if torrent.abcengine:
                            self.list.SetStringItem(index, rank, torrent.abcengine.numpeertext)
                        elif torrent.numcopy != '?' or torrent.scrapedonce:
                            self.list.SetStringItem(index, rank, '0 (' + torrent.peer + ')')
                    elif ID == 16:  # Copies
                        if torrent.numcopy != '?':
                            self.list.SetStringItem(index, rank, torrent.numcopy)
                    elif ID == 17:  # Peer avg progress
                        if torrent.abcengine:
                            self.list.SetStringItem(index, rank, torrent.abcengine.peeravgtext)
                    elif ID == 18:  # DL size
                        self.list.SetStringItem(index, rank, self.utility.formatedSize(torrent.downsize)[0])
                    elif ID == 19:  # UL size
                        self.list.SetStringItem(index, rank, self.utility.formatedSize(torrent.upsize)[0])
                    elif ID == 20:  # Total speed
                        if torrent.abcengine:
                            self.list.SetStringItem(index, rank, torrent.abcengine.totalspeedtext[0])
                    elif ID == 21:  # Torrent file name
                        self.list.SetStringItem(index, rank, path.split(torrent.src)[1])
                    elif ID == 22:  # Stop seeding conditions
                        if torrent.uploadoption == 0:
                            self.list.SetStringItem(index, rank, "oo")
                        elif torrent.uploadoption == 1:
                            self.list.SetStringItem(index, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False))
                        elif torrent.uploadoption == 2:
                            self.list.SetStringItem(index, rank, str(torrent.uploadratio) + "%")
                        else:
                            self.list.SetStringItem(index, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False)
                                                    + '/' + str(torrent.uploadratio) + "%")
                    elif ID == 23:  # Short label
                        self.list.SetStringItem(index, rank, torrent.shortlabel)
                    elif ID == 24:  # Label
                        self.list.SetStringItem(index, rank, torrent.label)
                    elif ID == 25:  # Activity
                        self.list.SetStringItem(index, rank, torrent.activityMarker())
                    elif ID == 26:  # Destination
                        self.list.SetStringItem(index, rank, torrent.getRebuiltDest())
                    elif ID == 27:  # Seeding time
                        self.list.SetStringItem(index, rank, self.utility.time_value(torrent.seedingtime, maxvalue = False))
                    elif ID == 28:  # Total seeding time
                        self.list.SetStringItem(index, rank, self.utility.time_value(torrent.totalseedingtime, maxvalue = False))
            else:
                for rank in tobeupdated:
                    ID = old_colid[rank]
                    if ID == 4:     # Name
                        self.list.SetStringItem(index, rank, torrent.name)
                    elif ID == 6:   # Status
                        if torrent.ismovingdata:
                            self.list.SetStringItem(index, rank, self.localize('moving'))
                        else:
                            self.list.SetStringItem(index, rank, self.localize(torrent.status))
                    elif ID == 7:   # Priority
                        self.list.SetStringItem(index, rank, self.utility.priorities_s[torrent.priority])
                    elif ID == 8:   # ETA
                        if torrent.status != 'finished' and torrent.complete:
                            if torrent.uploadoption == 0:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + '     oo')                       
                            elif torrent.abcengine.forceseeding and torrent.abcengine.normalseedingover:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + " "
                                                        + self.localize('tillcc'))
                    elif ID == 9:   # Size
                        if torrent.magnet:
                            self.list.SetStringItem(index, rank, '?')
                        else:
                            self.list.SetStringItem(index, rank, self.utility.formatedSize(torrent.requestedsize)[0])
                    elif ID == 13:  # Message
                        torrent.clearMessage(rank = rank)
                    elif ID == 21:  # Torrent file name
                        self.list.SetStringItem(index, rank, path.split(torrent.src)[1])
                    elif ID == 22:  # Stop seeding conditions
                        if torrent.uploadoption == 0:
                            self.list.SetStringItem(index, rank, "oo")
                        elif torrent.uploadoption == 1:
                            self.list.SetStringItem(index, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False))
                        elif torrent.uploadoption == 2:
                            self.list.SetStringItem(index, rank, str(torrent.uploadratio) + "%")
                        else:
                            self.list.SetStringItem(index, rank, self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False)
                                                    + '/' + str(torrent.uploadratio) + "%")
                    elif ID == 23:  # Short label
                        self.list.SetStringItem(index, rank, torrent.shortlabel)
                    elif ID == 24:  # Label
                        self.list.SetStringItem(index, rank, torrent.label)
                    elif ID == 25:  # Activity
                        self.list.SetStringItem(index, rank, torrent.activityMarker())
                    elif ID == 26:  # Destination
                        self.list.SetStringItem(index, rank, torrent.getRebuiltDest())
                    elif ID == 27:  # Seeding time
                        self.list.SetStringItem(index, rank, self.utility.time_value(torrent.seedingtime, maxvalue = False))
                    elif ID == 28:  # Total seeding time
                        self.list.SetStringItem(index, rank, self.utility.time_value(torrent.totalseedingtime, maxvalue = False))

    def updateMetaDetailWinSize(self):
        for torrent in self.proctab:
            if torrent.infowin is not None:
                torrent.infowin.setSize((self.guiman.getMetaWidth(), self.guiman.getMetaHeight()))
            if torrent.detailwin is not None:
                torrent.detailwin.setSize((self.guiman.getDetailWidth(), self.guiman.getDetailHeight()))

    def closeAllDetailWindows(self):
        for torrent in self.proctab:
            if torrent.detailwin is not None:
                torrent.detailwin.killAdv()                
            if torrent.infowin is not None:
                torrent.infowin.killInfo()

    def clearScheduler(self, exitflag):
        # Stop DHT
        if self.dht:
            self.stopDHT()

        # Stop cyclical garbage collecting timer
        if self.utility.cyclicalgcrate != 0:
            try:
                self.timercyclicalgc.cancel()
            except:
                pass

        # Stop cyclical expired buffers timer
        try:
            self.timerexpirebuffers.cancel()
        except:
            pass

        # Stop standby seeding
        if self.timerstandbyseedingtime:
            self.stopStandbySeeding()

        # Stop cyclical tasks timer
        try:
            self.timercyclicaltasks.cancel()
        except:
            pass

        # Stop cyclical torrent list file updating timer
        try:
            self.timerupdatetorrentlist.cancel()
        except:
            pass

        # Close popup loaded and stop timer
        if self.popuploaded:
            self.popuploaded.Destroy()
            self.popuploaded = None
            try:
                self.popuploadedtimer.cancel()
            except:
                pass

        # Stop all torrents
        # Start shutting down in advance all raw servers to speed up queuing of the list of torrents
        for torrent in self.proctab:
            if torrent.abcengine:
                torrent.abcengine.stopBT()
        now = clock()
        for torrent in self.proctab:
            if torrent.abcengine:
                if torrent.isCheckingOrAllocating() or torrent.isWaiting():
                    # Restore terminate event
                    torrent.termevt = torrent.prevtermevt
                else:
                    torrent.progress = torrent.abcengine.progress
                if torrent.abcengine.onlycheck:
                    # To remember the manual checking status even after the abcengine is terminated
                    torrent.status = 'onlycheck'
                else:
                    torrent.inactivitytime, torrent.trackererrortime = torrent.abcengine.getTimeoutTimes(now)
                    if torrent.status == 'completed':
                        # To remember the seeding status even after the abcengine is terminated
                        torrent.status = 'seeding'
                    elif torrent.abcengine.sbwaitingforleechers:
                        self.standbychecking = False
                        self.torrentwaitingforleechers = None
                        # To remember the standby status even after the abcengine is terminated
                        torrent.status = 'sbwaitingforleechers'
                torrent.abcengine.stop()
                if torrent in self.torrentalive:
                    self.torrentalive.remove(torrent)
                if torrent.status != 'onlycheck':
                    self.torrentrunning -= 1
            # Save progress file
            torrent.writeFileProgress()
            # Save peers
            torrent.writePeers()
            if torrent.delayedresumetimer:
                torrent.delayedresumetimer.cancel()
                torrent.delayedresumetimer = None
                if torrent.status == 'completed':
                    torrent.status = 'queue'

        # Stop peer listener
        if self.pl:
            self.stopPL()

        # Volume counters and volume check cycle
        self.abcparams['downvol'] = str(self.downvolfixedwin)
        self.abcparams['upvol'] = str(self.upvolfixedwin)
        self.abcparams['downvolrate'] = str(self.downvolmeasure.get_rate())
        self.abcparams['upvolrate'] = str(self.upvolmeasure.get_rate())
        self.abcparams['downvollast'] = str(self.downvolmeasure.last + self.utility.time0)
        self.abcparams['upvollast'] = str(self.upvolmeasure.last + self.utility.time0)
        self.abcparams['downvolstart'] = str(self.downvolmeasure.start + self.utility.time0)
        self.abcparams['upvolstart'] = str(self.upvolmeasure.start + self.utility.time0)

        self.startqueue.append('END')

        exitflag.set()

    def changeProcDest(self, torrent, dest, rentorrent):
        if torrent not in self.proctab:
            return
        torrent.updateDestination(dest)
        if rentorrent:
            # Update torrent name if necessary
            torrent.updateName(path.split(dest)[1])

    def getScrapeData(self, torrent, autoscraping = False, standby = False, ctrlscrape = False):
        torrent.lastscrapetime = clock()
        if autoscraping:
            if not standby:
                torrent.lastgetscrape = torrent.lastscrapetime
        else:
            torrent.lastgetscrapemanual = torrent.lastscrapetime
        if torrent.getTracker() is None:
            self.invokeLater(self.displayScrapeResult, [torrent, self.localize('notracker'), self.utility.notrackermarker,
                             self.utility.notrackermarker, autoscraping, standby])
        else:
            torrent.scraping = True
            torrent.scrapingcontext = [autoscraping, standby]
            if not autoscraping and not standby:
                if self.abcparams['displayscrapingmsg'] == '1':
                    msg = strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('scraping')
                else:
                    msg = None
                torrent.updateScrapeData(self.utility.scrapingmarker, self.utility.scrapingmarker,
                                         False, msg)
            torrent.scrapethread = ScrapeThread(self, torrent, torrent.lastscrapetime, autoscraping = autoscraping, standby = standby, ctrlscrape = ctrlscrape)
            torrent.scrapethread.start()

    def displayScrapeResult(self, torrent, status, seed, peer, autoscraping = False, standby = False, trackerok = None, external = False, scrapetime = None):
        # Display scrape result in list
        if torrent not in self.proctab:
            # The torrent is no longer in the list
            if standby:
                self.standbychecking = False
            return
        index = torrent.index

        if scrapetime and scrapetime < torrent.lastscrapetime:
            # Garbage answers from obsolete scrapes
            return

        torrent.scraping = False

        if trackerok is not None and not torrent.abcengine and torrent.multitracker and not external:
            torrent.setDefaultTracker(trackerok)

        if self.abcparams['displayscrapestatus'] == '1':
            msg = strftime(self.utility.timestamp) + self.utility.messageheader + status
            oldmsg = None
        else:
            msg = ''
            oldmsg = self.localize('scraping')

        torrent.updateScrapeData(seed, peer, autoscraping, msg, oldmsg)

        # If #seeds or #peers are not displayed, also display results in standalone window
        if (torrent.detailwin is None or not torrent.detailwin.IsShown()) and not autoscraping \
            and (self.getRankfromID(14) == -1 or self.getRankfromID(15) == -1):
            parent = wx.GetActiveWindow()
            if parent is None:
                parent = self.parent
            scrapedlg = ScrapeDialog(parent, -1, self.localize('currentseedpeerontracker'), torrent, status, seed, peer)
            scrapedlg.ShowModal()
            scrapedlg.Destroy()

        if standby:
            if torrent.status == 'standby' and self.abcparams['mode'] == '1' and self.canBeScheduled(torrent):
                # Check seeders and leechers for restarting torrent
                if torrent.complete:
                    if torrent.peer == self.utility.notrackermarker:
                        if self.dht and not torrent.private:
                            self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                        else:
                            self.standbychecking = False
                    elif torrent.peer == self.utility.notrackeranswermarker:
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                    elif int(torrent.peer) > 0:
                        self.standbychecking = False
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True)
                    elif self.dht and not torrent.private:
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                    else:
                        self.standbychecking = False
                else:
                    if torrent.seed == self.utility.notrackermarker:
                        if self.dht and not torrent.private:
                            self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                        else:
                            self.standbychecking = False
                    elif torrent.seed == self.utility.notrackeranswermarker:
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                    elif int(torrent.seed) > 0:
                        self.standbychecking = False
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True)
                    elif int(torrent.peer) > 0 or self.dht and not torrent.private:
                        self.procRESUMEsingle(index, int(self.abcparams['fastresume']), fromstandby = True, sbwaitingforleechers = True)
                    else:
                        self.standbychecking = False
            else:
                self.standbychecking = False

    def displayCheckResult(self, torrent, completedstatus = None, error = False):
        # Display check result in list
        if torrent not in self.proctab:
            # The torrent is no longer in the list
            return
        index = torrent.index

        if not torrent.magnet:
            if torrent.complete:
                if completedstatus == 'completed':
                    # Torrent is completed after a check
                    if error:
                        torrent.status = completedstatus
                    elif torrent.oldstatusbeforecheck == 'queue':
                        torrent.status = 'queue'
                        self.torrentqueued += 1
                    elif torrent.oldstatusbeforecheck == 'standby':
                        torrent.status = 'standby'
                        self.addTorrentStandby(torrent)
                    else:
                        torrent.status = completedstatus
                    torrent.displayStatus()
                    if self.abcparams['keepeta'] == "1":
                        # Display ETA
                        rank = self.getRankfromID(8)
                        if rank != -1:
                            if torrent.uploadoption == 0:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + "     oo")
                            elif torrent.uploadoption == 2:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + "      ?")
                            else:
                                self.list.SetStringItem(index, rank, self.localize('etaS') + " " + self.utility.eta_value(torrent.seedingtimeleft))
                elif completedstatus == 'finished':
                    # Torrent is finished after a check
                    torrent.displayStatus()
                    torrent.displayFinished()
            else:
                # Set status after checking
                if torrent.oldstatusbeforecheck == 'finished' or torrent.oldstatusbeforecheck == 'completed':
                    torrent.status = 'stop'
                else:
                    torrent.status = torrent.oldstatusbeforecheck
                    if torrent.oldstatusbeforecheck == 'queue':
                        self.torrentqueued += 1
                    elif torrent.oldstatusbeforecheck == 'standby':
                        self.addTorrentStandby(torrent)
                torrent.displayStatus()
                torrent.termevt = torrent.prevtermevt = ''
                rank = self.getRankfromID(5)
                if rank != -1:
                    self.list.SetStringItem(index, rank, torrent.formatedProgress())
                # Update detail windows
                if torrent.infowin is not None:
                    torrent.infowin.totalprogress.SetLabel(torrent.formatedProgress())
                if torrent.detailwin is not None:
                    torrent.detailwin.totalprogress.SetLabel(torrent.formatedProgress())
                if self.abcparams['keepeta'] == "1":
                    # Display ETA
                    rank = self.getRankfromID(8)
                    if rank != -1:
                        self.list.SetStringItem(index, rank, self.localize('etaD') + "      ?")

        # Refresh progress for files inside the torrent in the metainfo window
        if torrent.infowin is not None and not torrent.singlefile:
            torrent.infowin.refreshFileProgress()
        # Checking done message
        if self.abcparams['displaycheckstatus'] == '1':
            torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('checkingdone'))

    def grabTorrentFieldsForWeb(self, fieldidlist = None):
        if fieldidlist == None:
            fieldidlist = xrange(4, self.guiman.maxid)
        tillcc = " " + self.localize('tillcc')
        retmsg = ''
        for torrent in self.proctab:
            if torrent.abcengine and torrent.status != 'pause' and torrent.status != 'onhold':
                retmsg += torrent.abcengine.getInfo(fieldidlist)
            else:
                for field in fieldidlist:
                    if field == 4:
                        retmsg += torrent.name + "|"
                    elif field == 5:
                        retmsg += torrent.formatedProgress() + "|"
                    elif field == 6:
                        if torrent.ismovingdata:
                            retmsg += self.localize('moving') + "|"
                        else:
                            retmsg += self.localize(torrent.status) + "|"
                    elif field == 7:
                        retmsg += self.utility.priorities_s[torrent.priority] + "|"
                    elif field == 8:
                        if torrent.status != 'finished':
                            if torrent.complete:
                                if torrent.uploadoption == 0:
                                    retmsg += self.localize('etaS') + '     oo|'
                                elif torrent.uploadtillcopycomplete and \
                                     ((torrent.uploadoption == 1 or torrent.uploadoption == 3) and torrent.seedingtimeleft == 0 \
                                     or (torrent.uploadoption == 2 or torrent.uploadoption == 3) and torrent.ratioIsReached()):
                                    retmsg += self.localize('etaS') + tillcc + "|"
                                elif torrent.uploadoption == 2:
                                    retmsg += self.localize('etaS') + "      ?|"
                                else:
                                    retmsg += self.localize('etaS') + " " + self.utility.eta_value(torrent.seedingtimeleft) + "|"
                            else:
                                retmsg += self.localize('etaD') + "      ?|"
                        else:
                            retmsg += "|"
                    elif field == 9:
                        if torrent.magnet:
                            retmsg += "?|"
                        else:
                            retmsg += self.utility.sizeWithUnit(self.utility.formatedSize(torrent.requestedsize)) + "|"
                    elif field == 12:
                        if torrent.requestedsize <= 0 and not torrent.magnet:
                            retmsg += '999.9%|'
                        elif torrent.progress <= 0:
                            retmsg += '0.0%|'
                        else:
                            retmsg += str(.1 * int(torrent.upsize / torrent.requestedsize
                                                / torrent.progress * 100000)) + "%|"
                    elif field == 13:
                        retmsg += torrent.message + "|"
                    elif field == 14:
                        if torrent.abcengine:
                            retmsg += torrent.abcengine.numseedtext + "|"
                        elif torrent.numcopy != '?' or torrent.scrapedonce:
                            retmsg += "0 (" + torrent.seed + ")|"
                        else:
                            retmsg += "|"
                    elif field == 15:
                        if torrent.abcengine:
                            retmsg += torrent.abcengine.numpeertext + "|"
                        elif torrent.numcopy != '?' or torrent.scrapedonce:
                            retmsg += "0 (" + torrent.peer + ")|"
                        else:
                            retmsg += "|"
                    elif field == 16:
                        if torrent.numcopy != '?':
                            retmsg += torrent.numcopy + "|"
                        else:
                            retmsg += "|"
                    elif field == 17:
                        if torrent.abcengine:
                            retmsg += torrent.abcengine.peeravgtext + "|"
                        else:
                            retmsg += "|"
                    elif field == 18:
                        retmsg += self.utility.sizeWithUnit(self.utility.formatedSize(torrent.downsize)) + "|"
                    elif field == 19:
                        retmsg += self.utility.sizeWithUnit(self.utility.formatedSize(torrent.upsize)) + "|"
                    elif field == 20:
                        if torrent.abcengine:
                            retmsg += torrent.abcengine.totalspeedtext[0] + "|"
                        else:
                            retmsg += "|"
                    elif field == 21:
                        retmsg += path.split(torrent.src)[1] + "|"
                    elif field == 22:
                        if torrent.uploadoption == 0:
                            retmsg += "oo|"
                        elif torrent.uploadoption == 1:
                            retmsg += self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False) + "|"
                        elif torrent.uploadoption == 2:
                            retmsg += str(torrent.uploadratio) + "%|"
                        else:
                            retmsg += self.utility.time_value(torrent.uploadtimemax, seconds = False, maxvalue = False) + \
                                      str(torrent.uploadratio) + "%|"
                    elif field == 23:
                        retmsg += torrent.shortlabel + "|"
                    elif field == 24:
                        retmsg += torrent.label + "|"
                    elif field == 25:
                        retmsg += torrent.activityMarker() + "|"
                    elif field == 26:
                        retmsg += torrent.getRebuiltDest() + "|"
                    elif field == 27:
                        retmsg += self.utility.time_value(torrent.seedingtime, maxvalue = False) + "|"
                    elif field == 28:
                        retmsg += self.utility.time_value(torrent.totalseedingtime, maxvalue = False) + "|"
                    else:
                        retmsg += "|"
            retmsg += hexlify(torrent.infohash) + "\n"
        return retmsg

    def queryField(self, fieldlist, connection):
        fieldidlist = []
        retmsg = ""
        for req_field in fieldlist:
            try:
                req_field_int = int(req_field)
                if 3 < req_field_int < self.guiman.maxid:
                    retmsg += self.guiman.getText(req_field_int) + "|"
                    fieldidlist.append(req_field_int)
                else:
                    # Error
                    connection.send("Feedback\nError=Unknown field identifier : " + self.utility.encodeWebServiceData(req_field))
                    connection.close()
                    return
            except:
                # Old style query, only old column names are supported
                if req_field == "filename":
                    retmsg += self.guiman.getText(4) + "|"
                    fieldidlist.append(4)
                elif req_field == "progress":
                    retmsg += self.guiman.getText(5) + "|"
                    fieldidlist.append(5)
                elif req_field == "btstatus":
                    retmsg += self.guiman.getText(6) + "|"
                    fieldidlist.append(6)
                elif req_field == "priority":
                    retmsg += self.guiman.getText(7) + "|"
                    fieldidlist.append(7)
                elif req_field == "eta":
                    retmsg += self.guiman.getText(8) + "|"
                    fieldidlist.append(8)
                elif req_field == "totalsize":
                    retmsg += self.guiman.getText(9) + "|"
                    fieldidlist.append(9)
                elif req_field == "dlspeed":
                    retmsg += self.guiman.getText(10) + "|"
                    fieldidlist.append(10)
                elif req_field == "ulspeed":
                    retmsg += self.guiman.getText(11) + "|"
                    fieldidlist.append(11)
                elif req_field == "ratio":
                    retmsg += self.guiman.getText(12) + "|"
                    fieldidlist.append(12)
                elif req_field == "seeds":
                    retmsg += self.guiman.getText(14) + "|"
                    fieldidlist.append(14)
                elif req_field == "peers":
                    retmsg += self.guiman.getText(15) + "|"
                    fieldidlist.append(15)
                elif req_field == "copies":
                    retmsg += self.guiman.getText(16) + "|"
                    fieldidlist.append(16)
                elif req_field == "peeravgprogress":
                    retmsg += self.guiman.getText(17) + "|"
                    fieldidlist.append(17)
                elif req_field == "dlsize":
                    retmsg += self.guiman.getText(18) + "|"
                    fieldidlist.append(18)
                elif req_field == "ulsize":
                    retmsg += self.guiman.getText(19) + "|"
                    fieldidlist.append(19)
                elif req_field == "totalspeed":
                    retmsg += self.guiman.getText(20) + "|"
                    fieldidlist.append(20)
                else:
                    # error
                    connection.send("Feedback\nError=Unknown field identifier : " + self.utility.encodeWebServiceData(req_field))
                    connection.close()
                    return
        retmsg += "Info Hash\n" + self.grabTorrentFieldsForWeb(fieldidlist)
        connection.send(self.utility.encodeWebServiceData(retmsg))
        connection.close()

    def queryAll(self, connection):
        retmsg = ""
        for id in xrange(4, self.guiman.maxid):
            retmsg += self.guiman.getText(id) + "|"
        retmsg += "Info Hash\n" + self.grabTorrentFieldsForWeb()
        connection.send(self.utility.encodeWebServiceData(retmsg))
        connection.close()

    def getIDFromInfoHash(self, req_info_hash):
        try:
            req_info_hash = unhexlify(req_info_hash)
        except:
            return -1
        for t in self.proctab:
            if t.infohash == req_info_hash:
                return t.index
        return -1

    def getTorrentFromInfoHash(self, req_info_hash):
        try:
            req_info_hash = unhexlify(req_info_hash)
        except:
            return None
        for torrent in self.proctab:
            if torrent.infohash == req_info_hash:
                return torrent
        return None

    def checkActivity(self, now):
        oldtis = self.torrentinactiveseed
        oldti = self.torrentinactive
        for t in self.torrentalive:
            if t.activity:
                if t.abcengine.timeractivityon and (t.abcengine.uprate <= self.activitythup or t.status != 'completed') \
                                               and t.abcengine.downrate <= self.activitythdown:
                    if now - t.abcengine.timeractivityon > self.activitydelay:
                        t.setActivity(0, countextra = True)
                else:
                    t.abcengine.timeractivityon = now
            else:
                if t.abcengine.timeractivityoff and (t.abcengine.uprate > self.activitythup and t.status == 'completed'
                                                     or t.abcengine.downrate > self.activitythdown):
                    if now - t.abcengine.timeractivityoff > self.activitydelay:
                        t.setActivity(1, countextra = True)
                else:
                    t.abcengine.timeractivityoff = now
        if self.abcparams['trigwhenfinishseed'] == "1":
            if self.torrentinactiveseed + self.torrentinactive != oldtis + oldti:
                self.scheduler(True)
            else:
                if self.torrentinactiveseed != oldtis or self.torrentinactive != oldti:
                    self.updateRunningTorrentCounters()
                    self.extratobestopped = 0
                    self.extratobestoppedseed = 0
                self.freshactivetorrents[:] = []
                self.freshactivetorrentsseed[:] = []
        elif self.torrentinactiveseed != oldtis or self.torrentinactive != oldti:
            self.scheduler(True)
        else:
            self.freshactivetorrents[:] = []
            self.freshactivetorrentsseed[:] = []

    def distributeBandwidth(self, now):
        # Bandwidth distribution - Algo by Old King Cole
        if not self.torrentalive:
            # No alive torrent, no need to calculate
            return

        deadbandup = self.utility.bdcalcupth1 * self.utility.bdcalcupth2
        deadbanddn = self.utility.bdcalcdownth1 * self.utility.bdcalcdownth2

        self.dispatchTorrents(deadbandup, deadbanddn)
        if not self.allactive:
            # No active torrent, no need to calculate
            return

        #print "====================== BEGINNING ALGO ===================================="

        #####################################################################################################################################
        # Upload
        #####################################################################################################################################

        maxup = self.maxUploadRate()

        #####################################################################################################################################
        # isuploadunlimited
        #####################################################################################################################################
        if maxup == 0:
            # Unlimited static upload rate
            #print "UNLIMITED UP RATE"
            for t in self.allactive:
                t.maxuploadrate = t.maxlocaluploadrate
                if self.utility.upwithoutdown2 and t.abcengine.overmaxupwithoutdown2 \
                   and (not t.maxuploadrate or t.maxuploadrate > self.utility.upwithoutdownrate):
                    if t.maxuploadrate:
                        t.maxuploadrate = min(t.maxuploadrate, self.utility.upwithoutdownrate)
                    else:
                        t.maxuploadrate = self.utility.upwithoutdownrate
                # Send to BitTornado
                t.abcengine.dow.setUploadRate(t.maxuploadrate)
                #print t.name, "updated to", t.maxuploadrate

        else:
            #####################################################################################################################################
            # distributeUpload
            #####################################################################################################################################
            #print "DISTRIBUTE UPLOAD"
            #####################################################################################################################################
            # First treat special torrents before going on sharing and distributing
            #print "PHASE 0"
            ###########################################################################
            # Treat in priority the torrents with up rate below minuprate and that want to get a higher rate
            # May not concern any torrent with some values of bdcalcupth1 et bdcalcupth2
            grantedfortoberaisedbelowmin = 0.
            for t in self.toberaisedbelowmin_u:
                t.maxuploadrate = self.utility.minuprate
                grantedfortoberaisedbelowmin += t.maxuploadrate - t.abcengine.uprate
                #print "FOR BELOW MIN UPRATE TO BE RAISED :", t.name, "updated to", t.maxuploadrate

            availableuprate = maxup - self.totalupload - grantedfortoberaisedbelowmin
            #print "==============="
            #print "UP BELOWMIN"
            #print "maxup", maxup
            #print "grantedfortoberaisedbelowmin", grantedfortoberaisedbelowmin
            #print "availableuprate", availableuprate

            step = 0.
            if availableuprate < 0:
                ###########################################################################
                #print "PHASE 1"
                ###########################################################################
                # Phase 1 : There's no more available upload bandwidth to be distributed and measured up rate is over max.
                #     Each not prioritized torrent will have its up rate lowered to compensate the bandwidth
                #     given in priority to torrents with up rate below minuprate kB/s and prioritized torrents.
                #     This lowering must not bring the reserved up rate of a torrent below minuprate kB/s.
                #     Torrents will be sorted by their up rate and treated from the lower to the bigger. If a part of the lowering
                #     cannot be achieved with a torrent, it will be dispatched to the pool of the remaining torrents to be treated.

                # prio
                prioforup = self.priotoberaised_u + self.priotobelowered_u + self.prionottobechanged_u
                p = len(prioforup)

                # not prio
                notprioforup = self.toberaised_u + self.tobelowered_u + self.nottobechanged_u
                np = len(notprioforup)

                if p or np:
                    # not prio will be lowered priofactup times prio
                    step = availableuprate / (p + self.priofactup * np)
                    margin_torrent = []
                    for t in prioforup:
                        margin_torrent.append((t.abcengine.uprate - self.utility.minuprate, t, True))
                    for t in notprioforup:
                        margin_torrent.append((t.abcengine.uprate - self.utility.minuprate, t, False))
                    margin_torrent.sort()

                    for m, t, prio in margin_torrent:
                        if prio:
                            p -= 1
                            rateforeach = step
                        else:
                            np -= 1
                            rateforeach = self.priofactup * step
                        # (step and rateforeach are negative)
                        if rateforeach < -m:
                            t.maxuploadrate = self.utility.minuprate
                            # If some up rate lowering could not be dispatched, it will be distributed to the next torrents
                            # in the list (which are higher in up rate because the list is sorted this way)
                            if p or np:
                                step += (rateforeach + m) / (p + self.priofactup * np)
                        else:
                            t.maxuploadrate = t.abcengine.uprate + rateforeach
                        #print t.name, "lowered to", t.maxuploadrate

            else:
                ###########################################################################
                #print "PHASE 2"
                ###########################################################################
                # Phase 2 :
                #     As long as there's available upload bandwidth below the total max to be distributed, I give some
                #     to any torrents that asks for it.
                #     There's a special case with torrents to be raised that have a local max up rate : they must be
                #     topped to their max local up rate and the surplus part of the reserved bandwidth may be reused.
                # To sum up : In Phase 2, the measured up rate is leading the reserved up rate.

                # prio
                if self.priotoberaised_u:
                    prioforup = self.priotoberaised_u
                elif self.prionottobechanged_u:
                    prioforup = self.prionottobechanged_u
                else:
                    prioforup = self.priotobelowered_u
                p = len(prioforup)

                # not prio
                if self.toberaised_u:
                    notprioforup = self.toberaised_u
                elif self.nottobechanged_u:
                    notprioforup = self.nottobechanged_u
                else:
                    notprioforup = self.tobelowered_u
                np = len(notprioforup)

                if p or np:
                    # prio will be raised priofactup times not prio
                    step = availableuprate / (self.priofactup * p + np)
                    margin_torrent = []
                    withoutlocalmax = []
                    for t in prioforup:
                        if t.maxlocaluploadrate:
                            margin_torrent.append((t.maxlocaluploadrate - t.abcengine.uprate, t, True))
                        else:
                            withoutlocalmax.append((t, True))
                    for t in notprioforup:
                        if t.maxlocaluploadrate:
                            margin_torrent.append((t.maxlocaluploadrate - t.abcengine.uprate, t, False))
                        else:
                            withoutlocalmax.append((t, False))
                    margin_torrent.sort()

                    for m, t, prio in margin_torrent:
                        if prio:
                            p -= 1
                            rateforeach = self.priofactup * step
                        else:
                            np -= 1
                            rateforeach = step
                        if rateforeach > m:
                            t.maxuploadrate = t.maxlocaluploadrate
                            if p or np:
                                step += (rateforeach - m) / (self.priofactup * p + np)
                        else:
                            t.maxuploadrate = t.abcengine.uprate + rateforeach
                        #print t.name, "updated to", t.maxuploadrate
                    for t, prio in withoutlocalmax:
                        if prio:
                            rateforeach = self.priofactup * step
                        else:
                            rateforeach = step
                        t.maxuploadrate = t.abcengine.uprate + rateforeach
                        #print t.name, "updated to", t.maxuploadrate

            if not self.toberaised_u and self.nottobechanged_u:
                ###########################################################################
                #print "PHASE 3"
                ###########################################################################
                ###########################################################################
                #print "PHASE 3 mean upload rate algo"
                ###########################################################################
                # -a- Compute the mean max upload rate for the pool of not prioritized torrents that don't want to be raised
                #     and list torrents below and above that mean value.
                # -b- The regulation for torrents that want to have their upload rate raised is computed this way :
                #     The idea is to target a mean upload rate for all torrents that want to raise their up rate or don't want
                #     to have it changed, taking into account the max up rates of local settings, and the fact that no torrent
                #     must be lowered down below minuprate kB/s.
                # To sum up : In Phase 3, the reserved up rate is leading the real up rate.

                # -a-
                # Mean reserved up rate calculation
                # All prioritized torrents and all torrents with a local up rate setting if prioritizelocal is set
                # are completely excluded from the upload bandwidth exchange phase between other torrents. This is because
                # this phase targets a mean upload rate between torrents that want to upload more, and these torrents
                # can be very far from this mean up rate and their own purpose is not to integrate the pool of other
                # torrents but to live their own life (!). These torrents will also more surely be granted some bandwidth if
                # they need it because they are served in priority.
                torrentpool = self.nottobechanged_u + self.tobelowered_u
                if torrentpool:
                    meanuprate = 0.
                    for t in torrentpool :
                        meanuprate += t.getMaxUpRate()
                    meanuprate /= len(torrentpool)
                    #print "Mean up rate over minuprate kB/s : %.1f" % meanuprate

                    # torrents to be raised or not to be changed and with reserved up rate below mean max upload rate
                    raisedbelowm = [t for t in self.nottobechanged_u if t.getMaxUpRate() < meanuprate]         

                    # -b-
                    if raisedbelowm:
                        # torrents to be raised or not to be changed and with reserved up rate above mean max upload rate
                        allabovem = [t for t in torrentpool if t.getMaxUpRate() > meanuprate]
                        # Available bandwidth exchange
                        up = 0.
                        down = 0.
                        for t in raisedbelowm:
                            if t.maxlocaluploadrate and t.maxlocaluploadrate <= meanuprate:
                                up += min(t.maxlocaluploadrate - t.getMaxUpRate(), deadbandup)
                            else:
                                up += min(meanuprate - t.getMaxUpRate(), deadbandup)
                        for t in allabovem:
                            down += min(t.getMaxUpRate() - meanuprate, deadbandup)
                        if up > down:
                            limitup = down / up
                        else:
                            limitup = 1
                        # Speed up slow torrents that want their up rate to be raised
                        for t in raisedbelowm:
                            if t.maxlocaluploadrate and t.maxlocaluploadrate <= meanuprate:
                                t.maxuploadrate = t.getMaxUpRate() + min(t.maxlocaluploadrate - t.getMaxUpRate(), deadbandup) * limitup
                            else:
                                t.maxuploadrate = t.getMaxUpRate() + min(meanuprate - t.getMaxUpRate(), deadbandup) * limitup
                            #print "torrent :", t.name, "; up rate raised to", t.maxuploadrate
                        # Slow down fast torrents
                        if up < down:
                            limitdown = up / down
                        else:
                            limitdown = 1
                        for t in allabovem:
                            t.maxuploadrate = t.getMaxUpRate() - min(t.getMaxUpRate() - meanuprate, deadbandup) * limitdown
                            #print "torrent :", t.name, "; up rate lowered to", t.maxuploadrate

            if self.abcparams['urmlowpriority'] == "1" and self.nonextratoberaised_u:
                # Extra torrents with upload rate > minuprate and not prioritized
                extra = [t for t in self.allextra if t.abcengine.uprate > self.utility.minuprate
                         and not t.prioritizedown and not t.prioritizeup]
                if extra:
                    ###########################################################################
                    #print "PHASE 3 low priority urm algo"
                    ###########################################################################
                    # This is the algorithm which is used when the Preferences/URM/"Low rate priority" is ticked.
                    # For this algo to kick in, there must be active extra torrents and active non extra torrents that want to raise their up rate
                    # This is meant to give a higher priority to non extra torrents in the distribution of upload bandwidth. If non extra torrents
                    # need upload bandwidth, it will be taken from the extra torrents, and if there's not enough there, extra torrents can be stopped
                    # to fulfill the demand in bandwidth.
                    # -a- Compute the amount of up bandwidth the non extra torrents with up rate > minuprate kB/s would like to grab
                    #     and taking into account the local upload rate settings if they exist.
                    # -b- Lower the reserved upload bandwidth of all extra torrents the closer possible to this amount while keeping them above minuprate kB/s
                    #     (Same algo as in phase 2-a-).
                    # -c- Raise the reserved upload bandwidth for each non extra torrent of the real global amount that could be taken from the
                    #     lowering of the extra torrents.
                    # -d- If the non extra torrents would have liked to take more than they were given and if the upload rate is in the dead band,
                    #     on-hold (or queue) the active extra torrent inducing the smaller gap between the max upload rate and the upload rate.

                    # -a-
                    claimeduprate = 0.
                    for t in self.nonextratoberaised_u:
                        if t.maxlocaluploadrate:
                            claimeduprate += min(t.maxlocaluploadrate - t.getMaxUpRate(), deadbandup)
                        else:
                            claimeduprate += deadbandup
                    #print "Claimed up rate :", claimeduprate

                    if claimeduprate > 0:
                        # -b-
                        margin_torrent = [(t.getMaxUpRate() - self.utility.minuprate, t) for t in extra]
                        upratetobedistributed = claimeduprate
                        margin_torrent.sort()
                        rateforeach = upratetobedistributed / len(margin_torrent)
                        i = len(margin_torrent) - 1
                        for m, t in margin_torrent:
                            if rateforeach > m:
                                t.maxuploadrate = self.utility.minuprate
                                upratetobedistributed -= m
                                # If some up rate lowering could not be dispatched, it will be distributed to the next torrents
                                # in the list (which are higher in max up rate because the list is sorted this way)
                                if i:
                                    rateforeach += (rateforeach - m) / i
                            else:
                                t.maxuploadrate = t.getMaxUpRate() - rateforeach
                                upratetobedistributed -= rateforeach
                            i -= 1

                        # -c-
                        realupfactor = (claimeduprate - upratetobedistributed) / claimeduprate
                        for t in self.nonextratoberaised_u:
                            if t.maxlocaluploadrate:
                                t.maxuploadrate = t.getMaxUpRate() + min(t.maxlocaluploadrate - t.getMaxUpRate(), deadbandup) * realupfactor
                            else:
                                t.maxuploadrate = t.getMaxUpRate() + deadbandup * realupfactor
                            #print "torrent :", t.name, "; up rate raised to", t.maxuploadrate
                        #print "Up rate not recovered :", upratetobedistributed

                        # -d-
                        if self.abcparams['urmtrigger'] == '1':
                            urmtrigmargin = int(self.abcparams['urmtriggerval']) - self.totalupload
                        else:
                            urmtrigmargin =  maxup - self.totalupload
                        if upratetobedistributed > self.utility.missingupratetostopurm \
                           and (self.utility.urmthresholdstop == '-' or self.utility.urmthresholdstop <= urmtrigmargin) \
                           and urmtrigmargin <= self.utility.urmthresholdstart:
                            # Check if the last started torrent was started less than urmtorrentstartdelay seconds ago
                            if now - self.urmstartingtime >= self.utility.urmtorrentstartdelay:
                                # On-hold (or queue) an active extra torrent which will give back some upload bandwidth
                                haltedtorrent = self.haltExtraTorrentForUpload(self.allextra)
                                if haltedtorrent:
                                    self.allextra.remove(haltedtorrent)

            ################################################
            # Set new max upload rate to all active torrents
            ################################################
            for t in self.allactive:
                if t.maxuploadrate == -1:
                    if t.maxlocaluploadrate:
                        t.maxuploadrate = min(t.abcengine.uprate + deadbandup, t.maxlocaluploadrate)
                    else:
                        t.maxuploadrate = t.abcengine.uprate + deadbandup
                if self.utility.upwithoutdown2 and t.abcengine.overmaxupwithoutdown2 and t.maxuploadrate > self.utility.upwithoutdownrate:
                    t.maxuploadrate = min(t.maxuploadrate, self.utility.upwithoutdownrate)
                t.abcengine.dow.setUploadRate(t.maxuploadrate)
                #print "torrent :", t.name, "updated in up rate to", t.maxuploadrate

        if not self.allactivenotseeding:
            # No active torrent, no need to calculate
            return

        #####################################################################################################################################
        # Download
        #####################################################################################################################################

        maxdown = self.utility.maxdown

        #####################################################################################################################################
        # isdownloadunlimited
        #####################################################################################################################################
        if maxdown == 0:
            # Unlimited download rate
            #print "UNLIMITED DOWN RATE"
            for t in self.allactivenotseeding:
                t.maxdownloadrate = t.maxlocaldownloadrate
                # Send to BitTornado
                t.abcengine.dow.setDownloadRate(t.maxdownloadrate)
                #print t.name, "updated to", t.maxdownloadrate
            return

        #####################################################################################################################################
        # distributeDownload
        #####################################################################################################################################
        #print "DISTRIBUTE DOWNLOAD"
        ###########################################################################
        # First treat special torrents before going on sharing and distributing
        #print "PHASE 0"
        ###########################################################################
        # Treat in priority the torrents with down rate below mindownrate and that want to get a higher rate
        # May not concern any torrent with some values of bdcalcdownth1 et bdcalcdownth2
        grantedfortoberaisedbelowmin = 0.
        for t in self.toberaisedbelowmin_d:
            t.maxdownloadrate = self.utility.mindownrate
            grantedfortoberaisedbelowmin += t.maxdownloadrate - t.abcengine.downrate
            #print "FOR BELOW MIN DOWNRATE TO BE RAISED :", t.name, "updated to", t.maxdownloadrate

        availabledownrate = maxdown - self.totaldownload - grantedfortoberaisedbelowmin
        #print "==============="
        #print "DOWN BELOWMIN"
        #print "maxdown", maxdown
        #print "grantedfortoberaisedbelowmin", grantedfortoberaisedbelowmin
        #print "availabledownrate", availabledownrate

        step = 0.
        if availabledownrate < 0:
            ###########################################################################
            #print "PHASE 1"
            ###########################################################################
            # Phase 1 : There's no more available download bandwidth to be distributed and measured down rate is over max.
            #     Each not prioritized torrent will have its down rate lowered
            #     to compensate the bandwidth given in priority to prioritized torrents.
            #     Torrents will be sorted by their down rate and treated from the lower to the bigger. If a part of the lowering
            #     cannot be achieved with a torrent, it will be dispatched to the pool of the remaining torrents to be treated.

            # prio
            prioforup = self.priotoberaised_d + self.priotobelowered_d + self.prionottobechanged_d
            p = len(prioforup)

            # not prio
            notprioforup = self.toberaised_d + self.tobelowered_d + self.nottobechanged_d
            np = len(notprioforup)

            if p or np:
                # not prio will be lowered priofactup times prio
                step = availabledownrate / (p + self.priofactdn * np)
                margin_torrent = []
                for t in prioforup:
                    margin_torrent.append((t.abcengine.downrate - self.utility.mindownrate, t, True))
                for t in notprioforup:
                    margin_torrent.append((t.abcengine.downrate - self.utility.mindownrate, t, False))
                margin_torrent.sort()

                for m, t, prio in margin_torrent:
                    if prio:
                        p -= 1
                        rateforeach = step
                    else:
                        np -= 1
                        rateforeach = self.priofactdn * step
                    # (step and rateforeach are negative)
                    if rateforeach < -m:
                        t.maxdownloadrate = self.utility.mindownrate
                        # If some down rate lowering could not be dispatched, it will be distributed to the next torrents
                        # in the list (which are higher in down rate because the list is sorted this way)
                        if p or np:
                            step += (rateforeach + m) / (p + self.priofactdn * np)
                    else:
                        t.maxdownloadrate = t.abcengine.downrate + rateforeach
                    #print t.name, "lowered to", t.maxdownloadrate

        else:
            ###########################################################################
            #print "PHASE 2"
            ###########################################################################
            # Phase 2 :
            #     As long as there's available download bandwidth below the total max to be distributed, I give some
            #     to any torrents that asks for it.
            #     There's a special case with torrents to be raised that have a local max down rate : they must be
            #     topped to their max local down rate and the surplus part of the reserved bandwidth may be reused.
            # To sum up : In Phase 2, the measured down rate is leading the reserved down rate.

            # prio
            if self.priotoberaised_d:
                prioforup = self.priotoberaised_d
            elif self.prionottobechanged_d:
                prioforup = self.prionottobechanged_d
            else:
                prioforup = self.priotobelowered_d
            p = len(prioforup)

            # not prio
            if self.toberaised_d:
                notprioforup = self.toberaised_d
            elif self.nottobechanged_d:
                notprioforup = self.nottobechanged_d
            else:
                notprioforup = self.tobelowered_d
            np = len(notprioforup)

            if p or np:
                # prio will be raised priofactup times not prio
                step = availabledownrate / (self.priofactdn * p + np)
                margin_torrent = []
                withoutlocalmax = []
                for t in prioforup:
                    if t.maxlocaldownloadrate:
                        margin_torrent.append((t.maxlocaldownloadrate - t.abcengine.downrate, t, True))
                    else:
                        withoutlocalmax.append((t, True))
                for t in notprioforup:
                    if t.maxlocaldownloadrate:
                        margin_torrent.append((t.maxlocaldownloadrate - t.abcengine.downrate, t, False))
                    else:
                        withoutlocalmax.append((t, False))
                margin_torrent.sort()

                for m, t, prio in margin_torrent:
                    if prio:
                        p -= 1
                        rateforeach = self.priofactdn * step
                    else:
                        np -= 1
                        rateforeach = step
                    if rateforeach > m:
                        t.maxdownloadrate = t.maxlocaldownloadrate
                        if p or np:
                            step += (rateforeach - m) / (self.priofactdn * p + np)
                    else:
                        t.maxdownloadrate = t.abcengine.downrate + rateforeach
                    #print t.name, "updated to", t.maxdownloadrate
                for t, prio in withoutlocalmax:
                    if prio:
                        rateforeach = self.priofactdn * step
                    else:
                        rateforeach = step
                    t.maxdownloadrate = t.abcengine.downrate + rateforeach
                    #print t.name, "updated to", t.maxdownloadrate

        if not self.toberaised_d and self.nottobechanged_d:
            ###########################################################################
            #print "PHASE 3"
            ###########################################################################
            ###########################################################################
            #print "PHASE 3 mean download rate algo"
            ###########################################################################
            # -a- Compute the mean max download rate for the pool of not prioritized torrents that don't want to be raised
            #     and list torrents below and above that mean value.
            # -b- The regulation for torrents that want to have their download rate raised is computed this way :
            #     The idea is to target a mean download rate for all torrents that want to raise their down rate or don't want
            #     to have it changed, taking into account the max down rates of local settings, and the fact that no torrent
            #     must be lowered down below mindownrate kB/s.
            # To sum up : In Phase 3, the reserved down rate is leading the real down rate.

            # -a-
            # Mean reserved down rate calculation
            # All prioritized torrents and all torrents with a local down rate setting if prioritizelocal is set
            # are completely excluded from the download bandwidth exchange phase between other torrents. This is because
            # this phase targets a mean download rate between torrents that want to download more, and these torrents
            # can be very far from this mean down rate and their own purpose is not to integrate the pool of other
            # torrents but to live their own life (!). These torrents will also more surely be granted some bandwidth if
            # they need it because they are served in priority.
            torrentpool = self.nottobechanged_d + self.tobelowered_d
            if torrentpool:
                meandownrate = 0.
                for t in torrentpool :
                    meandownrate += t.getMaxDownRate()
                meandownrate /= len(torrentpool)
                #print "Mean down rate : %.1f" % meandownrate

                # torrents to be raised or not to be changed and with reserved down rate below mean max download rate
                raisedbelowm = [t for t in self.nottobechanged_d if t.getMaxDownRate() < meandownrate]

                # -b-
                if raisedbelowm:
                    # torrents to be raised or not to be changed and with reserved down rate above mean max download rate
                    allabovem = [t for t in torrentpool if t.getMaxDownRate() > meandownrate]
                    # Available bandwidth exchange
                    up = 0.
                    down = 0.
                    for t in raisedbelowm:
                        if t.maxlocaldownloadrate and t.maxlocaldownloadrate <= meandownrate:
                            up += min(t.maxlocaldownloadrate - t.getMaxDownRate(), deadbanddn)
                        else:
                            up += min(meandownrate - t.getMaxDownRate(), deadbanddn)
                    for t in allabovem:
                        down += min(t.getMaxDownRate() - meandownrate, deadbanddn)
                    if up > down:
                        limitup = down / up
                    else:
                        limitup = 1
                    # Speed up slow torrents that want their down rate to be raised
                    for t in raisedbelowm:
                        if t.maxlocaldownloadrate and t.maxlocaldownloadrate <= meandownrate:
                            t.maxdownloadrate = t.getMaxDownRate() + min(t.maxlocaldownloadrate - t.getMaxDownRate(), deadbanddn) * limitup
                        else:
                            t.maxdownloadrate = t.getMaxDownRate() + min(meandownrate - t.getMaxDownRate(), deadbanddn) * limitup
                        #print "torrent :", t.name, "; down rate raised to", t.maxdownloadrate
                    # Slow down fast torrents
                    if up < down:
                        limitdown = up / down
                    else:
                        limitdown = 1
                    for t in allabovem:
                        t.maxdownloadrate = t.getMaxDownRate() - min(t.getMaxDownRate() - meandownrate, deadbanddn) * limitdown
                        #print "torrent :", t.name, "; down rate lowered to", t.maxdownloadrate

        if self.abcparams['drmlowpriority'] == "1" and self.nonextratoberaised_d:
            # Extra torrents with download rate > mindownrate and not prioritized
            extra = [t for t in self.allextra if t.abcengine.downrate > self.utility.mindownrate
                     and not t.prioritizedown and not t.prioritizeup]
            if extra:
                ###########################################################################
                #print "PHASE 3 low priority drm algo"
                ###########################################################################
                # This is the algorithm which is used when the Preferences/DRM/"Low rate priority" is ticked.
                # For this algo to kick in, there must be active extra torrents and active non extra torrents that want to raise their down rate
                # This is meant to give a higher priority to non extra torrents in the distribution of download bandwidth. If non extra torrents
                # need download bandwidth, it will be taken from the extra torrents, and if there's not enough there, extra torrents can be stopped
                # to fulfill the demand in bandwidth..
                # -a- Compute the amount of down bandwidth the non extra torrents with down rate > mindownrate kB/s would like to grab
                #     and taking into account the local download rate settings if they exist.
                # -b- Lower the reserved download bandwidth of all extra torrents the closer possible to this amount while keeping them above mindownrate kB/s
                #     (Same algo as in phase 2-a-).
                # -c- Raise the reserved download bandwidth for each non extra torrent by the real global amount that could be taken from the
                #     lowering of the extra torrents.
                # -d- If the non extra torrents would have liked to take more than they were given and if the download rate is in the dead band,
                #     on-hold (or queue) the active extra torrent inducing the smaller gap between the max download rate and the download rate.

                # -a-
                claimeddownrate = 0.
                for t in self.nonextratoberaised_d:
                    if t.maxlocaldownloadrate:
                        claimeddownrate += min(t.maxlocaldownloadrate - t.getMaxDownRate(), deadbanddn)
                    else:
                        claimeddownrate += deadbanddn
                #print "Claimed down rate :", claimeddownrate

                if claimeddownrate > 0:
                    # -b-
                    margin_torrent = [(t.getMaxDownRate() - self.utility.mindownrate, t) for t in extra]
                    downratetobedistributed = claimeddownrate
                    margin_torrent.sort()
                    rateforeach = downratetobedistributed / len(margin_torrent)
                    i = len(margin_torrent) - 1
                    for m, t in margin_torrent:
                        if rateforeach > m:
                            t.maxdownloadrate = self.utility.mindownrate
                            downratetobedistributed -= m
                            # If some down rate lowering could not be dispatched, it will be distributed to the next torrents
                            # in the list (which are higher in max down rate because the list is sorted this way)
                            if i:
                                rateforeach += (rateforeach - m) / i
                        else:
                            t.maxdownloadrate = t.getMaxDownRate() - rateforeach
                            downratetobedistributed -= rateforeach
                        i -= 1

                    # -c-
                    realdownfactor = (claimeddownrate - downratetobedistributed) / claimeddownrate
                    for t in self.nonextratoberaised_d:
                        if t.maxlocaldownloadrate:
                            t.maxdownloadrate = t.getMaxDownRate() + min(t.maxlocaldownloadrate - t.getMaxDownRate(), deadbanddn) * realdownfactor
                        else:
                            t.maxdownloadrate = t.getMaxDownRate() + deadbanddn * realdownfactor
                        #print "torrent :", t.name, "; down rate raised to", t.maxdownloadrate
                    #print "Down rate not recovered :", downratetobedistributed

                    # -d-
                    if self.abcparams['drmtrigger'] == '1':
                        drmtrigmargin = int(self.abcparams['drmtriggerval']) - self.totaldownload
                    else:
                        drmtrigmargin =  maxdown - self.totaldownload
                    if downratetobedistributed > self.utility.missingdownratetostopdrm \
                       and (self.utility.drmthresholdstop == '-' or self.utility.drmthresholdstop <= drmtrigmargin) \
                       and drmtrigmargin <= self.utility.drmthresholdstart:
                        # Check if the last started torrent was started less than drmtorrentstartdelay seconds ago
                        if now - self.drmstartingtime >= self.utility.drmtorrentstartdelay:
                            # On-hold (or queue) an active extra torrent which will give back some download bandwidth
                            haltedtorrent = self.haltExtraTorrentForDownload(self.allextra)
                            if haltedtorrent:
                                self.allextra.remove(haltedtorrent)

        ################################################
        # Set new max download rate to all active torrents
        ################################################
        for t in self.allactivenotseeding:
            if t.maxdownloadrate == -1:
                if t.maxlocaldownloadrate:
                    t.maxdownloadrate = min(t.abcengine.downrate + deadbanddn, t.maxlocaldownloadrate)
                else:
                    t.maxdownloadrate = t.abcengine.downrate + deadbanddn
            t.abcengine.dow.setDownloadRate(t.maxdownloadrate)
            #print "torrent :", t.name, "updated in down rate to", t.maxdownloadrate
 