# Copyright 2002-2004 Andrew Loewenstern, All Rights Reserved
# Unless otherwise noted, all files are released under the MIT
# license, exceptions contain licensing information in them.
#
# Copyright (C) 2002-2003 Andrew Loewenstern
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, sublicense, and/or sell copies of the Software,
# and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# The Software is provided "AS IS", without warranty of any kind,
# express or implied, including but not limited to the warranties of
# merchantability, fitness for a particular purpose and
# noninfringement. In no event shall the authors or copyright holders
# be liable for any claim, damages or other liability, whether in an
# action of contract, tort or otherwise, arising from, out of or in
# connection with the Software or the use or other dealings in the
# Software.
# Updated and modified for ABC_OKC : Old King Cole

import sys, wx, re
from os import path, remove, rename
from time import clock
from random import randrange, randint
from hashlib import sha1
from binascii import hexlify, unhexlify
from threading import Event

from filemanager import TorrentConfigFileManager
from const import K, NULL_ID, KE_AGE, KE_INITIAL_DELAY, KE_DELAY, TOKEN_AGE,\
                  TOKEN_DELAY, CHECKPOINT_INTERVAL, MIN_PING_INTERVAL,\
                  BUCKET_STALENESS, NUM_PEERS
from ktable import KTable
from knode import KNodeBase, KNodeRead, KNodeWrite
from peerdict import PeerDict
from expiringdict import ExpiringDict
from khash import newID, newIDInRange
from actions import FindNode, GetValue, StoreValue, Expirer, SecretRenewer
from krpc import Hostbroker
from defer import Deferred
from crc32c import crc


# This is the base class, has base functionality and find node, no key-value mappings
class KhashmirBase:
    _Node = KNodeBase
    def __init__(self, utility, rawserver, sock, host, port, statusfunc):
        self.utility = utility
        self.ipv6 = (utility.abcparams['ipv6'] == '1')
        self.dhtnodes_file = path.join(utility.datapath, "dhtnode.lst")
        self.dhtpeers_file = path.join(utility.datapath, "dhtpeer.lst")
        self.dhtnodes6_file = path.join(utility.datapath, "dhtnode6.lst")
        self.dhtpeers6_file = path.join(utility.datapath, "dhtpeer6.lst")
        self.rawserver = rawserver
        self.sched = rawserver.add_task
        if utility.dhtsecurityextension and utility.abcparams['dhtip']:
            self.host = utility.abcparams['dhtip']
        else:
            self.host = utility.getHostByName(host)
        self.port = port
        id = unhexlify(utility.abcparams['dhtnodeid'])
        if id == '':
            id = self.computeDHTNodeID()
            utility.abcparams['dhtnodeid'] = hexlify(id)
            tcfm = TorrentConfigFileManager(path.join(utility.datapath, "abc.conf"), utility.abcparams)
            tcfm.writeAllConfig()
        # to protect against bugs in some clients' DHT Security Extension implementation
        self.ipchangecount = {}
        self.statusfunc = statusfunc
        self.statusdoneprocessing = Event()
        self.statusdoneprocessing.set()
        self.udp = Hostbroker(self, (self.host, port), sock, self.sched)
        self.node = self.Node().init(id, self.host, self.port)
        self.secret = self.prevsecret = sha1(newID())
        self.table = KTable(self.node)
        self.tables = [self.table]
        self.peers = PeerDict()
        if self.ipv6:
            self.table6 = KTable(self.node)
            self.tables.append(self.table6)
            self.peers6 = PeerDict()
        self.tokens = ExpiringDict()
        # Load routing table and peers
        self.loadNodes()
        self.loadPeers()
        SecretRenewer(self, self.sched)
        Expirer(self.peers, self.sched, KE_AGE, KE_INITIAL_DELAY, KE_DELAY)
        if self.ipv6:
            Expirer(self.peers6, self.sched, KE_AGE, KE_INITIAL_DELAY, KE_DELAY)
        Expirer(self.tokens, self.sched, TOKEN_AGE, TOKEN_DELAY, TOKEN_DELAY)
        self.sched(self.reportStatus, 0.5)
        self.refreshTable(force = 1)
        self.sched(self.checkpoint, 60, [1])
        # Init dht routers
        try:
            dhtrouterfile = open(path.join(utility.datapath, "dhtrouter.txt"), "r")
        except:
            pass
        else:
            rawnodelist = [s for s in re.split(r'[ \n]', dhtrouterfile.read()) if s]
            dhtrouterfile.close()
            nodelist = []
            for i in xrange(0, len(rawnodelist), 2):
                ip = rawnodelist[i]
                if ip.find(':') >= 0:
                    rawnodelist[i] = utility.expandIPv6(ip)
                nodelist.append((rawnodelist[i], int(rawnodelist[i + 1])))
            for ip, port in nodelist:
                resolvedips = utility.getIpsByName(ip)
                for ri in resolvedips:
                    self.sched(self.addContact, 0, [ri, port])

    def computeDHTNodeID(self):
        nodeid = []
        rand = randint(0, 255)
        r = rand & 0x7
        if self.ipv6:
            ip = [ord(i) & m for (i, m) in zip(list(unhexlify(''.join([self.host[i:i + 4] for i in xrange(0, 20, 5)]))),
                                               (0x01, 0x03, 0x07, 0x0f, 0x1f, 0x3f, 0x7f, 0xff))]
        else:
            ip = [int(i) & m for (i, m) in zip(self.host.split('.'), (0x03, 0x0f, 0x3f, 0xff))]
        ip[0] |= r << 5
        c = crc(ip)
        nodeid.append(chr((c >> 24) & 0xff))
        nodeid.append(chr((c >> 16) & 0xff))
        nodeid.append(chr(((c >> 8) & 0xf8) | randint(0, 255) & 0x7))
        for i in xrange(3, 19):
            nodeid.append(chr(randint(0, 255)))
        nodeid.append(chr(rand))
        return ''.join(nodeid)

    def checkIPDHT(self, ipdht, ipsender):
        if self.ipv6:
            host = self.decodePeerInfo6(ipdht)[0]
        else:
            host = self.decodePeerInfo(ipdht)[0]
        if host != self.host:
            self.ipchangecount[ipsender] = host
            for s in self.ipchangecount:
                if s == ipsender:
                    continue
                if self.ipchangecount[s] == host:
                    self.ipchangecount.clear()
                    self.setIPDHT(host)
                    break
        else:
            self.ipchangecount.clear()

    def setIPDHT(self, ip):
        self.host = ip
        id = self.computeDHTNodeID()
        #print "IP :", ip, "; NEW NODE :", hexlify(id)
        self.utility.abcparams['dhtnodeid'] = hexlify(id)
        self.utility.abcparams['dhtip'] = self.host
        tcfm = TorrentConfigFileManager(path.join(self.utility.datapath, "abc.conf"), self.utility.abcparams)
        tcfm.writeAllConfig()
        self.node.init(id, self.host, self.port)
        self.table.reset()
        if self.ipv6:
            self.table6.reset()
        self.refreshTable(force = 1)

    def reportStatus(self):
        if self.statusdoneprocessing.isSet():
            self.statusdoneprocessing.clear()
            self.statusfunc(self.statusdoneprocessing, self.udp.uprate.get_rate())
        self.sched(self.reportStatus, 0.5)

    def Node(self):
        n = self._Node()
        n.dht = self
        return n

    def checkpoint(self, auto = 0):
        self.saveNodes()
        self.savePeers()
        self.findCloseNodes(self.tables)
        self.refreshTable()
        if auto:
            self.sched(self.checkpoint, randrange(int(CHECKPOINT_INTERVAL * .9), int(CHECKPOINT_INTERVAL * 1.1)), [1])

    def loadNodes(self):
        """
            load nodes from file
        """
        # IPv4
        if path.exists(self.dhtnodes_file):
            try:
                f = open(self.dhtnodes_file, 'r+')
                firstline = True
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    nodeinfo = self.decodeNodeInfo(unhexlify(configline[:-1].decode(encoding)))
                    n = self.Node().init(nodeinfo[0], nodeinfo[1], nodeinfo[2])
                    self.table.insertNode(n, contacted = 0)
                f.close()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file dhtnode.lst"

        # IPv6
        if self.ipv6 and path.exists(self.dhtnodes6_file):
            try:
                f = open(self.dhtnodes6_file, 'r+')
                firstline = True
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    nodeinfo = self.decodeNodeInfo6(unhexlify(configline[:-1].decode(encoding)))
                    n = self.Node().init(nodeinfo[0], nodeinfo[1], nodeinfo[2])
                    self.table6.insertNode(n, contacted = 0)
                f.close()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file dhtnode6.lst"

    def saveNodes(self):
        """
            save nodes to file
        """
        # IPv4
        try:
            f = open(self.dhtnodes_file + '.part', 'w+')
            f.writelines('\xef\xbb\xbf')
            for bucket in self.table.buckets:
                for node in bucket.l:
                    f.writelines((hexlify(self.compactNodeInfo(node)) + "\n").encode('utf_8'))
        except:
            try:
                f.close()
            except:
                pass
        else:
            f.close()
            try:
                remove(self.dhtnodes_file)
            except:
                pass
            rename(self.dhtnodes_file + '.part', self.dhtnodes_file)

        # IPv6
        if self.ipv6:
            try:
                f = open(self.dhtnodes6_file + '.part', 'w+')
                f.writelines('\xef\xbb\xbf')
                for bucket in self.table6.buckets:
                    for node in bucket.l:
                        f.writelines((hexlify(self.compactNodeInfo6(node)) + "\n").encode('utf_8'))
            except:
                try:
                    f.close()
                except:
                    pass
            else:
                f.close()
                try:
                    remove(self.dhtnodes6_file)
                except:
                    pass
                rename(self.dhtnodes6_file + '.part', self.dhtnodes6_file)

    def loadPeers(self):
        """
            load peers from file
        """
        # IPv4
        if path.exists(self.dhtpeers_file):
            try:
                f = open(self.dhtpeers_file, 'r+')
                firstline = True
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    peerinfo = map(unhexlify, configline[:-1].decode(encoding).split('|'))
                    for peer in peerinfo[1:]:
                        self.peers[peerinfo[0]] = (peer, True)
                f.close()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file dhtpeer.lst"

        # IPv6
        if self.ipv6 and path.exists(self.dhtpeers6_file):
            try:
                f = open(self.dhtpeers6_file, 'r+')
                firstline = True
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    peerinfo = map(unhexlify, configline[:-1].decode(encoding).split('|'))
                    for peer in peerinfo[1:]:
                        self.peers6[peerinfo[0]] = (peer, True)
                f.close()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file dhtpeer6.lst"

    def savePeers(self):
        """
            save peers to file
        """
        # IPv4
        try:
            f = open(self.dhtpeers_file + '.part', 'w+')
            f.writelines('\xef\xbb\xbf')
            for infohash in self.peers:
                val = self.peers[infohash]
                if val[1]:
                    f.writelines(('|'.join(map(hexlify, [infohash] + val[0])) + "\n").encode('utf_8'))
        except:
            try:
                f.close()
            except:
                pass
        else:
            f.close()
            try:
                remove(self.dhtpeers_file)
            except:
                pass
            rename(self.dhtpeers_file + '.part', self.dhtpeers_file)

        # IPv6
        if self.ipv6:
            try:
                f = open(self.dhtpeers6_file + '.part', 'w+')
                f.writelines('\xef\xbb\xbf')
                for infohash in self.peers:
                    val = self.peers6[infohash]
                    if val[1]:
                        f.writelines(('|'.join(map(hexlify, [infohash] + val[0])) + "\n").encode('utf_8'))
            except:
                try:
                    f.close()
                except:
                    pass
            else:
                f.close()
                try:
                    remove(self.dhtpeers6_file)
                except:
                    pass
                rename(self.dhtpeers6_file + '.part', self.dhtpeers6_file)

    def compactNodeInfos(self, nodes):
        return ''.join(map(self.compactNodeInfo, nodes))

    def decodeNodeInfos(self, nodeinfos):
        # list of (id, ip, port) from compact node infos
        return [self.decodeNodeInfo(nodeinfos[i:i + 26]) for i in xrange(0, len(nodeinfos), 26)]

    def compactNodeInfo(self, node):
        return node.id + self.compactPeerInfo(node.host, node.port)

    def decodeNodeInfo(self, nodeinfo):
        # (id, ip, port) from compact node info
        ip_port = self.decodePeerInfo(nodeinfo[20:])
        return (nodeinfo[:20], ip_port[0], ip_port[1])

    def compactPeerInfo(self, host, port):
        return ''.join([chr(int(i)) for i in host.split('.')]) + chr((port & 0xFF00) >> 8) + chr(port & 0xFF)

    def decodePeerInfo(self, peerinfo):
        # (ip, port) from compact peer info
        return ('.'.join([str(ord(peerinfo[i])) for i in xrange(4)]), (ord(peerinfo[4]) << 8) + ord(peerinfo[5]))

    def compactNodeInfos6(self, nodes):
        return ''.join(map(self.compactNodeInfo6, nodes))

    def decodeNodeInfos6(self, nodeinfos):
        # list of (id, ip, port) from compact node infos
        return [self.decodeNodeInfo6(nodeinfos[i:i + 38]) for i in xrange(0, len(nodeinfos), 38)]

    def compactNodeInfo6(self, node):
        return node.id + self.compactPeerInfo6(node.host, node.port)

    def decodeNodeInfo6(self, nodeinfo):
        # (id, ip, port) from compact node info
        ip_port = self.decodePeerInfo6(nodeinfo[20:])
        return (nodeinfo[:20], ip_port[0], ip_port[1])

    def compactPeerInfo6(self, host, port):
        return unhexlify(''.join([host[i:i + 4] for i in xrange(0, 40, 5)])) \
               + chr((port & 0xFF00) >> 8) + chr(port & 0xFF)

    def decodePeerInfo6(self, peerinfo):
        # (ip, port) from compact peer info
        hexpeerinfo = hexlify(peerinfo[:16])
        return (':'.join([hexpeerinfo[i:i + 4] for i in xrange(0, 32, 4)]), (ord(peerinfo[16]) << 8) + ord(peerinfo[17]))

    def addPeer(self, info_hash, host = '', port = 0, public = False):
        ipv6 = (host.find(':') >= 0)
        if ipv6:
            if not self.ipv6:
                return
            peers = self.peers6
            try:
                peerinfo = self.compactPeerInfo6(host, port)
            except:
                return
        else:
            peers = self.peers
            try:
                peerinfo = self.compactPeerInfo(host, port)
            except:
                return
        peers[info_hash] = (peerinfo, public)

    def getToken(self, host, port):
        s = self.secret.copy()
        s.update(host + str(port))
        return s.digest()

    def checkToken(self, token, host, port):
        s = self.secret.copy()
        s.update(host + str(port))
        if token == s.digest():
            return True
        s = self.prevsecret.copy()
        s.update(host + str(port))
        if token == s.digest():
            return True
        return False

    ####### LOCAL INTERFACE - use these methods
    def addContact(self, host, port, callback = None):
        """
            ping this node and add the contact info to the table on pong !
        """
        n = self.Node().init(NULL_ID, host, port)
        self.sendPing(n, callback = callback)

    ## this call is async !
    def findNode(self, id, table, callback, errback = None):
        """
            return the contact info for node, or the k closest nodes, from the global table
        """
        # get K nodes out of local table/cache, or the node we want
        nodes = table.findNodes(id)
        d = Deferred()
        if errback:
            d.addCallbacks(callback, errback)
        else:
            d.addCallback(callback)
        if len(nodes) == 1 and nodes[0].id == id :
            d.callback(nodes)
        else:
            # create our search state
            state = FindNode(self, id, d.callback, self.sched)
            self.sched(state.goWithNodes, 0, [nodes])

    def insertNode(self, n, ipv6 = None, contacted = 1):
        """
            insert a node in our local table, pinging oldest contact in bucket, if necessary
            If all you have is a host/port, then use addContact, which calls this method after
            receiving the PONG from the remote node.  The reason for the seperation is we can't insert
            a node into the table without it's peer-ID.  That means of course the node passed into this
            method needs to be a properly formed Node object with a valid ID.
        """
        if ipv6 is None:
            ipv6 = n.isIPv6()
        if ipv6:
            if not self.ipv6:
                return
            table = self.table6
        else:
            table = self.table
        old = table.insertNode(n, contacted = contacted)
        if old and not old.pinging and (clock() - old.lastSeen) > MIN_PING_INTERVAL:
            # the bucket is full, check to see if old node is still around and if so, replace it

            # these are the callbacks used when we ping the oldest node in a bucket
            def _notStaleNodeHandler(dict, old = old, table = table):
                """
                    called when we get a pong from the old node
                """
                rsp = dict['rsp']
                if rsp['id'] == old.id:
                    table.justSeenNode(old.id)

            def _staleNodeHandler(err, oldnode = old, newnode = n, table = table):
                """
                    called if the pinged node never responds
                """
                table.replaceStaleNode(oldnode, newnode)

            df = old.ping(self.node.id)
            df.addCallbacks(_notStaleNodeHandler, _staleNodeHandler)

    def sendPing(self, node, callback = None):
        """
            ping a node
        """
        df = node.ping(self.node.id)
        # these are the callbacks we use when we issue a PING
        def _pongSuccessHandler(dict, node = node, callback = callback):
            node.pinging = False
            _krpc_sender = dict['_krpc_sender']
            rsp = dict['rsp']
            if rsp['id'] != self.node.id:
                n = self.Node().init(rsp['id'], _krpc_sender[0], _krpc_sender[1])
                self.insertNode(n)
            if callback:
                callback()
        def _pongFailureHandler(err, node = node, callback = callback):
            if node.isIPv6():
                table = self.table6
            else:
                table = self.table
            node.pinging = False
            table.nodeFailed(node)
            if callback:
                callback()

        df.addCallbacks(_pongSuccessHandler, _pongFailureHandler)

    def findCloseNodes(self, tables, callback = lambda a: None):
        """
            This does a findNode on the ID one away from our own.  
            This will allow us to populate our table with nodes on our network closest to our own.
            This is called as soon as we start up with an empty table
        """
        id = self.node.id[:-1] + chr((ord(self.node.id[-1]) + 1) % 256)
        for t in tables:
            self.findNode(id, t, callback)

    def refreshTable(self, force = 0):
        """
            force = 1 will refresh table regardless of last bucket access time
        """
        def callback(nodes):
            pass

        for t in self.tables:
            for bucket in t.buckets:
                if force or len(bucket.l) < K or clock() - bucket.lastAccessed >= BUCKET_STALENESS:
                    id = newIDInRange(bucket.min, bucket.max)
                    if id != self.node.id:
                        self.findNode(id, t, callback)

    def stats(self, table):
        """
            Returns (num_contacts, num_nodes)
            num_contacts: number contacts in our routing table
            num_nodes: number of nodes estimated in the entire dht
        """
        num_contacts = reduce(lambda a, b: a + len(b.l), table.buckets, 0)
        num_nodes = K * (2 ** (len(table.buckets) - 1))
        return (num_contacts, num_nodes)

    def krpc_ping(self, id, _krpc_sender):
        n = self.Node().init(id, _krpc_sender[0], _krpc_sender[1])
        self.insertNode(n, contacted = 0)
        return {'id': self.node.id}

    def krpc_find_node(self, target, id, _krpc_sender, want = None):
        n = self.Node().init(id, _krpc_sender[0], _krpc_sender[1])
        self.insertNode(n, contacted = 0)
        reply = {'id': self.node.id}
        if want is None:
            if n.isIPv6():
                want = ['n6']
            else:
                want = ['n4']
        if 'n4' in want:
            reply['nodes'] = self.compactNodeInfos(self.table.findNodes(target))
        if self.ipv6 and 'n6' in want:
            reply['nodes6'] = self.compactNodeInfos6(self.table6.findNodes(target))
        return reply


# This class provides read-only access to the DHT valueForKey
# You probably want to use this mixin and provide your own write methods
class KhashmirRead(KhashmirBase):
    _Node = KNodeRead
    def retrieveValues(self, key, want = None):
        if want is None:
            if self.ipv6:
                want = ['n4', 'n6']
            else:
                want = ['n4']
        if 'n4' in want:
            try:
                l = self.peers.sample(key, NUM_PEERS)
            except:
                l = []
        else:
            l = []
        if self.ipv6 and 'n6' in want:
            try:
                l6 = self.peers6.sample(key, NUM_PEERS)
            except:
                l6 = []
        else:
            l6 = []
        return l, l6

    # async
    def valueForKey(self, key, callback, internalcallback = None, searchlocal = True):
        """
            returns the values found for key in global table (get_peers from info_hash)
            callback will be called with a list of values for each peer that returns unique values
            final callback will be called with an empty list
        """
        # get locals
        if searchlocal:
            l, l6 = self.retrieveValues(key)
            if l or l6:
                self.sched(callback, 0, [map(self.decodePeerInfo, l) + map(self.decodePeerInfo6, l6)]) 
        else:
            l = l6 = []

        # create our search state
        action = GetValue(self, key, callback, internalcallback, self.sched)
        nodes = self.table.findNodes(key)
        if self.ipv6:
            nodes.extend(self.table6.findNodes(key))
        self.sched(action.goWithNodes, 0, [nodes, l + l6])

    def krpc_get_peers(self, info_hash, id, _krpc_sender, want = None, noseed = None, scrape = None):
        n = self.Node().init(id, _krpc_sender[0], _krpc_sender[1])
        self.insertNode(n, contacted = 0)
        if want is None:
            if n.isIPv6():
                want = ['n6']
            else:
                want = ['n4']
        l, l6 = self.retrieveValues(info_hash, want)
        reply = {'id': self.node.id, 'token': self.getToken(n.host, n.port)}
        if l or l6:
            reply['values'] = l + l6
        else:
            if 'n4' in want:
                reply['nodes'] = self.compactNodeInfos(self.table.findNodes(info_hash))
            if self.ipv6 and 'n6' in want:
                reply['nodes6'] = self.compactNodeInfos6(self.table6.findNodes(info_hash))
        return reply


# Provides a generic write method, you probably don't want to deploy something that allows
# arbitrary value storage
class KhashmirWrite(KhashmirRead):
    _Node = KNodeWrite
    # async, callback indicates nodes we got a response from (but no guarantee they didn't drop it on the floor)
    def storeValueForKey(self, key, value, callback = None):
        """
            stores the value for key in the global table (announce_peer with info_hash and port)
            returns immediately, no status in this implementation, peers respond but don't indicate status to storing values
            a key can have many values
        """
        def _storeValueForKey(nodes, key = key, value = value, callback = callback):
            action = StoreValue(self, key, value, callback, self.sched)
            self.sched(action.goWithNodes, 0, [nodes])

        for t in self.tables:
            if key != self.node.id:
                # this call is async
                self.findNode(key, t, _storeValueForKey)

    def valueForKeyAndStore(self, key, value, callback, searchlocal = True):
        """
            gets the values found for key in global table (get_peers from info_hash)
            and then stores value for each value found
            locally found values are not stored
        """
        def storeValue(nodes, key = key, value = value):
            action = StoreValue(self, key, value, None, self.sched)
            self.sched(action.goWithNodes, 0, [nodes])

        self.valueForKey(key, callback, storeValue, searchlocal)

    def krpc_announce_peer(self, info_hash, port, id, token, _krpc_sender, seed = None):
        if not self.checkToken(token, _krpc_sender[0], _krpc_sender[1]):
            raise Exception, "Invalid received token"
        self.addPeer(info_hash, _krpc_sender[0], port, public = True)
        n = self.Node().init(id, _krpc_sender[0], _krpc_sender[1])
        self.insertNode(n, contacted = 0)
        return {'id': self.node.id}


# the whole shebang, for testing
class Khashmir(KhashmirWrite):
    _Node = KNodeWrite
