##############################################################################
# Module : commandscheduler.py
# Author : Old King Cole
# Date   : 05/20/2006
#
# Description : Command scheduler
#
##############################################################################
import wx

from threading import Timer
from time import clock, time
from math import ceil, floor
from filemanager import CommandListFileManager


class ScheduledCommand:
    def __init__(self, val):
        self.activity = val[0]
        self.frequency = val[1]
        self.weekschedule = val[2]
        self.schedtime = int(val[3])
        self.enddate = int(val[4])
        # Null enddate means no end date
        self.command = val[5]
        self.value = val[6]
        self.nextruntime = 0
        self.lastruntime = 0

    def computeNextRunTime(self, reftime):
        if self.frequency == 'freweeklymulti':
            mindate_dt = wx.DateTimeFromTimeT(max(reftime, self.schedtime))
            reftime_dt = wx.DateTimeFromTimeT(reftime)
            schedtime_dt = wx.DateTimeFromTimeT(self.schedtime)
            reftime_t = wx.DateTimeFromHMS(reftime_dt.GetHour(), reftime_dt.GetMinute(), reftime_dt.GetSecond())
            schedtime_t = wx.DateTimeFromHMS(schedtime_dt.GetHour(), schedtime_dt.GetMinute(), schedtime_dt.GetSecond())
            minday = mindate_dt.GetWeekDay()
            nbdays = 0
            scheddayfound = False
            for i in xrange(minday, minday + 8):
                if self.weekschedule[(i - 1) % 7] == '1' and (not mindate_dt.IsSameDate(reftime_dt) or
                   (i != reftime_dt.GetWeekDay() or not reftime_t.IsLaterThan(schedtime_t))):
                    scheddayfound = True
                    break
                nbdays += 1
            if scheddayfound:
                schedtime_dt.SetYear(mindate_dt.GetYear())
                schedtime_dt.SetMonth(mindate_dt.GetMonth())
                schedtime_dt.SetDay(mindate_dt.GetDay())
                schedtime_dt += wx.TimeSpan(nbdays * 24)
                self.nextruntime = schedtime_dt.GetTicks()
            else:
                self.nextruntime = 0
        else:
            self.nextruntime = self.schedtime
            if self.frequency == 'fredaily':
                if reftime > self.schedtime:
                    self.nextruntime += ceil((reftime - self.schedtime) / 86400.) * 86400
            elif self.frequency == 'freweekly':
                if reftime > self.schedtime:
                    self.nextruntime += ceil((reftime - self.schedtime) / 604800.) * 604800
        # Expiration date
        if self.frequency != 'freonetime' and self.enddate:
            if self.nextruntime >=  self.enddate + 86400:
                self.nextruntime = 0
        #print 'command :', self, '; next run :', wx.DateTimeFromTimeT(self.nextruntime).Format()

    def computeLastRunTime(self, reftime):
        if self.schedtime >= reftime or (self.enddate and self.schedtime >= self.enddate + 86400) or self.activity == 'actdisabled' \
           or self.command == 'comnone' or self.command == 'comexit':
            # Never run yet or scheduled after expiration date or disabled or without command or exit command
            self.lastruntime = 0
        else:
            if self.enddate:
                lastpastvaliddate = min(self.enddate + 86400, reftime)
            else:
                lastpastvaliddate = reftime
            if self.frequency == 'freweeklymulti':
                maxdate_dt = wx.DateTimeFromTimeT(lastpastvaliddate)
                reftime_dt = wx.DateTimeFromTimeT(reftime)
                schedtime_dt = wx.DateTimeFromTimeT(self.schedtime)
                reftime_t = wx.DateTimeFromHMS(reftime_dt.GetHour(), reftime_dt.GetMinute(), reftime_dt.GetSecond())
                schedtime_t = wx.DateTimeFromHMS(schedtime_dt.GetHour(), schedtime_dt.GetMinute(), schedtime_dt.GetSecond())
                maxday = maxdate_dt.GetWeekDay()
                #print maxday
                nbdays = 0
                lastscheddayfound = False
                for i in xrange(maxday, maxday - 8, -1):
                    #print i, self.weekschedule[(i - 1)%7], reftime_dt.GetWeekDay(), reftime_t.IsLaterThan(schedtime_t)
                    if self.weekschedule[(i - 1)%7] == '1' and (not maxdate_dt.IsSameDate(reftime_dt) or
                                                          (i != reftime_dt.GetWeekDay() or reftime_t.IsLaterThan(schedtime_t))):
                        lastscheddayfound = True
                        break
                    nbdays += 1
                #print nbdays, lastscheddayfound
                if lastscheddayfound:
                    schedtime_dt.SetYear(maxdate_dt.GetYear())
                    schedtime_dt.SetMonth(maxdate_dt.GetMonth())
                    schedtime_dt.SetDay(maxdate_dt.GetDay())
                    schedtime_dt -= wx.TimeSpan(nbdays * 24)
                    self.lastruntime = schedtime_dt.GetTicks()
                    # Check if not below self.schedtime
                    if self.lastruntime < self.schedtime:
                        self.lastruntime = 0
                else:
                    self.lastruntime = 0
            else:
                self.lastruntime = self.schedtime
                if self.frequency == 'fredaily':
                    self.lastruntime += floor((lastpastvaliddate - self.schedtime) / 86400.) * 86400
                elif self.frequency == 'freweekly':
                    self.lastruntime += floor((lastpastvaliddate - self.schedtime) / 604800.) * 604800


class CommandScheduler:
    def __init__(self, parent):
        self.parent = parent
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.abcparams = self.utility.abcparams
        self.schedcommands = []
        self.nextschedcommand = None
        self.status = 0
        self.stopping = False
        self.act = ['actenabled', 'actdisabled']
        self.com = ['comnone', 'commode', 'comstoptraffic', 'comupratelimit', 'comdynuprate', 'comstatupratedown',
                    'comstatuprateseed', 'comdownrate', 'comnumsimdownload', 'comnumsimseed', 'comnumsimseednodown',
                    'comschedulermode', 'comscanner', 'comwebserv', 'comrestart', 'comexit']
        self.val = {'comnone':['valnone'],
                    'commode':['valmanual', 'valauto'],
                    'comstoptraffic':['valnone'],
                    'comupratelimit':['valstat', 'valdyn'],
                    'comdynuprate':['valnumber'],
                    'comstatupratedown':['valnumber'],
                    'comstatuprateseed':['valnumber'],
                    'comdownrate':['valnumber'],
                    'comnumsimdownload':['valnumber'],
                    'comnumsimseed':['valnumber'],
                    'comnumsimseednodown':['valnumber'],
                    'comschedulermode':['valtrigafterdown', 'valtrigafterseed', 'valonlyseed'],
                    'comscanner':['valoff', 'valon'],
                    'comwebserv':['valoff', 'valon'],
                    'comrestart':['valnone'],
                    'comexit':['valnone']}
        # Read scheduled command list file
        schedcomlistfile = CommandListFileManager(self)
        schedcomlistfile.open()
        index = -1
        while True:
            temp = schedcomlistfile.readList()
            if temp is None:
                break
            if len(temp) != 7:
                continue
            index += 1
            self.schedcommands.append(ScheduledCommand(temp))
        schedcomlistfile.close()

    def start(self):
        now = time()
        self.stopping = False
        self.status = 1
        # Replay last run if needed
        if self.utility.schedparams['schedrunpast'] == '1':
            self.playLastRuns(now)
        # Init all run times
        for schedcom in self.schedcommands:
            schedcom.computeNextRunTime(now)
        self.scheduleNext(now)
    
    def stop(self):
        self.stopping = True
        try:
            self.schedulertimer.cancel()
        except:
            pass
        self.status = 0
        self.nextschedcommand = None

    def postCommand(self, schedcommand):
        self.parent.queue.invokeLater(self.runScheduledCommand, [schedcommand])

    def runScheduledCommand(self, schedcommand):
        if self.stopping:
            return
        reftime = schedcommand.nextruntime
        schedcommand.nextruntime = 0
        self.scheduleNext(reftime)

        if schedcommand.command == 'commode':
            if schedcommand.value == 'valauto':
                self.parent.onModeToggle(mode = 1)
            elif schedcommand.value == 'valmanual':
                self.parent.onModeToggle(mode = 0)
        elif schedcommand.command == 'comstoptraffic':
            self.parent.onModeToggle(mode = 0, queue = 1)
        elif schedcommand.command == 'comupratelimit':
            if schedcommand.value == 'valstat':
                if self.abcparams['urmdynmaxuprate'] == '1':
                    self.abcparams['urmdynmaxuprate'] = '0'
                    self.parent.queue.urminitialstartingtime = clock()
            elif schedcommand.value == 'valdyn':
                if self.abcparams['urmdynmaxuprate'] == '0':
                    self.abcparams['urmdynmaxuprate'] = '1'
                    self.parent.queue.urminitialstartingtime = clock()
        elif schedcommand.command == 'comdynuprate':
            if self.abcparams['urmdynmaxuprate'] == '1' and int(schedcommand.value) > self.utility.maxsysup:
                self.parent.queue.urminitialstartingtime = clock()
            self.abcparams['urmmaxsysuprate'] = schedcommand.value
            self.utility.maxsysup = float(self.abcparams['urmmaxsysuprate'])
        elif schedcommand.command == 'comstatupratedown':
            uprateval = int(schedcommand.value)
            if self.abcparams['urmdynmaxuprate'] == '0' and uprateval > self.utility.maxup \
               and self.parent.queue.torrentdownloading:
                self.parent.queue.urminitialstartingtime = clock()
            if uprateval < self.utility.minuprate:
                self.abcparams['maxuploadrate'] = str(int(self.utility.minuprate))
            else:
                self.abcparams['maxuploadrate'] = schedcommand.value
            self.utility.maxup = float(self.abcparams['maxuploadrate'])
        elif schedcommand.command == 'comstatuprateseed':
            uprateval = int(schedcommand.value)
            if self.abcparams['urmdynmaxuprate'] == '0' and uprateval > self.utility.maxseedup \
               and not self.parent.queue.torrentdownloading:
                self.parent.queue.urminitialstartingtime = clock()
            if uprateval < self.utility.minuprate:
                self.abcparams['maxseeduploadrate'] = str(int(self.utility.minuprate))
            else:
                self.abcparams['maxseeduploadrate'] = schedcommand.value
            self.utility.maxseedup = float(self.abcparams['maxseeduploadrate'])
        elif schedcommand.command == 'comdownrate':
            downrateval = int(schedcommand.value)
            if downrateval > self.utility.maxdown:
                self.parent.queue.drminitialstartingtime = clock()
            if downrateval < self.utility.mindownrate:
                self.abcparams['maxdownloadrate'] = str(int(self.utility.mindownrate))
            else:
                self.abcparams['maxdownloadrate'] = schedcommand.value
            self.utility.maxdown = float(self.abcparams['maxdownloadrate'])
        elif schedcommand.command == 'comnumsimdownload':
            self.abcparams['numsimdownload'] = schedcommand.value
            self.parent.queue.maxnumsim = int(schedcommand.value)
            self.parent.queue.scheduler(True)
        elif schedcommand.command == 'comnumsimseed':
            self.abcparams['numsimseed'] = schedcommand.value
            self.parent.queue.maxnumsimseed = int(schedcommand.value)
            self.parent.queue.scheduler(True)
        elif schedcommand.command == 'comnumsimseednodown':
            self.abcparams['numsimseednodown'] = schedcommand.value
            self.parent.queue.maxnumsimseednodown = int(schedcommand.value)
            self.parent.queue.scheduler(True)
        elif schedcommand.command == 'comschedulermode':
            oldtrig = self.abcparams['trigwhenfinishseed']
            if schedcommand.value == 'valtrigafterdown':
                self.abcparams['trigwhenfinishseed'] = "0"
            elif schedcommand.value == 'valtrigafterseed':
                self.abcparams['trigwhenfinishseed'] = "1"
            elif schedcommand.value == 'valonlyseed' and self.abcparams['trigwhenfinishseed'] != "2":
                self.parent.queue.setSchedulerModeToSeeding(oldtrig)
                self.abcparams['trigwhenfinishseed'] = "2"
            if self.abcparams['trigwhenfinishseed'] != oldtrig:
                self.abcparams['schedonlyonhold'] = '0'
                trignexttorrentonlyseedind = self.localize('schedmodeseedcap')
                if trignexttorrentonlyseedind:
                    if self.abcparams['trigwhenfinishseed'] == '2':
                        self.utility.frame.SetTitle(self.localize('mainwindowtitle') + trignexttorrentonlyseedind)
                    elif oldtrig == '2':
                        self.utility.frame.SetTitle(self.localize('mainwindowtitle'))
            self.parent.queue.scheduler(True)
        elif schedcommand.command == 'comscanner':
            if schedcommand.value == 'valon':
                self.parent.onScannerToggle(status = 1)
            elif schedcommand.value == 'valoff':
                self.parent.onScannerToggle(status = 0)
        elif schedcommand.command == 'comwebserv':
            if schedcommand.value == 'valon':
                self.parent.onWebServiceToggle(status = 1)
            elif schedcommand.value == 'valoff':
                self.parent.onWebServiceToggle(status = 0)
        elif schedcommand.command == 'comrestart':
            self.parent.parent.onMenuRestart()
        elif schedcommand.command == 'comexit':
            self.parent.parent.onMenuExit(silent = True)
    
    def scheduleNext(self, reftime):
        # Find next command to be run
        # If different from activated one, disable then activate the new one
        activatedschedcommand = self.nextschedcommand
        # Find next to be run
        self.nextschedcommand = None
        for schedcommand in self.schedcommands:
            if schedcommand.activity == 'actenabled' and schedcommand.command != 'comnone' and schedcommand.nextruntime\
               and schedcommand.nextruntime >= reftime and (self.nextschedcommand == None or schedcommand.nextruntime < self.nextschedcommand.nextruntime):
                self.nextschedcommand = schedcommand

        if self.nextschedcommand is not None and self.nextschedcommand != activatedschedcommand:
            if activatedschedcommand is not None:
                try:
                    self.schedulertimer.cancel()
                except:
                    pass
            # Activate command
            if not self.stopping:
                self.schedulertimer = Timer(self.nextschedcommand.nextruntime - reftime, self.postCommand, [self.nextschedcommand])
                self.schedulertimer.start()
        #if self.nextschedcommand is not None:
        #    print 'Next sched command :', self.schedcommands.index(self.nextschedcommand), 'scheduled on :', wx.DateTimeFromTimeT(self.nextschedcommand.nextruntime).Format()
        #else:
        #    print 'Next sched command : None'

    def playLastRuns(self, reftime):
        for schedcom in self.schedcommands:
            schedcom.computeLastRunTime(reftime)
        lastcommands = [(c.lastruntime, c) for c in self.schedcommands if c.lastruntime]
        # Sort by lastruntime
        lastcommands.sort()
        for t in lastcommands:
            #print 'Replayed :', self.schedcommands.index(t[1]), '; lastrun :',  wx.DateTimeFromTimeT(t[0]).Format()
            self.postCommand(t[1])
