##############################################################################
# Module : abc_find.py
# Author : Old King Cole
# Date   : 12/18/2004
#
# Description : A utility to find torrents in the torrent list matching some
#               values
#
##############################################################################
import sys, wx
from os import path
from binascii import unhexlify

from utility import SpinBox


##############################################################
# Class : ABCFindPanel
##############################################################
class ABCFindPanel(wx.Panel):
    def __init__(self, parent):
        wx.Panel.__init__(self, parent, -1, style = wx.TAB_TRAVERSAL)

        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.abcparams = self.parent.abcparams
        self.proctab = self.parent.parent.window.queue.proctab
        self.mainlist = self.parent.parent.window.list
        self.foundinfohash = []
        self.sortdir = 0
        self.lastsortingcol = 0
        # List row selected by a left or a right click
        self.selectedrow = -1
        # Torrent list dragging mode :
        # 0 : no dragging
        # 1 : select torrent
        self.listdraggingmode = 0
        self.selectfound = self.abcparams['findselectfound'] == "1"

        buttonlabel = self.localize('findand')
        buttonwidth = self.GetTextExtent(buttonlabel)[0] + 12
        buttonlabel = buttonlabel.replace('&', '&&')
        self.findtorrentname = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(189, -1))
        self.findtorrentnamebtn = wx.ToggleButton(self, -1, buttonlabel, size = (buttonwidth, self.utility.buttonheight))
        self.findtorrentnamebtn.SetValue(self.abcparams['findtorrentnamebtn'] == "1")
        self.findtorrentnamesim = SpinBox(self, width = 28, height = self.utility.digitheight,
                                          value = self.abcparams['findtorrentnamesim'], defvalue = "1",
                                          min = -99, max = 99, raiseexcept = False, filterfunc = self.utility.onWriteDigitsMinus,
                                          rightbutfunc = self.onFindTorrent, style = wx.TE_CENTRE)
        self.findtorrentfilename = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(189, -1))
        self.findtorrentfilenamebtn = wx.ToggleButton(self, -1, buttonlabel, size = (buttonwidth, self.utility.buttonheight))
        self.findtorrentfilenamebtn.SetValue(self.abcparams['findtorrentfilenamebtn'] == "1")
        self.findtorrentfilenamesim = SpinBox(self, width = 28, height = self.utility.digitheight,
                                              value = self.abcparams['findtorrentfilenamesim'], defvalue = "1",
                                              min = -99, max = 99, raiseexcept = False, filterfunc = self.utility.onWriteDigitsMinus,
                                              rightbutfunc = self.onFindTorrent, style = wx.TE_CENTRE)
        self.findlabel = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(63, -1))
        self.findlabelbtn = wx.ToggleButton(self, -1, buttonlabel, size = (buttonwidth, self.utility.buttonheight))
        self.findlabelbtn.SetValue(self.abcparams['findlabelbtn'] == "1")
        self.findshortlabel = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(63, -1))
        self.findshortlabelbtn = wx.ToggleButton(self, -1, buttonlabel, size = (buttonwidth, self.utility.buttonheight))
        self.findshortlabelbtn.SetValue(self.abcparams['findshortlabelbtn'] == "1")
        self.findtrackername = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(126, -1))
        self.findtrackernamebtn = wx.ToggleButton(self, -1, buttonlabel, size = (buttonwidth, self.utility.buttonheight))
        self.findtrackernamebtn.SetValue(self.abcparams['findtrackernamebtn'] == "1")
        self.findinfohash = wx.TextCtrl(self, -1, '', wx.Point(-1, -1), wx.Size(126, -1))

        findstringbox = wx.BoxSizer(wx.HORIZONTAL)
        findstringleftbox = wx.FlexGridSizer(2, 2, 3, 3)
        findstringmidbox = wx.FlexGridSizer(2, 2, 3, 3)
        findstringrightbox = wx.FlexGridSizer(2, 2, 3, 3)

        findtorrentnamebox = wx.BoxSizer(wx.HORIZONTAL)
        findtorrentnamebox.Add(self.findtorrentname, 0, wx.ALIGN_CENTER_VERTICAL)
        findtorrentnamebox.Add(self.findtorrentnamebtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 1)
        findtorrentnamebox.Add(self.findtorrentnamesim.getSizer(), 0, wx.ALIGN_CENTER_VERTICAL)

        self.findtorrentnamelabel = wx.StaticText(self, -1, self.localize('findtorrentname'))
        findstringleftbox.Add(self.findtorrentnamelabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringleftbox.Add(findtorrentnamebox)

        findtorrentfilenamebox = wx.BoxSizer(wx.HORIZONTAL)
        findtorrentfilenamebox.Add(self.findtorrentfilename, 0, wx.ALIGN_CENTER_VERTICAL)
        findtorrentfilenamebox.Add(self.findtorrentfilenamebtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 1)
        findtorrentfilenamebox.Add(self.findtorrentfilenamesim.getSizer(), 0, wx.ALIGN_CENTER_VERTICAL)

        self.findtorrentfilenamelabel = wx.StaticText(self, -1, self.localize('findtorrentfilename'))
        findstringleftbox.Add(self.findtorrentfilenamelabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringleftbox.Add(findtorrentfilenamebox)

        findlabelbox = wx.BoxSizer(wx.HORIZONTAL)
        findlabelbox.Add(self.findlabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findlabelbox.Add(self.findlabelbtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 1)

        self.findlabellabel = wx.StaticText(self, -1, self.localize('findlabel'))
        findstringmidbox.Add(self.findlabellabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringmidbox.Add(findlabelbox)

        findshortlabelbox = wx.BoxSizer(wx.HORIZONTAL)
        findshortlabelbox.Add(self.findshortlabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findshortlabelbox.Add(self.findshortlabelbtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 1)

        self.findshortlabellabel = wx.StaticText(self, -1, self.localize('findshortlabel'))
        findstringmidbox.Add(self.findshortlabellabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringmidbox.Add(findshortlabelbox)

        findtrackernamebox = wx.BoxSizer(wx.HORIZONTAL)
        findtrackernamebox.Add(self.findtrackername, 0, wx.ALIGN_CENTER_VERTICAL)
        findtrackernamebox.Add(self.findtrackernamebtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 1)

        self.findtrackernamelabel = wx.StaticText(self, -1, self.localize('findtrackername'))
        findstringrightbox.Add(self.findtrackernamelabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringrightbox.Add(findtrackernamebox)

        self.findinfohashlabel = wx.StaticText(self, -1, self.localize('findinfohash'))
        findstringrightbox.Add(self.findinfohashlabel, 0, wx.ALIGN_CENTER_VERTICAL)
        findstringrightbox.Add(self.findinfohash, 1, wx.EXPAND)

        findstringbox.Add(findstringleftbox, 0, wx.ALL, 0)
        findstringbox.Add(findstringmidbox, 0, wx.LEFT, 10)
        findstringbox.Add(findstringrightbox, 0, wx.LEFT, 10)

        self.findstatuslabel = wx.StaticText(self, -1, self.localize('findstatus'))
        self.statusval = [self.localize('any'),
                          self.localize('finished'),
                          self.localize('completedseeding'),
                          self.localize('completed'),
                          self.localize('working'),
                          self.localize('onhold'),
                          self.localize('pause'),
                          self.localize('queue'),
                          self.localize('standby'),
                          self.localize('stop'),
                          self.localize('superseeding'),
                          self.localize('waiting'),
                          self.localize('checkingexistingdata2'),
                          self.localize('allocatingdiskspace2'),
                          self.localize('awaking')]
        self.findstatus = wx.ComboBox(self, -1, self.statusval[0],
                                      wx.Point(-1, -1), wx.Size(-1, -1), self.statusval, wx.CB_DROPDOWN | wx.CB_READONLY)
        self.findpriolabel = wx.StaticText(self, -1, self.localize('findprio'))
        prioval = [self.localize('any'),
                   self.localize('rate'),
                   self.localize('rateup'),
                   self.localize('ratedown'),
                   self.localize('high_s') + self.localize('highest_s'),
                   self.localize('highest_s'),
                   self.localize('high_s'),
                   self.localize('normal_s'),
                   self.localize('low_s'),
                   self.localize('lowest_s'),
                   self.localize('low_s') + self.localize('lowest_s')]
        self.findprio = wx.ComboBox(self, -1, prioval[0],
                                    wx.Point(-1, -1), wx.Size(-1, -1), prioval, wx.CB_DROPDOWN | wx.CB_READONLY)
        self.findavailabilitylabel = wx.StaticText(self, -1, self.localize('findavailability'))
        self.findavailability = wx.ComboBox(self, -1, self.localize('any'),
                                            wx.Point(-1, -1), wx.Size(-1, -1),
                                            [self.localize('any'), self.localize('complete'),
                                             self.localize('incomplete'), self.localize('withallpieces'),
                                             self.localize('withseeds')],
                                            wx.CB_DROPDOWN | wx.CB_READONLY)
        
        self.findpeers = wx.CheckBox(self, -1, self.localize('findpeers'), style = wx.CHK_3STATE | wx.CHK_ALLOW_3RD_STATE_FOR_USER)
        self.findpeers.SetToolTipString(self.localize('findpeerstt'))
        self.findpeers.Set3StateValue(2)
        self.findactivity = wx.CheckBox(self, -1, self.localize('findactivity'), style = wx.CHK_3STATE | wx.CHK_ALLOW_3RD_STATE_FOR_USER)
        self.findactivity.SetToolTipString(self.localize('findactivitytt'))
        self.findactivity.Set3StateValue(2)
        self.findprivate = wx.CheckBox(self, -1, self.localize('findprivate'), style = wx.CHK_3STATE | wx.CHK_ALLOW_3RD_STATE_FOR_USER)
        self.findprivate.SetToolTipString(self.localize('findprivatett'))
        self.findprivate.Set3StateValue(2)
        findstatusbox = wx.BoxSizer(wx.HORIZONTAL)
        findstatusbox.Add(self.findstatuslabel, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 3)
        findstatusbox.Add(self.findstatus, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 3)
        findstatusbox.Add(self.findpriolabel, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)
        findstatusbox.Add(self.findprio, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 3)
        findstatusbox.Add(self.findavailabilitylabel, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)
        findstatusbox.Add(self.findavailability, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 3)
        findstatusbox.Add(self.findpeers, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 24)
        findstatusbox.Add(self.findactivity, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)
        findstatusbox.Add(self.findprivate, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 10)

        matchsection = wx.StaticBoxSizer(wx.StaticBox(self, -1, ''), wx.VERTICAL)

        matchsection.Add(findstringbox, 0, wx.ALL, 3)
        matchsection.Add(findstatusbox, 0, wx.TOP, 3)

        findbtn = wx.Button(self, -1, self.localize('findtorrent'), size = (60, -1))
        findbtn.SetDefault()
        self.inselected = wx.CheckBox(self, -1, self.localize('inselected'))
        buttonlabel = self.localize('reset')
        buttonwidth = self.GetTextExtent(buttonlabel)[0] + 12
        buttonlabel = buttonlabel.replace('&', '&&')
        resetbtn = wx.Button(self, -1, buttonlabel, size = wx.Size(buttonwidth, self.utility.buttonheight))
        closebtn = wx.Button(self, -1, self.localize('close'), size = (60, -1))

        resetclosebox = wx.BoxSizer(wx.HORIZONTAL)
        resetclosebox.Add(resetbtn, 0, wx.ALIGN_CENTER_VERTICAL)
        resetclosebox.Add(closebtn, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 5)

        buttonbox = wx.BoxSizer(wx.VERTICAL)
        buttonbox.Add(findbtn, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP, 5)
        buttonbox.Add(self.inselected, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP, 5)
        buttonbox.Add(resetclosebox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP, 10)

        matchbuttonbox = wx.BoxSizer(wx.HORIZONTAL)
        matchbuttonbox.Add(matchsection, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 3)
        matchbuttonbox.Add(buttonbox, 1, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 8)

        # List for find result
        self.list = wx.ListCtrl(self, -1, size = (-1, 205), style = wx.LC_REPORT | wx.LC_VRULES | wx.LC_HRULES)
        self.list.SetFont(wx.Font(int(self.abcparams['listfontsize']), wx.DEFAULT, int(self.abcparams['listfontstyle']),
                                  int(self.abcparams['listfontweight']), faceName = self.abcparams['listfont']))
        self.list.InsertColumn(0, self.localize('findresultrow'), width = self.utility.abcfindlistwidth0)
        self.list.InsertColumn(1, self.parent.parent.guiman.getText(4))
        self.list.InsertColumn(2, self.parent.parent.guiman.getText(21))

        outerbox = wx.BoxSizer(wx.VERTICAL)
        outerbox.Add(matchbuttonbox, 0, wx.EXPAND | wx.TOP, -3)
        outerbox.Add(self.list, 1, wx.EXPAND | wx.TOP | wx.LEFT | wx.RIGHT, 3)

        self.SetAutoLayout(True)
        self.SetSizer(outerbox)
        self.Fit()

        # Size torrent name and file name columns
        colwidth = (self.list.GetClientSizeTuple()[0] - self.utility.abcfindlistwidth0) / 2
        self.list.SetColumnWidth(1, colwidth)
        self.list.SetColumnWidth(2, colwidth)

        # Add events
        self.Bind(wx.EVT_BUTTON, self.onFindTorrent, findbtn)
        self.Bind(wx.EVT_BUTTON, self.onReset, resetbtn)
        self.Bind(wx.EVT_BUTTON, self.onClose, closebtn)
        self.Bind(wx.EVT_CLOSE, self.onClose)
        self.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onListRightClick, self.list)
        self.findtorrentnamelabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelTorrentName)
        self.findtorrentfilenamelabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelTorrentFileName)
        self.findlabellabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelLabel)
        self.findtrackernamelabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelTrackerName)
        self.findshortlabellabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelShortLabel)
        self.findinfohashlabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelInfohash)
        self.findstatuslabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelStatus)
        self.findpriolabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelPrio)
        self.findavailabilitylabel.Bind(wx.EVT_RIGHT_DOWN, self.onLabelAvailability)
        self.findpeers.Bind(wx.EVT_RIGHT_DOWN, self.onLabelPeers)
        self.findactivity.Bind(wx.EVT_RIGHT_DOWN, self.onLabelActivity)
        self.findprivate.Bind(wx.EVT_RIGHT_DOWN, self.onLabelPrivate)
        self.list.Bind(wx.EVT_LEFT_DCLICK, self.onLeftDClick)
        self.list.Bind(wx.EVT_RIGHT_DOWN, self.onRightDown)
        self.list.Bind(wx.EVT_RIGHT_UP, self.onRightUp)
        self.list.Bind(wx.EVT_LEFT_UP, self.onLeftUp)
        self.list.Bind(wx.EVT_LEFT_DOWN, self.onLeftDown)
        self.list.Bind(wx.EVT_LEAVE_WINDOW, self.onLeaveList)
        self.list.Bind(wx.EVT_LIST_BEGIN_DRAG, self.onLeftDragList)
        self.list.Bind(wx.EVT_LIST_BEGIN_RDRAG, self.onRightDragList)
        self.list.Bind(wx.EVT_MIDDLE_DOWN, self.onMiddleDown)
        self.list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
        self.list.Bind(wx.EVT_LIST_COL_CLICK, self.onColLeftClick)
        self.list.Bind(wx.EVT_KEY_DOWN, self.onKeyInList)

        self.x = self.y = 0
        self.fromrightclick = False

    def onKeyInList(self, event):
        if event.ControlDown():
            keycode = event.GetKeyCode()
            if keycode == ord('a') or keycode == ord('A'):
                # Select all
                for index in xrange(self.list.GetItemCount()):
                    self.list.Select(index)
            elif keycode == self.utility.selinvert1 or keycode == self.utility.selinvert2:
                # Invert selection
                for index in xrange(self.list.GetItemCount()):
                    self.list.SetItemState(index, 4 - self.list.GetItemState(index, wx.LIST_STATE_SELECTED), wx.LIST_STATE_SELECTED)
            elif keycode == ord('z') or keycode == ord('Z'):
                # Unselect all
                for index in xrange(self.list.GetItemCount()):
                    self.list.Select(index, 0)
            else:
                event.Skip()
        else:
            event.Skip()

    def onItemSelected(self, event):
        if self.fromrightclick:
            return
        self.locateInMainList(event.GetIndex())

    def onLocateInMainList(self, event):
        self.locateInMainList(self.selectedrow)

    def locateInMainList(self, index):
        # Find and show the selected torrent in the find result list, in the main ABC list
        mainlistindex = self.findTorrentInMainList(index)
        if mainlistindex >= 0:
            self.mainlist.Select(mainlistindex)
            self.mainlist.EnsureVisible(mainlistindex)

    def onFindTorrent(self, event = None):
        infohashstring = self.findinfohash.GetValue()
        if infohashstring:
            if len(infohashstring) != 40:
                dlg = wx.MessageDialog(self, self.localize('infohashbadlen'), self.localize('abcokcerror'), wx.ICON_ERROR)
                dlg.ShowModal()
                dlg.Destroy()
                return
            try:
                infohash = unhexlify(infohashstring)
            except:
                dlg = wx.MessageDialog(self, self.localize('infohashinvalid'), self.localize('abcokcerror'), wx.ICON_ERROR)
                dlg.ShowModal()
                dlg.Destroy()
                return
        # Clear result list
        self.list.DeleteAllItems()
        self.foundinfohash[:] = []
        # Reset sorting order
        self.sortdir = 0
        self.lastsortingcol = 0
        self.parent.SetStatusText(self.localize('searching'), 0)
        namestrings = set(self.findtorrentname.GetValue().lower().split())
        filenamestrings = set(self.findtorrentfilename.GetValue().lower().split())
        labelstrings = set(self.findlabel.GetValue().lower().split())
        shortlabelstrings = set(self.findshortlabel.GetValue().lower().split())
        trackernamestrings = set(self.findtrackername.GetValue().lower().split())
        sim = self.findtorrentnamesim.getValue()
        if namestrings:
            if self.findtorrentnamebtn.GetValue():
                namesim = len(namestrings)
            elif sim > 0:
                namesim = min(sim, len(namestrings))
            else:
                namesim = max(len(namestrings) + sim, 0)
            namesimall = namesim == len(namestrings)
        sim = self.findtorrentfilenamesim.getValue()
        if filenamestrings:
            if self.findtorrentfilenamebtn.GetValue():
                filenamesim = len(filenamestrings)
            elif sim > 0:
                filenamesim = min(sim, len(filenamestrings))
            else:
                filenamesim = max(len(filenamestrings) + sim, 0)
            filenamesimall = filenamesim == len(filenamestrings)
        if labelstrings:
            if self.findlabelbtn.GetValue():
                labelsim = len(labelstrings)
                labelsimall = True
            else:
                labelsim = 1
                labelsimall = False
        if shortlabelstrings:
            if self.findshortlabelbtn.GetValue():
                shortlabelsim = len(shortlabelstrings)
                shortlabelsimall = True
            else:
                shortlabelsim = 1
                shortlabelsimall = False
        if trackernamestrings:
            if self.findtrackernamebtn.GetValue():
                trackernamesim = len(trackernamestrings)
                trackernamesimall = True
            else:
                trackernamesim = 1
                trackernamesimall = False
        status = self.findstatus.GetSelection()
        prio = self.findprio.GetSelection()
        availability = self.findavailability.GetSelection()
        peers = self.findpeers.Get3StateValue()
        activity = self.findactivity.Get3StateValue()
        private = self.findprivate.Get3StateValue()
        inselected = self.inselected.GetValue()

        nbfound = 0
        row = -1
        for torrent in self.proctab:
            row += 1
            # Search in whole list or selected torrents
            if inselected and not self.mainlist.IsSelected(row):
                continue
            if self.selectfound:
                self.mainlist.SetItemState(row, 0, wx.LIST_STATE_SELECTED)
            # Torrent info_hash
            if infohashstring:
                if torrent.infohash != infohash:
                    continue
            # Torrent name
            if namestrings and namesim:
                neededmatch = namesim
                if namesimall:
                    for substr in namestrings:
                        if torrent.name.lower().find(substr) < 0:
                            break
                        neededmatch -= 1
                else:
                    for substr in namestrings:
                        if torrent.name.lower().find(substr) >= 0:
                            neededmatch -= 1
                            if neededmatch == 0:
                                break
                if neededmatch:
                    continue
            # Torrent file name
            if filenamestrings and filenamesim:
                neededmatch = filenamesim
                if filenamesimall:
                    for substr in filenamestrings:
                        if path.split(torrent.src)[1].lower().find(substr) < 0:
                            break
                        neededmatch -= 1
                else:
                    for substr in filenamestrings:
                        if path.split(torrent.src)[1].lower().find(substr) >= 0:
                            neededmatch -= 1
                            if neededmatch == 0:
                                break
                if neededmatch:
                    continue
            # Torrent label
            if labelstrings:
                neededmatch = labelsim
                if labelsimall:
                    for substr in labelstrings:
                        if torrent.label.lower().find(substr) < 0:
                            break
                        neededmatch -= 1
                else:
                    for substr in labelstrings:
                        if torrent.label.lower().find(substr) >= 0:
                            neededmatch = 0
                            break
                if neededmatch:
                    continue
            # Torrent short label
            if shortlabelstrings:
                neededmatch = shortlabelsim
                if shortlabelsimall:
                    for substr in shortlabelstrings:
                        if torrent.shortlabel.lower().find(substr) < 0:
                            break
                        neededmatch -= 1
                else:
                    for substr in shortlabelstrings:
                        if torrent.shortlabel.lower().find(substr) >= 0:
                            neededmatch = 0
                            break
                if neededmatch:
                    continue
            # Torrent status
            if status:
                if status == 2:
                    if not torrent.abcengine or not torrent.complete or torrent.status == 'pause' or torrent.status == 'onhold' or torrent.abcengine.onlycheck:
                        continue
                elif status == 3:
                    if self.statusval[3] != self.localize(torrent.status) or torrent.abcengine:
                        continue
                elif status == 4:
                    if not torrent.abcengine or torrent.complete or torrent.status == 'pause' or torrent.status == 'onhold' or torrent.abcengine.onlycheck:
                        continue
                elif status == 10:
                    if not torrent.abcengine or not torrent.abcengine.superseed or torrent.status == 'pause' or torrent.status == 'onhold':
                        continue
                elif status == 11:
                    if not torrent.abcengine or torrent.abcengine.btstatus != "waiting" or torrent.status == 'pause' or torrent.status == 'onhold':
                        continue
                elif status == 12:
                    if not torrent.abcengine or not torrent.abcengine.btstatus.startswith("checkingexistingdata") or torrent.status == 'pause' or torrent.status == 'onhold':
                        continue
                elif status == 13:
                    if not torrent.abcengine or not torrent.abcengine.btstatus.startswith(("movingdata", "allocatingdiskspace")) or torrent.status == 'pause' or torrent.status == 'onhold':
                        continue
                elif status == 14:
                    if torrent.activity != 2 or torrent.status == 'pause' or torrent.status == 'onhold':
                        continue
                elif self.statusval[status] != self.localize(torrent.status):
                    continue
            # Torrent priority
            if prio:
                if prio == 1:
                    if not torrent.prioritizedown and not torrent.prioritizeup:
                        continue
                elif prio == 2:
                    if not torrent.prioritizeup:
                        continue
                elif prio == 3:
                    if not torrent.prioritizedown:
                        continue
                elif prio == 4:
                    if torrent.priority > 1:
                        continue
                elif prio == 5:
                    if torrent.priority != 0:
                        continue
                elif prio == 6:
                    if torrent.priority != 1:
                        continue
                elif prio == 7:
                    if torrent.priority != 2:
                        continue
                elif prio == 8:
                    if torrent.priority != 3:
                        continue
                elif prio == 9:
                    if torrent.priority != 4:
                        continue
                elif prio == 10:
                    if torrent.priority < 3:
                        continue
            # Availability
            if availability:
                if availability == 1:
                    if not torrent.complete:
                        continue
                elif availability == 2:
                    if torrent.complete:
                        continue
                elif availability == 3:
                    if (not torrent.abcengine or torrent.abcengine.numseed == '0') \
                        and (torrent.seed == self.utility.notrackeranswermarker or torrent.seed == self.utility.notrackermarker \
                        or torrent.seed == '0' or torrent.seed == self.utility.scrapingmarker) \
                        and (torrent.numcopy == '?' or float(torrent.numcopy) < 1):
                        continue
                elif availability == 4:
                    if (not torrent.abcengine or torrent.abcengine.numseed == '0') \
                        and (torrent.seed == self.utility.notrackeranswermarker or torrent.seed == self.utility.notrackermarker \
                        or torrent.seed == '0' or torrent.seed == self.utility.scrapingmarker):
                        continue
            # Peers
            if peers != 2:
                if (not torrent.abcengine or torrent.abcengine.numpeer == '0') \
                    and (torrent.peer == self.utility.notrackeranswermarker or torrent.peer == self.utility.notrackermarker \
                    or torrent.peer == '0' or torrent.peer == self.utility.scrapingmarker):
                    if peers == 1:
                        continue
                elif peers == 0:
                    continue
            # Activity
            if activity == 1 or activity != 2 and torrent.activity != 2:
                if torrent.activity != activity:
                    continue
            # Private
            if private != 2:
                if torrent.private != private:
                    continue
            # Tracker name
            if trackernamestrings:
                trackerlist = []
                metainfo = torrent.getResponse()
                if metainfo.has_key('announce'):
                    trackerlist.append(self.utility.decodeString(metainfo['announce']).lower())
                if metainfo.has_key('announce-list'):
                    for tier in metainfo['announce-list']:
                        for announce in tier:
                            trackerlist.append(self.utility.decodeString(announce).lower())
                # Add external tracker if needed
                if torrent.exttracker:
                    trackerlist.append(torrent.exttrackerurl.lower())
                if not trackerlist:
                    continue
                for tracker in trackerlist:
                    neededmatch = trackernamesim
                    if trackernamesimall:
                        for substr in trackernamestrings:
                            if tracker.find(substr) < 0:
                                break
                            neededmatch -= 1
                    else:
                        for substr in trackernamestrings:
                            if tracker.find(substr) >= 0:
                                neededmatch = 0
                                break
                    if neededmatch == 0:
                        break
                if neededmatch:
                    continue

            # Add torrent to result list
            self.list.InsertStringItem(nbfound, "")
            self.list.SetStringItem(nbfound, 0, str(row + 1))
            self.list.SetStringItem(nbfound, 1, torrent.name)
            self.list.SetStringItem(nbfound, 2, path.split(torrent.src)[1])
            nbfound += 1
            self.foundinfohash.append(torrent.infohash)
            if self.selectfound:
                # Select torrent in main ABC list
                self.mainlist.Select(row)

        self.parent.SetStatusText(" " + str(nbfound) + self.localize('torrentsfound'), 0)
        self.list.SetFocus()

    def onColLeftClick(self, event):
        col = event.m_col
        self.utility.sortcol = []
        if col == self.lastsortingcol:
            self.sortdir = 1 - self.sortdir
        else:
            self.sortdir = 0
        self.lastsortingcol = col

        if col:
            for index in xrange(self.list.GetItemCount()):
                self.utility.sortcol.append(self.list.GetItem(index, col).GetText().lower())
                self.list.SetItemData(index, index)
        else:
            for index in xrange(self.list.GetItemCount()):
                self.utility.sortcol.append(int(self.list.GetItem(index, col).GetText()))
                self.list.SetItemData(index, index)
        if self.sortdir == 0:
            self.list.SortItems(self.utility.sortList)
        else:
            self.list.SortItems(self.utility.sortListReverse)

        foundinfohash2 = self.foundinfohash
        self.foundinfohash = []
        for index in xrange(len(foundinfohash2)):
            self.foundinfohash.append(foundinfohash2[self.list.GetItemData(index)])

    def onReset(self, event = None):
        self.findtorrentname.Clear()
        self.findtorrentfilename.Clear()
        self.findlabel.Clear()
        self.findtrackername.Clear()
        self.findshortlabel.Clear()
        self.findinfohash.Clear()
        self.findstatus.SetSelection(0)
        self.findprio.SetSelection(0)
        self.findavailability.SetSelection(0)
        self.findpeers.Set3StateValue(2)
        self.findactivity.Set3StateValue(2)
        self.findprivate.Set3StateValue(2)
        self.list.DeleteAllItems()
        self.foundinfohash[:] = []
        self.parent.SetStatusText("", 0)
        self.findtorrentname.SetFocus()

    def onLabelTorrentName(self, event):
        self.findtorrentname.Clear()
        self.findtorrentname.SetFocus()

    def onLabelTorrentFileName(self, event):
        self.findtorrentfilename.Clear()
        self.findtorrentfilename.SetFocus()

    def onLabelLabel(self, event):
        self.findlabel.Clear()
        self.findlabel.SetFocus()

    def onLabelTrackerName(self, event):
        self.findtrackername.Clear()
        self.findtrackername.SetFocus()

    def onLabelShortLabel(self, event):
        self.findshortlabel.Clear()
        self.findshortlabel.SetFocus()

    def onLabelInfohash(self, event):
        self.findinfohash.Clear()
        self.findinfohash.SetFocus()

    def onLabelStatus(self, event):
        self.findstatus.SetSelection(0)
        self.findstatus.SetFocus()

    def onLabelPrio(self, event):
        self.findprio.SetSelection(0)
        self.findprio.SetFocus()

    def onLabelAvailability(self, event):
        self.findavailability.SetSelection(0)
        self.findavailability.SetFocus()

    def onLabelPeers(self, event):
        self.findpeers.Set3StateValue(2)
        self.findpeers.SetFocus()

    def onLabelActivity(self, event):
        self.findactivity.Set3StateValue(2)
        self.findactivity.SetFocus()

    def onLabelPrivate(self, event):
        self.findprivate.Set3StateValue(2)
        self.findprivate.SetFocus()

    def onClose(self, event):
        self.parent.OnExit()

    def onListRightClick(self, event):
        self.fromrightclick = False
        if wx.GetKeyState(wx.WXK_CONTROL) or wx.GetKeyState(wx.WXK_SHIFT) or wx.GetKeyState(wx.WXK_ALT):
            return
        if not hasattr(self, "popupfoundlist"):
            self.makePopupFoundList()
        self.PopupMenu(self.popupfoundlist, self.ScreenToClient(self.list.ClientToScreen((self.x, self.y))))

    def onLeftDown(self, event):
        self.x, self.y = event.GetPosition()
        self.selectedrow = self.list.HitTest((self.x, self.y))[0]
        event.Skip()

    def onRightDown(self, event):
        self.x, self.y = event.GetPosition()
        self.selectedrow = self.list.HitTest((self.x, self.y))[0]
        if self.selectedrow >= 0:
            self.fromrightclick = True
        event.Skip()

    def makePopupFoundList(self):
        self.popupfoundlist = wx.Menu()
        self.popupfoundlist.Append(900, self.localize('rtorrentlocate'))
        self.popupfoundlist.Append(901, self.localize('rcopy'))
        self.popupfoundlist.AppendSeparator()        
        self.popupfoundlist.Append(902, self.localize('rtorrentdetail'))
        self.popupfoundlist.Append(903, self.localize('radvdetail'))        
        self.Bind(wx.EVT_MENU, self.onLocateInMainList, id = 900)
        self.Bind(wx.EVT_MENU, self.onCopy, id = 901)
        self.Bind(wx.EVT_MENU, self.onFoundTorrentInfo, id = 902)
        self.Bind(wx.EVT_MENU, self.onFoundTorrentDetails, id = 903)

    def onCopy(self, event):
        clickedcol = self.utility.getListClickedColumn(self.list, self.x)
        if clickedcol == 1:
            colrange = [1]
        else:
            colrange = xrange(self.list.GetColumnCount())
        text = ''
        for index in self.getSelected():
            for col in colrange:
                text += self.list.GetItem(index, col).GetText() + '\t'
            text = text[:-1] + '\r\n'
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(text[:-1]))
            wx.TheClipboard.Close()

    def onFoundTorrentInfo(self, event = None):
        for index in self.getSelected():
            mainlistindex = self.findTorrentInMainList(index)
            if mainlistindex >= 0:
                self.parent.parent.window.onTorrentInfo(indexlist = [mainlistindex])

    def onFoundTorrentDetails(self, event = None):
        for index in self.getSelected():
            mainlistindex = self.findTorrentInMainList(index)
            if mainlistindex >= 0:
                self.parent.parent.window.onDetails(indexlist = [mainlistindex])

    def onLeftDClick(self, event):
        clickpoint = wx.Point(event.GetX(), event.GetY())
        clickedindex = self.list.HitTest(clickpoint)[0]
        if clickedindex >= 0:
            if self.abcparams['centerquickdetails'] == '0':
                position = clickpoint + self.parent.GetPosition()
                center = False
            else:
                position = self.list.ClientToScreen(clickpoint)
                center = True
            mainlistindex = self.findTorrentInMainList(clickedindex)
            if mainlistindex >= 0:
                self.parent.parent.window.onDetails(event, indexlist = [mainlistindex], position = position,
                                                    centeratpointer = center)

    def onMiddleDown(self, event):
        clickpoint = wx.Point(event.GetX(), event.GetY())
        clickedindex = self.list.HitTest(clickpoint)[0]
        if clickedindex >= 0:
            if self.abcparams['centerquickdetails'] == '0':
                position = clickpoint + self.parent.GetPosition()
                center = False
            else:
                position = self.list.ClientToScreen(clickpoint)
                center = True
            mainlistindex = self.findTorrentInMainList(clickedindex)
            if mainlistindex >= 0:
                self.parent.parent.window.onTorrentInfo(indexlist = [mainlistindex], position = position, centeratpointer = center)

    def findTorrentInMainList(self, index):
        # Search for torrent in main list at row "row" ; if not found there (maybe because the torrent was moved in the list),
        # search in list from top. Returns torrent and row.
        infohashtofind = self.foundinfohash[index]
        currentmainlistindex = int(self.list.GetItem(index, 0).GetText()) - 1
        if currentmainlistindex < len(self.proctab): 
            pointedtorrent = self.proctab[currentmainlistindex]
            if pointedtorrent.infohash == infohashtofind:
                if pointedtorrent.isbeingdeleted:
                    return -1
                # Update found list row
                self.list.SetStringItem(index, 1, pointedtorrent.name)
                self.list.SetStringItem(index, 2, path.split(pointedtorrent.src)[1])
                return currentmainlistindex
        mainlistindex = 0
        for torrent in self.proctab:
            if torrent.infohash == infohashtofind:
                if torrent.isbeingdeleted:
                    break
                # Update found list row
                self.list.SetStringItem(index, 0, str(mainlistindex + 1))
                self.list.SetStringItem(index, 1, torrent.name)
                self.list.SetStringItem(index, 2, path.split(torrent.src)[1])
                return mainlistindex
            mainlistindex += 1

        dlg = wx.MessageDialog(self, self.localize('errortorrentnolongerinlist'), self.localize('abcokcerror'), wx.ICON_ERROR)
        dlg.ShowModal()
        dlg.Destroy()
        return -1

    def getSelected(self):
        selected = []
        index = self.list.GetNextItem(-1, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        while index != -1:
            selected.append(index)
            index = self.list.GetNextItem(index, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        return selected

    def onLeftDragList(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 0
        else:
            self.listdraggingmode = 1

    def onRightDragList(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 1
        else:
            self.listdraggingmode = 0

    def endDraggingForSelecting(self, index):
        self.listdraggingmode = 0
        if self.selectedrow >= 0:
            if index < 0:
                index = self.list.GetItemCount() - 1
            if index > self.selectedrow:
                dir = 1
            else:
                dir = -1
            for i in xrange(self.selectedrow + dir, index + dir, dir):
                self.list.Select(i)

    def onRightUp(self, event):
        if self.abcparams['mousemode'] == '0':
            if self.listdraggingmode == 1:
                self.endDraggingForSelecting(self.list.HitTest((event.GetX(), event.GetY()))[0])
        else:
            self.listdraggingmode = 0

    def onLeftUp(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 0
        else:
            if self.listdraggingmode == 1:
                self.endDraggingForSelecting(self.list.HitTest((event.GetX(), event.GetY()))[0])

    def onLeaveList(self, event):
        # To handle dragging the mouse out of the list with the left or right button pressed
        if event.LeftIsDown() or event.RightIsDown():
            self.listdraggingmode = 0
        event.Skip()


##############################################################
# Class : ABCFindFrame
##############################################################
class ABCFindFrame(wx.Frame):
    def __init__(self, parent, ID, title, abcparams):
        wx.Frame.__init__(self, parent, ID, title,
                          style = wx.DEFAULT_FRAME_STYLE & ~ (wx.RESIZE_BORDER | wx.RESIZE_BOX | wx.MAXIMIZE_BOX))
        self.CreateStatusBar(1, style = 0)

        self.parent = parent
        self.SetAcceleratorTable(self.parent.accelerators)
        self.abcparams = abcparams
        self.utility = self.parent.utility
        if sys.platform == 'win32':
            self.icon = wx.Icon(path.join(self.utility.abcpath, 'icon_abc.ico'), wx.BITMAP_TYPE_ICO)
            self.SetIcon(self.icon)

        if self.abcparams['abcfinddlgx'] != '-1':
            self.x = int(self.abcparams['abcfinddlgx'])
            self.y = int(self.abcparams['abcfinddlgy'])
            self.MoveXY(self.x, self.y)
        else:
            self.x, self.y = self.GetPositionTuple()

        self.window = ABCFindPanel(self)
        self.SetClientSize(self.window.GetSize())
        wx.EVT_CLOSE(self, self.OnExit)
        self.Bind(wx.EVT_MOVE, self.onMove)

    def onMove(self, event):
        if not self.IsIconized():
            self.x, self.y = self.GetPositionTuple()
        event.Skip()

    def OnExit(self, event = None):
        self.abcparams['findtorrentnamebtn'] = str(int(self.window.findtorrentnamebtn.GetValue()))
        self.abcparams['findtorrentfilenamebtn'] = str(int(self.window.findtorrentfilenamebtn.GetValue()))
        self.abcparams['findlabelbtn'] = str(int(self.window.findlabelbtn.GetValue()))
        self.abcparams['findtrackernamebtn'] = str(int(self.window.findtrackernamebtn.GetValue()))
        self.abcparams['findshortlabelbtn'] = str(int(self.window.findshortlabelbtn.GetValue()))
        self.abcparams['findtorrentnamesim'] = str(self.window.findtorrentnamesim.getValue())
        self.abcparams['findtorrentfilenamesim'] = str(self.window.findtorrentfilenamesim.getValue())

        self.abcparams['abcfinddlgx'], self.abcparams['abcfinddlgy'] = str(self.x), str(self.y)
        self.parent.abcfind = None
        self.Destroy()
