##############################################################################
# Module : commandschedulerdlg.py
# Author : Old King Cole
# Date   : 05/20/2006
#
# Description : Dialog for command scheduler
#
##############################################################################
import wx
import wx.lib.masked as masked
from os import path, remove, rename
from time import time
from filemanager import TorrentConfigFileManager, CommandListFileManager
from commandscheduler import ScheduledCommand


class CommandSchedulerDialog(wx.Dialog):
    def __init__(self, parent, ID, title, commandscheduler, pos = wx.DefaultPosition, size = wx.DefaultSize,
                 style = wx.DEFAULT_DIALOG_STYLE):
        pre = wx.PreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        self.this = pre.this
        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.commandscheduler = commandscheduler
        self.schedcommands = commandscheduler.schedcommands

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)

        schedulerbox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.VERTICAL)

        scheduledcombox = wx.StaticBoxSizer(wx.StaticBox(p, -1, self.localize('schedcom')), wx.VERTICAL)

        self.commandlist = wx.ListCtrl(p, -1, size = (739, 200),
                                       style = wx.LC_REPORT | wx.LC_HRULES | wx.LC_VRULES)
        self.commandlist.InsertColumn(0, self.localize('comlistact'))
        self.commandlist.SetColumnWidth(0, 50)
        self.commandlist.InsertColumn(1, self.localize('comlistfreq'))
        self.commandlist.SetColumnWidth(1, 150)
        self.commandlist.InsertColumn(2, self.localize('comlistdate'))
        self.commandlist.SetColumnWidth(2, 70)
        self.commandlist.InsertColumn(3, self.localize('comlisttime'))
        self.commandlist.SetColumnWidth(3, 60)
        self.commandlist.InsertColumn(4, self.localize('comlistend'))
        self.commandlist.SetColumnWidth(4, 70)
        self.commandlist.InsertColumn(5, self.localize('comlistcom'))
        self.commandlist.SetColumnWidth(5, 250)
        self.commandlist.InsertColumn(6, self.localize('comlistval'))
        self.commandlist.SetColumnWidth(6, 75)
        self.commandlist.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.onRightClickList)
        self.commandlist.Bind(wx.EVT_RIGHT_DOWN, self.onRightDown)
        self.commandlist.Bind(wx.EVT_KEY_DOWN, self.onKeyInList)

        scheduledcombox.Add(self.commandlist, 0, wx.ALIGN_CENTER)

        schedulerbox.Add(scheduledcombox)

        bottombox = wx.BoxSizer(wx.HORIZONTAL)

        self.schedautostart = wx.CheckBox(p, -1, self.localize('schedautostart'))
        if self.utility.schedparams['schedautostart'] == "1":
            self.schedautostart.SetValue(True)
        bottombox.Add(self.schedautostart, 0, wx.ALIGN_CENTER_VERTICAL)

        self.schedrunpast = wx.CheckBox(p, -1, self.localize('schedrunpast'))
        if self.utility.schedparams['schedrunpast'] == "1":
            self.schedrunpast.SetValue(True)
        bottombox.Add(self.schedrunpast, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 40)

        bottombox.Add((0, 0), 1)

        deletebtn = wx.Button(p, -1, self.localize('scheddelete'))
        addbtn = wx.Button(p, -1, self.localize('schedadd'))
        self.Bind(wx.EVT_BUTTON, self.onAddSchedCommand, addbtn)
        self.Bind(wx.EVT_BUTTON, self.onDeleteSchedCommand, deletebtn)

        bottombox.Add(deletebtn, 0, wx.RIGHT, 20)
        bottombox.Add(addbtn, 0, wx.RIGHT)

        schedulerbox.Add(bottombox, 0, wx.EXPAND | wx.TOP, 3)

        outerbox.Add(schedulerbox, 0, wx.TOP, -4)

        if self.parent.window.commandscheduler.status:            
            self.actionbtn = wx.Button(p, -1, self.localize('stopscheduler'))
        else:
            self.actionbtn = wx.Button(p, -1, self.localize('startscheduler'))
        okbtn = wx.Button(p, -1, self.localize('ok'))
        cancelbtn = wx.Button(p, -1, self.localize('cancel'))

        self.Bind(wx.EVT_BUTTON, self.onAction, self.actionbtn)
        self.Bind(wx.EVT_BUTTON, self.onOK, okbtn)
        self.Bind(wx.EVT_BUTTON, self.onCancel, cancelbtn)

        buttonbox2 = wx.BoxSizer(wx.HORIZONTAL)
        buttonbox2.Add(self.actionbtn, 0, wx.ALL, 5)
        buttonbox2.Add(okbtn, 0, wx.ALL, 5)
        buttonbox2.Add(cancelbtn, 0, wx.ALL, 5)

        outerbox.Add(buttonbox2, 0, wx.ALIGN_CENTER)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

        if self.parent.abcparams['schedulerdlgx'] != '-1':
            self.MoveXY(int(self.parent.abcparams['schedulerdlgx']), int(self.parent.abcparams['schedulerdlgy']))

        index = 0
        for sc in self.schedcommands:
            dt = wx.DateTimeFromTimeT(sc.schedtime)
            if sc.enddate:
                dtend = wx.DateTimeFromTimeT(sc.enddate).FormatDate()
            else:
                dtend = self.localize('endnone')
            self.commandlist.InsertStringItem(index, '')
            self.commandlist.SetStringItem(index, 0, self.localize(sc.activity))
            if sc.frequency == 'freweeklymulti':
                strweeksched = self.localize('freweeklymulti') + '['
                for i in xrange(7):
                    if sc.weekschedule[i] == '0':
                        strweeksched += '.'
                    else:
                        strweeksched += self.localize('day_s' + str(i))
                strweeksched += ']' + self.localize('freweeklymulti2')
                self.commandlist.SetStringItem(index, 1, strweeksched)
            else:
                self.commandlist.SetStringItem(index, 1, self.localize(sc.frequency))
            self.commandlist.SetStringItem(index, 2, dt.FormatDate())
            self.commandlist.SetStringItem(index, 3, dt.FormatTime())
            self.commandlist.SetStringItem(index, 4, dtend)
            self.commandlist.SetStringItem(index, 5, self.localize(sc.command))
            if self.commandscheduler.val[sc.command] == ['valnumber']:
                self.commandlist.SetStringItem(index, 6, sc.value)
            else:
                self.commandlist.SetStringItem(index, 6, self.localize(sc.value))
            index += 1

    def getSelected(self):
        selected = []
        index = self.commandlist.GetNextItem(-1, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        while index != -1:
            selected.append(index)
            index = self.commandlist.GetNextItem(index, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        return selected

    def onRightDown(self, event):
        self.x, self.y = event.GetPosition()
        self.selectedrow = self.commandlist.HitTest(wx.Point(self.x, self.y))[0]
        event.Skip()

    def onRightClickList(self, event):
        if wx.GetKeyState(wx.WXK_CONTROL) or wx.GetKeyState(wx.WXK_SHIFT) or wx.GetKeyState(wx.WXK_ALT):
            return
        col = self.utility.getListClickedColumn(self.commandlist, self.x)
        popupxy = self.commandlist.ClientToScreen((self.x, self.y))
        if col == 0:
            popupmenu = wx.Menu()
            popuplength = len(self.commandscheduler.act)
            for i in xrange(popuplength):
                popupmenu.Append(957 + i, self.localize(self.commandscheduler.act[i]),
                                 self.localize(self.commandscheduler.act[i]), wx.ITEM_RADIO)
            self.Bind(wx.EVT_MENU, self.onChangeActivity, id = 957, id2 = 956 + popuplength)
            popupmenu.Check(self.commandscheduler.act.index(self.schedcommands[self.selectedrow].activity) + 957, True)
            self.PopupMenu(popupmenu, self.ScreenToClient(popupxy))
            popupmenu.Destroy()
        elif col == 1:
            changefreqdlg = ChangeFreqDialog(self, -1, self.schedcommands[self.selectedrow].frequency,
                                             self.schedcommands[self.selectedrow].weekschedule, popupxy)
            retcode = changefreqdlg.ShowModal()
            retfreq = changefreqdlg.freq
            retweeksched = changefreqdlg.weeksched
            changefreqdlg.Destroy()
            if retcode == wx.ID_OK:
                self.onChangeFreq(retfreq, retweeksched)
        elif col == 2:
            changedatedlg = ChangeDateDialog(self, -1, wx.DateTimeFromTimeT(self.schedcommands[self.selectedrow].schedtime),
                                             popupxy)
            retcode = changedatedlg.ShowModal()
            retval = changedatedlg.datetime
            changedatedlg.Destroy()
            if retcode == wx.ID_OK:
                self.onChangeDate(retval)
        elif col == 3:
            changetimedlg = ChangeTimeDialog(self, -1, wx.DateTimeFromTimeT(self.schedcommands[self.selectedrow].schedtime),
                                             popupxy)
            retcode = changetimedlg.ShowModal()
            retval = changetimedlg.datetime
            changetimedlg.Destroy()
            if retcode == wx.ID_OK:
                self.onChangeTime(retval)
        elif col == 4:
            if self.schedcommands[self.selectedrow].frequency == 'freonetime':
                return
            edt = self.schedcommands[self.selectedrow].enddate
            if edt:
                edt = wx.DateTimeFromTimeT(edt)
            else:
                edt = None
            changeenddatedlg = ChangeEndDateDialog(self, -1, edt, popupxy)
            retcode = changeenddatedlg.ShowModal()
            retval = changeenddatedlg.datetime
            changeenddatedlg.Destroy()
            if retcode == wx.ID_OK:
                self.onChangeEndDate(retval)
        elif col == 5:
            popupmenu = wx.Menu()
            popuplength = len(self.commandscheduler.com)
            for i in xrange(popuplength):
                popupmenu.Append(960 + i, self.localize(self.commandscheduler.com[i]),
                                 self.localize(self.commandscheduler.com[i]), wx.ITEM_RADIO)
            self.Bind(wx.EVT_MENU, self.onChangeCommand, id = 960, id2 = 959 + popuplength)
            popupmenu.Check(self.commandscheduler.com.index(self.schedcommands[self.selectedrow].command) + 960, True)
            self.PopupMenu(popupmenu, self.ScreenToClient(popupxy))
            popupmenu.Destroy()
        elif col == 6:
            selected = self.getSelected()
            firstcommand = self.schedcommands[selected[0]].command
            for index in selected[1:]:
                if self.schedcommands[index].command != firstcommand:
                    return
            values = self.commandscheduler.val[self.schedcommands[self.selectedrow].command]
            if values == ['valnumber']:
                changevaluedlg = ChangeValueDialog(self, -1, int(self.schedcommands[self.selectedrow].value),
                                                   popupxy)
                retcode = changevaluedlg.ShowModal()
                retval = changevaluedlg.value
                changevaluedlg.Destroy()
                if retcode == wx.ID_OK:
                    self.onChangeValueNumber(retval)
            elif values != ['valnone']:
                popupmenu = wx.Menu()
                popuplength = len(values)
                for i in xrange(popuplength):
                    popupmenu.Append(980 + i, self.localize(values[i]), self.localize(values[i]), wx.ITEM_RADIO)
                self.Bind(wx.EVT_MENU, self.onChangeValue, id = 980, id2 = 979 + popuplength)
                popupmenu.Check(values.index(self.schedcommands[self.selectedrow].value) + 980, True)
                self.PopupMenu(popupmenu, self.ScreenToClient(popupxy))
                popupmenu.Destroy()
        else:
            return

    def onChangeActivity(self, event):
        now = time()
        change = False
        newactivity = self.commandscheduler.act[event.GetId() - 957]
        for index in self.getSelected():
            if self.schedcommands[index].activity != newactivity:
                self.schedcommands[index].activity = newactivity
                self.commandlist.SetStringItem(index, 0, self.localize(newactivity))
                change = True
        if change and self.commandscheduler.status:
            self.commandscheduler.scheduleNext(now)

    def onChangeFreq(self, newfreq, newweeksched):
        now = time()
        change = False
        for index in self.getSelected():
            if self.schedcommands[index].frequency != newfreq:
                self.schedcommands[index].frequency = newfreq
                if self.schedcommands[index].weekschedule != newweeksched:
                    self.schedcommands[index].weekschedule = newweeksched
                if newfreq == 'freonetime':
                    # Reset end date to None
                    self.schedcommands[index].enddate = 0
                    self.commandlist.SetStringItem(index, 4, self.localize('endnone'))
                    strweeksched = self.localize(newfreq) 
                elif newfreq == 'fredaily':
                    strweeksched = self.localize(newfreq) 
                elif newfreq == 'freweekly':
                    strweeksched = self.localize('freweekly')
                else:
                    strweeksched = self.localize('freweeklymulti') + '['
                    for i in xrange(7):
                        if newweeksched[i] == '0':
                            strweeksched += '.'
                        else:
                            strweeksched += self.localize('day_s' + str(i))
                    strweeksched += ']' + self.localize('freweeklymulti2')
                change = True
            elif self.schedcommands[index].weekschedule != newweeksched:
                self.schedcommands[index].weekschedule = newweeksched
                if newfreq == 'freweekly':
                    change = True
                    strweeksched = self.localize('freweekly')
                elif newfreq == 'freweeklymulti':
                    change = True
                    strweeksched = self.localize('freweeklymulti') + '['
                    for i in xrange(7):
                        if newweeksched[i] == '0':
                            strweeksched += '.'
                        else:
                            strweeksched += self.localize('day_s' + str(i))
                    strweeksched += ']' + self.localize('freweeklymulti2')
            if change:
                self.commandlist.SetStringItem(index, 1, strweeksched)
                if self.commandscheduler.status:
                    self.schedcommands[index].computeNextRunTime(now)
        if change and self.commandscheduler.status:
                self.commandscheduler.scheduleNext(now)

    def onChangeDate(self, newdate):
        now = time()
        change = False
        for index in self.getSelected():
            currentdate = wx.DateTimeFromTimeT(self.schedcommands[index].schedtime)
            if not newdate.IsSameDate(currentdate):
                currentdate = newdate
                self.schedcommands[index].schedtime = currentdate.GetTicks()
                self.commandlist.SetStringItem(index, 2, currentdate.FormatDate())
                if self.commandscheduler.status:
                    self.schedcommands[index].computeNextRunTime(now)
                change = True
        if change:
            self.sortCommands()
            if self.commandscheduler.status:
                self.commandscheduler.scheduleNext(now)

    def onChangeTime(self, newtime):
        now = time()
        change = False
        for index in self.getSelected():
            currenttime = wx.DateTimeFromTimeT(self.schedcommands[index].schedtime)
            if not newtime.IsSameTime(currenttime):
                # If time has really changed
                currenttime.SetHour(newtime.GetHour())
                currenttime.SetMinute(newtime.GetMinute())
                currenttime.SetSecond(newtime.GetSecond())
                self.schedcommands[index].schedtime = currenttime.GetTicks()
                self.commandlist.SetStringItem(index, 3, currenttime.FormatTime())
                if self.commandscheduler.status:
                    self.schedcommands[index].computeNextRunTime(now)
                change = True
        if change:
            self.sortCommands()
            if self.commandscheduler.status:
                self.commandscheduler.scheduleNext(now)

    def onChangeEndDate(self, newdate):
        now = time()
        change = False
        if newdate == None:
            newdate = 0
        else:
            newdate = newdate.GetTicks()
        for index in self.getSelected():
            if self.schedcommands[index].frequency == 'freonetime':
                continue
            if newdate != self.schedcommands[index].enddate:
                # If date has really changed
                self.schedcommands[index].enddate = newdate
                if newdate:
                    self.commandlist.SetStringItem(index, 4, wx.DateTimeFromTimeT(newdate).FormatDate())
                else:
                    self.commandlist.SetStringItem(index, 4, self.localize('endnone'))
                self.schedcommands[index].computeNextRunTime(now)
                change = True
        if change:
            self.commandscheduler.scheduleNext(now)

    def sortCommands(self):
        self.utility.sortcol = []
        nbitem = len(self.schedcommands)
        for index in xrange(nbitem):
            self.utility.sortcol.append(self.schedcommands[index].schedtime)
            self.commandlist.SetItemData(index, index)
        self.commandlist.SortItems(self.utility.sortList)
        schedcommands2 = self.schedcommands[:]
        self.schedcommands[:] = []
        for index in xrange(nbitem):
            self.schedcommands.append(schedcommands2[self.commandlist.GetItemData(index)])

    def onChangeCommand(self, event):
        now = time()
        rescheduleneeded = False
        newcom = self.commandscheduler.com[event.GetId() - 960]
        for index in self.getSelected():
            if newcom != self.schedcommands[index].command:
                # If command has really changed
                if newcom == 'comnone' or self.schedcommands[index].command == 'comnone':
                    rescheduleneeded = True
                self.schedcommands[index].command = newcom
                self.commandlist.SetStringItem(index, 5, self.localize(newcom))
                values = self.commandscheduler.val[self.schedcommands[index].command]
                if values == ['valnumber']:
                    self.schedcommands[index].value = '0'
                    self.commandlist.SetStringItem(index, 6, '0')
                else:
                    self.schedcommands[index].value = values[0]
                    self.commandlist.SetStringItem(index, 6, self.localize(values[0]))
        if rescheduleneeded and self.commandscheduler.status:
            self.commandscheduler.scheduleNext(now)

    def onChangeValue(self, event):
        newval = self.commandscheduler.val[self.schedcommands[self.selectedrow].command][event.GetId() - 980]
        for index in self.getSelected():
            self.schedcommands[index].value = newval
            self.commandlist.SetStringItem(index, 6, self.localize(newval))

    def onChangeValueNumber(self, value):
        for index in self.getSelected():
            strvalue = str(value)
            self.schedcommands[index].value = strvalue
            self.commandlist.SetStringItem(index, 6, strvalue)

    def onAddSchedCommand(self, event):
        now = int(time())
        newcommand = ScheduledCommand(['actenabled', 'freonetime', '0000000', now, 0, 'comnone', 'valnone'])
        index = 0
        while index < len(self.schedcommands) and self.schedcommands[index].schedtime <= now:
            index += 1
        self.schedcommands.insert(index, newcommand)
        dt = wx.DateTimeFromTimeT(now)
        self.commandlist.InsertStringItem(index, '')
        self.commandlist.SetStringItem(index, 0, self.localize('actenabled'))
        self.commandlist.SetStringItem(index, 1, self.localize('freonetime'))
        self.commandlist.SetStringItem(index, 2, dt.FormatDate())
        self.commandlist.SetStringItem(index, 3, dt.FormatTime())
        self.commandlist.SetStringItem(index, 4, self.localize('endnone'))
        self.commandlist.SetStringItem(index, 5, self.localize('comnone'))
        self.commandlist.SetStringItem(index, 6, self.localize('valnone'))
        # Unselect all
        for ind in self.getSelected():
            self.commandlist.SetItemState(ind, 0, wx.LIST_STATE_SELECTED)
        # Select new command
        self.commandlist.Select(index)
        self.commandlist.Focus(index)
        self.commandlist.SetFocus()

    def onDeleteSchedCommand(self, event):
        now = time()
        tobedeleted = self.getSelected()
        tobedeleted.reverse()
        for index in tobedeleted:
            del self.schedcommands[index]
            self.commandlist.DeleteItem(index)
        self.commandlist.SetFocus()
        if self.commandscheduler.status:
            self.commandscheduler.scheduleNext(now)

    def saveCommandList(self):
        comfileman = CommandListFileManager(self)
        comfileman.open(mode = 'w+')
        try:
            comfileman.writeList(self.schedcommands)
        except:
            comfileman.close()
        else:
            comfileman.close()
            try:
                remove(comfileman.filename)
            except:
                pass
            rename(comfileman.filename + '.part', comfileman.filename)

    def onCancel(self, event = None):
        self.saveCommandList()
        self.EndModal(wx.ID_CANCEL)

    def onOK(self, event):
        self.inputData()
        # Save conf
        self.writeCommandSchedulerConf()
        self.saveCommandList()
        self.EndModal(wx.ID_OK)

    def onAction(self, event):
        if self.parent.window.commandscheduler.status:
            try:
                self.actionbtn.SetLabel(self.localize('startscheduler'))
            except:
                pass
            self.parent.window.schedulerbutton.SetValue(False)
            self.parent.window.commandscheduler.stop()
        else:
            if self.parent.window.torrentlistloading:
                return
            self.inputData()
            self.actionbtn.SetLabel(self.localize('stopscheduler'))
            self.parent.window.schedulerbutton.SetValue(True)
            self.writeCommandSchedulerConf()
            self.parent.window.commandscheduler.start()

    def forceStatus(self, status):
        if status:
            self.actionbtn.SetLabel(self.localize('stopscheduler'))
        else:
            self.actionbtn.SetLabel(self.localize('startscheduler'))

    def writeCommandSchedulerConf(self):
        # Record new config to scheduler.conf
        tcfm = TorrentConfigFileManager(path.join(self.utility.datapath, "scheduler.conf"), self.utility.schedparams)
        tcfm.writeAllConfig()

    def inputData(self):
        # Update schedparams
        if self.schedrunpast.GetValue():
            self.utility.schedparams['schedrunpast'] = "1"
        else:
            self.utility.schedparams['schedrunpast'] = "0"
        if self.schedautostart.GetValue():
            self.utility.schedparams['schedautostart'] = "1"
        else:
            self.utility.schedparams['schedautostart'] = "0"

    def onKeyInList(self, event):
        if event.ControlDown():
            keycode = event.GetKeyCode()
            if keycode == ord('a') or keycode == ord('A'):
                # Select all commands
                for index in xrange(self.commandlist.GetItemCount()):
                    self.commandlist.Select(index)
            elif keycode == self.utility.selinvert1 or keycode == self.utility.selinvert2:
                # Invert command selection
                for index in xrange(self.commandlist.GetItemCount()):
                    self.commandlist.SetItemState(index, 4 - self.commandlist.GetItemState(index, wx.LIST_STATE_SELECTED), wx.LIST_STATE_SELECTED)
            elif keycode == ord('z') or keycode == ord('Z'):
                # Unselect all commands
                for index in xrange(self.commandlist.GetItemCount()):
                    self.commandlist.Select(index, 0)
            else:
                event.Skip()
        else:
            event.Skip()


class ChangeFreqDialog(wx.Dialog):
    def __init__(self, parent, ID, freq, weeksched, pos):
        pre = wx.PreDialog()
        pre.Create(parent, ID, '', pos, (-1, -1), wx.DEFAULT_DIALOG_STYLE & ~ wx.CAPTION)
        self.this = pre.this
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.freq = freq
        self.weeksched = weeksched

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        vertsizer = wx.BoxSizer(wx.VERTICAL)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        self.okbtn = wx.Button(p, -1, self.localize('ok'), style = wx.BU_EXACTFIT)
        smallfont = self.okbtn.GetFont()
        smallfont.SetPointSize(7)
        self.okbtn.SetFont(smallfont)
        self.okbtn.SetDefault()
        self.cancelbtn = wx.Button(p, -1, self.localize('cancel'), style = wx.BU_EXACTFIT)
        self.cancelbtn.SetFont(smallfont)
        self.Bind(wx.EVT_BUTTON, self.onOK, self.okbtn)
        self.SetEscapeId(self.cancelbtn.GetId())
        buttonbox.Add(self.okbtn)
        buttonbox.Add(self.cancelbtn, 0, wx.LEFT, 5)

        changefreqbox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.VERTICAL)

        rbonetime = wx.RadioButton(p, -1, self.localize('freonetime_p'), style = wx.RB_GROUP)
        self.rbdaily = wx.RadioButton(p, -1, self.localize('fredaily_p'))
        self.rbweekly = wx.RadioButton(p, -1, self.localize('freweekly_p'))
        self.rbweeklymulti = wx.RadioButton(p, -1, self.localize('freweeklymulti_p'))
        if self.freq == 'freonetime':
            rbonetime.SetValue(True)
        elif self.freq == 'fredaily':
            self.rbdaily.SetValue(True)
        elif self.freq == 'freweekly':
            self.rbweekly.SetValue(True)
        elif self.freq == 'freweeklymulti':
            self.rbweeklymulti.SetValue(True)

        changefreqbox.Add(rbonetime, 0, wx.TOP, 2)
        changefreqbox.Add(self.rbdaily, 0, wx.TOP, 3)
        changefreqbox.Add(self.rbweekly, 0, wx.TOP, 3)
        changefreqbox.Add(self.rbweeklymulti, 0, wx.TOP, 3)

        daybtnsizer = wx.BoxSizer(wx.HORIZONTAL)
        smallbtnheight = self.okbtn.GetSizeTuple()[1]
        self.daybtns = []
        for i in xrange(7):
            self.daybtns.append(wx.ToggleButton(p, -1, self.localize('day_s' + str(i)),
                                                size = (smallbtnheight, smallbtnheight), style = wx.BU_EXACTFIT))
            self.daybtns[i].SetValue(self.weeksched[i] == '1')
            self.daybtns[i].SetFont(smallfont)
            daybtnsizer.Add(self.daybtns[i])

        changefreqbox.Add(daybtnsizer, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP, 3)

        vertsizer.Add(changefreqbox)

        vertsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP | wx.BOTTOM, 5)

        outerbox.Add(vertsizer, 0, wx.LEFT | wx.RIGHT, 5)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onOK(self, event):
        if self.rbdaily.GetValue():
            self.freq = 'fredaily'
        elif self.rbweekly.GetValue():
            self.freq = 'freweekly'
        elif self.rbweeklymulti.GetValue():
            self.freq = 'freweeklymulti'
        else:
            self.freq = 'freonetime'
        self.weeksched = ''
        for i in xrange(7):
            self.weeksched += str(int(self.daybtns[i].GetValue()))
        self.EndModal(wx.ID_OK)


class ChangeDateDialog(wx.Dialog):
    def __init__(self, parent, ID, datetime, pos):
        pre = wx.PreDialog()
        pre.Create(parent, ID, '', pos, (-1, -1), wx.DEFAULT_DIALOG_STYLE & ~ wx.CAPTION)
        self.this = pre.this
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.datetime = datetime

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        vertsizer = wx.BoxSizer(wx.VERTICAL)

        changedatebox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.HORIZONTAL)

        self.dp = wx.DatePickerCtrl(p, size = (80, -1), dt = datetime, style = wx.DP_DROPDOWN | wx.DP_SHOWCENTURY)

        changedatebox.Add(self.dp, 0, wx.TOP, 2)

        vertsizer.Add(changedatebox)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        self.okbtn = wx.Button(p, -1, self.localize('ok'), style = wx.BU_EXACTFIT)
        deffont = self.okbtn.GetFont()
        deffont.SetPointSize(7)
        self.okbtn.SetFont(deffont)
        self.okbtn.SetDefault()
        self.cancelbtn = wx.Button(p, -1, self.localize('cancel'), style = wx.BU_EXACTFIT)
        self.cancelbtn.SetFont(deffont)
        self.Bind(wx.EVT_BUTTON, self.onOK, self.okbtn)
        self.SetEscapeId(self.cancelbtn.GetId())
        buttonbox.Add(self.okbtn)
        buttonbox.Add(self.cancelbtn, 0, wx.LEFT, 5)

        vertsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP | wx.BOTTOM, 5)

        outerbox.Add(vertsizer, 0, wx.LEFT | wx.RIGHT, 5)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onOK(self, event):
        self.datetime = self.dp.GetValue()
        try:
            ticks = self.datetime.GetTicks()
        except:
            self.datetime.SetTimeT(0)
        self.EndModal(wx.ID_OK)


class ChangeTimeDialog(wx.Dialog):
    def __init__(self, parent, ID, datetime, pos):
        pre = wx.PreDialog()
        pre.Create(parent, ID, '', pos, (-1, -1), wx.DEFAULT_DIALOG_STYLE & ~ wx.CAPTION)
        self.this = pre.this
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.datetime = datetime

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        vertsizer = wx.BoxSizer(wx.VERTICAL)

        changetimebox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.HORIZONTAL)

        self.tc = masked.TimeCtrl(p, -1, fmt24hr = True)
        self.tc.SetFont(parent.commandlist.GetFont())

        h = self.tc.GetSize().height
        spin = wx.SpinButton(p, -1, (-1, -1), (h * 2 / 3, h), wx.SP_VERTICAL)
        self.tc.BindSpinButton(spin)
        self.tc.SetValue(datetime)

        changetimebox.Add(self.tc, 0, wx.TOP, 2)
        changetimebox.Add(spin, 0, wx.TOP | wx.ALIGN_CENTRE_VERTICAL, 2)

        vertsizer.Add(changetimebox)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        self.okbtn = wx.Button(p, -1, self.localize('ok'), style = wx.BU_EXACTFIT)
        deffont = self.okbtn.GetFont()
        deffont.SetPointSize(7)
        self.okbtn.SetFont(deffont)
        self.okbtn.SetDefault()
        self.cancelbtn = wx.Button(p, -1, self.localize('cancel'), style = wx.BU_EXACTFIT)
        self.cancelbtn.SetFont(deffont)
        self.Bind(wx.EVT_BUTTON, self.onOK, self.okbtn)
        self.SetEscapeId(self.cancelbtn.GetId())
        buttonbox.Add(self.okbtn)
        buttonbox.Add(self.cancelbtn, 0, wx.LEFT, 5)

        vertsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP | wx.BOTTOM, 5)

        outerbox.Add(vertsizer, 0, wx.LEFT | wx.RIGHT, 5)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onOK(self, event):
        self.datetime = self.tc.GetWxDateTime()
        self.EndModal(wx.ID_OK)


class ChangeEndDateDialog(wx.Dialog):
    def __init__(self, parent, ID, datetime, pos):
        pre = wx.PreDialog()
        pre.Create(parent, ID, '', pos, (-1, -1), wx.DEFAULT_DIALOG_STYLE & ~ wx.CAPTION)
        self.this = pre.this
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.datetime = datetime

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        vertsizer = wx.BoxSizer(wx.VERTICAL)

        changedatebox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.VERTICAL)

        rbnoenddate = wx.RadioButton(p, -1, self.localize('rbnoenddate'), style = wx.RB_GROUP)
        self.rbenddate = wx.RadioButton(p, -1, '')
        if self.datetime is None:
            rbnoenddate.SetValue(True)
        else:
            self.rbenddate.SetValue(True)

        changedatebox.Add(rbnoenddate, 0, wx.TOP, 2)

        enddatesizer = wx.BoxSizer(wx.HORIZONTAL)

        enddatesizer.Add(self.rbenddate, 0, wx.ALIGN_CENTER_VERTICAL)

        self.dp = wx.DatePickerCtrl(p, size = (80, -1), style = wx.DP_DROPDOWN | wx.DP_SHOWCENTURY)
        self.dp.Bind(wx.EVT_DATE_CHANGED, self.onChangeDate)
        if datetime is not None:
            self.dp.SetValue(datetime)

        enddatesizer.Add(self.dp, 0, wx.LEFT, 5)

        changedatebox.Add(enddatesizer, 0, wx.TOP, 3)

        vertsizer.Add(changedatebox)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        self.okbtn = wx.Button(p, -1, self.localize('ok'), style = wx.BU_EXACTFIT)
        deffont = self.okbtn.GetFont()
        deffont.SetPointSize(7)
        self.okbtn.SetFont(deffont)
        self.okbtn.SetDefault()
        self.cancelbtn = wx.Button(p, -1, self.localize('cancel'), style = wx.BU_EXACTFIT)
        self.cancelbtn.SetFont(deffont)
        self.Bind(wx.EVT_BUTTON, self.onOK, self.okbtn)
        self.SetEscapeId(self.cancelbtn.GetId())
        buttonbox.Add(self.okbtn)
        buttonbox.Add(self.cancelbtn, 0, wx.LEFT, 5)

        vertsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP | wx.BOTTOM, 5)

        outerbox.Add(vertsizer, 0, wx.LEFT | wx.RIGHT, 5)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onChangeDate(self, event):
        self.rbenddate.SetValue(True)

    def onOK(self, event):
        if self.rbenddate.GetValue():
            self.datetime = self.dp.GetValue()
            try:
                ticks = self.datetime.GetTicks()
            except:
                self.datetime.SetTimeT(0)
        else:
            self.datetime = None
        self.EndModal(wx.ID_OK)


class ChangeValueDialog(wx.Dialog):
    def __init__(self, parent, ID, value, pos):
        pre = wx.PreDialog()
        pre.Create(parent, ID, '', pos, (-1, -1), wx.DEFAULT_DIALOG_STYLE & ~ wx.CAPTION)
        self.this = pre.this
        self.utility = parent.utility
        self.localize = self.utility.lang.get
        self.value = value

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)
        vertsizer = wx.BoxSizer(wx.VERTICAL)

        changevaluebox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.HORIZONTAL)

        self.valuetext = wx.TextCtrl(p, -1, str(value), (-1, -1), (60, -1), wx.TE_RIGHT)
        self.valuetext.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.valuetext.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        h = self.valuetext.GetSize().height
        self.valuespin = wx.SpinButton(p, -1, (-1, -1), (h * 2 / 3, h), wx.SP_ARROW_KEYS | wx.SP_VERTICAL)
        self.valuespin.SetRange(0, 99999)
        self.valuespin.SetValue(value)
        self.Bind(wx.EVT_SPIN_UP, self.onValueSpinUp, self.valuespin)
        self.Bind(wx.EVT_SPIN_DOWN, self.onValueSpinDown, self.valuespin)
        valuebox = wx.BoxSizer(wx.HORIZONTAL)
        valuebox.Add(self.valuetext, 0, wx.ALIGN_CENTER_VERTICAL)
        valuebox.Add(self.valuespin, 0, wx.ALIGN_CENTER_VERTICAL)

        changevaluebox.Add(valuebox, 0, wx.TOP, 2)

        vertsizer.Add(changevaluebox)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        self.okbtn = wx.Button(p, -1, self.localize('ok'), style = wx.BU_EXACTFIT)
        deffont = self.okbtn.GetFont()
        deffont.SetPointSize(7)
        self.okbtn.SetFont(deffont)
        self.okbtn.SetDefault()
        self.cancelbtn = wx.Button(p, -1, self.localize('cancel'), style = wx.BU_EXACTFIT)
        self.cancelbtn.SetFont(deffont)
        self.Bind(wx.EVT_BUTTON, self.onOK, self.okbtn)
        self.SetEscapeId(self.cancelbtn.GetId())
        buttonbox.Add(self.okbtn)
        buttonbox.Add(self.cancelbtn, 0, wx.LEFT, 5)

        vertsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.TOP | wx.BOTTOM, 5)

        outerbox.Add(vertsizer, 0, wx.LEFT | wx.RIGHT, 5)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onRightButtonText(self, event):
        pass

    def onValueSpinUp(self, event):
        try:
            val = int(self.valuetext.GetValue())
        except:
            # Not integer
            self.valuespin.SetValue(0)
            self.valuetext.SetValue('0')
            return
        if val >= 99999:
            self.valuespin.SetValue(99999)
            self.valuetext.SetValue('99999')
            return
        if val < 0:
            self.valuespin.SetValue(0)
            self.valuetext.SetValue('0')
            return
        val += 1
        self.valuespin.SetValue(val)
        self.valuetext.SetValue(str(val))

    def onValueSpinDown(self, event):
        try:
            val = int(self.valuetext.GetValue())
        except:
            # Not integer
            self.valuespin.SetValue(0)
            self.valuetext.SetValue('0')
            return
        if val > 99999:
            self.valuespin.SetValue(99999)
            self.valuetext.SetValue('99999')
            return
        if val <= 0:
            self.valuespin.SetValue(0)
            self.valuetext.SetValue('0')
            return
        val -= 1
        self.valuespin.SetValue(val)
        self.valuetext.SetValue(str(val))

    def onOK(self, event):
        self.value = int(self.valuetext.GetValue())
        if self.value > 99999:
            self.value = 99999
        self.EndModal(wx.ID_OK)
