##############################################################################
# Module : buffer.py
# Author : Old King Cole
# Date   : 06/21/2011
#
# Description : Buffer pools
#
##############################################################################

from threading import Lock
from time import sleep, clock
from math import log


class Buffer:
    def __init__(self, pool, size):
        self.bufferpool = pool
        self.size = size
        self.buf = memoryview(bytearray(size))
        self.start = 0
        self.length = 0
        self.lastreleasedate = clock()

    def init(self, s = ''):
        self.start = 0
        self.length = 0
        if s:
            self.append(s)

    def shift(self, amount):
        if amount < self.length:
            self.start += amount
            self.length -= amount
        elif self.length:
            self.start = self.length - 1
            self.length = 1

    def append(self, s):
        l = self.start + self.length + len(s)
        self.buf[(self.start + self.length):l] = s
        self.length = l - self.start

    def __len__(self):
        return self.length

    def setLength(self, length):
        self.length = length

    def __getitem__(self, index):
        return self.buf[self.start + index]

    def __setitem__(self, index, val):
        self.buf[self.start + index] = val

    def viewSlice(self, a = 0, b = None):
        a += self.start
        if b is None or b > self.length:
            b = self.start + self.length
        elif b >= 0:
            b += self.start
        else:
            b += self.start + self.length
        return self.buf[a:b]

    def __getslice__(self, a, b):
        # Returns a string allowing easy external testing or converting of a small
        # part of the buffer
        # As this allocates new memory for this string it's not adequate for returning
        # a big amount of data from the buffer : in this case use viewSlice
        return self.viewSlice(a, b).tobytes()

    def __setslice__(self, a, b, s):
        self.buf[a:b] = s

    def release(self):
        self.bufferpool.release(self)


class CatBufferPool:
    def __init__(self, name, catmaxmem = 67108864, catmaxnumber = 10, expire = 900, catshift = 13):
        # Default catshift value (13) to align buffer sizes to block send message length
        self.name = name
        self.catmaxmem = catmaxmem
        self.catmaxnumber = catmaxnumber
        self.expire = expire
        self.catshift = catshift
        self.pool = {}
        self.catnumber = {}
        self.sem = Lock()
        self.overloaded = False

    def setCatMaxMem(self, catmaxmem):
        self.catmaxmem = catmaxmem

    def setCatMaxNumber(self, catmaxnumber):
        self.catmaxnumber = catmaxnumber

    def setExpire(self, expire):
        self.expire = expire

    def bufferSize(self, size, base):
        if size <= base:
            return base
        return 2 ** (int(log(size - self.catshift - 1, 2)) + 1) + self.catshift

    def catSize(self, bufsize):
        cn = self.catnumber.get(bufsize)
        if cn is None:
            return 0
        return cn * bufsize

    def isOverloaded(self, bufsize):
        self.sem.acquire()
        cat = self.pool.get(bufsize)
        if cat is None:
            cat = self.pool[bufsize] = []
            self.catnumber[bufsize] = 0
        if self.catmaxnumber:
            catmaxnumber = min(self.catmaxnumber, self.catmaxmem / bufsize)
        else:
            catmaxnumber = self.catmaxmem / bufsize
        overloaded = (self.catnumber[bufsize] - len(cat) + 1 > catmaxnumber)
        if overloaded:
            self.overloaded = True
        self.sem.release()
        return overloaded

    def new(self, data = '', size = 0, checkoverload = False, base = 32781):
        if data:
            size = max(size, len(data))
        bufsize = self.bufferSize(size, base)
        self.sem.acquire()
        cat = self.pool.get(bufsize)
        if cat is None:
            cat = self.pool[bufsize] = []
            self.catnumber[bufsize] = 0
        if checkoverload:
            if self.catmaxnumber:
                catmaxnumber = min(self.catmaxnumber, self.catmaxmem / bufsize)
            else:
                catmaxnumber = self.catmaxmem / bufsize
            if self.catnumber[bufsize] - len(cat) + 1 > catmaxnumber:
                self.overloaded = True
                self.sem.release()
                return None
        if cat:
            x = cat.pop()
        else:
            x = Buffer(self, bufsize)
            self.catnumber[bufsize] += 1
        self.sem.release()
        x.init(data)
        return x

    def release(self, x):
        self.sem.acquire()
        x.lastreleasedate = clock()
        self.pool[x.size].append(x)
        self.sem.release()

    def totalSize(self):
        totalsize = 0
        for bufsize in self.catnumber:
            totalsize += self.catnumber[bufsize] * bufsize
        return totalsize

    def report(self):
        rep = self.name + '\n'
        total = 0.
        self.sem.acquire()
        sizelist = self.pool.keys()
        sizelist.sort()
        for size in sizelist:
            catnb = self.catnumber[size]
            catsize = size * catnb
            total += catsize
            rep += '%.2f Kb : %i/%i ---> %.2f Mb\n' % (size / 1024.,  catnb - len(self.pool[size]), catnb, catsize / 1048576.)
        self.sem.release()
        total /= 1048576.
        rep += '---> %.2f Mb\n' % total
        return rep, total

    def cleanUp(self, size):
        freed = 0
        self.sem.acquire()
        now = clock()
        for bufsize in self.pool:
            if bufsize >= size:
                buflist = self.pool[bufsize]
                # Delete unused expired buffers
                i = 0
                for x in buflist:
                    if now - x.lastreleasedate <= self.expire:
                        break
                    i += 1
                del buflist[:i]
                freed += bufsize * i
                self.catnumber[bufsize] -= i
        self.sem.release()
        return freed


GENBUFFERPOOL = CatBufferPool('GEN', catshift = 13)

def newDataBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 525)

def newMsgPartBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 525)

def newMsgBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 29)

def newSendMetaDataBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 17000)

def newMetadataBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 525)

def newMetadataPartBuffer(data = '', size = 0):
    return GENBUFFERPOOL.new(data = data, size = size, base = 16397)

def newReceiveBuffer(readsize = 524288):
    return GENBUFFERPOOL.new(base = readsize + 13)


PIECEREADBUFFERPOOL = CatBufferPool('PIECEREAD', catshift = 0)

def newPieceReadBuffer(data = '', size = 0, checkoverload = False):
    return PIECEREADBUFFERPOOL.new(data = data, size = size, checkoverload = checkoverload, base = 16384)
