# Written by Bram Cohen
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

from binascii import b2a_hex
from socket import error as socketerror
from urllib import quote
from time import clock
try:
    from select import POLLIN
except ImportError:
    from BitTornado.selectpoll import POLLIN

from BitTornado.BTcrypto import Crypto
from BitTornado.buffer import newMsgPartBuffer, newMsgBuffer, newReceiveBuffer
from BitTornado.utility import getABCUtility

DEBUG = False

MAX_INCOMPLETE = 8

TCTRYMAX = [15, 10, 7, 5, 1]

protocol_name = 'BitTorrent protocol'


def toint(s):
    return long(b2a_hex(s), 16)

def tobinary16(i):
    return chr((i >> 8) & 0xFF) + chr(i & 0xFF)


class Connection:
    def __init__(self, encoder, connection, option_pattern, pl, encrypted = None, encrypter = None, peer = None):
        self.encoder = encoder
        self.config = encoder.config
        self.connection = connection
        self.connecter = encoder.connecter
        if peer is None:
            self.id = None
        else:
            self.id = peer[1]
        self.option_pattern = option_pattern
        self.locally_initiated = (self.id != None)
        self.pl = pl
        self.frompl = pl and not self.locally_initiated
        self.complete = False
        self.keepalive = lambda: None
        self.closed = False
        self.buffer = ''
        self.bufferlen = None
        self.log = None
        self.read = self._read
        self.write = self._write
        self.cryptmode = 0
        self.encrypter = None
        self.uses_dht = False
        self.uses_extended = False
        self.cryptoallowed = None
        self.remoteport = None
        self.retryconnection = self.config['retry_connection']

        if self.locally_initiated:
            self.peer = peer
            self.tc = self.encoder.currenttc
            self.encoder.incompletecounter += 1
            if encrypted:
                self.encrypted = True
                self.encrypter = Crypto(True)
                self.write((newMsgBuffer(self.encrypter.pubkey + self.encrypter.padding()), 0, None))
            else:
                self.encrypted = False
                self.write((newMsgBuffer(chr(len(protocol_name)) + protocol_name + 
                           self.option_pattern + self.encoder.download_id), 0, None))
            if self.tc == 4:
                autoclosetimeout = self.config['max_peer_connection_timeout']
            else:
                autoclosetimeout = self.config['min_peer_connection_timeout']
        else:
            self.encrypted = encrypted
            if pl and encrypter:
                 self.encrypter = encrypter
                 self._start_crypto()
            autoclosetimeout = self.config['max_peer_connection_timeout']

        self.next_len, self.next_func = 1 + len(protocol_name), self.read_header
        self.encoder.sched(self._auto_close, autoclosetimeout)

    def _log_start(self):   # only called with DEBUG = True
        self.log = open('peerlog.' + self.get_ip() + '.txt', 'a')
        self.log.write('connected - ')
        if self.locally_initiated:
            self.log.write('outgoing\n')
        else:
            self.log.write('incoming\n')
        self._logwritefunc = self.write
        self.write = self._log_write

    def _log_write(self, s):
        self.log.write('w:' + b2a_hex(s[:]) + '\n')
        self._logwritefunc(s)

    def get_ip(self, real = False):
        return self.connection.get_ip(real)

    def get_port(self, real = False):
        return self.connection.get_port(real)

    def get_remoteport(self):
        return self.remoteport

    def set_port(self, port):
        self.connection.set_port(port)

    def get_dns(self, real = False):
        return self.connection.get_dns(real)

    def get_dns_remote(self, real = False):
        return (self.connection.get_ip(real), self.remoteport)

    def get_id(self):
        return self.id

    def is_locally_initiated(self):
        return self.locally_initiated

    def is_encrypted(self):
        return bool(self.encrypted)

    def buffered(self):
        return self.connection.buffered()

    def initReading(self, buffer, uses_dht, uses_extended, cryptmode):
        self.buffer = buffer
        self.uses_dht = uses_dht
        self.uses_extended = uses_extended
        self.cryptmode = cryptmode
        self.next_len, self.next_func = 20, self.read_peer_id
        self.connecter.external_connection_made += 1
        self.write((newMsgBuffer(chr(len(protocol_name)) + protocol_name + 
                   self.option_pattern + self.encoder.download_id + self.encoder.my_id), 0, None))
        if buffer:
            self.read(newReceiveBuffer())

    def _read_header(self, s):
        if s == chr(len(protocol_name)) + protocol_name:
            return 8, self.read_options
        return None

    def read_header(self, s):
        if self._read_header(s):
            if self.encrypted or self.config['crypto_stealth']:
                return None
            return 8, self.read_options
        if self.locally_initiated and not self.encrypted:
            return None
        elif not self.config['crypto_allowed']:
            return None
        if not self.encrypted:
            self.encrypted = True
            self.encrypter = Crypto(self.locally_initiated)
        self._write_buffer(s)
        return self.encrypter.keylength, self.read_crypto_header

    ################## ENCRYPTION SUPPORT ######################

    def _start_crypto(self):
        self.encrypter.setrawaccess(self._read, self._write)
        self.write = self.encrypter.write
        self.read = self.encrypter.read
        if self.buffer:
            self.buffer = self.encrypter.decrypt(self.buffer)

    def _end_crypto(self):
        self.read = self._read
        self.write = self._write
        self.encrypter = None

    def read_crypto_header(self, s):
        self.encrypter.received_key(s)
        self.encrypter.set_skey(self.encoder.download_id)
        if self.locally_initiated:
            if self.config['crypto_only']:
                cryptmode = '\x00\x00\x00\x02'    # full stream encryption
            else:
                cryptmode = '\x00\x00\x00\x03'    # header or full stream
            padc = self.encrypter.padding()
            self.write((newMsgBuffer(
                       self.encrypter.block3a
                       + self.encrypter.block3b
                       + self.encrypter.encrypt(
                       ('\x00' * 8)               # VC
                       + cryptmode                # acceptable crypto modes
                       + tobinary16(len(padc))
                       + padc                     # PadC
                       + '\x00\x00')), 0, None))            # no initial payload data
            self._max_search = 520
            return 1, self.read_crypto_block4a
        self.write((newMsgBuffer(self.encrypter.pubkey + self.encrypter.padding()), 0, None))
        self._max_search = 520
        return 0, self.read_crypto_block3a

    def _search_for_pattern(self, s, pat):
        p = s.find(pat)
        if p < 0:
            if len(s) >= len(pat):
                self._max_search -= len(s) + 1 - len(pat)
            if self._max_search < 0:
                self.close()
                return False
            self._write_buffer(s[1 - len(pat):])
            return False
        self._write_buffer(s[p + len(pat):])
        return True

    ### INCOMING CONNECTION ###
    def read_crypto_block3a(self, s):
        if not self._search_for_pattern(s, self.encrypter.block3a):
            return -1, self.read_crypto_block3a     # wait for more data
        return 20, self.read_crypto_block3b

    def read_crypto_block3b(self, s):
        if s != self.encrypter.block3b:
            return None
        self.connecter.external_connection_made += 1
        self._start_crypto()
        return 14, self.read_crypto_block3c

    def read_crypto_block3c(self, s):
        if s[:8] != ('\x00' * 8):           # check VC
            return None
        try:
            self.cryptmode = toint(s[8:12]) % 4
        except:
            return None
        if self.cryptmode == 0:
            return None                     # no encryption selected
        if (self.cryptmode == 1            # only header encryption
            and self.config['crypto_only']):
            return None
        padlen = (ord(s[12]) << 8) + ord(s[13])
        if padlen > 512:
            return None
        return padlen + 2, self.read_crypto_pad3

    def read_crypto_pad3(self, s):
        s = s[-2:]
        ialen = (ord(s[0]) << 8) + ord(s[1])
        if ialen > 65535:
            return None
        if self.cryptmode == 1:
            cryptmode = '\x00\x00\x00\x01'    # header only encryption
        else:
            cryptmode = '\x00\x00\x00\x02'    # full stream encryption
        padd = self.encrypter.padding()
        self.write((newMsgBuffer(
                   ('\x00' * 8)            # VC
                   + cryptmode             # encryption mode
                   + tobinary16(len(padd))
                   + padd), 0, None))                 # PadD
        if ialen:
            return ialen, self.read_crypto_ia
        return self.read_crypto_block3done()

    def read_crypto_ia(self, s):
        if DEBUG:
            self._log_start()
            self.log.write('r:' + b2a_hex(s) + '(ia)\n')
            if self.buffer:
                self.log.write('r:' + b2a_hex(self.buffer) + '(buffer)\n')
        return self.read_crypto_block3done(s)

    def read_crypto_block3done(self, ia = ''):
        if DEBUG:
            if not self.log:
                self._log_start()
        if self.cryptmode == 1:     # only handshake encryption
            if not self.buffer:  # oops; check for exceptions to this
                return None
            self._end_crypto()
        if ia:
            self._write_buffer(ia)
        return 1 + len(protocol_name), self.read_encrypted_header

    ### OUTGOING CONNECTION ###
    def read_crypto_block4a(self, s):
        if not self._search_for_pattern(s, self.encrypter.VC_pattern()):
            return -1, self.read_crypto_block4a     # wait for more data
        self._start_crypto()
        return 6, self.read_crypto_block4b

    def read_crypto_block4b(self, s):
        try:
            self.cryptmode = toint(s[:4]) % 4
        except:
            return None
        if self.cryptmode == 1:             # only header encryption
            if self.config['crypto_only']:
                return None
        elif self.cryptmode != 2:
            return None                     # unknown encryption
        padlen = (ord(s[4]) << 8) + ord(s[5])
        if padlen > 512:
            return None
        if padlen:
            return padlen, self.read_crypto_pad4
        return self.read_crypto_block4done()

    def read_crypto_pad4(self, s):
        # discard data
        return self.read_crypto_block4done()

    def read_crypto_block4done(self):
        if DEBUG:
            self._log_start()
        if self.cryptmode == 1:     # only handshake encryption
            if not self.buffer:  # oops; check for exceptions to this
                return None
            self._end_crypto()
        self.write((newMsgBuffer(chr(len(protocol_name)) + protocol_name + 
                   self.option_pattern + self.encoder.download_id), 0, None))
        return 1 + len(protocol_name), self.read_encrypted_header

    ### START PROTOCOL OVER ENCRYPTED CONNECTION ###
    def read_encrypted_header(self, s):
        return self._read_header(s)

    ################################################

    def read_options(self, s):
        self.options = s
        if ord(self.options[7]) & 1:
            self.uses_dht = True
        if ord(self.options[5]) & 16:
            self.uses_extended = True
        return 20, self.read_download_id

    def read_download_id(self, s):
        if (s != self.encoder.download_id
            or not self.encoder.check_ip(ip = self.get_ip())):
            return None
        if not self.locally_initiated:
            if not self.encrypted:
                self.connecter.external_connection_made += 1
            self.write((newMsgBuffer(chr(len(protocol_name)) + protocol_name + 
                       self.option_pattern + self.encoder.download_id + self.encoder.my_id), 0, None))
        return 20, self.read_peer_id

    def read_peer_id(self, s):
        if not self.encrypted and self.config['crypto_only']:
            return None     # allows older trackers to ping,
                            # but won't proceed w/ connections
        if not self.id:
            self.id = s
        elif s != self.id:
            return None
        self.complete = self.encoder.got_id(self)
        if not self.complete:
            return None
        if self.locally_initiated:
            self.write((newMsgBuffer(self.encoder.my_id), 0, None))
            self.encoder.incompletecounter -= 1
        self._switch_to_read2()
        c = self.connecter.connection_made(self)
        if c is None:
            return None
        self.keepalive = c.send_keepalive
        return 4, self.read_len

    def read_len(self, s):
        try:
            l = toint(s[:])
        except:
            return None
        if l > self.encoder.max_len:
            return None
        return l, self.read_message

    def read_message(self, s):
        if len(s):
            self.connecter.got_message(self, s)
        return 4, self.read_len

    def read_dead(self, s):
        return None

    def _auto_close(self):
        if not self.closed and not self.complete:
            if self.retryconnection and self.locally_initiated and self.tc != 4:
                # Retry with higher timeout
                self.encoder.start_connections([self.peer], 4)
            self.close()

    def close(self, shutdown = False):
        if self.closed:
            return
        self.connection.close()
        self.sever(shutdown)

    def sever(self, shutdown = False):
        if self.closed:
            return
        if type(self.buffer) is list:
            for b in self.buffer:
                b.release()
            self.buffer[:] = []
        if self.log:
            self.log.write('closed\n')
            self.log.close()
        self.closed = True
        del self.encoder.connections[self.connection]
        if self.complete:
            self.connecter.connection_lost(self, shutdown)
        elif self.locally_initiated:
            self.encoder.incompletecounter -= 1
        self.connection = None
        if self.encoder.cnxformetadata is self and self.connecter.curmetadatapiece > -2 and not shutdown:
            self.encoder.nextConnection()

    def send_message_raw(self, message):
        self.write(message)

    def _write(self, message):
        if self.closed:
            message[0].release()
            return
        self.connection.write(message)

    def data_came_in(self, s):
        self.read(s)

    def _write_buffer(self, s):
        self.buffer = s + self.buffer

    def _read(self, s):
        if self.closed:
            s.release()
            return
        if self.log:
            self.log.write('r:' + b2a_hex(s[:]) + '\n')
        # self.encoder.measurefunc(len(s))  # not used
        self.buffer += s[:]
        s.release()
        while True:
            if self.closed:
                return
            # self.next_len = # of characters function expects
            # or 0 = all characters in the buffer
            # or -1 = wait for next read, then all characters in the buffer
            # not compatible w/ keepalives, switch out after all negotiation complete
            if self.next_len <= 0:
                m = self.buffer
                self.buffer = ''
            elif len(self.buffer) >= self.next_len:
                m = self.buffer[:self.next_len]
                self.buffer = self.buffer[self.next_len:]
            else:
                return
            try:
                x = self.next_func(m)
            except:
                self.next_len, self.next_func = 1, self.read_dead
                raise
            if x is None:
                self.close()
                return
            self.next_len, self.next_func = x
            if self.next_len < 0:  # already checked buffer, wait for additional data
                return
            if self.bufferlen is not None:
                self._read2('')
                return

    def _switch_to_read2(self):
        self._write_buffer = None
        if self.encrypter:
            self.encrypter.setrawaccess(self._read2, self._write)
        else:
            self.read = self._read2
        self.bufferlen = len(self.buffer)
        self.buffer = [newMsgPartBuffer(self.buffer)]

    def _read2(self, s):    # more efficient, requires buffer['',''] & bufferlen
        if self.closed:
            if len(s):
                s.release()
            return
        if self.log:
            self.log.write('r:' + b2a_hex(s[:]) + '\n')
        # self.encoder.measurefunc(len(s))  # not used
        while True:
            if self.closed:
                if len(s):
                    s.release()
                return
            p = self.next_len - self.bufferlen
            if self.next_len == 0:
                m = newMsgBuffer()
            elif len(s):
                if p > len(s):
                    self.buffer.append(newMsgPartBuffer(s.viewSlice()))
                    self.bufferlen += len(s)
                    s.release()
                    return
                self.bufferlen = len(s) - p
                self.buffer.append(newMsgPartBuffer(s.viewSlice(0, p)))
                messagesize = 0
                for msgpart in self.buffer:
                    messagesize += len(msgpart)
                m = newMsgBuffer(size = messagesize)
                for msgpart in self.buffer:
                    m.append(msgpart.viewSlice())
                    msgpart.release()
                self.buffer[:] = []
                if p != len(s):
                    self.buffer.append(newMsgPartBuffer(s.viewSlice(p)))
                s.release()
                s = ''
            elif p <= 0:
                # assert len(self.buffer) == 1
                b = self.buffer[0]
                self.bufferlen = len(b) - self.next_len
                m = newMsgBuffer(b.viewSlice(0, self.next_len))
                self.buffer[:] = []
                if p < 0:
                    self.buffer.append(newMsgPartBuffer(b.viewSlice(self.next_len)))
                b.release()
            else:
                return

            try:
                x = self.next_func(m)
            except:
                m.release()
                if len(s):
                    s.release()
                self.next_len, self.next_func = 1, self.read_dead
                raise
            m.release()
            if x is None:
                if len(s):
                    s.release()
                self.close()
                return
            self.next_len, self.next_func = x
            if self.next_len < 0:  # already checked buffer, wait for additional data
                if len(s):
                    s.release()
                return 

    def connection_flushed(self):
        if self.complete:
            self.connecter.connection_flushed(self)

    def connection_lost(self):
        self.sever()


class _dummy_banlist:
    def includes(self, x):
        return False


class Encoder:
    def __init__(self, connecter, rawserver, my_id,
                 dht, pl, sched, download_id, measurefunc,
                 upfunc, downfunc, localdisplay,
                 config, bans = _dummy_banlist()):
        self.config = config
        self.rawserver = rawserver
        self.sched = rawserver.add_task
        self.sockethandler = rawserver.sockethandler
        self.connecter = connecter
        self.my_id = my_id
        self.max_len = self.config['max_message_length']
        self.dht = dht
        self.pl = pl
        self.sched = sched
        self.keepalive_delay = self.config['keepalive_interval']
        self.download_id = download_id
        self.measurefunc = measurefunc
        self.upfunc = upfunc
        self.downfunc = downfunc
        self.localdisplay = localdisplay
        self.connections = {}
        self.banned = {}
        self.external_bans = bans
        # Default priorities for to_connect lists :
        # 0 : manual, trackers (1rst call), DHT (1rst call)
        # 1 : trackers, DHT
        # 2 : PEX (1rst reception)
        # 3 : PEX
        # 4 : autoclose retry and old peers
        self.to_connect = [[], [], [], [], []]
        self.tc_try = [0, 0, 0, 0, 0]
        self.currenttc = 0
        self.no_metadata_support = []
        self.completed = []
        self.paused = False
        if self.config['max_connections'] == 0:
            self.max_connections = 2 ** 30
            self.toconnectmaxsize = getABCUtility().maxpeerqueue
        else:
            self.max_connections = self.config['max_connections']
            self.toconnectmaxsize = max(getABCUtility().maxpeerqueue, self.max_connections)
        sched(self.send_keepalives, self.keepalive_delay)
        if self.dht:
            # extended messages + dht
            self.option_pattern = chr(0) * 5 + chr(16) + chr(0) + chr(1)
        else:
            # extended messages
            self.option_pattern = chr(0) * 5 + chr(16) + chr(0) * 2
        self.incompletecounter = 0
        self.cnxformetadata = None
        self.updatestorageflag = None
        # Min slowdown
        self.slowdown = self.config['raw_server_slowdown']
        self.eventsfrompl = False
        self.lastevent = clock()

    def getRawServerParams(self, time, events):
        if self.localdisplay.isSet():
            return self.slowdown, False
        if self.connections:
            if self.downfunc() > 0:
                self.lastevent = time
                self.eventsfrompl = False
                return self.slowdown, True
            if self.upfunc() > 0:
                self.lastevent = time
                self.eventsfrompl = False
                return self.slowdown, False
        if events or self.eventsfrompl:
            self.lastevent = time
            self.eventsfrompl = False
            return 0.1, False
        if time - self.lastevent < 5:
            return 0.1, False
        return 1.0, False

    def send_keepalives(self):
        self.sched(self.send_keepalives, self.keepalive_delay)
        if self.paused:
            return
        for c in self.connections.values():
            c.keepalive()

    def initPeers(self):
        if self.connecter.peers:
            self.sched(self.start_connections, 0, [[(p, 0, None, None, 0) for p in self.connecter.peers], 0])

    def getFirstNonEmptyTC(self):
        for tc in self.to_connect:
            if tc:
                return tc
        return None

    def getNextTC(self):
        nbtc = len(self.to_connect)
        if self.to_connect[self.currenttc] and self.tc_try[self.currenttc] < TCTRYMAX[self.currenttc] - 1:
            self.tc_try[self.currenttc] += 1
            return self.to_connect[self.currenttc]
        nexttc = (self.currenttc + 1) % nbtc
        for tci in [i % nbtc for i in xrange(nexttc, nexttc + nbtc)]:
            if self.to_connect[tci]:
                self.currenttc = tci
                self.tc_try[tci] = 0
                return self.to_connect[tci]
        return None

    def isMaxConnectionReached(self):
        if self.connecter.external_connection_made:
            max_initiate = self.config['max_initiate']
        else:
            max_initiate = int(self.config['max_initiate'] * 1.5)
        if len(self.connections) >= min(self.max_connections, max_initiate):
            return True
        return False

    def start_connections(self, peers, prio):
        # print 'start_connections', peers
        if self.getFirstNonEmptyTC() is None and not self.cnxformetadata:
            self.sched(self._start_connection_from_queue)
        self.to_connect[prio][0:0] = peers
        del self.to_connect[prio][self.toconnectmaxsize:]

    def _start_connection_from_queue(self):
        #print '_start_connection_from_queue'
        if self.getFirstNonEmptyTC() is None:
            return
        if self.isMaxConnectionReached():
            #print 'case 1 :', len(self.connections)
            delay = 60
        elif self.paused or self.incompletecounter >= MAX_INCOMPLETE:
            #print 'case 2 :', self.paused, self.incompletecounter >= MAX_INCOMPLETE
            delay = 1
        else:
            #print 'case 3'
            delay = 0
            self.start_connection(self.getNextTC().pop(0))
        if self.getFirstNonEmptyTC() and not self.cnxformetadata:
            self.sched(self._start_connection_from_queue, delay)

    def start_connection(self, peer):
        # print 'start_connection', peer
        dns, id, encrypted, cryptoallowed, source = peer
        sec = self.config['security']
        if id == self.my_id or not self.check_ip(ip = dns[0]):
            return True
        if self.config['crypto_only']:
            if encrypted == 0 or cryptoallowed == 0:
                return True
            encrypted = 1
        if self.config['magnet']:
            for d, i, e, c, s in self.no_metadata_support:
                if id and i == id or d == dns:
                    #print 'Skipping already contacted peer with no metadata support'
                    return True
                if sec and d[0] != 'unknown' and d[0] == dns[0]:
                    #print 'Skipping already contacted peer with no metadata support'
                    return True
        else:
            if self.connecter.downloader.storage.am_I_complete() and dns in self.connecter.downloader.disconnectedseeds:
                self.connecter.downloader.add_disconnected_seed(dns)
                return True
        for v in self.connections.values():
            condns = v.get_dns_remote()
            if id and v.id == id or condns == dns:
                if v.remoteport is None:
                    v.remoteport = dns[1]
                if v.cryptoallowed is None:
                    v.cryptoallowed = cryptoallowed
                return True
            if sec and condns[0] != 'unknown' and condns[0] == dns[0]:
                return True
        try:
            c = self.rawserver.start_connection(dns)
            #print 'Creating connection'
            cnx = Connection(self, c, self.option_pattern, self.pl, encrypted = encrypted, peer = peer)
            self.connections[c] = cnx
            c.set_handler(cnx)
        except socketerror:
            return False

        if self.config['magnet']:
            self.cnxformetadata = cnx

        return True

    def nextConnection(self):
        #print 'Next connection'
        if self.connecter.extconwithmeta:
            con = self.connecter.extconwithmeta.pop(0)
            self.cnxformetadata = con.connection
            if self.connecter.curmetadatapiece == -1:
                self.connecter.curmetadatapiece = 0
            con.send_metadatarequest(self.connecter.curmetadatapiece)
        else:
            self.cnxformetadata = None
            self._start_connection_from_queue()

    def metadataReport(self, report):
        self.sched(self._metadataReport, 0, [report])

    def _metadataReport(self, report):
        if report:
            self.connecter.curmetadatapiece = -3
        else:
            self.connecter.resetMetadata()
            self.cnxformetadata.close()

    def onUpdateStorageComplete(self):
        if self.no_metadata_support:
            self.to_connect[0][0:0] = self.no_metadata_support[:]
            self.no_metadata_support[:] = []
        del self.to_connect[0][self.toconnectmaxsize:]
        self.config['magnet'] = 0
        self.connecter.updateConnections()
        self.connecter.extconwithmeta[:] = []
        self.nextConnection()

    def storePeerWithNoMetadataSupport(self, peer):
        #print 'Storing peer with no metadata support'
        if len(self.no_metadata_support) < 100:
            self.no_metadata_support.append(peer)

    def check_ip(self, connection = None, ip = None):
        if not ip:
            ip = connection.get_ip(True)
        if self.banned.has_key(ip):
            return False
        if self.external_bans.includes(ip):
            return False
        return True

    def got_id(self, connection):
        if connection.id == self.my_id:
            self.connecter.external_connection_made -= 1
            return False
        ip = connection.get_ip(True)
        sec = self.config['security']
        for v in self.connections.values():
            if v is connection:
                continue
            if v.id == connection.id:
                # There will be at most 1 duplicate id
                if ip == v.get_ip(True):
                    if connection.remoteport is None:
                        connection.remoteport = v.remoteport
                    if connection.cryptoallowed is None:
                        connection.cryptoallowed = v.cryptoallowed
                    v.close()
                return False
            if sec and ip != 'unknown' and ip == v.get_ip(True):
                v.close()
        return True

    def external_connection_made(self, connection, buffer = None, uses_dht = None, uses_extended = None, encrypted = None,
                                 encrypter = None, cryptmode = None, switchtotorrentflag = None):
        if self.paused or len(self.connections) >= self.max_connections \
           or not self.check_ip(connection = connection):
            connection.close()
            if self.pl:
                switchtotorrentflag.set()
            return
        cnx = Connection(self, connection, self.option_pattern, self.pl, encrypted = encrypted, encrypter = encrypter)
        connection.set_handler(cnx)
        self.connections[connection] = cnx
        if self.pl:
            connection.set_socket_handler(self.sockethandler)
            switchtotorrentflag.set()            
            cnx.initReading(buffer, uses_dht, uses_extended, cryptmode)

    def ban(self, ip, ban):
        if ban:
            self.banned[ip] = 1
        elif ip in self.banned:
            del self.banned[ip]

    def pause(self, flag):
        if not flag:
            self.initPeers()
        self.paused = flag

    def shutdown(self):
        for c in self.connections.values():
            c.close(shutdown = True)
        self.cnxformetadata = None
