##############################################################################
# Module : scanner.py
# Author : Old King Cole
# Date   : 10/15/2005
#
# Description : Directory scanner (searches for torrents in a directory and
#               load them)
#
##############################################################################
import wx

from threading import Timer
from os import path, listdir, access, R_OK, mkdir, rename, remove
from shutil import copyfile


class DirectoryScanner:
    def __init__(self, parent):
        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.status = 0
        self.scantimer = None
        self.scannedfiles = 0
        self.loadedfiles = 0
        self.buttontooltip = self.parent.scannerbutton.GetToolTip()
        self.stopping = False
        self.loadedstring = self.localize('loaded')
        self.scannedstring = self.localize('scanned')

    def start(self):
        self.stopping = False
        self.status = 1
        self.buttontooltip.SetTip(str(self.loadedfiles) + self.loadedstring +
                                  str(self.scannedfiles) + self.scannedstring)
        self.scanDirectory()

    def stop(self, reset = True):
        self.stopping = True
        try:
            self.scantimer.cancel()
        except:
            pass
        if reset:
            self.parent.scannerbutton.SetBitmapSelected(self.parent.scannerbutton.bitmapselected)
            self.buttontooltip.SetTip(self.localize('dirscanner'))
            self.scannedfiles = self.loadedfiles = 0
        self.status = 0

    def restart(self):
        self.stop(reset = False)
        self.start()

    def canScan(self):
        self.scandir = self.utility.completePath(self.utility.scanparams['scandir'])

        errormsg = ''
        if self.utility.abcparams['setextdefaultfolder'] == '0':
            # There must be a default download folder for the scanner to run
            errormsg = self.localize('scandefdownfolderr')
        elif not path.isdir(self.scandir):
            # Check is folder to be scanned exists
            errormsg = self.localize('noscanfolderr')
        else:
            normscandirlower = path.normpath(self.scandir).lower()
            if normscandirlower == path.normpath(self.utility.abcpath).lower() \
               or normscandirlower == path.normpath(self.utility.datapath).lower():
                # Folder to be scanned can't be ABC installation directory
                errormsg = self.localize('badscanfolderr')

        if errormsg:
            dlg = wx.MessageDialog(self.parent, errormsg, self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False
        return True

    def addNewProc(self, scanned):
        if self.utility.scanparams['scanstartpos'] == '0':
            movetop = True
        else:
            movetop = False
        self.loadeddir = path.join(self.scandir, 'loaded')
        self.baddir = path.join(self.scandir, 'bad')
        self.dupdir = path.join(self.scandir, 'duplicate')
        if self.utility.scanparams['scanstatus'] == '0':
            status = 'stop'
            startnow = False
        elif self.utility.scanparams['scanstatus'] == '1':
            status = 'queue'
            startnow = False
        else:
            status = 'stop'
            startnow = True
        for filename in scanned:
            dest = self.utility.findUniqueFileName(path.join(self.utility.datapath, 'torrent', filename), 'scan')
            if dest is None:
                continue
            src = path.join(self.scandir, filename)
            copyfile(src, dest)
            addreport, msg = self.parent.queue.addNewProc(dest, filename, False, caller = "scan", prio = int(self.utility.scanparams['scanprio']),
                                                          status = status, startnow = startnow, movetop = movetop)
            if addreport:
                # Good torrent
                self.loadedfiles += 1
                if not self.stopping:
                    self.buttontooltip.SetTip(str(self.loadedfiles) + self.loadedstring +
                                              str(self.scannedfiles) + self.scannedstring)
                if self.utility.scanparams['scanmovetor'] == '0':
                    try:
                        remove(src)
                    except:
                        pass
                else:
                    if not path.exists(self.loadeddir):
                        mkdir(self.loadeddir)
                    elif path.isfile(self.loadeddir):
                        try:
                            remove(src)
                        except:
                            pass
                        continue
                    dest = self.utility.findUniqueFileName(path.join(self.loadeddir, filename), 'scan', False)
                    if dest is None:
                        try:
                            remove(src)
                        except:
                            pass
                    else:
                        try:
                            rename(src, dest)
                        except:
                            pass
            elif msg == 'Duplicate torrent' or msg == 'Duplicate destination':
                # Duplicate torrent
                if not path.exists(self.dupdir):
                    mkdir(self.dupdir)
                elif path.isfile(self.dupdir):
                    try:
                        remove(src)
                    except:
                        pass
                    continue
                dest = self.utility.findUniqueFileName(path.join(self.dupdir, filename), 'scan', False)
                if dest is None:
                    try:
                        remove(src)
                    except:
                        pass
                else:
                    try:
                        rename(src, dest)
                    except:
                        pass
            else:
                # Bad torrent
                if not path.exists(self.baddir):
                    mkdir(self.baddir)
                elif path.isfile(self.baddir):
                    try:
                        remove(src)
                    except:
                        pass
                    continue
                dest = self.utility.findUniqueFileName(path.join(self.baddir, filename), 'scan', False)
                if dest is None:
                    try:
                        remove(src)
                    except:
                        pass
                else:
                    try:
                        rename(src, dest)
                    except:
                        pass

    def scanDirectory(self):
        if self.stopping:
            return
        if self.canScan():
            scanned = [entry for entry in listdir(self.scandir) if path.isfile(path.join(self.scandir, entry))
                       and access(path.join(self.scandir, entry), R_OK)]
            if scanned and not self.stopping:
                if self.scannedfiles == 0:
                    self.parent.scannerbutton.SetBitmapSelected(self.parent.scannerbutton.bitmapdetected)
                    self.parent.scannerbutton.Refresh()
                self.scannedfiles += len(scanned)
                self.buttontooltip.SetTip(str(self.loadedfiles) + self.loadedstring +
                                          str(self.scannedfiles) + self.scannedstring)
                self.parent.queue.invokeLater(self.addNewProc, [scanned])
        if not self.stopping:
            self.scantimer = Timer(int(self.utility.scanparams['scanfreqh']) * 3600 + int(self.utility.scanparams['scanfreqm']) * 60,
                                   self.scanDirectory)
            self.scantimer.start()
