#!/usr/bin/env python

# Written by Bram Cohen
# multitracker extensions by John Hoffman
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

import sys
from os.path import getsize, split, join, abspath, isdir, normpath
from os import listdir
from hashlib import sha1
from copy import copy
from threading import Event
from time import time

from BitTornado.bencode import bencode
from BitTornado.parseargs import parseargs, formatDefinitions
from BitTornado.utility import exceptionArgsToString

defaults = [
    ('announce_list', '',
        'a list of announce URLs - explained below'),
    ('nodes', '',
        'a list of pairs of DHT nodes address and port'),
    ('piece_size_pow2', 0,
        "which power of 2 to set the piece size to (0 = automatic)"),
    ('comment', '',
        "optional human-readable comment to put in .torrent"),
    ('target', '',
        "optional target file for the torrent"),
    ('private', 0,
        "Private torrent (if set to 1 only accept peers from the trackers)")
    ]

default_piece_len_exp = 18

ignore = ['core', 'CVS']

def print_announcelist_details():
    print ('    announce_list = optional list of redundant/backup tracker URLs, in the format:')
    print ('           url[,url...][|url[,url...]...]')
    print ('                where URLs separated by commas are all tried first')
    print ('                before the next group of URLs separated by the pipe is checked.')
    print ("                If none is given, it is assumed you don't want one in the metafile.")
    print ('                If announce_list is given, clients which support it')
    print ('                will ignore the <announce> value.')
    print ('           Examples:')
    print ('                http://tracker1.com|http://tracker2.com|http://tracker3.com')
    print ('                     (tries trackers 1-3 in order)')
    print ('                http://tracker1.com,http://tracker2.com,http://tracker3.com')
    print ('                     (tries trackers 1-3 in a randomly selected order)')
    print ('                http://tracker1.com|http://backup1.com,http://backup2.com')
    print ('                     (tries tracker 1 first, then tries between the 2 backups randomly)')

def dummy(v):
    pass

def make_meta_data(file, url, dht, params, nofile = False, flag = Event(), progress = dummy, progress_percent = 1):
    if dht and params.has_key('addclosestdhtnodes') and params['addclosestdhtnodes'] != 0:
        addclosestdhtnodes = 1
    else:
        addclosestdhtnodes = 0

    info = params.get('info')
    if info is None:
        if params.has_key('piece_size_pow2'):
            piece_len_exp = params['piece_size_pow2']
        else:
            piece_len_exp = default_piece_len_exp
        if piece_len_exp == 0:              # automatic
            size = calcsize(file)
            if size > 64L*1024*1024*1024:   # > 64 gig =
                piece_len_exp = 24          #   16 meg pieces
            elif size > 32L*1024*1024*1024: # > 32 gig =
                piece_len_exp = 23          #   8 meg pieces
            elif size > 4L*1024*1024*1024:  # > 4 gig =
                piece_len_exp = 22          #   4 meg pieces
            elif size > 2L*1024*1024*1024:  # > 2 gig =
                piece_len_exp = 21          #   2 meg pieces
            elif size > 1024*1024*1024:     # > 1 gig =
                piece_len_exp = 20          #   1 meg pieces
            elif size > 512*1024*1024:      # > 512M =
                piece_len_exp = 19          #   512K pieces
            elif size > 256*1024*1024:      # > 256M =
                piece_len_exp = 18          #   256K pieces
            elif size > 128*1024*1024:      # > 128M =
                piece_len_exp = 17          #   128K pieces
            elif size > 64*1024*1024:       # > 64M =
                piece_len_exp = 16          #   64K pieces
            elif size > 32*1024*1024:       # > 32M =
                piece_len_exp = 15          #   32K pieces
            else:                           # < 32M =
                piece_len_exp = 14          #   16K pieces
        piece_length = 2 ** piece_len_exp

        if params.has_key('private') and params['private'] != 0:
            private = 1
        else:
            private = 0

        info = makeinfo(file, piece_length, private, nofile, flag, progress, progress_percent)

    if flag.isSet():
        return

    data = {'info': info, 'creation date': long(time())}
    if params.has_key('real_url'):
        data['announce'] = params['real_url']
    else:
        url = url.strip()
        if url:
            data['announce'] = url.encode('utf_8')
    if params.has_key('real_announce_list'):    # shortcut for progs calling in from outside
        data['announce-list'] = params['real_announce_list']
    elif params.has_key('announce_list') and params['announce_list'] != '':
        list = []
        for tier in params['announce_list'].split('|'):
            sublist = []
            for tracker in tier.split(','):
                sublist += [tracker.encode('utf_8')]
            list += [sublist]
        data['announce-list'] = list
    if params.has_key('comment') and params['comment'] != '':
        data['comment'] = params['comment'].encode('utf_8')
    if params.has_key('dht_nodes_list'):
        nodelist = params['dht_nodes_list'][:]
    else:
        nodelist = []
    closestnodes = []
    if addclosestdhtnodes:
        for t in dht.tables:
            closestnodes.extend(t.findNodes(sha1(bencode(info)).digest(), alwaysk = True))
        nodelist.extend([[node.host, node.port] for node in closestnodes])
    if nodelist or not url and not params.has_key('real_announce_list'):
        data['nodes'] = nodelist

    if params.has_key('infohash'):
        data['infohash'] = params['infohash']
    if params.has_key('validmagnetname'):
        data['validmagnetname'] = params['validmagnetname']
    if params.has_key('keepmagnetname'):
        data['keepmagnetname'] = params['keepmagnetname']

    return bencode(data), len(closestnodes)

def make_meta_file(file, url, dht, params = {}, flag = Event(),
                    progress = dummy, progress_percent = 1):
    if params.has_key('target') and params['target'] != '':
        f = join(params['target'], split(normpath(file))[1] + '.torrent')
    else:
        a, b = split(file)
        if b == '':
            f = a + '.torrent'
        else:
            f = join(a, b + '.torrent')

    metadata, lenclosestnodes = make_meta_data(file, url, dht, params, flag = flag, progress = progress,
                                               progress_percent = progress_percent)
    h = open(f, 'wb')
    h.write(metadata)
    h.close()

    return lenclosestnodes

def calcsize(file):
    if not isdir(file):
        return getsize(file)
    total = 0L
    for s in subfiles(abspath(file)):
        total += getsize(s[1])
    return total

def makeinfo(file, piece_length, private, nofile, flag, progress, progress_percent = 1):
    if nofile:
        info = {'pieces': '',
                'piece length': piece_length, 'length': 0, 
                'name': file.encode('utf_8')}
    else:
        file = abspath(file)
        if isdir(file):
            subs = subfiles(file)
            subs.sort()
            pieces = []
            sh = sha1()
            done = 0L
            fs = []
            totalsize = 0.0
            totalhashed = 0L
            for p, f in subs:
                totalsize += getsize(f)
    
            for p, f in subs:
                pos = 0L
                size = getsize(f)
                for subpath in p:
                    subpath = subpath.encode('utf_8')
                fs.append({'length': size, 'path': p})
                h = open(f, 'rb')
                while pos < size:
                    a = min(size - pos, piece_length - done)
                    sh.update(h.read(a))
                    if flag.isSet():
                        return
                    done += a
                    pos += a
                    totalhashed += a
    
                    if done == piece_length:
                        pieces.append(sh.digest())
                        done = 0
                        sh = sha1()
                    if progress_percent:
                        progress(totalhashed / totalsize)
                    else:
                        progress(a)
                h.close()
            if done > 0:
                pieces.append(sh.digest())
            info = {'pieces': ''.join(pieces),
                    'piece length': piece_length, 'files': fs, 
                    'name': split(file)[1].encode('utf_8')}
        else:
            size = getsize(file)
            pieces = []
            p = 0L
            h = open(file, 'rb')
            while p < size:
                x = h.read(min(piece_length, size - p))
                if flag.isSet():
                    return
                pieces.append(sha1(x).digest())
                p += piece_length
                if p > size:
                    p = size
                if progress_percent:
                    progress(float(p) / size)
                else:
                    progress(min(piece_length, size - p))
            h.close()
            info = {'pieces': ''.join(pieces), 
                    'piece length': piece_length, 'length': size, 
                    'name': split(file)[1].encode('utf_8')}
    if private == 1:
        info['private'] = 1
    return info

def subfiles(d):
    r = []
    stack = [([], d)]
    while len(stack) > 0:
        p, n = stack.pop()
        if isdir(n):
            for s in listdir(n):
                if s not in ignore and s[:1] != '.':
                    stack.append((copy(p) + [s], join(n, s)))
        else:
            r.append((p, n))
    return r

def prog(amount):
    print '%.1f%% complete\r' % (amount * 100),

if __name__ == '__main__':
    if len(sys.argv) < 3:
        a, b = split(sys.argv[0])
        print 'Usage: ' + b + ' <trackerurl> <file> [file...] [params...]'
        print
        print formatDefinitions(defaults, 80)
        print_announcelist_details()
        print ('')
        sys.exit(2)

    try:
        config, args = parseargs(sys.argv[1:], defaults, 2, None)
        for file in args[1:]:
            write_meta_file(file, args[0], None, config, progress = prog)
    except ValueError, e:
        print 'error: ' + exceptionArgsToString(e)
        print 'run with no args for parameter explanations'
