##############################################################################
# Module : abcengine.py
# Author :
# Date   :
# Updated and modified for ABC_OKC : Old King Cole
#
# Description : Engine for a running torrent. Interfaces the output of a
#               BitTornado thread with ABC_OKC GUI.
#
##############################################################################
import sys, wx

from os import path, remove, rename
from threading import Event, Lock, Thread, Timer
from time import clock, strftime
from traceback import print_exc
from StringIO import StringIO
from socket import error as socketerror
from hashlib import sha1
from random import random

from BitTornado.download_bt1 import BT1Download, parse_params, get_response
from BitTornado.RawServer import RawServer
from BitTornado.natpunch import UPnP_test
from BitTornado.__init__ import createPeerID
from BitTornado.utility import exceptionArgsToString
from BitTornado.bencode import bdecode
from btmakemetafile import make_meta_data


FILESEM = Lock()
CHECKWAITING = Event()
CHECKWAITING.set()

wxEVT_INVOKE = wx.NewEventType()


def EVT_INVOKE(win, func):
    win.Connect(-1, -1, wxEVT_INVOKE, func)

    
def DELEVT_INVOKE(win):
    win.Disconnect(-1, -1, wxEVT_INVOKE)


class InvokeEvent(wx.PyEvent):
    def __init__(self, func, args, kwargs, forcepost):
        wx.PyEvent.__init__(self)
        self.SetEventType(wxEVT_INVOKE)
        self.func = func
        self.args = args
        self.kwargs = kwargs
        self.forcepost = forcepost


class ABCEngine(wx.EvtHandler):
    def __init__(self, parent, guiman, torrent, tracker, auto, onlycheck,
                 fastresume, reseed, btparams, sbwaitingforleechers, dht, pl, startqueue):
        wx.EvtHandler.__init__(self)
        now = clock()
        self.dow = None
        self.parent = parent
        self.guiman = guiman
        self.torrent = torrent
        self.listid = torrent.index
        self.tracker = tracker
        self.btparams = btparams
        self.fastresume = fastresume
        self.reseed = reseed
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.abcparams = self.utility.abcparams
        self.getRankfromID = self.guiman.getRankfromID
        self.list = parent.list
        self.doneflag = Event()
        self.fin = False
        self.stoppedonrequest = False
        # Seeding and total seeding pseudo timers
        # 0 : stopped
        # 1 : running
        self.seedingtimer = 0
        self.totalseedingtimer = 0
        # Last seeding times before current seeding
        self.lastseedingtime = torrent.seedingtime
        self.lasttotalseedingtime = torrent.totalseedingtime
        self.workingtimeleft = 0
        self.detailwin = torrent.detailwin
        self.infowin = torrent.infowin
        self.torrent.alloc_type = int(self.abcparams['alloc_type'])
        self.superseed = False
        self.reannouncelast = 0
        self.olddownsize = torrent.downsize
        self.oldupsize = torrent.upsize
        self.newdownsize = 0.
        self.newupsize = 0.
        # Down and up size from previous refresh
        self.prevdownsize = 0.
        self.prevupsize = 0.
        self.shareratio = 0.
        self.forceseeding = torrent.uploadtillcopycomplete
        self.normalseedingover = False
        self.numpeer = '0'
        self.numseed = '0'
        self.numpeertext = '0 (' + torrent.peer + ')'
        self.numseedtext = '0 (' + torrent.seed + ')'
        self.guilastupdate = self.spewwait = now
        self.lastsavepeers = now - 480
        self.lastreloadpeers = now - 60
        self.networkcnxretrycount = 0
        self.networkdisconnected = False
        self.networkdisconnectedtime = -1
        # Used to prevent all active torrents to reconnect at the same time
        self.networkdisconnecteddelay = 15 * random()
        if self.utility.resettimeoutsonstart:
            self.lasttorrentactive = now
            self.firsterrortracker = 0
        else:
            self.lasttorrentactive = now - self.torrent.inactivitytime
            if self.torrent.trackererrortime == 0:
                self.firsterrortracker = 0
            else:
                self.firsterrortracker = now - self.torrent.trackererrortime
        self.torrent.lastgetscrape = now - self.utility.scrapeauto + 60
        self.checkinginttrack = False
        self.startcheckinginttrack = -1
        self.btstatus = 'init'
        self.oldbtstatus = None
        self.awaken = False
        if onlycheck:
            self.progress = 0.
        else:
            self.progress = torrent.progress
        self.progressongoing = self.abcparams['progressongoing']
        self.progressongoing2 = self.abcparams['progressongoing2']
        self.ongoing = ''
        self.downrate = 0.
        self.uprate = 0.
        self.extradatauprate = 0.
        self.shareratiotext = "0.0%"
        self.numconnections = 0
        self.numdownloads = 0
        self.numuploads = 0
        self.numdeaddown = 0
        self.peeravgtext = "0.0%"
        self.totalspeed = 0.
        self.downloadrate = self.utility.formatedRate(0, self.torrent.prioritizedown)
        self.uploadrate = self.utility.formatedRate(0, self.torrent.prioritizeup)
        self.totalspeedtext = self.utility.formatedRate(0)
        self.dlsizetext = self.utility.formatedSize(self.olddownsize)
        self.ulsizetext = self.utility.formatedSize(self.oldupsize)
        self.workingtimelastcheck = 0.
        self.seedingtimelastcheck = 0.
        self.seedingtimerreset = 0.
        self.totalseedingtimerreset = 0.
        self.statistics = None
        self.spew = None
        # To compute mean of the last values of uprate and downrate over 10 s
        self.meansize = int(10. / self.utility.GUIupdaterate_fast)
        self.pastdownrate = self.meansize * [0.]
        self.pastuprate = self.meansize * [0.]
        self.pastextradatauprate = self.meansize * [0.]
        self.meandownrate = 0.
        self.meanuprate = 0.
        self.stopafterfirstrefresh = False
        self.btcomplete = False
        self.dht = dht
        self.pl = pl
        self.startqueue = startqueue
        self.initializingfiles = True
        self.paused = False
        # Max uploading time without downloading
        self.upwithoutdown = 0.
        self.lastdown = now
        self.overmaxupwithoutdown = False
        self.overmaxupwithoutdown2 = False

        # Keep track of if we have any connections
        self.hasconnections = False

        # Automatically or manually started
        self.auto = auto

        self.colour = self.utility.colnoeng
        self.trackercolour = self.utility.colnocon

        # Counters for activity
        self.timeractivityoff = self.timeractivityon = 0

        # True if only check the torrent
        self.onlycheck = onlycheck

        # True if timeouts are currently canceled
        self.timeoutscanceled = False

        # To keep track of change in uploadtimemax
        torrent.olduploadtimemax = torrent.uploadtimemax

        # True once magnet has got its metadata and starts updating its storage
        self.restarted = False

        # Last time a standby uncomplete torrent with leechers was resumed
        # to check the number of available copies and leechers progress
        # Before being initiated with this delay only when a torrent possibly ends checking its data,
        # it's True or False to say that this mode must be engaged.
        # False means the torrent is not waiting for connections from leechers and
        # must not be checked for the number of available copies and leechers progress
        self.sbwaitingforleechers = sbwaitingforleechers

        # Flag set when BitTornado thread is finished
        self.shutdownflag = Event()

        # To keep trace of a completed event received while in pause
        self.skippedcompleted = False

        # Flag enabling reading of the blocked waiting status (dow.blockedwaiting)
        self.readwaiting = Event()
        self.readwaiting.set()

        EVT_INVOKE(self, self.onInvoke)

    def onInvoke(self, event):
        if event.forcepost or not self.doneflag.isSet():
            event.func(*event.args, **event.kwargs)

    def invokeLater(self, func, args = [], kwargs = {}, forcepost = False):
        if forcepost or not self.doneflag.isSet():
            wx.PostEvent(self, InvokeEvent(func, args, kwargs, forcepost))

    def formatedProgress(self):
        if self.progress == -1:
            return '  ?  %' + self.torrent.partialMarker()
        progressstring = str(.1 * int(10 * self.progress))
        if progressstring == '100.0':
            if self.fin or self.torrent.isWaiting() or self.torrent.isCheckingOrAllocating() or self.paused:
                return ''.join(('100%', self.ongoing, self.torrent.partialMarker()))
            return ''.join(('99.9%', self.ongoing, self.torrent.partialMarker()))
        return ''.join((progressstring, '%', self.ongoing, self.torrent.partialMarker()))

    def updateDetailWindowNow(self, detwin):
        self.detailwin = detwin
        self.spewwait = 0
        self.updateDetailWindow()

    def updateStatus(self, dpflag = None, fractionDone = None, ongoing = None, sizeWanted = None, sizeDone = None, downRate = None,
                     upRate = None, extradataUpRate = None, activity = None, statistics = None, spew = None, totalspeed = None,
                     kicked = None, banned = None):
        self.invokeLater(self.onUpdateStatus, [dpflag, fractionDone, ongoing, sizeWanted, sizeDone, downRate, upRate, extradataUpRate,
                                               activity, statistics, spew, totalspeed, kicked, banned])

    def onUpdateStatus(self, dpflag, fractionDone, ongoing, sizeWanted, sizeDone, downRate, upRate, extradataUpRate, activity,
                       statistics, spew, totalspeed, kicked, banned):
        now = clock()
        self.statistics = statistics
        self.spew = spew
        self.kicked = kicked
        self.banned = banned
        if not self.fin:
            if activity is None:
                if self.hasconnections:
                    self.btstatus = 'working'
                else:
                    self.btstatus = 'connectingtopeers'
            else:
                self.btstatus = activity

        # Always update up and down rates at highest refresh rate
        if self.guilastupdate + self.utility.GUIupdaterate_fast <= now:
            if self.fin:
                self.downrate = 0.
                self.downloadrate = self.utility.formatedRate(None, self.torrent.prioritizedown)
            elif downRate is None:
                self.downrate = 0.
                if self.restarted:
                    self.downloadrate = self.utility.formatedRate(0, self.torrent.prioritizedown)
                else:
                    self.downloadrate = self.utility.formatedRate(None, self.torrent.prioritizedown)
            else:
                self.downrate = downRate / 1024
                self.downloadrate = self.utility.formatedRate(self.downrate, self.torrent.prioritizedown)

            if upRate is None:
                self.uprate = 0.
                if self.restarted:
                    self.uploadrate = self.utility.formatedRate(0, self.torrent.prioritizeup)
                else:
                    self.uploadrate = self.utility.formatedRate(None, self.torrent.prioritizeup)
            else:
                self.uprate = upRate / 1024
                self.uploadrate = self.utility.formatedRate(self.uprate, self.torrent.prioritizeup)

            if extradataUpRate is None:
                self.extradatauprate = 0.
            else:
                self.extradatauprate = extradataUpRate / 1024
    
            # Compute mean uprate and downrate
            self.meanuprate += (self.uprate - self.pastuprate.pop(0)) / self.meansize
            if self.meanuprate < 0:
                self.meanuprate = 0.
            self.pastuprate.append(self.uprate)
            self.meandownrate += (self.downrate - self.pastdownrate.pop(0)) / self.meansize
            if self.meandownrate < 0:
                self.meandownrate = 0.
            self.pastdownrate.append(self.downrate)

        if self.guilastupdate + self.utility.GUIupdaterate > now \
           and self.btstatus == self.oldbtstatus \
           and (ongoing is None or (not self.progressongoing and not self.progressongoing2)):
            if dpflag is not None:
                dpflag.set()
            return

        self.guilastupdate = now
        self.oldbtstatus = self.btstatus

        # Set colour
        oldcolour = self.colour
        if not self.paused:
            if self.statistics is None:
                # Start up
                self.colour = self.utility.colnoeng
                self.hasconnections = False
            elif self.statistics.numPeers + self.statistics.numSeeds + self.statistics.numOldSeeds == 0:
                if self.statistics.last_failed:
                    # Disconnected
                    self.colour = self.utility.coldisc
                else:
                    # No connections
                    self.colour = self.utility.colnocon
                self.hasconnections = False
            elif not self.statistics.external_connection_made:
                # No incoming
                self.colour = self.utility.colnoinc
                self.hasconnections = True
            elif self.fin and self.statistics.numSeeds + self.statistics.numOldSeeds == 0 and self.statistics.numCopies < 1 \
                 or not self.fin and self.statistics.numSeeds == 0 and self.statistics.numCopies2 < 1:
                # No completes
                self.colour = self.utility.colnocom
                self.hasconnections = True
            else:
                # All good
                self.colour = self.utility.colok
                self.hasconnections = True

        if not self.fin:
            if self.torrent.magnet:
                self.progress = 0.0
            elif fractionDone is not None:
                self.progress = fractionDone * 100
                if ongoing is None or ongoing == '':
                    self.ongoing = ''
                elif ongoing:
                    self.ongoing = self.progressongoing
                else:
                    self.ongoing = self.progressongoing2

        if self.statistics is None:
            if self.spew is None:
                self.numpeer = "0"
            else:
                self.numpeer = str(len(self.spew))
            self.numseed = "0"
        else:
            self.numpeer = str(self.statistics.numPeers)
            self.peeravgtext = str(.1 * int(10 * self.statistics.percentDone)) + '%'
            self.dlsizetext = self.utility.formatedSize(self.olddownsize + self.statistics.downTotal)
            self.ulsizetext = self.utility.formatedSize(self.oldupsize + self.statistics.upTotal)
            self.totalspeed = totalspeed / 1024
            self.totalspeedtext = self.utility.formatedRate(self.totalspeed)
            self.numconnections = self.statistics.numPeers
            if self.fin:
                self.numseed = str(self.statistics.numOldSeeds)
                self.torrent.numcopy = '%.3f' % (.001 * int(1000 * self.statistics.numCopies))
            else:
                self.numseed = str(self.statistics.numSeeds)
                self.numconnections += self.statistics.numSeeds
                self.torrent.numcopy = '%.3f' % (.001 * int(1000 * self.statistics.numCopies2))
            self.newdownsize = float(self.statistics.downTotal)
            self.parent.downvolmeasure.update_rate(self.newdownsize - self.prevdownsize)
            if self.parent.downvolcheckcycle == 0:
                self.parent.downvolfixedwin += self.newdownsize - self.prevdownsize
            self.prevdownsize = self.newdownsize
            self.newupsize = float(self.statistics.upTotal)
            self.parent.upvolmeasure.update_rate(self.newupsize - self.prevupsize)
            if self.parent.upvolcheckcycle == 0:
                self.parent.upvolfixedwin += self.newupsize - self.prevupsize
            self.prevupsize = self.newupsize
            if self.torrent.requestedsize <= 0 and not self.torrent.magnet:
                self.shareratio = 999.9
            elif self.progress <= 0:
                self.shareratio = 0.
            else:
                self.shareratio = float(self.oldupsize + self.newupsize) / self.torrent.requestedsize / self.progress * 10000
            self.shareratiotext = str(.1 * int(10 * self.shareratio)) + "%"
        self.numpeertext = self.numpeer + ' (' + self.torrent.peer + ')'
        self.numseedtext = self.numseed + ' (' + self.torrent.seed + ')'

        if self.colour != oldcolour:
            self.updateColour()

        if self.fin:
            self.countSeedingTimes(now)
        else:
            self.countWorkingTimeLeft(now, sizeWanted, sizeDone)

        self.refreshDisplay()

        self.updateInfoWindow()
        self.updateDetailWindow()

        if dpflag is not None:
            dpflag.set()

        if not self.paused:
            if self.stopafterfirstrefresh:
                self.parent.procSTOP([self.listid])
                return

            if self.torrent.isWaiting() or self.torrent.isCheckingOrAllocating():
                if self.utility.resettimeoutsonstart:
                    self.lasttorrentactive = now
                    self.firsterrortracker = 0.
                else:
                    self.lasttorrentactive = now - self.torrent.inactivitytime
                    if self.torrent.trackererrortime == 0:
                        self.firsterrortracker = 0.
                    else:
                        self.firsterrortracker = now - self.torrent.trackererrortime
                self.torrent.lastgetscrape = now - self.utility.scrapeauto + 60
                self.lastsavepeers = now - 480
                self.lastreloadpeers = now - 60
                self.lastdown = now - self.upwithoutdown
            else:
                # Check if torrent is completed
                if not self.fin and self.torrent.unwantedfiles and self.torrent.checkComplete():
                    self.onCompletedEvent()
                    return

                # Check timeouts
                if self.timeoutscanceled:
                    self.lasttorrentactive = now
                else:
                    # Check timeout when seeding
                    if self.fin:
                        if upRate is not None:
                            if upRate == 0:
                                if self.torrent.timeoutswitch and self.torrent.timeoutseed and self.torrent.timeoutseedup \
                                   and now - self.lasttorrentactive > self.torrent.timeoutupload * 60 \
                                   and self.parent.isTimeoutAllowed():
                                    self.lasttorrentactive = now
                                    self.torrent.inactivitytime = 0.
                                    if self.onTimeout(self.localize('timeoutupmsg')):
                                        return
                            else:
                                self.lasttorrentactive = now

                    # Check timeout when working
                    else:
                        if downRate is not None:
                            if downRate == 0:
                                if self.torrent.timeoutswitch and self.torrent.timeoutwork and self.torrent.timeoutworkdown \
                                   and now - self.lasttorrentactive > self.torrent.timeoutdownload * 60 \
                                   and self.parent.isTimeoutAllowed():
                                    self.lasttorrentactive = now
                                    self.torrent.inactivitytime = 0.
                                    if self.onTimeout(self.localize('timeoutdownmsg')):
                                        return
                            else:
                                self.lasttorrentactive = now

                # Check max uploading time without downloading
                if not self.fin and downRate is not None and upRate is not None:
                    if self.utility.upwithoutdown and downRate == 0:
                        if upRate == 0:
                            self.lastdown = now - self.upwithoutdown
                        else:
                            self.upwithoutdown = now - self.lastdown
                        if self.upwithoutdown > self.utility.upwithoutdowntime:
                            if not self.overmaxupwithoutdown:
                                self.dow.setConns(0)
                                self.overmaxupwithoutdown = True
                        elif self.overmaxupwithoutdown:
                            self.dow.setConns(self.torrent.maxupload)
                            self.overmaxupwithoutdown = False
                    else:
                        if self.overmaxupwithoutdown:
                            self.dow.setConns(self.torrent.maxupload)
                            self.overmaxupwithoutdown = False
                        self.lastdown = now
                        self.upwithoutdown = 0.
                    if self.utility.upwithoutdown2 and downRate == 0:
                        self.overmaxupwithoutdown2 = (upRate > 0)
                    else:
                        self.overmaxupwithoutdown2 = False

                # Check for reloading peers
                if self.fin:
                    self.startcheckinginttrack = -1
                    if self.networkdisconnected and upRate is not None:
                        if upRate == 0:
                            if self.networkdisconnected < 2:
                                self.torrent.reannounce(tracker = 'reset', force = True)
                                self.networkdisconnected = 3
                                self.torrent.setDHTDisconnected(True)
                                self.lastreloadpeers = now
                        else:
                            self.torrent.setDHTDisconnected(False)
                            self.networkdisconnected = False
                            self.networkdisconnectedtime = -1
                            self.networkcnxretrycount = 0
                            self.lastreloadpeers = now - 60
                else:
                    if self.networkdisconnected:
                        self.startcheckinginttrack = -1
                        if upRate is not None and upRate == 0 and downRate is not None and downRate == 0 \
                           and self.networkcnxretrycount < 20 and self.parent.networkdisconnection >= 0:
                            if self.networkdisconnectedtime == -1:
                                self.networkdisconnectedtime = now
                            if now - self.networkdisconnectedtime > self.networkdisconnecteddelay:
                                if self.networkdisconnected < 2:
                                    self.loadPeers()
                                    if self.torrent.exttracker and not self.checkinginttrack \
                                       and (self.utility.checkinttrack == 1
                                            or self.utility.checkinttrack == 2 and (self.torrent.private or not self.dht)):
                                        self.torrent.switchTracker(0)
                                        self.networkdisconnected = 2
                                    else:
                                        self.torrent.reannounce(tracker = 'reset', force = True)
                                        self.networkdisconnected = 3
                                    self.torrent.setDHTDisconnected(True)
                                    self.lastreloadpeers = now
                                    self.networkcnxretrycount += 1
                                elif now - self.lastreloadpeers > 90:
                                    self.loadPeers()
                                    self.lastreloadpeers = now
                                    self.networkcnxretrycount += 1
                        elif upRate is not None and upRate > 0 or downRate is not None and downRate > 0 \
                             or self.networkcnxretrycount >= 20:
                            if self.networkdisconnected == 2:
                                self.torrent.switchTracker(1)
                            self.torrent.setDHTDisconnected(False)
                            self.networkdisconnected = False
                            self.networkdisconnectedtime = -1
                            self.networkcnxretrycount = 0
                            self.lastreloadpeers = now - 60
                        elif self.parent.networkdisconnection < 0 \
                             and now + self.parent.networkdisconnection > 120 + self.networkdisconnecteddelay:
                            # Next 3 lines to deal with special case when connected with tracker with direct IP and with peers through VPN
                            self.loadPeers()
                            self.torrent.reannounce(tracker = 'reset', force = True)
                            if self.networkdisconnected == 2:
                                self.torrent.switchTracker(1)
                            self.torrent.setDHTDisconnected(False)
                            self.networkdisconnected = False
                            self.networkdisconnectedtime = -1
                            self.networkcnxretrycount = 0
                            self.lastreloadpeers = now - 60
                    else:
                        if self.checkinginttrack and downRate is not None and downRate > 0:
                            self.torrent.switchTracker(1)
                        elif self.torrent.exttracker:
                            if downRate is not None and downRate == 0:
                                if self.startcheckinginttrack == -1:
                                    self.startcheckinginttrack = now
                                if now - self.startcheckinginttrack > self.utility.checkinttrackautodelay:
                                    self.loadPeers()
                                    if self.utility.checkinttrack == 1 \
                                       or self.utility.checkinttrack == 2 and (self.utility.checkinttrackautopublic
                                                                               or self.torrent.private or not self.dht):
                                        if self.torrent.checkinttrackwait:
                                            self.checkinginttrack = True
                                            self.torrent.switchTracker(0)
                                        else:
                                            self.torrent.switchTrackerExternal()
                                    self.startcheckinginttrack = -1
                            else:
                                self.startcheckinginttrack = -1
                        else:
                            self.startcheckinginttrack = -1

                # Check for #seeds and #copies to set forceseeding
                if self.fin:
                    # Check if must terminate
                    if (self.torrent.uploadoption == 1 or self.torrent.uploadoption == 3) and self.torrent.seedingtimeleft == 0 \
                       or (self.torrent.uploadoption == 2 or self.torrent.uploadoption == 3) and self.torrent.ratioIsReached():
                        if self.normalseedingover:
                            if not self.forceseeding:
                                self.terminateUpload()
                        else:
                            # First time normal seeding is over
                            self.checkForTerminating()
                    else:
                        self.normalseedingover = False

                    if self.btstatus != 'finished' and self.torrent.uploadtillcopycomplete:
                        if self.statistics is None or (self.statistics.numSeeds + self.statistics.numOldSeeds == 0 and self.statistics.numCopies < 1):
                            if not self.torrent.scraping and self.torrent.scrapeisfresh \
                               and self.torrent.seed != self.utility.notrackeranswermarker \
                               and self.torrent.seed != self.utility.notrackermarker \
                               and int(self.torrent.seed) > 1:
                                self.forceseeding = False
                            else:
                                self.forceseeding = True
                        else:
                            self.forceseeding = False

                # Get scrape data every scrapeauto minutes
                if self.btstatus != 'finished' \
                     and (self.abcparams['scrape'] == "1" or (self.fin and self.normalseedingover and self.forceseeding)):
                    if self.statistics is not None and not self.torrent.scraping and (self.torrent.lastgetscrape == 0
                       or now - self.torrent.lastgetscrape >= self.utility.scrapeauto):
                        self.parent.getScrapeData(self.torrent, autoscraping = True)

                # Save peers
                if self.utility.autosavepeers and now - self.lastsavepeers > 600:
                    self.savePeers()
                    self.lastsavepeers = now

    def refreshIfNeeded(self, rank, field):
        if self.list.GetItem(self.listid, rank).GetText() != field:
            self.list.SetStringItem(self.listid, rank, field)

    def refreshDisplay(self):
        # Progress
        rank = self.getRankfromID(5)
        if rank != -1:
            self.refreshIfNeeded(rank, self.formatedProgress())
        # Status
        rank = self.getRankfromID(6)
        if rank != -1:
            if self.torrent.status == 'pause' or self.torrent.status == 'onhold':
                self.refreshIfNeeded(rank, self.localize(self.torrent.status))
            else:
                self.refreshIfNeeded(rank, self.localize(self.btstatus))
        # Time remaining
        rank = self.getRankfromID(8)
        if rank != -1:
            if self.fin:
                if self.torrent.uploadoption != 0 and not (self.forceseeding and self.normalseedingover):
                    self.refreshIfNeeded(rank, self.localize('etaS') + " " + self.utility.eta_value(self.torrent.seedingtimeleft))
                else:
                    self.refreshIfNeeded(rank, self.localize('etaS') + "      ?")
            elif self.torrent.isWaiting() or self.torrent.isCheckingOrAllocating():
                if self.torrent.complete:
                    self.refreshIfNeeded(rank, self.localize('etaS') + "      ?")
                else:
                    self.refreshIfNeeded(rank, self.localize('etaD') + "      ?")
            else:
                self.refreshIfNeeded(rank, self.localize('etaD') + " " + self.utility.eta_value(self.workingtimeleft))
        # DL speed
        rank = self.getRankfromID(10)
        if rank != -1:
            #self.refreshIfNeeded(rank, self.downloadrate[0] + '/%.1f/%.1f' % (self.meandownrate, self.torrent.maxdownloadrate))
            if not self.torrent.complete:
                self.refreshIfNeeded(rank, self.downloadrate[0])
        # UL speed
        rank = self.getRankfromID(11)
        if rank != -1:
            #self.refreshIfNeeded(rank, self.uploadrate[0] + '/%.1f/%.1f' % (self.meanuprate, self.torrent.maxuploadrate))
            self.refreshIfNeeded(rank, self.uploadrate[0])
        # # seeds
        rank = self.getRankfromID(14)
        if rank != -1:
            self.refreshIfNeeded(rank, self.numseedtext)
        # # peers
        rank = self.getRankfromID(15)
        if rank != -1:
            self.refreshIfNeeded(rank, self.numpeertext)
        if self.statistics is not None:
            # Share ratio
            rank = self.getRankfromID(12)
            if rank != -1:
                self.refreshIfNeeded(rank, self.shareratiotext)
            # # copies
            rank = self.getRankfromID(16)
            if rank != -1:
                self.refreshIfNeeded(rank, self.torrent.numcopy)
            # Peer avg progress
            rank = self.getRankfromID(17)
            if rank != -1:
                self.refreshIfNeeded(rank, self.peeravgtext)
            # Download size
            rank = self.getRankfromID(18)
            if rank != -1:
                self.refreshIfNeeded(rank, self.dlsizetext[0])
            # Upload size
            rank = self.getRankfromID(19)
            if rank != -1:
                self.refreshIfNeeded(rank, self.ulsizetext[0])
            # Total speed
            rank = self.getRankfromID(20)
            if rank != -1:
                self.refreshIfNeeded(rank, self.totalspeedtext[0])
        if self.fin:
            # Seeding time
            rank = self.getRankfromID(27)
            if rank != -1:
                self.refreshIfNeeded(rank, self.utility.time_value(self.torrent.seedingtime, seconds = False, maxvalue = False))
            # Total seeding time
            rank = self.getRankfromID(28)
            if rank != -1:
                self.refreshIfNeeded(rank, self.utility.time_value(self.torrent.totalseedingtime, seconds = False, maxvalue = False))

    def updateDetailWindow(self):
        if self.detailwin is None:
            # Only compute connections activity
            numdownloads = numuploads = numdeaddown = 0
            if self.spew is not None:
                for peer in self.spew:
                    if peer['uprate'] > 100:
                        numuploads += 1
                    if peer['downrate'] > 100:
                        numdownloads += 1
                    elif not self.torrent.complete and peer['dinterested'] \
                       and not peer['dchoked']:
                        numdeaddown += 1
            self.numdownloads = numdownloads
            self.numuploads = numuploads
            self.numdeaddown = numdeaddown
            return

        self.detailwin.totalprogress.SetLabel(self.formatedProgress())
        if self.torrent.status == 'pause' or self.torrent.status == 'onhold':
            self.detailwin.status.SetLabel(self.localize(self.torrent.status))
            self.detailwin.colour.SetBackgroundColour(self.utility.colnoeng)
        else:
            self.detailwin.status.SetLabel(self.localize(self.btstatus))
            self.detailwin.colour.SetBackgroundColour(self.colour)
        self.detailwin.colour.Refresh()
        self.detailwin.trackercolour.SetBackgroundColour(self.trackercolour)
        self.detailwin.trackercolour.Refresh()

        self.detailwin.downsize.SetLabel(self.utility.sizeWithUnit(self.dlsizetext))
        self.detailwin.upsize.SetLabel(self.utility.sizeWithUnit(self.ulsizetext))
        if self.fin:
            self.detailwin.seedtitle.SetLabel(self.localize('dseenseed'))
        else:
            self.detailwin.seedtitle.SetLabel(self.localize('dnumconnectedseed'))
        self.detailwin.numseed.SetLabel(self.numseedtext)
        self.detailwin.numpeer.SetLabel(self.numpeertext)
        self.detailwin.numcopy.SetLabel(self.torrent.numcopy)
        self.detailwin.totalspeed.SetLabel(self.utility.rateWithUnit(self.totalspeedtext))
        self.detailwin.downrate.SetLabel(self.utility.rateWithUnit(self.downloadrate, self.torrent.prioritizedown))
        self.detailwin.uprate.SetLabel(self.utility.rateWithUnit(self.uploadrate, self.torrent.prioritizeup))
        self.detailwin.avgprogress.SetLabel(self.peeravgtext)

        if self.spew is None:
            self.numdownloads = 0
            self.numuploads = 0
            self.numdeaddown = 0
            spewlist = self.detailwin.spewlist
            for i in xrange(spewlist.GetItemCount() - 3, 0, -1):
                spewlist.DeleteItem(i)
        elif clock() - self.spewwait > 1:
            self.updateSpewList()

        if self.torrent.magnet:
            self.detailwin.storagestats1.SetLabel(self.localize('downloadingmetadata'))
            self.detailwin.storagestats2.SetLabel("")
        elif self.initializingfiles or self.statistics is None:
            self.detailwin.storagestats1.SetLabel("")
            self.detailwin.storagestats2.SetLabel("")
        else:
            if self.fin:
                if self.torrent.unwantedfiles:
                    self.detailwin.storagestats1.SetLabel(self.localize('seedingpartial'))
                else:
                    self.detailwin.storagestats1.SetLabel(self.localize('seeding'))
            else:
                self.detailwin.storagestats1.SetLabel(self.localize('detailline1')
                                                      % (self.statistics.storage_active, self.statistics.storage_new,
                                                         self.statistics.storage_dirty))
            self.detailwin.storagestats2.SetLabel(self.localize('detailline2')
                                                  % (self.statistics.storage_numcomplete, self.statistics.storage_totalpieces,
                                                     self.statistics.storage_justdownloaded, self.statistics.storage_numflunked))

    def updateSpewList(self):
        self.spewwait = clock()
        spewlist = self.detailwin.spewlist
        spewlen = len(self.spew) + 2
        if self.utility.advdetdbuf:
            spewlen += 1
        self.detailwin.spew = self.spew
        self.detailwin.kicked = self.kicked
        self.detailwin.banned = self.banned

        kickbanlen = len(self.kicked) + len(self.banned)
        if kickbanlen:
            spewlen += kickbanlen + 1

        for x in xrange(spewlist.GetItemCount(), spewlen):
            spewlist.InsertStringItem(x, '')
        for x in xrange(spewlist.GetItemCount() - 1, spewlen - 1, -1):
            spewlist.DeleteItem(x)

        numdownloads = numuploads = numdeaddown = 0
        if self.utility.advdetdbuf:
            row = 1
        else:
            row = 0
        for peer in self.spew:
            spewlist.SetStringItem(row, 0, self.utility.starflag[peer['optimistic']])
            spewlist.SetStringItem(row, 1, peer['dns'][0])
            spewlist.SetStringItem(row, 2, peer['direction'])

            downrate = peer['downrate']
            if downrate > 100:
                spewlist.SetStringItem(row, 3, ('%.0f ' % (downrate / 1024)) + self.utility.kbps)
                numdownloads += 1
            else:
                spewlist.SetStringItem(row, 3, ' ')
                if not self.torrent.complete and peer['dinterested'] and not peer['dchoked']:
                    numdeaddown += 1

            spewlist.SetStringItem(row, 4, self.utility.starflag[peer['dinterested']])
            spewlist.SetStringItem(row, 5, self.utility.starflag[peer['dchoked']])
            spewlist.SetStringItem(row, 6, self.utility.starflag[peer['snubbed']])

            uprate = peer['uprate']
            if uprate > 100:
                spewlist.SetStringItem(row, 7, '%.0f ' % (uprate / 1024) + self.utility.kbps)
                numuploads += 1
            else:
                spewlist.SetStringItem(row, 7, ' ')

            spewlist.SetStringItem(row, 8, self.utility.starflag[peer['uinterested']])
            spewlist.SetStringItem(row, 9, self.utility.starflag[peer['uchoked']])

            spewlist.SetStringItem(row, 10, ('%.2f ' % (float(peer['dtotal']) / 1048576)) + self.utility.mb)

            if peer['utotal'] is None:
                spewlist.SetStringItem(row, 11, '')
            else:
                spewlist.SetStringItem(row, 11, '%.2f ' % (float(peer['utotal']) / 1048576) + self.utility.mb)

            if peer['completed'] is None:
                spewlist.SetStringItem(row, 12, '?')
            else:
                spewlist.SetStringItem(row, 12, str(.1 * int(1000 * peer['completed'])) + '%')

            if peer['speed'] is None:
                spewlist.SetStringItem(row, 13, '?')
            else:
                spewlist.SetStringItem(row, 13, ('%.0f ' % (peer['speed'] / 1024)) + self.utility.kbps)

            row += 1

        self.numdownloads = numdownloads
        self.numuploads = numuploads
        self.numdeaddown = numdeaddown

        for i in xrange(14):
            spewlist.SetStringItem(row, i, '')

        row += 1
        for i in [0, 2, 3, 4, 5, 6, 7, 8, 9, 12, 13]:
            spewlist.SetStringItem(row, i, '')
        spewlist.SetStringItem(row, 1, self.localize('TOTALS'))

        if self.statistics is not None:
            spewlist.SetStringItem(row, 10, ('%.2f ' % (float(self.statistics.downTotal) / 1048576)) + self.utility.mb)
            spewlist.SetStringItem(row, 11, ('%.2f ' % (float(self.statistics.upTotal) / 1048576)) + self.utility.mb)
        else:
            spewlist.SetStringItem(row, 10, '')
            spewlist.SetStringItem(row, 11, '')

        if kickbanlen:
            row += 1
            for i in xrange(14):
                spewlist.SetStringItem(row, i, '')
            for ip in self.kicked:
                row += 1
                spewlist.SetStringItem(row, 1, ip)
                spewlist.SetStringItem(row, 3, self.localize('KICKED'))
                for i in [0, 2, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]:
                    spewlist.SetStringItem(row, i, '')
            for ip in self.banned:
                row += 1
                spewlist.SetStringItem(row, 1, ip)
                spewlist.SetStringItem(row, 3, self.localize('BANNED'))
                for i in [0, 2, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13]:
                    spewlist.SetStringItem(row, i, '')

    def reportProgress(self, statistics = None):
        if statistics is None:
            statistics = self.statistics
        if statistics is not None and statistics.multifile:
            if self.torrent.filesprogress is None:
                self.torrent.filesprogress = self.torrent.filesnumber * ['']
            for i in xrange(self.torrent.filesnumber):
                if self.torrent.filespriority is None or self.torrent.filespriority[i] >= 0:
                    if statistics.fileinplace[i]:
                        self.torrent.filesprogress[i] = 'Done'
                    elif statistics.filecomplete[i]:
                        self.torrent.filesprogress[i] = '100.0'
                    else:
                        self.torrent.filesprogress[i] = str(max(statistics.fileamtdone[i] * 100, 0.))
                else:
                    self.torrent.filesprogress[i] = '0.0'
        else:
            self.torrent.filesprogress = None

    def updateInfoWindow(self):
        if self.infowin is None: 
            return

        self.infowin.totalprogress.SetLabel(self.formatedProgress())
        if self.torrent.status == 'pause' or self.torrent.status == 'onhold':
            self.infowin.status.SetLabel(self.localize(self.torrent.status))
            self.infowin.colour.SetBackgroundColour(self.utility.colnoeng)
        else:
            self.infowin.status.SetLabel(self.localize(self.btstatus))
            self.infowin.colour.SetBackgroundColour(self.colour)
        self.infowin.colour.Refresh()

        self.infowin.downrate.SetLabel(self.utility.rateWithUnit(self.downloadrate, self.torrent.prioritizedown))
        self.infowin.uprate.SetLabel(self.utility.rateWithUnit(self.uploadrate, self.torrent.prioritizeup))

        if self.statistics is not None and self.statistics.multifile \
           and self.statistics.filelistupdated.isSet():
            progresscol = 3 + self.infowin.originalfilenamesshown
            for i in xrange(self.torrent.filesnumber):
                if self.torrent.filespriority is None or self.torrent.filespriority[i] >= 0:
                    if self.statistics.fileinplace[i]:
                        progress = self.localize('done')
                    elif self.statistics.filecomplete[i]:
                        progress = "100%"
                    else:
                        progress = '%d%%' % max(0., self.statistics.fileamtdone[i] * 100)
                else:
                    progress = "0%"
                if self.infowin.filelist.GetItem(i, progresscol).GetText() != progress:
                    self.infowin.filelist.SetStringItem(i, progresscol, progress)
            self.statistics.filelistupdated.clear()

    def getDownUpSize(self):
        return self.olddownsize + self.newdownsize, self.oldupsize + self.newupsize

    def getTimeoutTimes(self, t):
        if self.firsterrortracker == 0:
            trackererrortime = 0.
        else:
            trackererrortime = t - self.firsterrortracker
        return (t - self.lasttorrentactive, trackererrortime)

    def getInfo(self, fieldidlist):
        retmsg = ""
        for field in fieldidlist:
            if field == 4:
                retmsg += self.torrent.name + "|"
            elif field == 5:
                retmsg += self.formatedProgress() + "|"
            elif field == 6:
                if self.torrent.ismovingdata:
                    retmsg += self.localize('moving') + "|"
                else:
                    retmsg += self.localize(self.btstatus) + "|"
            elif field == 7:
                retmsg += self.utility.priorities_s[self.torrent.priority] + "|"
            elif field == 8:
                if self.fin:
                    if self.torrent.uploadoption == 0:
                        retmsg += self.localize('etaS') + '     oo|'
                    elif self.forceseeding and self.normalseedingover:
                        retmsg += self.localize('etaS') + " " + self.localize('tillcc') + "|"
                    else:
                        retmsg += self.localize('etaS') + " " + self.utility.eta_value(self.torrent.seedingtimeleft) + "|"
                elif self.torrent.isWaiting() or self.torrent.isCheckingOrAllocating():
                    if self.torrent.complete:
                        retmsg += self.localize('etaS') + "      ?|"
                    else:
                        retmsg += self.localize('etaD') + "      ?|"
                else:
                    retmsg += self.localize('etaD') + " " + self.utility.eta_value(self.workingtimeleft) + "|"
            elif field == 9:
                if self.torrent.magnet:
                    retmsg += "?|"
                else:
                    retmsg += self.utility.sizeWithUnit(self.utility.formatedSize(self.torrent.requestedsize)) + "|"
            elif field == 10:
                retmsg += self.utility.formatedRate(self.downrate)[0] + "|"
            elif field == 11:
                retmsg += self.utility.formatedRate(self.uprate)[0] + "|"
            elif field == 12:
                retmsg += self.shareratiotext + "|"
            elif field == 13:
                retmsg += self.torrent.message + "|"
            elif field == 14:
                retmsg += self.numseedtext + "|"
            elif field == 15:
                retmsg += self.numpeertext + "|"
            elif field == 16:
                retmsg += self.torrent.numcopy + "|"
            elif field == 17:
                retmsg += self.peeravgtext + "|"
            elif field == 18:
                retmsg += self.utility.sizeWithUnit(self.dlsizetext) + "|"
            elif field == 19:
                retmsg += self.utility.sizeWithUnit(self.ulsizetext) + "|"
            elif field == 20:
                retmsg += self.totalspeedtext[0] + "|"
            elif field == 21:
                retmsg += path.split(self.torrent.src)[1] + "|"
            elif field == 22:
                if self.torrent.uploadoption == 0:
                    retmsg += "oo|"
                elif self.torrent.uploadoption == 1:
                    retmsg += self.utility.time_value(self.torrent.uploadtimemax, seconds = False, maxvalue = False) + "|"
                elif self.torrent.uploadoption == 2:
                    retmsg += str(self.torrent.uploadratio) + "%|"
                else:
                    retmsg += self.utility.time_value(self.torrent.uploadtimemax, seconds = False, maxvalue = False) + \
                              str(self.torrent.uploadratio) + "%|"
            elif field == 23:
                retmsg += self.torrent.shortlabel + "|"
            elif field == 24:
                retmsg += self.torrent.label + "|"
            elif field == 25:
                retmsg += self.torrent.activityMarker() + "|"
            elif field == 26:
                retmsg += self.torrent.getRebuiltDest() + "|"
            elif field == 27:
                retmsg += self.utility.time_value(self.torrent.seedingtime, maxvalue = False) + "|"
            elif field == 28:
                retmsg += self.utility.time_value(self.torrent.totalseedingtime, maxvalue = False) + "|"
        return retmsg

    def completed(self):
        self.btcomplete = True
        self.invokeLater(self.onCompletedEvent)

    def uncompleted(self):
        self.invokeLater(self.onUncompletedEvent)

    def failed(self):
        if self.dow.storage:
            for handle in self.dow.storage.handles.values():
                try:
                    handle.close()
                except:
                    continue
        self.invokeLater(self.onFailEvent, forcepost = True)

    def handleLimitedTorrent(self):
        self.invokeLater(self.onLimitedTorrent)

    def error(self, errormsg, rejectedonfirstreqcycle = False, extra = False, warning = False):
        self.invokeLater(self.onError, [errormsg, rejectedonfirstreqcycle, extra, warning], forcepost = True)

    def trackerOK(self, connected, tracker, extra = False):
        self.invokeLater(self.onTrackerOK, [connected, tracker, extra])

    def trackerConnecting(self):
        self.invokeLater(self.onTrackerConnecting)

    def onError(self, errormsg, rejectedonfirstreqcycle, extra, warning):
        # extra : An extra URL is used in place of the internal tracker or list of trackers (the tracker is external or extra internal)
        now = clock()
        trackercnxerror = self.utility.messageIsTrackerCnxError(errormsg)
        # Display error message
        if warning:
            if self.abcparams['displaytrackcnxwng'] == "1":
                self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + errormsg)
            else:
                self.torrent.clearMessage(oldmsg = self.localize('trackerconnecting'))
        elif trackercnxerror:
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.utility.trackercat[extra] + errormsg)
        else:
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + errormsg)
        if trackercnxerror:
            self.trackercolour = self.utility.colnocon
            if self.detailwin is not None:
                self.detailwin.trackercolour.SetBackgroundColour(self.trackercolour)
                self.detailwin.trackercolour.Refresh()

        if warning:
            # Warning messages don't trigger timeout
            return

        # If DHT is on and the torrent is not private, all tracker timeout features are disabled
        trackertimeoutcanceled = (self.dht and not self.torrent.private)

        # If an automatically started torrent gets rejected by all trackers on first request cycle and option set, stop it
        # just after the next update (to get the global progress and the progress for each file for a checking torrent)
        if not trackertimeoutcanceled and self.auto and self.abcparams['stoprejected'] == "1" and rejectedonfirstreqcycle:
            self.stopafterfirstrefresh = True

        # Check if all timeouts must be canceled because the tracker message contains
        # one of the strings specified in Preferences/Queue
        if self.abcparams['canceltimeouts'] == "1":
            for s in self.parent.canceltimeoutsmsg:
                if errormsg.find(s) >= 0:
                    self.firsterrortracker = 0
                    self.lasttorrentactive = now
                    self.timeoutscanceled = True
                    break
            else:
                self.timeoutscanceled = False

        # If failed connecting tracker in abcparams['timeouttracker'] mins
        # reduce its priority and force to queue
        if trackercnxerror:
            if self.torrent.multitracker and not extra:
                self.torrent.setDefaultTracker('')
            if not self.timeoutscanceled and not trackertimeoutcanceled:
                if self.firsterrortracker == 0:
                    self.firsterrortracker = now
                elif self.uprate == 0 and self.downrate == 0 \
                     and self.torrent.timeoutswitch \
                     and (self.fin and self.torrent.timeoutseed and self.torrent.timeoutseedtrack \
                     or not self.fin and self.torrent.timeoutwork and self.torrent.timeoutworktrack) \
                     and now - self.firsterrortracker > self.torrent.timeouttracker * 60 \
                     and self.parent.isTimeoutAllowed():
                    self.firsterrortracker = 0
                    self.torrent.trackererrortime = 0.
                    self.onTimeout(self.localize('timeouttrackermsg'))

    def onTimeout(self, msg):
        if self.abcparams['displaytimeout'] == '1':
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + msg)
        if (self.torrent.timeoutaction == 1  or self.torrent.timeoutaction == 4 and not self.fin) and self.torrent.priority == 4 \
           or self.torrent.timeoutaction == 2 or self.torrent.timeoutaction == 5 and not self.fin:
            self.parent.procSTOP([self.listid])
            return True
        if self.torrent.timeoutaction == 0 or self.torrent.timeoutaction == 3 and not self.fin:
            if self.abcparams['sbqueued'] == '0':
                self.parent.procSTANDBY([self.listid])
                return True
            if self.parent.isNextToBeResumed(self.listid, self.torrent.priority + 1):
                self.torrent.incDecPriority(-1)
                return False
            self.parent.procSTANDBY([self.listid])
            return True
        if self.torrent.timeoutaction == 1 or self.torrent.timeoutaction == 4 and not self.fin:
            self.torrent.incDecPriority(-1)
            if self.parent.isNextToBeResumed(self.listid, self.torrent.priority):
                return False
            self.parent.procQUEUE([self.listid])
            return True
        if self.torrent.timeoutaction >= 3:
            self.parent.procFINISHED(self.listid, self.onlycheck)
            return True

    def onTrackerOK(self, connected, tracker, extra):
        if connected :
            self.trackercolour = self.utility.colok
            if self.torrent.multitracker and not extra:
                self.torrent.setDefaultTracker(tracker)
        else:
            self.trackercolour = self.utility.colnocon
        if self.detailwin is not None:
            self.detailwin.trackercolour.SetBackgroundColour(self.trackercolour)
            self.detailwin.trackercolour.Refresh()
        self.firsterrortracker = 0
        self.timeoutscanceled = False
        if self.torrent.message: 
            if self.abcparams['erasetrackal'] == '1' \
               and self.utility.messageIsTrackerCnxError(self.torrent.message[self.utility.timestamplen + self.utility.messageheaderlen + self.utility.trackercatlen:]):
                self.torrent.clearMessage()
            else:
                self.torrent.clearMessage(oldmsg = self.localize('trackerconnecting'))

    def onTrackerConnecting(self):
        if self.abcparams['displaytrackcnxmsg'] == '1':
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('trackerconnecting'))
        self.trackercolour = self.utility.coldisc
        if self.detailwin is not None:
            self.detailwin.trackercolour.SetBackgroundColour(self.trackercolour)
            self.detailwin.trackercolour.Refresh()

    def countWorkingTimeLeft(self, t, sizewanted, sizedone):
        # Working time left
        if self.meandownrate > 0:
            workingtimeleft = max(self.workingtimeleft - t + self.workingtimelastcheck, 0)
            if sizewanted is None:
                newworkingtimeleft = 0.
            else:
                newworkingtimeleft = (sizewanted - sizedone) / (self.meandownrate * 1024)
            if newworkingtimeleft > 86400:
                delta = newworkingtimeleft / 20
            elif newworkingtimeleft > 60:
                delta = newworkingtimeleft / 10
            elif newworkingtimeleft > 10:
                delta = 2
            else:
                delta = 1
            if abs(workingtimeleft - newworkingtimeleft) > delta:
                # If timer value deviates too much from theoretical value, reset it to theoretical value
                workingtimeleft = newworkingtimeleft
        else:
            # Set to 100 days (= infinite)
            workingtimeleft = 8640000
        self.workingtimeleft = workingtimeleft
        self.workingtimelastcheck = t

    def countSeedingTimes(self, t):
        # Seeding times
        if self.seedingtimer:
            self.torrent.seedingtime = self.getCurrentSeedingTime(t)
        if self.totalseedingtimer:
            self.torrent.totalseedingtime = self.getCurrentTotalSeedingTime(t)
        self.countSeedingTimeLeft(t)

    def countSeedingTimeLeft(self, t):
        # Seeding time left
        if self.torrent.uploadoption == 1 or self.torrent.uploadoption == 3:
            timeshift = self.torrent.uploadtimemax - self.torrent.seedingtime
            # Correct seeding times shifting because of check rate
            if timeshift > 0:
                self.torrent.olduploadtimemax = self.torrent.uploadtimemax
            elif self.torrent.uploadtimemax >= self.torrent.olduploadtimemax:
                self.torrent.seedingtime = self.torrent.uploadtimemax
                self.torrent.totalseedingtime += timeshift
            seedingtimeleft_time = max(timeshift, 0)
        if self.torrent.uploadoption == 2 or self.torrent.uploadoption == 3:
            if self.torrent.ratioIsReached():
                seedingtimeleft_ratio = 0
            else:
                if self.meanuprate > 0:
                    seedingtimeleft = max(self.torrent.seedingtimeleft - t + self.seedingtimelastcheck, 0)
                    down, up = self.getDownUpSize()
                    if down <= 0:
                        down = self.torrent.requestedsize
                    newseedingtimeleft = max((self.torrent.uploadratio * down / 100 - up) / (self.meanuprate * 1024), 0)
                    if newseedingtimeleft > 86400:
                        delta = newseedingtimeleft / 20
                    elif newseedingtimeleft > 60:
                        delta = newseedingtimeleft / 10
                    elif newseedingtimeleft > 10:
                        delta = 2
                    else:
                        delta = 1
                    if abs(seedingtimeleft - newseedingtimeleft) > delta:
                        # If timer value deviates too much from theoretical value, reset it to theoretical value
                        seedingtimeleft = newseedingtimeleft
                else:
                    # Set to 100 days (= infinite)
                    seedingtimeleft = 8640000
                seedingtimeleft_ratio = seedingtimeleft
        if self.torrent.uploadoption == 1:
            self.torrent.seedingtimeleft = seedingtimeleft_time
        elif self.torrent.uploadoption == 2:
            self.torrent.seedingtimeleft = seedingtimeleft_ratio
        elif self.torrent.uploadoption == 3:
            self.torrent.seedingtimeleft = min(seedingtimeleft_time, seedingtimeleft_ratio)
        self.seedingtimelastcheck = t

    def startSeedingTimers(self, resettotalseeding = True):
        now = clock()
        if self.utility.seedingtimecount == "0" or self.torrent.uploadoption == 1 \
           or self.torrent.uploadoption == 3:
            self.startSeedingTimer(now)
        if resettotalseeding:
            self.startTotalSeedingTimer(now)

    def startSeedingTimer(self, t):
        self.seedingtimer = 1
        self.seedingtimerreset = self.seedingtimelastcheck = t
        self.lastseedingtime = self.torrent.seedingtime

    def startTotalSeedingTimer(self, t):
        self.totalseedingtimer = 1
        self.totalseedingtimerreset = t
        self.lasttotalseedingtime = self.torrent.totalseedingtime

    def stopSeedingTimers(self, resettotalseeding = True):
        now = clock()
        if self.seedingtimer:
            self.stopSeedingTimer(now)
        if self.totalseedingtimer and resettotalseeding:
            self.stopTotalSeedingTimer(now)

    def stopSeedingTimer(self, t):
        self.torrent.seedingtime = self.getCurrentSeedingTime(t)
        self.seedingtimer = 0

    def stopTotalSeedingTimer(self, t):
        self.torrent.totalseedingtime = self.getCurrentTotalSeedingTime(t)
        self.totalseedingtimer = 0

    def getCurrentSeedingTime(self, t):
        return self.lastseedingtime + t - self.seedingtimerreset

    def getCurrentTotalSeedingTime(self, t):
        return self.lasttotalseedingtime + t - self.totalseedingtimerreset

    def initForceSeeding(self):
        self.normalseedingover = True
        if not self.torrent.scraping:
            self.torrent.scrapeisfresh = False
            self.torrent.lastgetscrape = 0
        etarank = self.getRankfromID(8)
        if etarank != -1:
            self.list.SetStringItem(self.listid, etarank, self.localize('etaS') + " " + self.localize('tillcc'))

    def checkForTerminating(self, queueback = False):
        if self.forceseeding:
            if queueback:
                # Torrent is limited and must not seed
                self.onLimitedTorrent()
            else:
                self.initForceSeeding()
        else:
            self.terminateUpload()

    def terminateUpload(self):
        ####################################################
        # change :  5:Progress 6:BT Status 13:Message
        # untouch : 4:Title 7:Priority 9:Size 12:%U/DSize
        #           18:DownloadSize 19:UploadSize 21:torrentname
        #           22:seeding option
        # clear :   8:ETA 10:DLSpeed 11:ULspeed
        #           14:#seed 15:#peer 16:#copies 17:peer avg
        #           20:total speed
        #####################################################
        self.btstatus = 'finished'
        self.progress = 100.
        self.torrent.seedingtimeleft = 0
        self.colour = self.utility.colnoeng
        self.updateColour()

        self.parent.procFINISHED(self.listid, self.onlycheck)

    def onCompletedEvent(self):
        ####################################################
        # change :  5:Progress 6:BT Status 8:ETA 12:%U/DSize
        #           13:Message
        # untouch : 4:Title 7:Priority 9:Size
        #           18:DownloadSize 19:UploadSize 11:ULspeed
        #           14:#seed 15:#peer 16:#copies 17:peer avg
        #           20:total speed 21:torrentname
        #           22:seeding option
        # clear :   10:DLSpeed  
        #####################################################
        if self.paused:
            self.skippedcompleted = True
            return
        if self.overmaxupwithoutdown:
            self.dow.setConns(self.torrent.maxupload)
            self.overmaxupwithoutdown = False
        self.overmaxupwithoutdown2 = False
        if self.networkdisconnected > 1:
            if self.networkdisconnected == 2:
                self.torrent.switchTracker(1)
            self.networkdisconnected = 1
        if self.checkinginttrack:
            self.torrent.switchTracker(1)
        self.fin = True
        self.btstatus = 'completedseeding'
        self.progress = 100.
        self.normalseedingover = False

        rank = self.getRankfromID(5)
        if rank != -1:
            self.list.SetStringItem(self.listid, rank, self.formatedProgress())
        if not self.onlycheck:
            rank = self.getRankfromID(8)
            if rank != -1:
                if self.torrent.uploadoption == 0:
                    self.list.SetStringItem(self.listid, rank, self.localize('etaS') + "     oo")
                elif self.torrent.uploadoption == 2:
                    self.list.SetStringItem(self.listid, rank, self.localize('etaS') + "      ?")
                else:
                    self.list.SetStringItem(self.listid, rank, self.localize('etaS') + " " + self.utility.eta_value(self.torrent.seedingtimeleft))
            rank = self.getRankfromID(10)
            if rank != -1:
                self.list.SetStringItem(self.listid, rank, self.utility.formatedRate(None, self.torrent.prioritizedown)[0])
        rank = self.getRankfromID(12)
        if rank != -1:
            self.list.SetStringItem(self.listid, rank, self.shareratiotext)

        # Update detail windows
        if self.infowin:
            if not self.onlycheck:
                self.infowin.downrate.SetLabel(self.utility.rateWithUnit('', self.torrent.prioritizedown))
            if not self.torrent.singlefile:
                if self.torrent.unwantedfiles:
                    self.infowin.refreshFileProgress()
                else:
                    # Force display of progress of all files to Done
                    donestr = self.localize('done')
                    progresscol = 3 + self.infowin.originalfilenamesshown
                    for x in xrange(self.torrent.filesnumber):
                        self.infowin.filelist.SetStringItem(x, progresscol, donestr)
            self.infowin.totalprogress.SetLabel(self.formatedProgress())
        if self.detailwin:
            if not self.onlycheck:
                self.detailwin.downrate.SetLabel(self.utility.rateWithUnit('', self.torrent.prioritizedown))
            self.detailwin.totalprogress.SetLabel(self.formatedProgress())

        if self.parent.procCOMPLETED(self.listid, self.onlycheck, self.sbwaitingforleechers, self.auto) \
           and self.torrent.status != 'finished' and self.torrent.status != 'queue' and not self.onlycheck:
            # Display seeding status
            rank = self.getRankfromID(6)
            if rank != -1:
                self.list.SetStringItem(self.listid, rank, self.localize(self.btstatus))
            if self.infowin:
                self.infowin.status.SetLabel(self.localize(self.btstatus))
            if self.detailwin:
                self.detailwin.status.SetLabel(self.localize(self.btstatus))
            # Start seeding
            self.startSeedingTimers()

    def onUncompletedEvent(self):
        self.parent.procUNCOMPLETED(self.listid, self.sbwaitingforleechers, self.onlycheck, self.auto)

    def onFailEvent(self):
        self.parent.procSTOP([self.listid], torrentfailed = True)

    def onLimitedTorrent(self):
        if self.sbwaitingforleechers:
            self.parent.procSTANDBY([self.listid])
        else:
            self.parent.procQUEUE([self.listid])

    def chooseFile(self, saveas):
        if saveas != '':       
            if self.torrent.requestedsize <= 0 and not self.torrent.magnet:
                self.shareratio = 999.9
            elif self.progress <= 0:
                self.shareratio = 0.
            else:
                self.shareratio = float(self.oldupsize) / self.torrent.requestedsize / self.progress * 10000
            self.shareratiotext = str(.1 * int(10 * self.shareratio)) + "%"
            return self.utility.completePath(saveas)
        return None

    def start(self):
        self.torrent.displayStatus('init')
        self.fastresume = self.fastresume and (self.torrent.checkedonce or int(self.abcparams['fastresumefirst']))
        if self.parent.networkdisconnection >= 0:
            self.networkdisconnected = True

        switchbacktoext = False
        if not self.onlycheck:
            if self.torrent.exttracker and self.tracker == 1:
                self.torrent.switchTracker(0)
            else:
                mustcheckinternal = (self.tracker == 0 and self.utility.checkinttrack == 1
                                     or self.tracker == 4
                                     or self.tracker == 6
                                     or self.tracker == 8
                                     or (self.tracker == 2 or self.tracker == 7 or self.tracker == 9
                                         or self.tracker == 0 and self.utility.checkinttrack == 2)
                                     and (self.torrent.private or not self.dht))
                if mustcheckinternal:
                    mustcheckinternalandwait = (self.tracker != 0 or not self.torrent.complete) \
                                               and (self.torrent.checkinttrackwait and self.tracker != 8 and self.tracker != 9
                                                    or self.tracker == 6 or self.tracker == 7)

                if self.torrent.exttracker:
                    if mustcheckinternal:
                        self.checkinginttrack = True
                        if not mustcheckinternalandwait:
                            switchbacktoext = True
                        self.torrent.switchTracker(0)
                elif self.tracker >= 2:
                    self.torrent.switchTracker(1)
                    if mustcheckinternal:
                        self.checkinginttrack = True
                        if not mustcheckinternalandwait:
                            switchbacktoext = True
                        self.torrent.switchTracker(0)

            self.torrent.setActivity()

        rawserver = self.initBT()
        if rawserver:
            self.startqueue.append(self.torrent.infohash)
            thread = Thread(target = self.startBT, args = [rawserver, switchbacktoext])
            thread.daemon = False
            thread.start()
        else:
            self.shutdownflag.set()
            self.onFailEvent()

    def initBT(self):
        try:
            config = parse_params(self.btparams, {})
        except ValueError, e:
            self.error(self.localize('error') + ' : ' + exceptionArgsToString(e) + self.localize('invalidbtconfig'))
            return
        if not config:
            self.error(self.localize('error') + ' : ' + self.localize('invalidbtconfig'))
            return

        myid = createPeerID()
        wakeupflag = Event()

        rawserver = RawServer(self.doneflag, config['timeout_check_interval'],
                              config['timeout'], ipv6_enable = config['ipv6_enabled'],
                              failfunc = self.error, errorfunc = self.error,
                              readsize = 524288, wakeupflag = wakeupflag)

        if self.onlycheck:
            upnp_type = listen_port = None
        else:
            upnp_type = UPnP_test(config['upnp_nat_access'])
            if self.pl:
                listen_port = self.utility.plport
            elif self.pl is None:
                try:
                    listen_port = rawserver.find_and_bind(config['minport'], config['maxport'], config['bind'],
                                                          ipv6_socket_style = config['ipv6_binds_v4'], upnp = upnp_type,
                                                          randomizer = config['random_port'])
                except socketerror, e:
                    self.error(self.localize('cantlisten') + " - " + exceptionArgsToString(e))
                    return
            else:
                listen_port = None

        response = get_response(config['responsefile'], self.error)
        if not response:
            return

        if self.torrent.exttracker and not self.checkinginttrack:
            extannounce = self.torrent.exttrackerurl
        elif self.torrent.extrainttracker:
            extannounce = self.torrent.inttrackerurl
        else:
            extannounce = None

        self.dow = BT1Download(self.updateStatus, self.completed, self.error, self.doneflag, wakeupflag,
                               config, response, self.torrent.infohash, myid, rawserver, listen_port,
                               self.fastresume, self.dht, self.pl, self.startqueue,
                               extann = extannounce,
                               trackerokfunc = self.trackerOK,
                               trackerconnectingfunc = self.trackerConnecting,
                               endallocfunc = self.endAllocation,
                               rejectexceptions = self.parent.rejectexceptions,
                               filesnamefunc = self.torrent.getFileNames,
                               metadatacompletefunc = self.metadataComplete,
                               reseed = self.reseed,
                               checkfreediskspacefunc = self.checkFreeDiskSpace,
                               updateseenpeersfunc = self.updateSeenPeers,
                               displayongoingfunc = self.torrent.displayOnGoing)

        return rawserver

    def startBT(self, rawserver, switchbacktoext):
        limitedtorrent = False
        if self.onlycheck:
            checkover = False

        try:
            while True:
                if not self.dow.saveAs(self.chooseFile):
                    self.startqueue.remove(self.torrent.infohash)
                    break
                if not self.dow.initFiles(FILESEM, CHECKWAITING, self.readwaiting, self.onlycheck):
                    self.dow.shutdown()
                    break
                if not self.torrent.magnet:
                    self.torrent.checkedonce = True

                now = clock()
                if self.utility.resettimeoutsonstart:
                    self.lasttorrentactive = now
                    self.firsterrortracker = 0
                else:
                    self.lasttorrentactive = now - self.torrent.inactivitytime
                    if self.torrent.trackererrortime == 0:
                        self.firsterrortracker = 0
                    else:
                        self.firsterrortracker = now - self.torrent.trackererrortime

                if not self.dow.startEngine():
                    self.dow.shutdown()
                    break

                self.dow._init_stats()

                # Check real completed status after checking
                statsaftercheck = self.dow.statistics.update()
                if self.torrent.unwantedfiles:
                    if self.torrent.checkComplete(statsaftercheck):
                        self.completed()
                    elif self.torrent.complete:
                        self.uncompleted()
                elif not self.btcomplete and self.torrent.complete:
                    self.uncompleted()

                if self.onlycheck:
                    # Manual data checking
                    if not self.btcomplete:
                        # Retrieve and display checked progress
                        obtained, desired = self.dow.storagewrapper.get_stats()
                        if desired > 0:
                            self.torrent.progress = float(obtained) / desired * 100
                        else:
                            self.torrent.progress = 100.

                    self.reportProgress(statistics = statsaftercheck)

                    # Save resume data and stop torrent
                    self.dow.shutdown()

                    if not self.btcomplete:
                        self.parent.invokeLater(self.parent.displayCheckResult, [self.torrent])

                    checkover = True
                    break

                # Fix complete status if needed
                self.torrent.applyRealCompleteStatusToLimiters()
                if self.auto and (self.torrent.isOverSeedingLimit() or self.torrent.isOverDownloadingLimit()):
                    # The real complete status makes the torrent limited in connections (only for automatically started torrents)
                    self.handleLimitedTorrent()

                if self.sbwaitingforleechers:
                    self.sbwaitingforleechers = now

                # Starting torrents are not immediately added to torrentalive
                # They will be added in next run of cyclicalTasks with correct index
                # according to priority.

                self.reannouncelast = now
                self.dow.startRerequester()
                if self.dht and not self.torrent.private:
                    self.dow.startDHTRerequester()

                self.loadPeers()

                if switchbacktoext:
                    self.invokeLater(self.torrent.switchTracker, [1])

                self.dow.autoStats()

                if not self.dow.am_I_finished():
                    self.updateStatus(activity = 'connectingtopeers')

                # Set the spew flag if the detail window is shown
                if self.torrent.detailwin is not None:
                    self.dow.spewflag.set()

                self.initializingfiles = False

                if self.pl:
                    self.pl.regTorrent(self.dow)

                exception = rawserver.listen_forever(self.dow.getPortHandler())

                if self.pl:
                    self.pl.unregTorrent(self.dow)

                self.newdownsize, self.newupsize = self.dow.shutdown(exception = exception)
                break
        except:
            data = StringIO()
            print_exc(file = data)
            sys.stderr.write(data.getvalue())

        self.readwaiting.set()
        if self.onlycheck:
            if not self.stoppedonrequest:
                if checkover:
                    if not self.btcomplete:
                        self.invokeLater(self.stopEngine)
                else:
                    self.failed()
        else:
            try:
                rawserver.shutdown()
            except:
                pass
            if not self.stoppedonrequest:
                self.failed()

        self.shutdownflag.set()

    def stop(self):
        if self.utility.autosavepeers:
            self.savePeers()
        self.stopSeedingTimers()
        if not self.stoppedonrequest:
            self.stopBT()
        self.shutdownflag.wait()
        self.torrent.downsize, self.torrent.upsize = self.getDownUpSize()
        if not self.torrent.singlefile and not self.fin:
            self.reportProgress()
        self.stopEngine()
        self.torrent.forceseeding = self.normalseedingover and self.forceseeding

    def stopBT(self):
        self.stoppedonrequest = True
        self.doneflag.set()

    def stopEngine(self):
        DELEVT_INVOKE(self)
        self.dow = None
        self.torrent.abcengine = None

    def pause(self):
        if self.dow:
            if not self.onlycheck:
                now = clock()
                self.torrent.inactivitytime, self.torrent.trackererrortime = self.getTimeoutTimes(now)
                self.upwithoutdown = now - self.lastdown
            self.paused = True
            self.dow.Pause()
            if self.utility.autosavepeers:
                self.savePeers()
            self.colour = self.utility.colnoeng
            return True
        return False

    def unpause(self):
        if self.parent.networkdisconnection >= 0:
            self.networkdisconnected = True
        if self.dow:
            # To force status refresh
            self.oldbtstatus = None
            self.paused = False
            if not self.onlycheck:
                now = clock()
                self.lasttorrentactive = now - self.torrent.inactivitytime
                if self.torrent.trackererrortime == 0:
                    self.firsterrortracker = 0.
                else:
                    self.firsterrortracker = now - self.torrent.trackererrortime
                self.lastdown = now - self.upwithoutdown
            self.dow.Unpause()
            if self.skippedcompleted:
                # BT generated the completed event but it could not be processed because of pausing
                self.skippedcompleted = False
                self.invokeLater(self.onCompletedEvent)

    def updateColour(self):
        self.list.SetItemTextColour(self.listid, self.colour)

    def endAllocation(self, onrequest = False):
        if onrequest:
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                        + self.localize('allocationdone'))

    def updateSeenPeers(self, numseenseeds, numseenpeers):
        self.invokeLater(self.onUpdateSeenPeers, [numseenseeds, numseenpeers])

    def onUpdateSeenPeers(self, numseenseeds, numseenpeers):
        if not self.torrent.scraping:
            self.torrent.updateScrapeData(numseenseeds, numseenpeers, False, None)

    def metadataComplete(self, metadata):
        self.invokeLater(self.onMetadataComplete, [metadata])

    def onMetadataComplete(self, metadata):
        if sha1(metadata.viewSlice()).digest() != self.torrent.infohash:
            # Invalid metadata
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                        + self.localize('error') + ' : '
                                        + self.localize('failedinvalidtorrent_short'))
            metadata.release()
            self.dow.encoder.metadataReport(False)
            return
        self.dow.encoder.metadataReport(True)

        try:
            metadatadict = bdecode(metadata[:], sloppy = 1)
            newtorrentname = metadatadict.get('name.utf-8')
            if newtorrentname is None:
                newtorrentname = metadatadict['name']
        except:
            # Invalid metadata
            self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                        + self.localize('error') + ' : '
                                        + self.localize('failedinvalidtorrent_short'))
            metadata.release()
            self.onFailEvent()
            return

        metadata.release()

        newtorrentname = self.utility.decodeString(newtorrentname).replace('|', self.abcparams['tornamevb'])
        fixednewtorrentfile = self.utility.fixWindowsName(newtorrentname)
        if fixednewtorrentfile:
            newtorrentfile = fixednewtorrentfile
        else:
            newtorrentfile = newtorrentname

        # Build new torrent
        params = {'addclosestdhtnodes': 1, 'info': metadatadict}
        # Trackers
        announce = self.dow.response.get('announce')
        if announce is not None:
            params['real_url'] = announce
        else:
            announcelist = self.dow.response.get('announce-list')
            if announcelist is not None:
                params['real_announce_list'] = announcelist
        # New complete torrent metadata
        btmetafile = make_meta_data('', '', self.dht, params)[0]
            
        newtorrentfilepath = self.utility.findUniqueFileName(path.join(self.utility.datapath, 'torrent', newtorrentfile + '.torrent'), src = 'queue')
        if newtorrentfilepath is None:
            dfmessage = strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('error') + ' : ' + self.localize('failedwritetorrentfrommagnet')
            self.torrent.displayMessage(dfmessage)
            self.onFailEvent()
            return

        # Write new torrent file
        try:
            f = open(newtorrentfilepath, "wb")
            f.write(btmetafile)
            f.close()
        except:
            try:
                f.close()
            except:
                pass
            dfmessage = strftime(self.utility.timestamp) + self.utility.messageheader + self.localize('error') + ' : ' + self.localize('failedwritetorrentfrommagnet')
            self.torrent.displayMessage(dfmessage)
            self.onFailEvent()
            return

        # Replace torrent metadata with new ones
        self.initializingfiles = True
        oldtorrentfile = self.torrent.src
        success, mesg = self.parent.addNewProc(newtorrentfilepath, newtorrentname, False, caller = 'magnet', replace = self.listid)
        if not success:
            # Torrent could not be loaded
            self.onFailEvent()
            return

        # Metadata are valid and were successfully loaded

        # Remove old torrent file
        try:
            remove(oldtorrentfile)
        except:
            pass

        # Rename old peer file
        try:
            rename(oldtorrentfile + '.prs', newtorrentfilepath + '.prs')
        except:
            pass

        if self.utility.autosavepeers:
            self.savePeers()

        # The torrent must be stopped because manual modifications in content or destination are needed
        # or because it was started only to get the metadata
        if mesg == 'Stop torrent' or self.torrent.onlygetmetadata:
            if self.torrent.onlygetmetadata:
                self.torrent.onlygetmetadata = 0
                self.torrent.displayMessage(strftime(self.utility.timestamp) + self.utility.messageheader
                                            + self.localize('metadatacomplete'))
            self.onFailEvent()
            return

        # Check if torrent is private without trackers
        self.torrent.checkPrivateWithoutTrackers()

        self.reannouncelast = 0
        self.restarted = True
        if not self.paused and self.torrent.activity != 2 and self.torrent.setActivity(1, countextra = True):
            self.parent.scheduler(updatecount = True)

        # Note :
        # If torrent was queued by scheduler, updateStorage is not run
        # If torrent was set on-hold by scheduler, thread started by updateStorage creates folders and then is paused
        if self.dow:
            self.startqueue.append(self.torrent.infohash)
            self.dow.updateStorage(self.chooseFile, FILESEM, CHECKWAITING, self.readwaiting, self.updateStorageComplete)

    def updateStorageComplete(self):
        if self.hasconnections:
            self.btstatus = 'working'
        else:
            self.btstatus = 'connectingtopeers'
        self.initializingfiles = False
        self.torrent.checkedonce = True

    def checkFreeDiskSpace(self, nexttobeallocated):
        if not self.torrent.magnet and not self.torrent.complete \
           and not self.parent.diskFullWatchdog(self.torrent, nexttobeallocated):
            return False
        return True

    def savePeers(self):
        if self.dow:
            lps = self.torrent.lastpeers
            if not self.spew:
                return
            for peer in [(p['dns'][0], p['rp']) for p in self.spew]:
                if peer[1] is None:
                    continue
                try:
                    lps.remove(peer)
                except:
                    pass
                lps.append(peer)
            del lps[:max(0, len(lps) - self.utility.maxlastpeers)]

    def loadPeers(self):
        if self.dow:
            self.dow.addPeers([(peer, 0, None, None, 0) for peer in self.torrent.lastpeers[::-1]], 4)

    def clearSavedPeers(self):
        self.torrent.lastpeers[:] = []
        try:
            remove(self.torrent.src + '.prs')
        except:
            pass
