##############################################################################
# Module : abc_tweak.py
# Author : Choopan RATTANAPOKA
# Date   : 10/24/2003
# Updated and modified for ABC_OKC : Old King Cole
#
# Description : A utility for tweaking ABC_OKC display (window sizes, torrent
#               list columns)
#
##############################################################################
import sys, wx
from os import path, remove, rename

from BitTornado.utility import sysencoding

F_ID = 0
F_COL = 1
F_VAL1 = 2
F_VAL2 = 3
F_DESC = 4
F_ODEF = 5
F_OLOCK = 6


##############################################################
# Class : IniManager
##############################################################
class IniManager:
    ###########################################################
    # abc.ini pattern
    # id|column|val1 (text or width)|val2 (startingwidth or height)|description|default order|order lock
    # (column -1 = no display)
    # 0 = app_width,app_height              1 = metawin_width, metawin_height
    # 2 = detailwin_width,detailwin_height  3 = Not used
    # 4 = Title                             5 = Progress
    # 6 = BT status                         7 = Priority
    # 8 = ETA                               9 = Size
    #10 = DL speed                         11 = UL speed
    #12 = %U/D Size                        13 = Error message
    #14 = #connected seed                  15 = #connected peer
    #16 = #seeing copies                   17 = Peer Avg progress
    #18 = Download Size                    19 = Upload Size
    #20 = Total Speed                      21 = Torrent name
    #22 = Seeding option                   23 = Short label
    #24 = Label                            25 = Activity
    #26 = Destination                      27 = Seeding time
    #28 = Total seeding time
    ###########################################################
    def __init__(self, datapath):
        self.datapath = datapath
        self.readIni()

    # Read setting from abc.ini
    ################################
    def readIni(self):
        self.configdata = []
        self.maxid = 0
        ini = open(path.join(self.datapath, "abc.ini"), "r")
        firstline = True
        while True:
            iniline = ini.readline()
            if iniline == "" or iniline == "\n":
                break
            if firstline:
                if iniline[:3] == '\xef\xbb\xbf':
                    self.encoding = 'utf_8'
                    # Skip BOM
                    iniline = iniline[3:]
                    if iniline == "" or iniline == "\n":
                        break
                else:
                    self.encoding = sysencoding
                firstline = False
            iniarg = iniline.decode(self.encoding).split("|")
            self.configdata.append([int(iniarg[0]), int(iniarg[1]), iniarg[2], int(iniarg[3]), iniarg[4], int(iniarg[5]), int(iniarg[6][:-1])])
            self.maxid += 1
        ini.close()
        for i in xrange(4):
            self.configdata[i][F_VAL1] = int(self.configdata[i][F_VAL1])

    # Save setting to abc.ini
    ################################
    def saveIni(self, selected_list):
        filename = path.join(self.datapath, "abc.ini")
        try:
            ini = open(filename + '.part', "w")
            ini.writelines('\xef\xbb\xbf')
            for i in xrange(self.maxid):
                if i < 4:
                    val1 = str(self.configdata[i][F_VAL1])
                else:
                    val1 = self.configdata[i][F_VAL1]
                ini.writelines((str(self.configdata[i][F_ID]) + "|" + str(self.getRank(selected_list, i))
                               + "|" + val1 + "|" + str(self.configdata[i][F_VAL2])
                               + "|" + self.configdata[i][F_DESC] + "|" + str(self.configdata[i][F_ODEF])
                               + "|" + str(self.configdata[i][F_OLOCK]) + "\n").encode('utf_8'))
        except:
            try:
                ini.close()
            except:
                pass
        else:
            ini.close()
            try:
                remove(filename)
            except:
                pass
            rename(filename + '.part', filename)

    # Get column rank
    ################################
    def getRank(self, selected_list, id):            
        for i in xrange(len(selected_list)):
            if selected_list[i] == id:
                return i
        return -1


##############################################################
# Class : ABCTweakPanel
##############################################################
class ABCTweakPanel(wx.Panel):
    def __init__(self, parent):
        wx.Panel.__init__(self, parent, -1)

        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.ini = IniManager(self.utility.datapath)

        outerbox = wx.BoxSizer(wx.VERTICAL)

        columns = wx.StaticBoxSizer(wx.StaticBox(self, -1, self.localize('columns')), wx.VERTICAL)

        columnschoice = wx.BoxSizer(wx.HORIZONTAL)

        # Unselected columns
        self.unselected_list = wx.ListCtrl(self, -1, size = (140, 122), style = wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.unselected_list.InsertColumn(0, self.localize('unselectedcolumns'))
        self.unselected_list.SetColumnWidth(0, 124)

        columnschoice.Add(self.unselected_list, 0)

        # Add & remove column buttons
        addbtn = wx.Button(self, -1, self.localize('addto'), size = (50, 32))
        removebtn = wx.Button(self, -1, self.localize('removefrom'), size = (50, 32))

        addrembuttonbox = wx.BoxSizer(wx.VERTICAL)
        addrembuttonbox.Add(addbtn, 0, wx.ALL, 0)
        addrembuttonbox.Add(removebtn, 0, wx.TOP, 5)

        columnschoice.Add(addrembuttonbox, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 4)

        # Selected columns
        self.selected_list = wx.ListCtrl(self, -1, size = (140, 122), style = wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.selected_list.InsertColumn(0, self.localize('selectedcolumns'))
        self.selected_list.SetColumnWidth(0, 124)

        columnschoice.Add(self.selected_list, 0)

        # Up & Down button
        upbtn = wx.Button(self, -1, self.localize('up'), size = (15, 32))
        downbtn = wx.Button(self, -1, self.localize('down'), size = (15, 32))

        updownbuttonbox = wx.BoxSizer(wx.VERTICAL)
        updownbuttonbox.Add(upbtn, 0, wx.ALL, 0)
        updownbuttonbox.Add(downbtn, 0, wx.TOP, 5)

        columnschoice.Add(updownbuttonbox, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT, 4)

        columns.Add(columnschoice, 0, wx.TOP, 0)

        # Column Setting
        self.displaycolbx = wx.StaticBox(self, -1, self.localize('columnsetting'))
        columnsetting = wx.StaticBoxSizer(self.displaycolbx, wx.VERTICAL)

        displaydescbox = wx.BoxSizer(wx.HORIZONTAL)
        self.displaydesc = wx.StaticText(self, -1, "")
        self.displaydesctext = wx.StaticText(self, -1, self.localize('displaydesc'))
        displaydescbox.Add(self.displaydesctext, 0, wx.ALIGN_CENTER_VERTICAL)
        displaydescbox.Add(self.displaydesc, 1, wx.EXPAND | wx.LEFT, 5)

        displaynamebox = wx.BoxSizer(wx.HORIZONTAL)
        self.displayname = wx.TextCtrl(self, -1, "")
        self.displaynametext = wx.StaticText(self, -1, self.localize('displayname'))
        displaynamebox.Add(self.displaynametext, 0, wx.ALIGN_CENTER_VERTICAL)
        displaynamebox.Add(self.displayname, 1, wx.EXPAND | wx.LEFT, 5)

        columnwidthbox = wx.BoxSizer(wx.HORIZONTAL)
        self.displaywidth = wx.TextCtrl(self, -1, "", size = wx.Size(40, self.utility.digitheight), style = wx.TE_RIGHT)
        self.displaywidth.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.displaywidth.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.displaywidthtext = wx.StaticText(self, -1, self.localize('columnwidth'))
        columnwidthbox.Add(self.displaywidthtext, 0, wx.ALIGN_CENTER_VERTICAL)
        columnwidthbox.Add(self.displaywidth, 0, wx.LEFT | wx.RIGHT, 5)
        self.pixeltext = wx.StaticText(self, -1, self.localize('pixel'))
        columnwidthbox.Add(self.pixeltext, 0, wx.ALIGN_CENTER_VERTICAL)

        self.sortingtext = wx.StaticText(self, -1, self.localize('sorting'))
        columnwidthbox.Add(self.sortingtext, 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 20)
        self.orderdef = wx.CheckBox(self, -1, self.localize('orderdef'))
        columnwidthbox.Add(self.orderdef, 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 10)
        self.orderlock = wx.CheckBox(self, -1, self.localize('orderlock'))
        columnwidthbox.Add(self.orderlock, 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 10)

        columnsetting.Add(displaydescbox, 0, wx.EXPAND | wx.TOP, 0)
        columnsetting.Add(displaynamebox, 0, wx.EXPAND | wx.TOP, 3)
        columnsetting.Add(columnwidthbox, 0, wx.EXPAND | wx.TOP, 3)
        columns.Add(columnsetting, 0, wx.EXPAND | wx.TOP, 3)

        windowsizes = wx.StaticBoxSizer(wx.StaticBox(self, -1, self.localize('windowsizes')), wx.HORIZONTAL)

        sizecol1 = wx.BoxSizer(wx.VERTICAL)

        abcwbox = wx.BoxSizer(wx.HORIZONTAL)
        self.abcwinwidth = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.abcwinwidth.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.abcwinwidth.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        abcwbox.Add(wx.StaticText(self, -1, self.localize('abcwinwidth')), 0, wx.ALIGN_CENTER_VERTICAL)
        abcwbox.Add(self.abcwinwidth, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        abcwbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol1.Add(abcwbox, 0, wx.TOP, 0)

        metainfowbox = wx.BoxSizer(wx.HORIZONTAL)
        self.metainfowinwidth = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.metainfowinwidth.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.metainfowinwidth.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.metainfowinwidth.SetToolTipString(self.localize('0isdefminsizett'))
        metainfowbox.Add(wx.StaticText(self, -1, self.localize('metainfowinwidth')), 0, wx.ALIGN_CENTER_VERTICAL)
        metainfowbox.Add(self.metainfowinwidth, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        metainfowbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol1.Add(metainfowbox, 0, wx.TOP, 3)

        detailwbox = wx.BoxSizer(wx.HORIZONTAL)
        self.detailwinwidth = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.detailwinwidth.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.detailwinwidth.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        detailwbox.Add(wx.StaticText(self, -1, self.localize('detailwinwidth')), 0, wx.ALIGN_CENTER_VERTICAL)
        detailwbox.Add(self.detailwinwidth, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        detailwbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol1.Add(detailwbox, 0, wx.TOP, 3)

        sizecol2 = wx.BoxSizer(wx.VERTICAL)

        abchbox = wx.BoxSizer(wx.HORIZONTAL)
        self.abcwinheight = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.abcwinheight.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.abcwinheight.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        abchbox.Add(wx.StaticText(self, -1, self.localize('abcwinheight')), 0, wx.ALIGN_CENTER_VERTICAL)
        abchbox.Add(self.abcwinheight, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        abchbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol2.Add(abchbox, 0, wx.TOP, 0)

        metainfohbox = wx.BoxSizer(wx.HORIZONTAL)
        self.metainfowinheight = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.metainfowinheight.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.metainfowinheight.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.metainfowinheight.SetToolTipString(self.localize('0isdefminsizett'))
        metainfohbox.Add(wx.StaticText(self, -1, self.localize('metainfowinheight')), 0, wx.ALIGN_CENTER_VERTICAL)
        metainfohbox.Add(self.metainfowinheight, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        metainfohbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol2.Add(metainfohbox, 0, wx.TOP, 3)

        detailhbox = wx.BoxSizer(wx.HORIZONTAL)
        self.detailwinheight = wx.TextCtrl(self, -1, "", (-1, -1), (43, self.utility.digitheight), style = wx.TE_RIGHT)
        self.detailwinheight.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.detailwinheight.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        detailhbox.Add(wx.StaticText(self, -1, self.localize('detailwinheight')), 0, wx.ALIGN_CENTER_VERTICAL)
        detailhbox.Add(self.detailwinheight, 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        detailhbox.Add(wx.StaticText(self, -1, self.localize('pixel')), 0, wx.ALIGN_CENTER_VERTICAL)
        sizecol2.Add(detailhbox, 0, wx.TOP, 3)

        windowsizes.Add(sizecol1, 0)
        windowsizes.Add(sizecol2, 0, wx.LEFT, 15)

        # Save&exit, cancel button
        buttonbox = wx.BoxSizer(wx.HORIZONTAL)

        readbtn = wx.Button(self, -1, self.localize('readfromabc'))
        applybtn = wx.Button(self, -1, self.localize('apply'))
        okbtn = wx.Button(self, -1, self.localize('ok'))
        cancelbtn = wx.Button(self, -1, self.localize('cancel'))

        buttonbox.Add(readbtn, 0, wx.LEFT, 0)
        buttonbox.Add(applybtn, 0, wx.LEFT, 15)
        buttonbox.Add(okbtn, 0, wx.LEFT, 15)
        buttonbox.Add(cancelbtn, 0, wx.LEFT, 15)

        outerbox.Add(columns, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.RIGHT, 3)
        outerbox.Add(windowsizes, 0, wx.EXPAND | wx.TOP | wx.LEFT | wx.RIGHT, 3)
        outerbox.Add(buttonbox, 0, wx.BOTTOM | wx.TOP | wx.ALIGN_CENTER_HORIZONTAL, 6)

        self.SetAutoLayout(True)
        self.SetSizer(outerbox)
        self.Fit()

        # Add events
        self.unselected_list.Bind(wx.EVT_KEY_DOWN, self.OnKeyInUnselectedList)
        self.unselected_list.Bind(wx.EVT_LEFT_DOWN, self.OnLeftUnselected)
        self.selected_list.Bind(wx.EVT_KEY_DOWN, self.OnKeyInSelectedList)
        self.selected_list.Bind(wx.EVT_LEFT_DOWN, self.OnLeftSelected)
        self.Bind(wx.EVT_BUTTON, self.OnAdd, addbtn)
        self.Bind(wx.EVT_BUTTON, self.OnRemove, removebtn)
        self.Bind(wx.EVT_BUTTON, self.OnUp, upbtn)
        self.Bind(wx.EVT_BUTTON, self.OnDown, downbtn)
        self.Bind(wx.EVT_BUTTON, self.OnReadFromAbc, readbtn)
        self.Bind(wx.EVT_BUTTON, self.OnApply, applybtn)
        self.Bind(wx.EVT_BUTTON, self.OnOk, okbtn)
        self.Bind(wx.EVT_BUTTON, self.OnCancel, cancelbtn)
        self.displayname.Bind(wx.EVT_CHAR, self.utility.onWriteFilteredText)
        self.displayname.Bind(wx.EVT_TEXT_PASTE, self.utility.onPasteText)
        self.displaywidth.Bind(wx.EVT_CHAR, self.utility.onWriteFilteredText)

        self.displayData()

    def onRightButtonText(self, event):
        pass

    def displayData(self):
        self.leftid = []
        self.rightid = []
        self.leftindex = -1
        self.rightindex = -1
        self.DisplayInfo()
        # Set Default value from abc.ini
        self.abcwinwidth.SetValue(str(self.ini.configdata[0][F_VAL1]))
        self.abcwinheight.SetValue(str(self.ini.configdata[0][F_VAL2]))
        self.metainfowinwidth.SetValue(str(self.ini.configdata[1][F_VAL1]))
        self.metainfowinheight.SetValue(str(self.ini.configdata[1][F_VAL2]))
        self.detailwinwidth.SetValue(str(self.ini.configdata[2][F_VAL1]))
        self.detailwinheight.SetValue(str(self.ini.configdata[2][F_VAL2]))
        self.max_selected = 0
        self.max_unselected = 0
        for i in xrange(4, self.ini.maxid):
            numcol = self.ini.configdata[i][F_COL]
            if numcol == -1:
                self.unselected_list.InsertStringItem(self.max_unselected, self.ini.configdata[i][F_VAL1])
                self.leftid.append(self.ini.configdata[i][F_ID])
                self.max_unselected += 1
            else:
                if numcol >= self.max_selected:
                    for j in xrange(self.max_selected, numcol + 1):
                        self.selected_list.InsertStringItem(j, "")
                        self.rightid.append("")
                        self.max_selected += 1
                self.selected_list.SetStringItem(numcol, 0, self.ini.configdata[i][F_VAL1])
                self.rightid[numcol] = self.ini.configdata[i][F_ID]

    def OnKeyInSelectedList(self, event):
        if not self.selected_list.GetItemCount():
            return
        keycode = event.GetKeyCode()
        if keycode == wx.WXK_UP:
            self.OnLeftSelected(index = max(0, self.rightindex - 1))
        elif keycode == wx.WXK_DOWN:
            self.OnLeftSelected(index = min(self.selected_list.GetItemCount() - 1, self.rightindex + 1))
        elif keycode == wx.WXK_PRIOR:
            self.OnLeftSelected(index = max(0, self.rightindex - self.selected_list.GetCountPerPage()))
        elif keycode == wx.WXK_NEXT:
            self.OnLeftSelected(index = min(self.selected_list.GetItemCount() - 1, self.rightindex + self.selected_list.GetCountPerPage()))
        elif keycode == wx.WXK_HOME:
            self.OnLeftSelected(index = 0)
        elif keycode == wx.WXK_END:
            self.OnLeftSelected(index = self.selected_list.GetItemCount() - 1)

    def OnKeyInUnselectedList(self, event):
        if not self.unselected_list.GetItemCount():
            return
        keycode = event.GetKeyCode()
        if keycode == wx.WXK_UP:
            self.OnLeftUnselected(index = max(0, self.leftindex - 1))
        elif keycode == wx.WXK_DOWN:
            self.OnLeftUnselected(index = min(self.unselected_list.GetItemCount() - 1, self.leftindex + 1))
        elif keycode == wx.WXK_PRIOR:
            self.OnLeftUnselected(index = max(0, self.leftindex - self.unselected_list.GetCountPerPage()))
        elif keycode == wx.WXK_NEXT:
            self.OnLeftUnselected(index = min(self.unselected_list.GetItemCount() - 1, self.leftindex + self.unselected_list.GetCountPerPage()))
        elif keycode == wx.WXK_HOME:
            self.OnLeftUnselected(index = 0)
        elif keycode == wx.WXK_END:
            self.OnLeftUnselected(index = self.unselected_list.GetItemCount() - 1)

    def DisplayInfo(self):
        if self.rightindex == -1 and self.leftindex == -1:
            self.displaycolbx.SetLabel(self.localize('columnsetting'))
            self.displayname.SetValue('')
            self.displaywidth.SetValue('')
            self.orderdef.SetValue(False)
            self.orderlock.SetValue(False)
            self.displaydesc.SetLabel('')
            self.displaynametext.Disable()
            self.displayname.Disable()
            self.displaywidthtext.Disable()
            self.displaywidth.Disable()
            self.pixeltext.Disable()
            self.sortingtext.Disable()
            self.orderdef.Disable()
            self.orderlock.Disable()
            self.displaydesctext.Disable()
            self.displaydesc.Disable()
        else:
            self.displaynametext.Enable()
            self.displayname.Enable()
            self.displaywidthtext.Enable()
            self.displaywidth.Enable()
            self.pixeltext.Enable()
            self.sortingtext.Enable()
            self.orderdef.Enable()
            self.orderlock.Enable()
            self.displaydesctext.Enable()
            self.displaydesc.Enable()
            if self.rightindex == -1:
                self.displaycolbx.SetLabel(self.localize('unselcolsetting'))
                self.displayname.SetValue(self.ini.configdata[self.leftid[self.leftindex]][F_VAL1])
                self.displaywidth.SetValue(str(self.ini.configdata[self.leftid[self.leftindex]][F_VAL2]))
                self.orderdef.SetValue(self.ini.configdata[self.leftid[self.leftindex]][F_ODEF])
                self.orderlock.SetValue(self.ini.configdata[self.leftid[self.leftindex]][F_OLOCK])
                self.displaydesc.SetLabel(self.ini.configdata[self.leftid[self.leftindex]][F_DESC])
            elif self.leftindex == -1:
                self.displaycolbx.SetLabel(self.localize('selcolsetting') + ' (' + str(self.rightindex + 1) + ')')
                self.displayname.SetValue(self.ini.configdata[self.rightid[self.rightindex]][F_VAL1])
                self.displaywidth.SetValue(str(self.ini.configdata[self.rightid[self.rightindex]][F_VAL2]))
                self.orderdef.SetValue(self.ini.configdata[self.rightid[self.rightindex]][F_ODEF])
                self.orderlock.SetValue(self.ini.configdata[self.rightid[self.rightindex]][F_OLOCK])
                self.displaydesc.SetLabel(self.ini.configdata[self.rightid[self.rightindex]][F_DESC])

    def OnUp(self, event):
        if self.rightindex == -1:
            return

        self.selected_list.SetFocus()

        if self.rightindex == 0:
            return

        id = self.rightid[self.rightindex - 1]
        self.rightid[self.rightindex - 1] = self.rightid[self.rightindex]
        self.rightid[self.rightindex] = id

        # Update display
        self.selected_list.SetStringItem(self.rightindex - 1, 0,
                                         self.ini.configdata[self.rightid[self.rightindex - 1]][F_VAL1])

        self.selected_list.SetStringItem(self.rightindex, 0,
                                         self.ini.configdata[self.rightid[self.rightindex]][F_VAL1])

        self.rightindex -= 1
        self.selected_list.Select(self.rightindex)
        self.selected_list.Focus(self.rightindex)
        self.DisplayInfo()

    def OnDown(self, event):
        if self.rightindex == -1:
            return

        self.selected_list.SetFocus()

        if self.rightindex == self.max_selected - 1:
            return

        id = self.rightid[self.rightindex + 1]
        self.rightid[self.rightindex + 1] = self.rightid[self.rightindex]
        self.rightid[self.rightindex] = id

        # Update display
        self.selected_list.SetStringItem(self.rightindex + 1, 0,
                                         self.ini.configdata[self.rightid[self.rightindex + 1]][F_VAL1])

        self.selected_list.SetStringItem(self.rightindex, 0,
                                         self.ini.configdata[self.rightid[self.rightindex]][F_VAL1])

        self.rightindex += 1
        self.selected_list.Select(self.rightindex)
        self.selected_list.Focus(self.rightindex)
        self.DisplayInfo()

    def applyChanges(self):
        if self.leftindex == -1 and self.rightindex == -1:
            return True

        try:
            displaywidth = int(self.displaywidth.GetValue())
        except:
            dlg = wx.MessageDialog(self, self.localize('displaywidthwarning'), self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.displaywidth.SetValue(str(self.ini.configdata[self.rightid[self.rightindex]][F_VAL2]))
            return False

        if self.rightindex == -1:
            # Change in memory
            id = self.leftid[self.leftindex]
            self.ini.configdata[id][F_VAL1] = self.displayname.GetValue()
            self.ini.configdata[id][F_VAL2] = displaywidth
            self.ini.configdata[id][F_ODEF] = int(self.orderdef.GetValue())
            self.ini.configdata[id][F_OLOCK] = int(self.orderlock.GetValue())
            # Change in display
            self.unselected_list.SetStringItem(self.leftindex, 0, self.displayname.GetValue())
        elif self.leftindex == -1:
            # Change in memory
            id = self.rightid[self.rightindex]
            self.ini.configdata[id][F_VAL1] = self.displayname.GetValue()
            self.ini.configdata[id][F_VAL2] = displaywidth
            self.ini.configdata[id][F_ODEF] = int(self.orderdef.GetValue())
            self.ini.configdata[id][F_OLOCK] = int(self.orderlock.GetValue())
            # Change in display
            self.selected_list.SetStringItem(self.rightindex, 0, self.displayname.GetValue())

        return True

    def OnAdd(self, event):
        if self.leftindex == -1:
            return
        # Delete from left side
        cid = self.leftid[self.leftindex]
        del self.leftid[self.leftindex]
        self.unselected_list.DeleteItem(self.leftindex)
        self.max_unselected -= 1

        # Add in right side
        self.selected_list.InsertStringItem(self.max_selected, self.ini.configdata[cid][F_VAL1])
        self.rightid.append(cid)
        self.leftindex = -1
        self.rightindex = self.max_selected
        self.max_selected += 1
        self.selected_list.Select(self.rightindex)
        self.selected_list.Focus(self.rightindex)
        self.displaycolbx.SetLabel(self.localize('selcolsetting') + ' (' + str(self.rightindex + 1) + ')')

    def OnRemove(self, event):
        if self.rightindex == -1:
            return
        # Delete from right side
        cid = self.rightid[self.rightindex]
        del self.rightid[self.rightindex]
        self.selected_list.DeleteItem(self.rightindex)
        self.max_selected -= 1

        # Add in left side
        self.unselected_list.InsertStringItem(self.max_unselected, self.ini.configdata[cid][F_VAL1])
        self.leftid.append(cid)
        self.rightindex = -1
        self.leftindex = self.max_unselected
        self.max_unselected += 1
        self.unselected_list.Select(self.leftindex)
        self.unselected_list.Focus(self.leftindex)
        self.displaycolbx.SetLabel(self.localize('unselcolsetting'))

    def OnLeftSelected(self, event = None, index = None):
        if not self.applyChanges():
            return
        if index is None:
            self.rightindex = self.selected_list.HitTest(wx.Point(event.GetX(), event.GetY()))[0]
        else:
            self.rightindex = index
        self.leftindex = -1
        if self.rightindex >= 0:
            self.selected_list.Select(self.rightindex)
            self.selected_list.Focus(self.rightindex)
        elif event is not None:
            event.Skip()
        self.selected_list.SetFocus()
        self.DisplayInfo()

    def OnLeftUnselected(self,  event = None, index = None):
        if not self.applyChanges():
            return
        if index is None:
            self.leftindex = self.unselected_list.HitTest(wx.Point(event.GetX(), event.GetY()))[0]
        else:
            self.leftindex = index
        self.rightindex = -1
        if self.leftindex >= 0:
            self.unselected_list.Select(self.leftindex)
            self.unselected_list.Focus(self.leftindex)
        elif event is not None:
            event.Skip()
        self.unselected_list.SetFocus()
        self.DisplayInfo()

    def OnCancel(self, event = None):
        self.parent.OnExit()

    def OnApply(self, event):
        if not self.applyChanges():
            return False
        # Update w, h value
        self.ini.configdata[0][F_VAL1] = int(self.abcwinwidth.GetValue())
        self.ini.configdata[0][F_VAL2] = int(self.abcwinheight.GetValue())
        self.ini.configdata[1][F_VAL1] = int(self.metainfowinwidth.GetValue())
        self.ini.configdata[1][F_VAL2] = int(self.metainfowinheight.GetValue())
        self.ini.configdata[2][F_VAL1] = int(self.detailwinwidth.GetValue())
        self.ini.configdata[2][F_VAL2] = int(self.detailwinheight.GetValue())

        # Save ini        
        self.ini.saveIni(self.rightid)

        # Apply on-the-fly
        self.parent.parent.updateABCDisplay()

        return True

    def OnReadFromAbc(self, event):
        # Re-read data from abc.ini
        self.ini.readIni()
        # Set main window size
        self.ini.configdata[0][F_VAL1], self.ini.configdata[0][F_VAL2] = self.parent.parent.GetSizeTuple()
        # Set details and  advanced details window size from the first one open in the torrent list
        infowinfound = False
        detailwinfound = False
        for torrent in self.parent.parent.window.queue.proctab:
            if not infowinfound and torrent.infowin:
                self.ini.configdata[1][F_VAL1], self.ini.configdata[1][F_VAL2] = torrent.infowin.GetSizeTuple()
                infowinfound = True
            if not detailwinfound and torrent.detailwin:
                self.ini.configdata[2][F_VAL1], self.ini.configdata[2][F_VAL2] = torrent.detailwin.GetSizeTuple()
                detailwinfound = True
            if infowinfound and detailwinfound:
                break
        if not infowinfound:
            self.ini.configdata[1][F_VAL1] = self.parent.parent.window.queue.guiman.getMetaWidth()
            self.ini.configdata[1][F_VAL2] = self.parent.parent.window.queue.guiman.getMetaHeight()
        if not detailwinfound:
            self.ini.configdata[2][F_VAL1] = self.parent.parent.window.queue.guiman.getDetailWidth()
            self.ini.configdata[2][F_VAL2] = self.parent.parent.window.queue.guiman.getDetailHeight()
        # Set column widths and window sizes from ABC GUI
        for colid in xrange(4, self.ini.maxid):
            col = self.ini.configdata[colid][F_COL]
            if col != -1:
                self.ini.configdata[colid][F_VAL2] = self.parent.parent.window.list.GetColumnWidth(col)
        # Display data
        self.unselected_list.DeleteAllItems()
        self.selected_list.DeleteAllItems()
        self.displayData()

    def OnOk(self, event):
        if not self.OnApply(event):
            return
        self.parent.OnExit()


##############################################################
# Class : ABCTweakFrame
##############################################################
class ABCTweakFrame(wx.Frame):
    def __init__(self, parent, ID, title):
        wx.Frame.__init__(self, parent, ID, title,
                          style = wx.DEFAULT_FRAME_STYLE & ~ (wx.RESIZE_BORDER | wx.RESIZE_BOX | wx.MAXIMIZE_BOX))

        self.parent = parent
        self.SetAcceleratorTable(self.parent.accelerators)
        self.utility = self.parent.utility
        if (sys.platform == 'win32'):
            self.icon = wx.Icon(path.join(self.utility.abcpath, 'icon_abc.ico'), wx.BITMAP_TYPE_ICO)
            self.SetIcon(self.icon)

        if self.parent.abcparams['abctweakdlgx'] != '-1':
            self.x = int(self.parent.abcparams['abctweakdlgx'])
            self.y = int(self.parent.abcparams['abctweakdlgy'])
            self.MoveXY(self.x, self.y)
        else:
            self.x, self.y = self.GetPositionTuple()

        self.window = ABCTweakPanel(self)
        self.SetClientSize(self.window.GetSize())
        wx.EVT_CLOSE(self, self.OnExit)
        self.Bind(wx.EVT_MOVE, self.onMove)

    def onMove(self, event):
        if not self.IsIconized():
            self.x, self.y = self.GetPositionTuple()
        event.Skip()

    def OnExit(self, event = None):
        self.parent.abcparams['abctweakdlgx'], self.parent.abcparams['abctweakdlgy'] = str(self.x), str(self.y)
        self.parent.abctweak = None
        self.Destroy()
