#-----------------------------------------------------------------------------
# Name:        g3rss.py
# Purpose:     rss feed reader
# Author:      d0c 54v4g3, Jeremy Arendt
# Created:     2004/23/02
# Copyright:   (c) 2002
# Licence:     See G3.LICENCE.TXT
# Modified by : roee88
# Updated and modified for ABC_OKC : Old King Cole
#-----------------------------------------------------------------------------
import sys, wx, re
import webbrowser, urlparse
import wx.lib.scrolledpanel
import wx.html
from threading import Thread, Timer, Lock
from os import remove, rename
from os.path import join, dirname, exists
from time import sleep, strftime
from urlfinder import UrlFinder, ParserThread
from html2text import html2text
from string import whitespace

from BitTornado.zurllib import urlopen
from feedparser import parse
from HTMLParser import HTMLParser
from Dialogs.rssscanfreqdlg import RSSScanFreqDialog
from BitTornado.utility import sysencoding
from utility import ScrolledMessageDialog

# Semaphore to manage writing to url.lst
URLFILESEM = Lock()

# Semaphore to manage the simultaneously running feed auto-grabbing threads
AUTOGRABTHREADING = Lock()

wxEVT_RSS = wx.NewEventType()

def EVT_RSS(win, func):
    win.Connect(-1, -1, wxEVT_RSS, func)


class RSSEvent(wx.PyEvent):
    def __init__(self, func, args, kwargs):
        wx.PyEvent.__init__(self)
        self.SetEventType(wxEVT_RSS)
        self.func = func
        self.args = args
        self.kwargs = kwargs


class LogPanel(wx.Panel):
    def __init__(self, parent, utility):
        wx.Panel.__init__(self, parent, -1, style = 0)
        self.parent = parent
        self.utility = utility
        self.localize = self.utility.lang.get
        self.loglastdir = self.utility.userpath

        self.log = wx.ListCtrl(self, -1, style = wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.log.InsertColumn(0, self.localize('rsslog'), width = 1200)
        self.log.Bind(wx.EVT_RIGHT_DOWN, self.onRightInList)
        self.log.Bind(wx.EVT_RIGHT_DCLICK, self.onRightInList)

        sizer = wx.BoxSizer(wx.VERTICAL)
        btnsizer = wx.BoxSizer(wx.HORIZONTAL)

        clearallbtn = wx.Button(self, -1, self.localize('clearlog'))
        self.Bind(wx.EVT_BUTTON, self.clearLog, clearallbtn)
        btnsizer.Add(clearallbtn, 0, wx.LEFT | wx.RIGHT, 5)

        savelogbtn = wx.Button(self, -1, self.localize('savelog'))
        self.Bind(wx.EVT_BUTTON, self.saveLog, savelogbtn)
        btnsizer.Add(savelogbtn, 0, wx.LEFT | wx.RIGHT, 5)

        sizer.Add(self.log, 1, wx.EXPAND, 0)
        sizer.Add(btnsizer, 0)

        self.SetSizerAndFit(sizer)

    def onRightInList(self, event):
        pass

    def clearLog(self, event):
        self.log.DeleteAllItems()

    def saveLog(self, event = None):
        # Ask where to save the file to
        date = strftime('%Y-%m-%d')
        date = date.replace("/", "-")
        dlg = wx.FileDialog(self.parent, 
                            message = self.localize('savelogas'),
                            defaultDir = self.loglastdir,
                            defaultFile = "RSS_" + date + ".log",
                            wildcard = self.localize('logfileswildcard') + ' (*.log)|*.log',
                            style = wx.SAVE)
        result = dlg.ShowModal()
        dlg.Destroy()
        if result != wx.ID_OK:
            return

        dest = dlg.GetPath()
        self.loglastdir = dirname(dest)

        try:
            logfile = open(dest, "w")
            logfile.writelines('\xef\xbb\xbf')
            for i in xrange(self.log.GetItemCount()):
                logfile.writelines((self.log.GetItemText(i) + '\n').encode('utf_8'))
            logfile.close()
        except:
            dialog = wx.MessageDialog(self.parent,
                                      self.localize('error_savelog'),
                                      self.localize('abcokcerror'),
                                      wx.ICON_ERROR)
            dialog.ShowModal()
            dialog.Destroy()

    def append(self, msg, url, msgtype):
        if msgtype == "error":
            color = self.utility.colnocon
        elif msgtype == "info":
            color = self.utility.colnocom
        else:
            color = self.utility.colnoeng
        item = wx.ListItem()
        item.SetText(strftime('%Y/%m/%d-%H:%M') + " - " + self.handleMessage(msg) + " : " + url)
        item.SetId(self.log.GetItemCount())
        item.SetTextColour(color)
        index = self.log.InsertItem(item)
        # Colour redraw problem without following refresh
        self.log.RefreshItem(index)

    def handleMessage(self, msg):
        if msg == "Error=Can't get torrent from URL":
            msg = self.localize('log_cantgettorrent')
        elif msg == "Error=Invalid torrent file":
            msg = self.localize('log_invalidtorrent')
        elif msg == "Error=Duplicate torrent":
            msg = self.localize('log_duplicate')
        elif msg == "Error=Duplicate destination":
            msg = self.localize('log_duplicatedest')
        elif msg == "Error=No default download folder":
            msg = self.localize('log_nodest')
        elif msg == "Error=Local torrent adding pending":
            msg = self.localize('log_loctoradding')
        elif msg == "Error=Torrent URL doesn't start with http":
            msg = self.localize('log_httponly')
        elif msg == "Error=Can't find a unique file name for torrent":
            msg = self.localize('log_nounique')
        return msg


class HistoryPanel(wx.Panel):
    def __init__(self, parent, utility, urls):
        wx.Panel.__init__(self, parent, -1, style = 0)
        self.parent = parent
        self.invokeLater = self.parent.invokeLater
        self.utility = utility
        self.abcparams = self.utility.abcparams
        self.localize = self.utility.lang.get
        self.urls = urls
        self.urlsprevindex = len(urls) * [1000000000]
        self.loglastdir = self.utility.userpath
        self.sortdir = 2
        # To avoid one useles update of the list header when a selected item is deleted
        self.skipupdateheader = False
        # History list row selected by a left or a right click
        self.selectedrow = -1
        # History list dragging mode :
        # 0 : no dragging
        # 1 : select torrent
        self.listdraggingmode = 0
        self.scrolltimer = None

        self.history = wx.ListCtrl(self, -1, style = wx.LC_REPORT)
        self.history.InsertColumn(0, self.localize('rsshist'), width = 1200)
        self.history.Bind(wx.EVT_LIST_COL_CLICK, self.onColLeftClick)
        self.history.Bind(wx.EVT_RIGHT_DOWN, self.onLeftRightDown)
        self.history.Bind(wx.EVT_RIGHT_UP, self.onRightUp)
        self.history.Bind(wx.EVT_LEFT_UP, self.onLeftUp)
        self.history.Bind(wx.EVT_LEFT_DOWN, self.onLeftRightDown)
        self.history.Bind(wx.EVT_KEY_DOWN, self.onKeyInList)
        self.history.Bind(wx.EVT_LIST_ITEM_SELECTED, self.updateHeader)
        self.history.Bind(wx.EVT_LIST_ITEM_DESELECTED, self.updateHeader)
        self.history.Bind(wx.EVT_LEAVE_WINDOW, self.onLeaveList)
        self.history.Bind(wx.EVT_ENTER_WINDOW, self.onEnterList)
        self.history.Bind(wx.EVT_LIST_BEGIN_DRAG, self.onLeftDragList)
        self.history.Bind(wx.EVT_LIST_BEGIN_RDRAG, self.onRightDragList)
        self.history.Bind(wx.EVT_MOTION, self.onMotionInList)

        sizer = wx.BoxSizer(wx.VERTICAL)
        btnsizer = wx.BoxSizer(wx.HORIZONTAL)

        deleteallbtn = wx.Button(self, -1, self.localize('clearhist'))
        self.Bind(wx.EVT_BUTTON, self.clear, deleteallbtn)
        btnsizer.Add(deleteallbtn, 0, wx.LEFT | wx.RIGHT, 5)

        deletebtn = wx.Button(self, -1, self.localize('delselhist'))
        self.Bind(wx.EVT_BUTTON, self.deleteSelected, deletebtn)
        btnsizer.Add(deletebtn, 0, wx.LEFT | wx.RIGHT, 5)

        sizer.Add(self.history, 1, wx.EXPAND, 0)
        sizer.Add(btnsizer, 0)

        self.SetSizerAndFit(sizer)

        i = 0
        for url in self.urls:
            self.history.InsertStringItem(i, url)
            i += 1
        self.updateHeader()

    def onKeyInList(self, event):
        if event.ControlDown() and event.GetKeyCode() == wx.WXK_NUMPAD_DECIMAL:
            self.sortdir = 2
            self.parent.parent.displayStatusTextTemp(self.localize('listsortingstatusunsorted'))
        else:
            event.Skip()

    def onLeftRightDown(self, event):
        self.selectedrow = self.history.HitTest(event.GetPosition())[0]
        if self.selectedrow >= 0:
            event.Skip()
        else:
            for index in self.getSelected():
                self.history.SetItemState(index, 0, wx.LIST_STATE_SELECTED)

    def updateHeader(self, event = None):
        if self.skipupdateheader:
            return
        colitem = wx.ListItem()
        colitem.SetText(self.localize('rsshist') + ' (' + str(self.history.GetSelectedItemCount())
                        + '/' + str(self.history.GetItemCount()) + ')')
        self.history.SetColumn(0, colitem)

    def getSelected(self):
        selected = []
        index = self.history.GetNextItem(-1, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        while index != -1:
            selected.append(index)
            index = self.history.GetNextItem(index, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
        return selected

    def deleteSelected(self, event):
        selected = self.getSelected()
        selected.reverse()
        self.skipupdateheader = True
        for index in selected:
            self.history.DeleteItem(index)
            del self.urls[index]
            del self.urlsprevindex[index]
        self.skipupdateheader = False
        self.updateHeader()
        self.parent.SaveURLs()

    def clear(self, event):
        if not self.urls:
            return
        dialog = wx.MessageDialog(None,
                                  self.localize('reseturlsmsg'),
                                  self.localize('abcokcwarning'),
                                  wx.YES_NO | wx.ICON_EXCLAMATION)
        result = dialog.ShowModal()
        dialog.Destroy()
        if result == wx.ID_NO:
            return

        self.skipupdateheader = True
        self.history.DeleteAllItems()
        self.urls[:] = []
        self.urlsprevindex[:] = []
        self.skipupdateheader = False
        self.updateHeader()
        self.parent.SaveURLs()

    def append(self, url):
        self.urls.append(url)
        self.urlsprevindex.append(1000000000)
        self.history.InsertStringItem(self.history.GetItemCount(), url)
        self.updateHeader()

    def onColLeftClick(self, event):
        self.utility.sortcol = []
        nbitem = self.history.GetItemCount()
        if self.sortdir == 2:
            self.urlsprevindex[:] = range(nbitem)
            self.sortdir = 0
        else:
            self.sortdir += 1
        
        if self.sortdir == 2:
            for index in xrange(nbitem):
                self.utility.sortcol.append(self.urlsprevindex[index])
                self.history.SetItemData(index, index)
        else:
            for index in xrange(nbitem):
                self.utility.sortcol.append(self.urls[index].lower())
                self.history.SetItemData(index, index)

        if self.sortdir == 0 or self.sortdir == 2:
            self.history.SortItems(self.utility.sortList)
        else:
            self.history.SortItems(self.utility.sortListReverse)
        urls2 = self.urls[:]
        urlsprevindex2 = self.urlsprevindex[:]
        self.urls[:] = []
        self.urlsprevindex[:] = []
        for index in xrange(nbitem):
            oldindex = self.history.GetItemData(index)
            self.urls.append(urls2[oldindex])
            self.urlsprevindex.append(urlsprevindex2[oldindex])

    def onLeftDragList(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 0
        else:
            self.listdraggingmode =1

    def onRightDragList(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 1
        else:
            self.listdraggingmode = 0

    def onMotionInList(self, event):
        selectedrow = self.history.HitTest(event.GetPosition())[0]
        # List scrolling
        if self.utility.scrolldelay and self.listdraggingmode:
            topitem = self.history.GetTopItem()
            botitem = max(topitem + self.history.GetCountPerPage() - 1, 0)
            if selectedrow == topitem:
                if not self.scrolltimer:
                    self.scrolltimer = Timer(self.utility.scrolldelay, self.invokeLater, [self.scrollList, [1, selectedrow]])
                    self.scrolltimer.start()
            elif selectedrow >= botitem:
                if not self.scrolltimer:
                    self.scrolltimer = Timer(self.utility.scrolldelay, self.invokeLater, [self.scrollList, [-1, selectedrow - (selectedrow > botitem)]])
                    self.scrolltimer.start()
            elif self.scrolltimer:
                self.scrolltimer.cancel()
                self.scrolltimer = None
        event.Skip()

    def scrollList(self, dir, row):
        if self.scrolltimer:
            row -= dir
            if 0 <= row < self.history.GetItemCount():
                self.history.EnsureVisible(row)
                self.scrolltimer = Timer(self.utility.scrolldelay, self.invokeLater, [self.scrollList, [dir, row]])
                self.scrolltimer.start()

    def endDraggingForSelecting(self, index):
        self.listdraggingmode = 0
        if self.scrolltimer:
            self.scrolltimer.cancel()
            self.scrolltimer = None
        if index >= 0 and self.selectedrow >= 0:
            if index > self.selectedrow:
                dir = 1
            else:
                dir = -1
            for i in xrange(self.selectedrow + dir, index + dir, dir):
                self.history.Select(i)

    def onRightUp(self, event):
        if self.abcparams['mousemode'] == '0':
            if self.listdraggingmode == 1:
                self.endDraggingForSelecting(self.history.HitTest((event.GetX(), event.GetY()))[0])
        else:
            self.listdraggingmode = 0

    def onLeftUp(self, event):
        if self.abcparams['mousemode'] == '0':
            self.listdraggingmode = 0
        else:
            if self.listdraggingmode == 1:
                self.endDraggingForSelecting(self.history.HitTest((event.GetX(), event.GetY()))[0])

    def onLeaveList(self, event):
        # To handle dragging the mouse out of the list with the left or right button pressed
        if self.scrolltimer:
            self.scrolltimer.cancel()
            self.scrolltimer = None
        event.Skip()

    def onEnterList(self, event):
        # To handle dragging the mouse back into the list without the left or right button pressed
        if self.listdraggingmode and not (event.LeftIsDown() or event.RightIsDown()) :
            self.listdraggingmode = 0
            if self.scrolltimer:
                self.scrolltimer.cancel()
                self.scrolltimer = None
        event.Skip()


class FeedsDialog(wx.Dialog):
    def __init__(self, filterspanel):
        self.filterspanel = filterspanel
        self.feeders = self.filterspanel.feedalias[:]
        if '' in self.feeders:
            self.feeders.remove('') 
        self.utility = self.filterspanel.utility
        self.localize = self.utility.lang.get

        pre = wx.PreDialog()
        pre.Create(filterspanel, -1, self.localize('choosefeeds'))
        self.this = pre.this

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)

        # Don't use buggy CheckListBox that makes checkmarks disappear from menus
        #self.checklistbox = wx.CheckListBox(p, -1, choices = self.feeders, style = wx.LB_SINGLE)
        #i = 0
        #for f in self.feeders:
        #    if f in self.filterspanel.currentfeeds:
        #        self.checklistbox.Check(i)
        #    i += 1
        self.checklistpanel = wx.lib.scrolledpanel.ScrolledPanel(p, -1, size = (-1, 150), style = wx.TAB_TRAVERSAL | wx.SUNKEN_BORDER)
        checklist = wx.BoxSizer(wx.VERTICAL)
        self.checkboxes = []
        for f in self.feeders:
            checkbox = wx.CheckBox(self.checklistpanel, -1, f)
            if f in self.filterspanel.currentfeeds:
                checkbox.SetValue(True)
            self.checkboxes.append(checkbox)
            checklist.Add(checkbox, 0, wx.TOP | wx.LEFT, 3)
        self.checklistpanel.SetAutoLayout(True)
        self.checklistpanel.SetSizer(checklist)
        self.checklistpanel.SetupScrolling()

        selectallbtn = wx.Button(p, -1, self.localize('selectall'))
        self.Bind(wx.EVT_BUTTON, self.onSelectAll, selectallbtn)

        unselectallbtn = wx.Button(p, -1, self.localize('unselectall'))
        self.Bind(wx.EVT_BUTTON, self.onUnSelectAll, unselectallbtn)

        applybtn = wx.Button(p, -1, self.localize('apply'))
        self.Bind(wx.EVT_BUTTON, self.onApply, applybtn)

        okbtn = wx.Button(p, -1, self.localize('ok'))
        self.Bind(wx.EVT_BUTTON, self.onOK, okbtn)

        cancelbtn = wx.Button(p, wx.ID_CANCEL, self.localize('cancel'))

        selectbuttonbox = wx.BoxSizer(wx.HORIZONTAL)
        selectbuttonbox.Add(selectallbtn, 0, wx.ALL, 5)
        selectbuttonbox.Add(unselectallbtn, 0, wx.ALL, 5)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        buttonbox.Add(applybtn, 0, wx.ALL, 5)
        buttonbox.Add(okbtn, 0, wx.ALL, 5)
        buttonbox.Add(cancelbtn, 0, wx.ALL, 5)

        outerbox.Add(self.checklistpanel, 0, wx.EXPAND | wx.ALL, 5)
        outerbox.Add(selectbuttonbox, 0, wx.ALIGN_CENTER)
        outerbox.Add(buttonbox, 0, wx.ALIGN_CENTER)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

    def onOK(self, event):
        if self.onApply():
            self.EndModal(wx.ID_OK)

    def onApply(self, event = None):
        i = 0
        self.filterspanel.currentfeeds[:] = []
        for f in self.feeders:
            #if self.checklistbox.IsChecked(i):
            if self.checkboxes[i].IsChecked():
                self.filterspanel.currentfeeds.append(f)
            i += 1
        self.filterspanel.populateCombo()
        self.filterspanel.setSaveButtonStatus(self.filterspanel.rule[1], newvalue = self.filterspanel.currentfeeds)
        self.checklistpanel.SetFocus()
        return True

    def onSelectAll(self, event):
        for i in xrange(len(self.feeders)):
            #self.checklistbox.Check(i)
            self.checkboxes[i].SetValue(True)
        self.checklistpanel.SetFocus()

    def onUnSelectAll(self, event):
        for i in xrange(len(self.feeders)):
            #self.checklistbox.Check(i, 0)
            self.checkboxes[i].SetValue(False)
        self.checklistpanel.SetFocus()


class T_GetFeed(Thread):
    def __init__(self, invokelater, callback, feeder, auto, owntimer = False, filters = None):
        Thread.__init__(self)
        self.daemon = True
        self.callback = callback
        self.InvokeLater = invokelater
        self.feeder = feeder
        self.auto = auto
        self.owntimer = owntimer
        self.filters = filters

    def run(self):
        request_headers = {}
        if self.feeder.cookiesw:
            request_headers['Cookie'] = self.feeder.cookie
        try:
            parseresult = parse(self.feeder.url, agent = 'Mozilla/4.0 (compatible; MSIE 5.5; Windows NT)', request_headers = request_headers)
            if self.auto:
                data = (parseresult, self.owntimer, self.filters)
            else:
                data = parseresult
        except:
            if self.auto:
                AUTOGRABTHREADING.acquire()
                self.feeder.rsspanel.nbfeedautothread -= 1
                if self.owntimer:
                    self.feeder.scheduleNextTimerCycle()
                else:
                    self.feeder.rsspanel.nbglobalfeedautothread -= 1
                    if self.feeder.rsspanel.nbglobalfeedautothread == 0:
                        self.feeder.rsspanel.scheduleNextGlobalTimerCycle()
                if self.feeder.rsspanel.nbfeedautothread == 0:
                    self.feeder.rsspanel.searchbutt.Enable(True)
                AUTOGRABTHREADING.release()
            else:
                self.feeder.rsspanel.editfeedalias.Enable()
                self.feeder.rsspanel.updatebutt.Enable()
                self.feeder.rsspanel.clearlistbutt.Enable()
            return
        try:
            self.InvokeLater(self.callback, data)
        except wx.PyDeadObjectError:
            pass


class RSSListHandler(wx.EvtHandler):
    def __init__(self, articlelist):
        wx.EvtHandler.__init__(self)
        self.list = articlelist
        EVT_RSS(self, self.onInvoke)

    def onInvoke(self, event):
        event.func(*event.args, **event.kwargs)

    def invokeLater(self, func, args = [], kwargs = {}):
        wx.PostEvent(self, RSSEvent(func, args, kwargs))


class RSSList(wx.ListCtrl):
    def __init__(self, parent, rsspanel, utility, pos = wx.DefaultPosition, size = wx.DefaultSize,
                 style = wx.LC_REPORT | wx.LC_VRULES | wx.LC_SINGLE_SEL, onclickfunc = None, ondblclickfunc = None,
                 onbrowfunc = None, onsearchfortitlefunc = None, oncopytitlefunc = None, oncopytitlelinkfunc = None,
                 oncopyurlfunc = None, onstopurl = None):
        wx.ListCtrl.__init__(self, parent, -1, style = style)
        self.utility = utility
        self.localize = self.utility.lang.get
        self.parent = parent
        self.rsspanel = rsspanel
        self.CmdOnClick = onclickfunc
        self.CmdDblClick = ondblclickfunc
        self.CmdBrowFunc = onbrowfunc
        self.CmdSearchForTitleFunc = onsearchfortitlefunc
        self.CmdCopyTitleFunc = oncopytitlefunc
        self.CmdCopyTitleLinkFunc = oncopytitlelinkfunc
        self.CmdCopyURLFunc = oncopyurlfunc
        self.CmdStopUrl = onstopurl
        # To limit html refresh to once a selection
        self.previtem = None
        self.listfont = self.GetFont()
        self.boldlistfont = self.GetFont()
        self.boldlistfont.SetWeight(wx.FONTWEIGHT_BOLD)
        # Currently displayed feed
        self.curfeed = None
        self.sortdir = 2
        self.lastsortingcol = -1

        col = 0
        for colinfo in self.utility.rsscolumns:
            self.InsertColumn(col, colinfo[0], format = colinfo[2], width = colinfo[1])
            col += 1

        wx.EVT_LIST_COL_END_DRAG(self, -1, self.onResizeColumn)
        wx.EVT_LIST_ITEM_ACTIVATED(self, -1, self.OnDblClick)
        wx.EVT_LEFT_DOWN(self, self.OnClick)
        wx.EVT_RIGHT_DOWN(self, self.OnRightDown)
        self.Bind(wx.EVT_LIST_ITEM_RIGHT_CLICK, self.OnListRightClick)
        wx.EVT_KEY_DOWN(self, self.onKeyInList)
        wx.EVT_LIST_COL_CLICK(self, -1, self.onColLeftClick)
        wx.EVT_LIST_COL_RIGHT_CLICK(self, -1, self.onColRightClick)
        
        self.initPopupMenu()

        self.x = 0
        self.y = 0

    def initPopupMenu(self):
        self.listmenu = wx.Menu()
        self.listmenu.Append(930, self.localize('downloadtorrent'), self.localize('downloadtorrent'))
        self.listmenu.Append(1039, self.localize('searchfortitle'), self.localize('searchfortitle'))
        self.listmenu.Append(1044, self.localize('searchforstrippedtitle'), self.localize('searchforstrippedtitle'))
        self.listmenu.Append(932, self.localize('copytitle'), self.localize('copytitle'))
        self.listmenu.Append(1045, self.localize('copystrippedtitle'), self.localize('copystrippedtitle'))
        self.listmenu.Append(939, self.localize('copytitlelink'), self.localize('copytitlelink'))
        self.listmenu.Append(931, self.localize('copyurl'), self.localize('copyurl'))
        self.listmenu.Append(933, self.localize('openurlindefaultbrowser'), self.localize('openurlindefaultbrowser'))
        stopscanningmenu = wx.Menu()
        stopscanningmenu.Append(935, self.localize('stopparsingurl'), self.localize('stopparsingurl'))
        stopscanningmenu.Append(936, self.localize('stopparsingallurls'), self.localize('stopparsingurl'))
        self.listmenu.AppendMenu(934, self.localize('stopparsing'), stopscanningmenu)
        wx.EVT_MENU(self, 930, self.MenuOnDownload)
        wx.EVT_MENU(self, 1039, self.MenuOnSearchForTitle)
        wx.EVT_MENU(self, 1044, self.MenuOnSearchForStrippedTitle)
        wx.EVT_MENU(self, 932, self.MenuOnCopyTitle)
        wx.EVT_MENU(self, 1045, self.MenuOnCopyStrippedTitle)
        wx.EVT_MENU(self, 939, self.MenuOnCopyTitleLink)
        wx.EVT_MENU(self, 931, self.MenuOnCopyURL)
        wx.EVT_MENU(self, 933, self.MenuOnBrowser)
        wx.EVT_MENU(self, 935, self.MenuOnStopUrlScan)
        wx.EVT_MENU(self, 936, self.MenuOnStopAllUrlScan)

    def onColLeftClick(self, event = None):
        if self.curfeed is None:
            return
        keys =[RSSArticle.getTitle, RSSArticle.getURL, RSSArticle.getTime, RSSArticle.getOld]
        self.utility.sortcol = []
        if event is None:
            col = 3
            for index in xrange(self.GetItemCount()):
                self.utility.sortcol.append(not self.curfeed.articletab[index].new)
                self.SetItemData(index, index)
        else:
            col = event.m_col
            if col == self.lastsortingcol:
                if self.sortdir == 2:
                    for index in xrange(self.GetItemCount()):
                        self.curfeed.articletab[index].previndex = index
                    self.sortdir = 0
                else:
                    self.sortdir += 1
            else:
                if self.lastsortingcol == -1:
                    for index in xrange(len(self.curfeed.articletab)):
                        self.curfeed.articletab[index].previndex = index
                self.sortdir = 0
            self.lastsortingcol = col

            if self.sortdir == 2:
                for index in xrange(self.GetItemCount()):
                    self.utility.sortcol.append(self.curfeed.articletab[index].previndex)
                    self.SetItemData(index, index)
            else:
                for index in xrange(self.GetItemCount()):
                    self.utility.sortcol.append(self.GetItem(index, col).GetText().lower())
                    self.SetItemData(index, index)

        if self.sortdir == 0 or self.sortdir == 2:
            self.SortItems(self.utility.sortList)
            if self.sortdir == 2:
                self.curfeed.rawarticletab.sort(key = RSSArticle.getPrevIndex)
            else:
                self.curfeed.rawarticletab.sort(key = keys[col])
        else:
            self.SortItems(self.utility.sortListReverse)
            self.curfeed.rawarticletab.sort(key = keys[col], reverse = True)
        articletab2 = self.curfeed.articletab[:]
        self.curfeed.articletab[:] = []
        for index in xrange(len(articletab2)):
            self.curfeed.articletab.append(articletab2[self.GetItemData(index)])

    def onColRightClick(self, event):
        self.onColLeftClick()

    def onKeyInList(self, event):
        keycode = event.GetKeyCode()
        if keycode in [wx.WXK_UP, wx.WXK_DOWN, wx.WXK_HOME, wx.WXK_END, wx.WXK_PRIOR, wx.WXK_NEXT]:
            if self.CmdOnClick:
                event.Skip()
                wx.CallAfter(self.afterOnKeyInList)
        elif event.ControlDown() and keycode == wx.WXK_NUMPAD_DECIMAL:
            self.sortdir = 2
            self.rsspanel.parent.displayStatusTextTemp(self.localize('listsortingstatusunsorted'))
        else:
            event.Skip()

    def afterOnKeyInList(self):
        selected = self.GetFirstSelected()
        if selected != -1:
            article = self.curfeed.articletab[selected]
            if article is not self.previtem:
                self.previtem = article
                self.CmdOnClick(article)

    def OnDblClick(self, event):
        if self.CmdDblClick and self.curfeed is not None and self.curfeed.articletab:
            thread = Thread(target = self.CmdDblClick, args = [self.curfeed.articletab[event.GetIndex()]])
            thread.daemon = False
            thread.start()

    def OnClick(self, event):
        if self.CmdOnClick and self.curfeed is not None and self.curfeed.articletab:
            if event is None:
                selected = self.HitTest((self.x, self.y))[0]
            else:
                selected = self.HitTest(event.GetPosition())[0]
                event.Skip()
            if selected != -1:
                article = self.curfeed.articletab[selected]
                if article is not self.previtem:
                    self.previtem = article
                    self.CmdOnClick(article)
                return True
            self.rsspanel.html.SetPage("")
            self.previtem = None
            return False
        return False

    def OnRightDown(self, event):
        self.x, self.y = event.GetPosition()
        if not self.OnClick(None):
            selected = self.GetNextItem(-1, wx.LIST_NEXT_ALL, wx.LIST_STATE_SELECTED)
            if selected != -1:
                self.SetItemState(selected, 0, wx.LIST_STATE_SELECTED)
            return
        event.Skip()

    def OnListRightClick(self, event):
        self.PopupMenu(self.listmenu, (self.x, self.y))

    def MenuOnStopUrlScan(self, event):
        if self.CmdStopUrl:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdStopUrl(article = self.curfeed.articletab[selected])

    def MenuOnStopAllUrlScan(self, event):
        if self.CmdStopUrl:
            self.CmdStopUrl(all = True)

    def MenuOnDownload(self, event):
        if self.CmdDblClick:
            selected = self.GetFirstSelected()
            if selected != -1:
                thread = Thread(target = self.CmdDblClick, args = [self.curfeed.articletab[selected]])
                thread.daemon = False
                thread.start()

    def MenuOnSearchForTitle(self, event):
        if self.CmdSearchForTitleFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdSearchForTitleFunc(self.curfeed.articletab[selected].title)
                self.SetFocus()

    def MenuOnSearchForStrippedTitle(self, event):
        if self.CmdSearchForTitleFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdSearchForTitleFunc(self.stripTitle(self.curfeed.articletab[selected].title))
                self.SetFocus()

    def MenuOnCopyTitle(self, event):
        if self.CmdCopyTitleFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdCopyTitleFunc(self.curfeed.articletab[selected].title)

    def MenuOnCopyStrippedTitle(self, event):
        if self.CmdCopyTitleFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdCopyTitleFunc(self.stripTitle(self.curfeed.articletab[selected].title))

    def MenuOnCopyTitleLink(self, event):
        if self.CmdCopyTitleLinkFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdCopyTitleLinkFunc(self.curfeed.articletab[selected].titlelink)

    def MenuOnCopyURL(self, event):
        if self.CmdCopyURLFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdCopyURLFunc(self.curfeed.articletab[selected].url)

    def MenuOnBrowser(self, event):
        if self.CmdBrowFunc:
            selected = self.GetFirstSelected()
            if selected != -1:
                self.CmdBrowFunc(self.curfeed.articletab[selected])

    def stripTitle(self, title):
        return re.sub(r"[^\dA-Za-z]+", ' ', title).strip()

    def clear(self):
        self.rsspanel.html.SetPage("")
        self.DeleteAllItems()
        self.previtem = None

    def filterArticles(self, event = None, feed = None, fromrequest = True):
        filterpattern = self.rsspanel.articlefilter.GetValue().lower()
        self.sortdir = 2
        if event is None:
            # Populate from an RSS update or an existing RSS data pool
            if not feed.rawarticletab:
                self.curfeed = feed
                if fromrequest:
                    self.InsertStringItem(0, self.localize('nodatainrssfeed'))
            else:
                filteredindex = 0
                for index in xrange(feed.nbnew):
                    article = feed.rawarticletab[index]
                    ident = article.getIdentity()
                    if ident[0].lower().find(filterpattern) >= 0:
                        item = wx.ListItem()
                        item.SetText(ident[0])
                        if article.status == 1:
                            item.SetTextColour(self.utility.colnocom)
                        elif article.status == 2:
                            item.SetTextColour(self.utility.colok)
                        elif article.status == 3:
                            item.SetTextColour(self.utility.colnocon)
                        elif article.titlelinksw and ident[3] in self.rsspanel.urls \
                             or not article.titlelinksw and ident[1] in self.rsspanel.urls:
                            item.SetTextColour(self.utility.colok)
                            article.status = 2
                        item.SetFont(self.boldlistfont)
                        item.SetId(filteredindex)
                        self.InsertItem(item)
                        # Without this RefreshItem, wx.Python doesn't correctly show font weight and colour
                        # SetItemFont and SetItemTextColour are still worse, because they override each other
                        self.RefreshItem(filteredindex)
                        self.SetStringItem(filteredindex, 1, ident[1])
                        self.SetStringItem(filteredindex, 2, ident[2])
                        feed.articletab.insert(filteredindex, article)
                        filteredindex += 1
                if feed is self.curfeed:
                    for index in xrange(filteredindex, len(feed.articletab)):
                        item = self.GetItem(index)
                        item.SetTextColour(self.GetItemTextColour(index))
                        item.SetFont(self.listfont)
                        self.SetItem(item)
                        self.RefreshItem(index)
                else:
                    self.curfeed = feed
                    for index in xrange(feed.nbnew, len(feed.rawarticletab)):
                        article = feed.rawarticletab[index]
                        ident = article.getIdentity()
                        if ident[0].lower().find(filterpattern) >= 0:
                            item = wx.ListItem()
                            item.SetText(ident[0])
                            if article.status == 1:
                                item.SetTextColour(self.utility.colnocom)
                            elif article.status == 2:
                                item.SetTextColour(self.utility.colok)
                            elif article.status == 3:
                                item.SetTextColour(self.utility.colnocon)
                            elif article.titlelinksw and ident[3] in self.rsspanel.urls \
                                 or not article.titlelinksw and ident[1] in self.rsspanel.urls:
                                item.SetTextColour(self.utility.colok)
                                article.status = 2
                            if article.new:
                                item.SetFont(self.boldlistfont)
                            item.SetId(filteredindex)
                            self.InsertItem(item)
                            self.RefreshItem(filteredindex)
                            self.SetStringItem(filteredindex, 1, ident[1])
                            self.SetStringItem(filteredindex, 2, ident[2])
                            feed.articletab.append(article)
                            filteredindex += 1
            self.rsspanel.editfeedalias.Enable()
            self.rsspanel.updatebutt.Enable()
            self.rsspanel.clearlistbutt.Enable()
        elif self.curfeed and self.curfeed.rawarticletab:
            # Filter pattern has changed
            selected = self.GetFirstSelected()
            if selected != -1:
                selectedarticle = self.curfeed.articletab[selected]
            else:
                selectedarticle = None
            self.DeleteAllItems()
            self.curfeed.articletab[:] = []
            filteredindex = 0
            for article in self.curfeed.rawarticletab:
                ident = article.getIdentity()
                if ident[0].lower().find(filterpattern) >= 0:
                    item = wx.ListItem()
                    item.SetText(ident[0])
                    if article.status == 1:
                        item.SetTextColour(self.utility.colnocom)
                    elif article.status == 2:
                        item.SetTextColour(self.utility.colok)
                    elif article.status == 3:
                        item.SetTextColour(self.utility.colnocon)
                    elif article.titlelinksw and ident[3] in self.rsspanel.urls \
                         or not article.titlelinksw and ident[1] in self.rsspanel.urls:
                        item.SetTextColour(self.utility.colok)
                        article.status = 2
                    if article.new:
                        item.SetFont(self.boldlistfont)
                    item.SetId(filteredindex)
                    self.InsertItem(item)
                    self.RefreshItem(filteredindex)
                    self.SetStringItem(filteredindex, 1, ident[1])
                    self.SetStringItem(filteredindex, 2, ident[2])
                    self.curfeed.articletab.append(article)
                    filteredindex += 1
            if selectedarticle is not None:
                try:
                    self.Select(self.curfeed.articletab.index(selectedarticle))
                except:
                    self.rsspanel.html.SetPage("")
                    self.previtem = None

    def setStatus(self, article, status):
        article.status = status
        try:
            index = self.curfeed.articletab.index(article)
        except:
            return
        item = self.GetItem(index)
        if status == 0:
            item.SetTextColour(self.utility.colnoeng)
        elif status == 1:
            item.SetTextColour(self.utility.colnocom)
        elif status == 2:
            item.SetTextColour(self.utility.colok)
        else:
            item.SetTextColour(self.utility.colnocon)
        if article.new:
            item.SetFont(self.boldlistfont)
        self.SetItem(item)
        self.RefreshItem(index)

    def updateColumnWidth(self, col):
        # Save a column width from the spew list
        newcolwidth = self.GetColumnWidth(col)
        self.utility.rsscolumns[col][1] = newcolwidth
        self.utility.abcparams[self.utility.rsscolinfo[col][1]] = str(newcolwidth)

    def onResizeColumn(self, event):
        event.Skip()
        wx.CallAfter(self.updateColumnWidth, event.GetColumn())


class FeedersPanel(wx.ScrolledWindow):
    def __init__(self, parent, utility, feedalias):
        wx.ScrolledWindow.__init__(self, parent, -1, style = wx.SUNKEN_BORDER | wx.FULL_REPAINT_ON_RESIZE)
        self.parent = parent
        self.utility = utility
        self.localize = self.utility.lang.get
        self.feedalias = feedalias
        self.SetScrollbars(0, 2, -1, -1)
        self.EnableScrolling(False, True)
        self.rssscanfreqdlg = None

        colsizer = wx.BoxSizer(wx.VERTICAL)

        setsizer = wx.BoxSizer(wx.HORIZONTAL)

        # Data edit boxes
        datasizer = wx.FlexGridSizer(5, 2, 0, 0)
        datasizer.AddGrowableCol(1)

        # Feeders box
        self.feederlist = wx.ListCtrl(self, -1, size = (250, -1), style = wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.feederlist.InsertColumn(0, self.localize('rssfeeders'), width = 246)
        self.feederlist.Bind(wx.EVT_LEFT_DOWN, self.OnListBox)
        self.feederlist.Bind(wx.EVT_KEY_DOWN, self.onKeyInList)
        self.feederlist.Bind(wx.EVT_RIGHT_DOWN, self.onRightInList)
        self.feederlist.Bind(wx.EVT_RIGHT_DCLICK, self.onRightInList)

        setsizer.Add(self.feederlist, 0, wx.EXPAND | wx.ALL, 2)

        self.oldselectedindex = self.selectedindex = -1
        self.populateFeederList()

        # Name of the feeder
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfeedername')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)
        self.feedername = wx.TextCtrl(self, -1, "")
        self.feedername.Bind(wx.EVT_CHAR, self.onUpdateFeederName)
        self.feedername.Bind(wx.EVT_TEXT, self.onUpdateFeederNameText)
        self.feedername.Bind(wx.EVT_TEXT_PASTE, self.onPasteFeederName)
        datasizer.Add(self.feedername, 0, wx.EXPAND | wx.ALL, 2)

        # URL
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfeederurl')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)
        self.feederurl = wx.TextCtrl(self, -1, "")
        self.feederurl.Bind(wx.EVT_CHAR, self.onUpdateFeederURL)
        self.feederurl.Bind(wx.EVT_TEXT_PASTE, self.onPasteFeederURL)
        datasizer.Add(self.feederurl, 0, wx.EXPAND | wx.ALL, 2)

        # Cookies
        self.cookieswitch = wx.CheckBox(self, -1, self.localize('rsscookies'))
        self.cookieswitch.Bind(wx.EVT_CHECKBOX, self.onUpdateCookieSwitch)
        datasizer.Add(self.cookieswitch, 0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)
        self.cookie = wx.TextCtrl(self, -1, "")
        self.cookie.Bind(wx.EVT_CHAR, self.onUpdateCookie)
        self.cookie.Bind(wx.EVT_TEXT_PASTE, self.onPasteCookie)
        datasizer.Add(self.cookie, 0, wx.EXPAND | wx.ALL, 2)

        # Timer
        self.timerswitch = wx.CheckBox(self, -1, self.localize('rsstimer'))
        self.timerswitch.Bind(wx.EVT_CHECKBOX, self.onUpdateTimerSwitch)
        datasizer.Add(self.timerswitch, 0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)
        timerbox = wx.BoxSizer(wx.HORIZONTAL)
        self.dtimer = wx.TextCtrl(self, -1, '', (-1, -1), (31, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.dtimer.Bind(wx.EVT_CHAR, self.onUpdateTimer)
        self.dtimer.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.dtimer.Bind(wx.EVT_TEXT_PASTE, self.onPasteTimer)
        timerbox.Add(self.dtimer, 0)
        timerbox.Add(wx.StaticText(self, -1, self.localize('l_day')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)
        self.htimer = wx.TextCtrl(self, -1, '', (-1, -1), (25, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.htimer.Bind(wx.EVT_CHAR, self.onUpdateTimer)
        self.htimer.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.htimer.Bind(wx.EVT_TEXT_PASTE, self.onPasteTimer)
        timerbox.Add(self.htimer, 0, wx.LEFT, 3)
        timerbox.Add(wx.StaticText(self, -1, self.localize('l_hour')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)
        self.mtimer = wx.TextCtrl(self, -1, '', (-1, -1), (25, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.mtimer.Bind(wx.EVT_CHAR, self.onUpdateTimer)
        self.mtimer.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        self.mtimer.Bind(wx.EVT_TEXT_PASTE, self.onPasteTimer)
        timerbox.Add(self.mtimer, 0, wx.LEFT, 3)
        timerbox.Add(wx.StaticText(self, -1, self.localize('l_minute')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)

        datasizer.Add(timerbox, 0, wx.ALL, 2)

        # Article title link
        self.titlelinkswitch = wx.CheckBox(self, -1, self.localize('rsstitlelink'))
        self.titlelinkswitch.Bind(wx.EVT_CHECKBOX, self.onUpdateTitleLinkSwitch)
        datasizer.Add(self.titlelinkswitch, 0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)

        setsizer.Add(datasizer, 1, wx.EXPAND)

        colsizer.Add(setsizer, 1, wx.EXPAND)

        # New feeder
        newbtn = wx.Button(self, -1, self.localize('rssnew'))
        self.Bind(wx.EVT_BUTTON, self.OnNewEntry, newbtn)

        # Save feeder
        self.svbtn = wx.Button(self, -1, self.localize('rsssave'))
        self.Bind(wx.EVT_BUTTON, self.OnSaveEntry, self.svbtn)

        # Delete feeder
        self.rmbtn = wx.Button(self, -1, self.localize('rssremove'))
        self.Bind(wx.EVT_BUTTON, self.OnRemoveEntry, self.rmbtn)

        # Set default timer value
        self.setdeftimerbtn = wx.Button(self, -1, self.localize('rsssetdeftimer'))
        self.Bind(wx.EVT_BUTTON, self.SetDefTimer, self.setdeftimerbtn)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        buttonbox.Add(newbtn)
        buttonbox.Add(self.svbtn, 0, wx.LEFT, 20)
        buttonbox.Add(self.rmbtn, 0, wx.LEFT, 20)
        buttonbox.Add(self.setdeftimerbtn, 0, wx.LEFT, 20)

        colsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.ALL, 5)
        self.SetAutoLayout(True)
        self.SetSizer(colsizer)
        colsizer.Fit(self)

        self.OnListBox()

    def onRightInList(self, event):
        pass

    def onRightButtonText(self, event):
        pass

    def setSaveButtonStatus(self, originalvalue, newvaluegetter):
        newval = newvaluegetter()
        if self.svbtn.IsEnabled():
            if newval == originalvalue and not self.feederEdited():
                self.svbtn.Disable()
        elif newval != originalvalue:
            self.svbtn.Enable()

    def onUpdateFeederName(self, event):
        if self.utility.onWriteFilteredText(event):
            wx.CallAfter(self.setSaveButtonStatus, self.feed.name, self.feedername.GetValue)

    def onPasteFeederName(self, event):
        self.utility.onPasteText(event)
        wx.CallAfter(self.setSaveButtonStatus, self.feed.name, self.feedername.GetValue)

    def onUpdateFeederURL(self, event):
        if self.utility.onWriteFilteredText(event):
            wx.CallAfter(self.setSaveButtonStatus, self.feed.url, self.feederurl.GetValue)

    def onPasteFeederURL(self, event):
        self.utility.onPasteText(event)
        wx.CallAfter(self.setSaveButtonStatus, self.feed.url, self.feederurl.GetValue)

    def onUpdateCookieSwitch(self, event):
        event.Skip()
        self.setSaveButtonStatus(self.feed.cookiesw, self.cookieswitch.GetValue)

    def onUpdateCookie(self, event):
        if self.utility.onWriteFilteredText(event):
            wx.CallAfter(self.setSaveButtonStatus, self.feed.cookie, self.cookie.GetValue)

    def onPasteCookie(self, event):
        self.utility.onPasteText(event)
        wx.CallAfter(self.setSaveButtonStatus, self.feed.cookie, self.cookie.GetValue)

    def onUpdateTimerSwitch(self, event):
        event.Skip()
        self.setSaveButtonStatus(self.feed.timersw, self.timerswitch.GetValue)

    def onUpdateTitleLinkSwitch(self, event):
        event.Skip()
        self.setSaveButtonStatus(self.feed.titlelinksw, self.titlelinkswitch.GetValue)

    def getTimerTotalMinutes(self):
        dtime = self.dtimer.GetValue()
        htime = self.htimer.GetValue()
        mtime = self.mtimer.GetValue()
        if not dtime:
            dtime = '0'
        if not htime:
            htime = '0'
        if not mtime:
            mtime = '0'
        timererr = False
        try:
            dtime_int = int(dtime)
            htime_int = int(htime)
            mtime_int = int(mtime)
        except:
            timererr = True
        if timererr or dtime_int < 0 or htime_int < 0 or mtime_int < 0:
            dlg = wx.MessageDialog(self, self.localize('errorfeedertimer'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return None
        return str(1440 * dtime_int + 60 * htime_int + mtime_int)

    def onUpdateTimer(self, event):
        if self.utility.onWriteDigits(event):
            wx.CallAfter(self.setSaveButtonStatus, self.feed.timer, self.getTimerTotalMinutes)

    def onPasteTimer(self, event):
        event.Skip()
        wx.CallAfter(self.setSaveButtonStatus, self.feed.timer, self.getTimerTotalMinutes)

    def onUpdateFeederNameText(self, event):
        # Catch exception to avoid silly assertion in wxPython 2.7.1.3
        try:
            self.feederlist.SetStringItem(self.selectedindex, 0, self.feedername.GetValue())
        except:
            pass

    def onKeyInList(self, event):
        if not self.feederlist.GetItemCount():
            return
        keycode = event.GetKeyCode()
        if keycode == wx.WXK_UP:
            self.OnListBox(index = max(0, self.selectedindex - 1))
        elif keycode == wx.WXK_DOWN:
            self.OnListBox(index = min(self.feederlist.GetItemCount() - 1, self.selectedindex + 1))
        elif keycode == wx.WXK_PRIOR:
            self.OnListBox(index = max(0, self.selectedindex - self.feederlist.GetCountPerPage()))
        elif keycode == wx.WXK_NEXT:
            self.OnListBox(index = min(self.feederlist.GetItemCount() - 1, self.selectedindex + self.feederlist.GetCountPerPage()))
        elif keycode == wx.WXK_HOME:
            self.OnListBox(index = 0)
        elif keycode == wx.WXK_END:
            self.OnListBox(index = self.feederlist.GetItemCount() - 1)

    def getIndexFromName(self, name, skipcurrent = False):
        # Find the index that exactly matches name (FindItem can't be used because it doesn't take into account the case)
        maxfeedernb = len(self.feedalias)
        index = 0
        while index < maxfeedernb and (self.feedalias[index] != name or (skipcurrent and index == self.selectedindex)):
            index += 1
        if index == maxfeedernb:
            return -1
        return index

    def OnListBox(self, event = None, index = None):
        savingneeded = False
        dropnewentry = False
        if index is not None or event is not None:
            if index is None:
                clickedindex = self.feederlist.HitTest(wx.Point(event.GetX(), event.GetY()))[0]
            else:
                clickedindex = index
            if clickedindex == self.selectedindex:
                if event is not None:
                    event.Skip()
                return
            if clickedindex >= 0:
                clickedname = self.feedalias[clickedindex]
            if self.svbtn.IsEnabled():
                dlg = wx.MessageDialog(self, self.localize('rssfeedersaveerror'),
                                       self.localize('rssfeedersaveerror_short'),
                                       wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION)
                dlgresult = dlg.ShowModal()
                dlg.Destroy()
                if dlgresult == wx.ID_CANCEL:
                    return
                if dlgresult == wx.ID_YES:
                    savingneeded = True
                elif not self.feed.name:
                    dropnewentry = True
                    self.OnRemoveEntry()

            if not dropnewentry:
                if not savingneeded:
                    if self.selectedindex >= 0:
                        self.feederlist.SetStringItem(self.selectedindex, 0, self.feed.name)
                elif not self.OnSaveEntry():
                    return

            self.oldselectedindex = self.selectedindex
            if clickedindex >= 0:
                self.selectedindex = self.getIndexFromName(clickedname)
            else:
                self.selectedindex = -1

        if self.selectedindex == -1:
            if event is not None:
                event.Skip()
            # Disable save/remove buttons and other fields/values if nothing is selected
            self.rmbtn.Disable()
            self.feedername.SetValue('')
            self.feederurl.SetValue('')
            self.cookieswitch.SetValue(False)
            self.cookie.SetValue('')
            self.timerswitch.SetValue(False)
            self.dtimer.SetValue('')
            self.htimer.SetValue('')
            self.mtimer.SetValue('')
            self.titlelinkswitch.SetValue(False)
            self.feedername.Disable()
            self.feederurl.Disable()
            self.cookieswitch.Disable()
            self.cookie.Disable()
            self.timerswitch.Disable()
            self.dtimer.Disable()
            self.htimer.Disable()
            self.mtimer.Disable()
            self.titlelinkswitch.Disable()
        else:
            self.feederlist.SetFocus()
            self.feederlist.Select(self.selectedindex)
            self.feederlist.Focus(self.selectedindex)
            self.feed = self.parent.feeds[self.feedalias[self.selectedindex]]
            # Re-enable fields
            if self.oldselectedindex == -1:
                self.feedername.Enable()
                self.feederurl.Enable()
                self.cookieswitch.Enable()
                self.cookie.Enable()
                self.timerswitch.Enable()
                self.dtimer.Enable()
                self.htimer.Enable()
                self.mtimer.Enable()
                self.titlelinkswitch.Enable()
                self.rmbtn.Enable()
            # Name
            self.feedername.SetValue(self.feed.name)
            # URL
            self.feederurl.SetValue(self.feed.url)
            # Cookies
            self.cookieswitch.SetValue(self.feed.cookiesw)
            self.cookie.SetValue(self.feed.cookie)
            # Timer
            self.timerswitch.SetValue(self.feed.timersw)
            dtime, mtime = divmod(int(self.feed.timer), 1440)
            htime, mtime = divmod(mtime, 60)
            self.dtimer.SetValue(str(dtime))
            self.htimer.SetValue(str(htime))
            self.mtimer.SetValue(str(mtime))
            # Article title link
            self.titlelinkswitch.SetValue(self.feed.titlelinksw)
        self.svbtn.Disable()

    def populateFeederList(self, keepselection = False):
        if keepselection and self.selectedindex != -1:
            selected = self.feedalias[self.selectedindex]
        self.feedalias.sort(lambda u, v: cmp(u.lower(), v.lower()) or cmp(u, v))
        self.feederlist.Freeze()
        self.feederlist.DeleteAllItems()
        for index in xrange(len(self.feedalias)):
            self.feederlist.InsertStringItem(index, self.feedalias[index])
        self.feederlist.Thaw()
        if keepselection and self.selectedindex != -1:
            self.selectedindex = self.feedalias.index(selected)
            self.feederlist.Select(self.selectedindex)

    def OnNewEntry(self, event):
        savingneeded = False
        dropnewentry = False
        if self.svbtn.IsEnabled():
            dlg = wx.MessageDialog(self, self.localize('rssfeedersaveerror'),
                                   self.localize('rssfeedersaveerror_short'),
                                   wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION)
            dlgresult = dlg.ShowModal()
            dlg.Destroy()
            if dlgresult == wx.ID_CANCEL:
                return
            if dlgresult == wx.ID_YES:
                savingneeded = True
            elif not self.feed.name:
                dropnewentry = True
                self.OnRemoveEntry()

        if not dropnewentry:
            if not savingneeded:
                if self.selectedindex >= 0:
                    self.feederlist.SetStringItem(self.selectedindex, 0, self.feed.name)
            elif not self.OnSaveEntry():
                return

        self.feedalias.append('')
        self.oldselectedindex = self.selectedindex
        self.selectedindex = self.feederlist.GetItemCount()
        self.parent.feeds[''] = RSSFeeder(self.utility, self.parent, '', '', False, '', False, '5', False)
        self.feederlist.InsertStringItem(self.selectedindex, '')
        self.feederlist.Select(self.selectedindex)
        self.feederlist.Focus(self.selectedindex)
        self.OnListBox()
        self.feedername.SetFocus()
        self.svbtn.Enable()

    def feederEdited(self):
        if not self.feed.name:
            return True
        if self.feedername.GetValue() == self.feed.name and self.feederurl.GetValue() == self.feed.url \
           and self.cookieswitch.GetValue() == self.feed.cookiesw and self.cookie.GetValue() == self.feed.cookie \
           and self.timerswitch.GetValue() == self.feed.timersw and self.titlelinkswitch.GetValue() == self.feed.titlelinksw:
            dtime = self.dtimer.GetValue()
            htime = self.htimer.GetValue()
            mtime = self.mtimer.GetValue()
            if not dtime:
                dtime = '0'
            if not htime:
                htime = '0'
            if not mtime:
                mtime = '0'
            if str(1440 * int(dtime) + 60 * int(htime) + int(mtime)) == self.feed.timer:
                return False
            return True
        return True

    def OnSaveEntry(self, event = None):
        newname = self.feedername.GetValue()

        # Name must not be blank
        if newname == '':
            dlg = wx.MessageDialog(self, self.localize('errorfeedername'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.feedername.SetFocus()
            return False

        # Check if name already exists
        if self.getIndexFromName(newname, skipcurrent = True) != -1:
            dlg = wx.MessageDialog(self, self.localize('errorfeederexists'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.feedername.SetFocus()
            self.feedername.SetSelection(-1, -1)
            return False

        newurl = self.feederurl.GetValue()
        # URL must not be blank
        if newurl == '':
            dlg = wx.MessageDialog(self, self.localize('errorfeederurl'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.feederurl.SetFocus()
            return False

        # Twice the same URL not allowed
        index = 0
        for f in self.feedalias:
            if self.parent.feeds[f].url == newurl and index != self.selectedindex:
                dlg = wx.MessageDialog(self, self.localize('errorurl') + '\n' + self.parent.feeds[f].name,
                                       self.localize('abcokcerror'), wx.ICON_ERROR)
                dlg.ShowModal()
                dlg.Destroy()
                self.feederurl.SetFocus()
                return False
            index += 1

        newtimer = self.getTimerTotalMinutes()
        if newtimer is None:
            return False
        newtimer_int = int(newtimer)
        # Timer must be >= 5 mn
        if newtimer_int < 5:
            dlg = wx.MessageDialog(self, self.localize('rssscanfreqwarning'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.mtimer.SetFocus()
            return False
        dtime_int, mtime_int = divmod(newtimer_int, 1440)
        htime_int, mtime_int = divmod(mtime_int, 60)
        self.dtimer.SetValue(str(dtime_int))
        self.htimer.SetValue(str(htime_int))
        self.mtimer.SetValue(str(mtime_int))

        oldname = self.feedalias[self.selectedindex]
        editedfeed = self.parent.feeds[oldname]
        oldtimer = editedfeed.timer
        oldtimersw = editedfeed.timersw

        # Save new feeder
        if newname != oldname:
            self.parent.feeds[newname] = editedfeed
            del self.parent.feeds[oldname]
        editedfeed.url = newurl
        editedfeed.name = newname
        editedfeed.cookiesw = self.cookieswitch.GetValue()
        editedfeed.cookie = self.cookie.GetValue()
        editedfeed.timersw = self.timerswitch.GetValue()
        editedfeed.timer = newtimer
        editedfeed.titlelinksw = self.titlelinkswitch.GetValue()
        self.feedalias[self.selectedindex] = newname
        self.populateFeederList()

        if newname != oldname:
            if oldname:
                # Not a new feeder
                # Rename feeder from feeder list for each filter if needed
                for rule in self.parent.filterspanel.rules:
                    if oldname in rule[1]:
                        rule[1].remove(oldname)
                        rule[1].append(newname)
                        rule[1].sort(lambda u, v: cmp(u.lower(), v.lower()) or cmp(u, v))
                # Populate feeder combo in filters panel for currently selected filter
                self.parent.filterspanel.renameFeed(oldname, newname)
            # Populate main feeder combo
            self.parent.populateMainFeederCombo()

        if oldname and self.utility.abcparams['rssautograb'] == '1':
            # Not a new feeder (If it's a new feeder no filter reference it yet)
            # If timer switch has changed
            if editedfeed.timersw != oldtimersw:
                if oldtimersw:
                    self.stopTimer()
                self.parent.manageTimers()
            elif oldtimersw and newtimer_int != int(oldtimer):
                # Timer value has changed
                editedfeed.resetTimer()
        else:
            self.parent.scanFeedsActivity()

        self.selectedindex = self.getIndexFromName(newname)
        if event is not None:
            self.feederlist.Select(self.selectedindex)
        self.parent.SaveFeeds()
        self.parent.SaveRules()
        self.svbtn.Disable()

        return True

    def OnRemoveEntry(self, event = None):
        if self.selectedindex >= 0:
            feedername = self.feedalias[self.selectedindex]
            feed = self.parent.feeds[feedername]
            del self.feedalias[self.selectedindex]
            del self.parent.feeds[feedername]
            if feedername:
                if self.utility.abcparams['rssautograb'] == '1':
                    # Delete feeder from active feeder with own timer list if needed
                    if feed in self.parent.activefeedswithowntimer:
                        feed.stopTimer()
                        self.parent.activefeedswithowntimer.remove(feed)
                    elif feed in self.parent.activefeedswithglobaltimer:
                        self.parent.activefeedswithglobaltimer.remove(feed)
                        if not self.parent.activefeedswithglobaltimer:
                            self.parent.stopGlobalTimer()
                    elif feed in self.parent.inactivefeeds:
                        self.parent.inactivefeeds.remove(feed)

                # Delete feeder from the feeder list of each filter if needed
                for rule in self.parent.filterspanel.rules:
                    if feedername in rule[1]:
                        rule[1].remove(feedername)

                # Update feeder combo in filters panel for currently selected filter
                # The removed feeder may be referenced in the combo without being saved
                self.parent.filterspanel.removeFeed(feedername)
    
                if self.parent.editfeedalias.GetValue() == feedername:
                    self.parent.editfeedalias.SetValue('')
    
                self.parent.SaveFeeds()
                self.parent.SaveRules()

            # Populate main feeder combo
            self.parent.populateMainFeederCombo()
            self.feederlist.DeleteItem(self.selectedindex)
            self.selectedindex = -1
            self.OnListBox()

    def SetDefTimer(self, event):
        self.rssscanfreqdlg = RSSScanFreqDialog(self, -1, self.localize('rssscanfreqtitle'))
        self.rssscanfreqdlg.ShowModal()
        x, y = self.rssscanfreqdlg.GetPositionTuple()
        self.utility.abcparams['rssscanfreqdlgx'], self.utility.abcparams['rssscanfreqdlgy'] = str(x), str(y)
        self.rssscanfreqdlg.Destroy()
        self.rssscanfreqdlg = None


class FiltersPanel(wx.ScrolledWindow):
    def __init__(self, parent, utility, feedalias, rules):
        wx.ScrolledWindow.__init__(self, parent, -1, style = wx.SUNKEN_BORDER | wx.FULL_REPAINT_ON_RESIZE)
        self.parent = parent
        self.rules = rules
        self.utility = utility
        self.localize = self.utility.lang.get
        self.feedalias = feedalias
        self.nbactivefilter = 0
        self.SetScrollbars(0, 2, -1, -1)
        self.EnableScrolling(False, True)

        colsizer = wx.BoxSizer(wx.VERTICAL)

        setsizer = wx.BoxSizer(wx.HORIZONTAL)

        # Data edit boxes
        datasizer = wx.FlexGridSizer(8, 3, 0, 0)
        datasizer.AddGrowableCol(1)

        # Rules box
        self.rulelist = wx.ListCtrl(self, -1, size = (180, -1), style = wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.rulelist.InsertColumn(0, self.localize('rssfilters'), width = 176)
        self.rulelist.Bind(wx.EVT_LEFT_DOWN, self.OnListBox)
        self.rulelist.Bind(wx.EVT_KEY_DOWN, self.onKeyInList)
        self.rulelist.Bind(wx.EVT_RIGHT_DOWN, self.onRightInList)
        self.rulelist.Bind(wx.EVT_RIGHT_DCLICK, self.onRightInList)

        setsizer.Add(self.rulelist, 0, wx.EXPAND | wx.ALL, 2)

        self.oldselectedindex = self.selectedindex = -1
        self.populateFilterList()

        # Label the auto-grab button
        self.parent.setAutoGrabButtonLabel(self.nbactivefilter)

        # Name of the rule
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfiltername')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.rulename = wx.TextCtrl(self, -1, "")
        self.rulename.Bind(wx.EVT_CHAR, self.onUpdateRuleName)
        self.rulename.Bind(wx.EVT_TEXT, self.onUpdateRuleNameText)
        self.rulename.Bind(wx.EVT_TEXT_PASTE, self.onPasteRuleName)
        datasizer.Add(self.rulename, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        datasizer.Add((16, -1))

        # Priority
        datasizer.Add(wx.StaticText(self, -1, self.localize('rsspriority')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.priorities = [self.localize('highest'), 
                           self.localize('high'), 
                           self.localize('normal'), 
                           self.localize('low'), 
                           self.localize('lowest')]
        self.defaultpriority = wx.ComboBox(self, -1, "", choices = self.priorities,
                                           style = wx.CB_DROPDOWN | wx.CB_READONLY, size = (120, -1))
        self.defaultpriority.Bind(wx.EVT_COMBOBOX, self.onUpdateDefaultPriority)
        datasizer.Add(self.defaultpriority, 0, wx.FIXED_MINSIZE | wx.ALL, 2)
        datasizer.Add((16, -1))

        # Status
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssstatus')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.statuses = [self.localize('queue'), 
                         self.localize('stop'), 
                         self.localize('working')]
        self.defaultstatus = wx.ComboBox(self, -1, "",
                                         choices = [self.localize('settostop'), self.localize('settoqueue'), self.localize('start')],
                                         style = wx.CB_DROPDOWN | wx.CB_READONLY, size = (120, -1))
        self.defaultstatus.Bind(wx.EVT_COMBOBOX, self.onUpdateDefaultStatus)
        datasizer.Add(self.defaultstatus, 0, wx.FIXED_MINSIZE | wx.ALL, 2)
        datasizer.Add((16, -1))

        # Feed choose
        self.currentfeeds = [] 
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfilterfeed')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.feedchoosebox = wx.ComboBox(self, -1, "", choices = self.currentfeeds,
                                         style = wx.CB_DROPDOWN | wx.CB_READONLY, size = (150, -1))
        datasizer.Add(self.feedchoosebox, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.columndlgbtn = wx.Button(self, -1, self.localize('feedchoosesetfeed'))
        self.columndlgbtn.SetToolTipString(self.localize('columndlgbtntt'))
        datasizer.Add(self.columndlgbtn, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        wx.EVT_BUTTON(self, self.columndlgbtn.GetId(), self.onFeedsDialog)

        # +Filter
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfilterpositive')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.filter = []
        self.edit3 = wx.ComboBox(self, -1, "", choices = self.filter,
                                 style = wx.CB_DROPDOWN | wx.TE_PROCESS_ENTER, size = (150, -1))
        datasizer.Add(self.edit3, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.edit3.Bind(wx.EVT_CHAR, self.utility.onWriteFilteredText2)
        self.edit3.Bind(wx.EVT_TEXT_PASTE, self.utility.onPasteText2)
        self.edit3.Bind(wx.EVT_TEXT_ENTER, self.onAddFilter)
        self.amendbutt = wx.Button(self, -1, self.localize('nfiltersavedel'))
        self.amendbutt.SetToolTipString(self.localize('sdfiltertt'))
        datasizer.Add(self.amendbutt, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        wx.EVT_BUTTON(self, self.amendbutt.GetId(), self.onAddRemoveFilter)

        # -Filter (negfilter)
        datasizer.Add(wx.StaticText(self, -1, self.localize('rssfilternegative')),
                      0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.negfilter = []
        self.edit4 = wx.ComboBox(self, -1, "", choices = self.negfilter,
                                 style = wx.CB_DROPDOWN | wx.TE_PROCESS_ENTER, size = (150, -1))
        datasizer.Add(self.edit4, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        self.edit4.Bind(wx.EVT_CHAR, self.utility.onWriteFilteredText2)
        self.edit4.Bind(wx.EVT_TEXT_PASTE, self.utility.onPasteText2)
        self.edit4.Bind(wx.EVT_TEXT_ENTER, self.onAddNegFilter)
        self.amendbutt2 = wx.Button(self, -1, self.localize('nfiltersavedel'))
        self.amendbutt2.SetToolTipString(self.localize('sdnegfiltertt'))
        datasizer.Add(self.amendbutt2, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        wx.EVT_BUTTON(self, self.amendbutt2.GetId(), self.onAddRemoveNegFilter)

        datasizer.Add((16, -1))

        # Regular expression
        self.regularexpcheck = wx.CheckBox(self, -1, self.localize('rssregularexp'))
        self.regularexpcheck.Bind(wx.EVT_CHECKBOX, self.onUpdateRegularExpCheck)
        datasizer.Add(self.regularexpcheck, 0, wx.ALIGN_LEFT | wx.ALL | wx.FIXED_MINSIZE, 2)
        datasizer.Add((16, -1))

        # Download location
        self.downlocswitch = wx.CheckBox(self, -1, self.localize('rssdownloc'))
        self.downlocswitch.Bind(wx.EVT_CHECKBOX, self.onUpdateDownLocSwitch)
        datasizer.Add(self.downlocswitch, 0, wx.ALIGN_CENTER_VERTICAL | wx.ALIGN_LEFT | wx.ALL, 2)
        self.downloc = wx.TextCtrl(self, -1, "")
        self.downloc.Bind(wx.EVT_CHAR, self.onUpdateDownLoc)
        datasizer.Add(self.downloc, 0, wx.EXPAND | wx.ALL, 2)
        self.browsebtn = wx.Button(self, -1, "...", (-1, -1), (-1, 20))
        self.browsebtn.SetToolTipString(self.localize('rssbrowsebtntt'))
        datasizer.Add(self.browsebtn, 0, wx.EXPAND | wx.FIXED_MINSIZE | wx.ALL, 2)
        wx.EVT_BUTTON(self, self.browsebtn.GetId(), self.onBrowseDownDir)

        setsizer.Add(datasizer, 1, wx.EXPAND)

        colsizer.Add(setsizer, 1, wx.EXPAND)

        # Activity
        self.active = wx.CheckBox(self, -1, self.localize('rssfilteractivity'))
        self.active.Bind(wx.EVT_CHECKBOX, self.onActivitySwitch)
        colsizer.Add(self.active, 0, wx.ALIGN_LEFT | wx.ALL, 2)

        # New rule
        newbtn = wx.Button(self, -1, self.localize('rssnew'))
        self.Bind(wx.EVT_BUTTON, self.OnNewEntry, newbtn)

        # Save rule
        self.svbtn = wx.Button(self, -1, self.localize('rsssave'))
        self.Bind(wx.EVT_BUTTON, self.OnSaveEntry, self.svbtn)

        # Delete rule
        self.rmbtn = wx.Button(self, -1, self.localize('rssremove'))
        self.Bind(wx.EVT_BUTTON, self.OnRemoveEntry, self.rmbtn)

        buttonbox = wx.BoxSizer(wx.HORIZONTAL)
        buttonbox.Add(newbtn)
        buttonbox.Add(self.svbtn, 0, wx.LEFT, 20)
        buttonbox.Add(self.rmbtn, 0, wx.LEFT, 20)

        colsizer.Add(buttonbox, 0, wx.ALIGN_CENTER_HORIZONTAL | wx.BOTTOM, 5)
        self.SetAutoLayout(True)
        self.SetSizer(colsizer)
        colsizer.Fit(self)

        self.OnListBox()

    def onRightInList(self, event):
        pass

    def setSaveButtonStatus(self, originalvalue, newvaluegetter = None, newvalue = None):
        if newvaluegetter is None:
            newval = newvalue
        else:
            newval = newvaluegetter()
        if self.svbtn.IsEnabled():
            if newval == originalvalue and not self.ruleEdited():
                self.svbtn.Disable()
        elif newval != originalvalue:
            self.svbtn.Enable()

    def onUpdateRuleName(self, event):
        if self.utility.onWriteFilteredText(event):
            wx.CallAfter(self.setSaveButtonStatus, self.rule[0], self.rulename.GetValue)

    def onPasteRuleName(self, event):
        self.utility.onPasteText(event)
        wx.CallAfter(self.setSaveButtonStatus, self.rule[0], self.rulename.GetValue)

    def onUpdateDefaultPriority(self, event):
        self.setSaveButtonStatus(self.rule[6], event.GetSelection)

    def onUpdateDefaultStatus(self, event):
        self.setSaveButtonStatus(self.rule[7], event.GetSelection)

    def onUpdateRegularExpCheck(self, event):
        event.Skip()
        self.setSaveButtonStatus(self.rule[5], self.regularexpcheck.GetValue)

    def onUpdateDownLocSwitch(self, event):
        event.Skip()
        self.setSaveButtonStatus(self.rule[8], self.downlocswitch.GetValue)

    def onUpdateDownLoc(self, event):
        event.Skip()
        wx.CallAfter(self.setSaveButtonStatus, self.rule[9], self.downloc.GetValue)

    def onUpdateRuleNameText(self, event):
        # Catch exception to avoid silly assertion in wxPython 2.7.1.3
        try:
            self.rulelist.SetStringItem(self.selectedindex, 0, self.rulename.GetValue())
        except:
            pass

    def onBrowseDownDir(self, event):
        folderdialog = wx.DirDialog(self, self.localize('choosedefaultdownloadfolder'), 
                                    style = wx.DD_DEFAULT_STYLE | wx.DD_NEW_DIR_BUTTON)
        if folderdialog.ShowModal() == wx.ID_OK:
            self.downloc.SetValue(folderdialog.GetPath())
            self.setSaveButtonStatus(self.rule[9], self.downloc.GetValue)
        folderdialog.Destroy()

    def onFeedsDialog(self, event = None):
        dialog = FeedsDialog(self)
        dialog.ShowModal()
        dialog.Destroy()

    def onAddFilter(self, event):
        item = self.edit3.GetValue().rstrip()
        if item and not item in self.filter:
            self.filter.append(item)
            self.edit3.Append(item)
            self.edit3.SetFocus()
            self.setSaveButtonStatus(self.rule[2], newvalue = self.filter)

    def onAddRemoveFilter(self, event):
        item = self.edit3.GetValue().rstrip()
        if item:
            if item in self.filter:
                self.filter.remove(item)
                self.edit3.Delete(self.edit3.FindString(item))
                if self.filter:
                    self.edit3.SetValue(self.filter[0])
                else:
                    self.edit3.SetValue("")
            else:
                self.filter.append(item)
                self.edit3.Append(item)
            self.edit3.SetFocus()
            self.setSaveButtonStatus(self.rule[2], newvalue = self.filter)

    def onAddNegFilter(self, event):
        item = self.edit4.GetValue().rstrip()
        if item and not item in self.negfilter:
            self.negfilter.append(item)
            self.edit4.Append(item)
            self.edit4.SetFocus()
            self.setSaveButtonStatus(self.rule[3], newvalue = self.negfilter)

    def onAddRemoveNegFilter(self, event):
        item = self.edit4.GetValue().rstrip()
        if item:
            if item in self.negfilter:
                self.negfilter.remove(item)
                self.edit4.Delete(self.edit4.FindString(item))
                if self.negfilter:
                    self.edit4.SetValue(self.negfilter[0])
                else:
                    self.edit4.SetValue("")
            else:
                self.negfilter.append(item)
                self.edit4.Append(item)
            self.edit4.SetFocus()
            self.setSaveButtonStatus(self.rule[3], newvalue = self.negfilter)

    def renameFeed(self, oldname, newname):
        if self.selectedindex >= 0:
            # Do not use rules to update currentfeeds because the user may have edited
            # choosebox content without having saved it yet into rules
            if oldname in self.currentfeeds:
                self.currentfeeds.remove(oldname)
                self.currentfeeds.append(newname)
                self.currentfeeds.sort(lambda u, v: cmp(u.lower(), v.lower()) or cmp(u, v))
                self.populateCombo()

    def removeFeed(self, feed):
        if self.selectedindex >= 0:
            if feed in self.currentfeeds:
                # Do not use rules to update currentfeeds because the user may have edited
                # choosebox content without having saved it yet into rules
                self.currentfeeds.remove(feed)
                self.populateCombo()

    def populateCombo(self):
        self.feedchoosebox.SetItems(self.currentfeeds)
        self.feedchoosebox.SetSelection(0)

    def onKeyInList(self, event):
        if not self.rulelist.GetItemCount():
            return
        keycode = event.GetKeyCode()
        if keycode == wx.WXK_UP:
            self.OnListBox(index = max(0, self.selectedindex - 1))
        elif keycode == wx.WXK_DOWN:
            self.OnListBox(index = min(self.rulelist.GetItemCount() - 1, self.selectedindex + 1))
        elif keycode == wx.WXK_PRIOR:
            self.OnListBox(index = max(0, self.selectedindex - self.rulelist.GetCountPerPage()))
        elif keycode == wx.WXK_NEXT:
            self.OnListBox(index = min(self.rulelist.GetItemCount() - 1, self.selectedindex + self.rulelist.GetCountPerPage()))
        elif keycode == wx.WXK_HOME:
            self.OnListBox(index = 0)
        elif keycode == wx.WXK_END:
            self.OnListBox(index = self.rulelist.GetItemCount() - 1)

    def getIndexFromName(self, name, skipcurrent = False):
        # Find the index that exactly matches name (FindItem can't be used because it doesn't take into account the case)
        maxrulenb = len(self.rules)
        index = 0
        while index < maxrulenb and (self.rules[index][0] != name or (skipcurrent and index == self.selectedindex)):
            index += 1
        if index == maxrulenb:
            return -1
        return index

    def OnListBox(self, event = None, index = None):
        savingneeded = False
        dropnewentry = False
        if index is not None or event is not None:
            if index is None:
                clickedindex = self.rulelist.HitTest(wx.Point(event.GetX(), event.GetY()))[0]
            else:
                clickedindex = index
            if clickedindex == self.selectedindex:
                if event is not None:
                    event.Skip()
                return
            if clickedindex >= 0:
                clickedname = self.rules[clickedindex][0]
            if self.svbtn.IsEnabled():
                dlg = wx.MessageDialog(self, self.localize('rssfiltersaveerror'),
                                       self.localize('rssfiltersaveerror_short'),
                                       wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION)
                dlgresult = dlg.ShowModal()
                dlg.Destroy()
                if dlgresult == wx.ID_CANCEL:
                    return
                if dlgresult == wx.ID_YES:
                    savingneeded = True
                elif not self.rule[0]:
                    dropnewentry = True
                    self.OnRemoveEntry()

            if not dropnewentry:
                if not savingneeded:
                    if self.selectedindex >= 0:
                        self.rulelist.SetStringItem(self.selectedindex, 0, self.rule[0])
                elif not self.OnSaveEntry():
                    return

            self.oldselectedindex = self.selectedindex
            if clickedindex >= 0:
                self.selectedindex = self.getIndexFromName(clickedname)
            else:
                self.selectedindex = -1

        if self.selectedindex == -1:
            try:
                if event is not None:
                    event.Skip()
                # Disable save/remove buttons and other fields/values if nothing is selected
                self.rmbtn.Disable()
                self.regularexpcheck.SetValue(False)
                self.active.SetValue(False)
                self.rulename.SetValue('')
                self.feedchoosebox.SetSelection(-1)
                self.edit3.SetValue("")
                self.edit4.SetValue("")
                self.defaultpriority.SetSelection(-1)
                self.defaultstatus.SetSelection(-1)
                self.downlocswitch.SetValue(False)
                self.downloc.SetValue("")
                self.rulename.Disable()
                self.feedchoosebox.Disable()
                self.edit3.Disable()
                self.edit4.Disable()
                self.amendbutt.Disable()
                self.amendbutt2.Disable()
                self.columndlgbtn.Disable()
                self.active.Disable()
                self.regularexpcheck.Disable()
                self.defaultpriority.Disable()
                self.defaultstatus.Disable()
                self.downlocswitch.Disable()
                self.downloc.Disable()
                self.browsebtn.Disable()
            except:
                pass
        else:
            try:
                self.rulelist.SetFocus()
                self.rulelist.Select(self.selectedindex)
                self.rulelist.Focus(self.selectedindex)
                self.rule = self.rules[self.selectedindex]
                self.currentfeeds = self.rule[1][:]
                self.filter = self.rule[2][:]
                self.negfilter = self.rule[3][:]
                # Re-enable fields
                if self.oldselectedindex == -1:
                    self.rulename.Enable()
                    self.defaultpriority.Enable()
                    self.defaultstatus.Enable()
                    self.feedchoosebox.Enable()
                    self.columndlgbtn.Enable()
                    self.edit3.Enable()
                    self.amendbutt.Enable()
                    self.edit4.Enable()
                    self.amendbutt2.Enable()
                    self.regularexpcheck.Enable()
                    self.downlocswitch.Enable()
                    self.downloc.Enable()
                    self.browsebtn.Enable()
                    self.active.Enable()
                    self.rmbtn.Enable()
                # Name
                self.rulename.SetValue(self.rule[0])
                # Prio
                self.currentprio = self.rule[6]
                self.defaultpriority.SetSelection(self.currentprio)
                # Status
                self.currentstatus = self.rule[7]
                self.defaultstatus.SetSelection(self.currentstatus)
                # Feeds
                self.populateCombo()
                # Filters and neg filters
                self.edit3.SetItems(self.filter)
                self.edit3.SetSelection(0)
                self.edit4.SetItems(self.negfilter)
                self.edit4.SetSelection(0)
                self.regularexpcheck.SetValue(self.rule[5])
                # Download location
                self.downlocswitch.SetValue(self.rule[8])
                self.downloc.SetValue(self.rule[9])
                # Activity
                self.active.SetValue(self.rule[4])
            except:
                pass
        self.svbtn.Disable()

    def onActivitySwitch(self, event):
        # If only activity changed
        if not self.rule[0]:
            return

        if self.active.GetValue() != self.rule[4] and not self.svbtn.IsEnabled():
            self.rule[4] = self.active.GetValue()
            if self.rule[4]:
                self.rulelist.SetItemTextColour(self.selectedindex, wx.Colour(0, 0, 0))
                self.nbactivefilter += 1
                self.parent.setAutoGrabButtonLabel(self.nbactivefilter)
            else:
                self.rulelist.SetItemTextColour(self.selectedindex, wx.Colour(255, 0, 0))
                self.nbactivefilter -= 1
                self.parent.setAutoGrabButtonLabel(self.nbactivefilter)
            if self.utility.abcparams['rssautograb'] == '1':
                self.parent.manageTimers()
            else:
                self.parent.scanFeedsActivity()
            self.parent.SaveRules()

    def populateFilterList(self):
        self.rules.sort(lambda u, v: cmp(u[0].lower(), v[0].lower()) or cmp(u[0], v[0]))
        self.rulelist.Freeze()
        self.rulelist.DeleteAllItems()
        self.nbactivefilter = 0
        for index in xrange(len(self.rules)):
            self.rulelist.InsertStringItem(index, self.rules[index][0])
            if self.rules[index][4]:
                self.rulelist.SetItemTextColour(index, wx.Colour(0, 0, 0))
                self.nbactivefilter += 1
            else:
                self.rulelist.SetItemTextColour(index, wx.Colour(255, 0, 0))
        self.rulelist.Thaw()
        self.parent.setAutoGrabButtonLabel(self.nbactivefilter)

    def OnNewEntry(self, event):
        savingneeded = False
        dropnewentry = False
        if self.svbtn.IsEnabled():
            dlg = wx.MessageDialog(self, self.localize('rssfiltersaveerror'),
                                   self.localize('rssfiltersaveerror_short'),
                                   wx.YES_NO | wx.CANCEL | wx.ICON_QUESTION)
            dlgresult = dlg.ShowModal()
            dlg.Destroy()
            if dlgresult == wx.ID_CANCEL:
                return
            if dlgresult == wx.ID_YES:
                savingneeded = True
            elif not self.rule[0]:
                dropnewentry = True
                self.OnRemoveEntry()

        if not dropnewentry:
            if not savingneeded:
                if self.selectedindex >= 0:
                    self.rulelist.SetStringItem(self.selectedindex, 0, self.rule[0])
            elif not self.OnSaveEntry():
                return

        self.rule = ['', self.feedalias[:], [], [], True, False, 2, 1, False, '']
        self.rules.append(self.rule)
        self.oldselectedindex = self.selectedindex
        self.selectedindex = self.rulelist.GetItemCount()
        self.rulelist.InsertStringItem(self.selectedindex, '')
        self.rulelist.Select(self.selectedindex)
        self.rulelist.Focus(self.selectedindex)
        self.OnListBox()
        self.rulename.SetFocus()
        self.svbtn.Enable()

    def ruleEdited(self):
        if not self.rule[0]:
            return True
        if self.rulename.GetValue() == self.rule[0] and self.regularexpcheck.GetValue() == self.rule[5] \
           and self.defaultpriority.GetSelection() == self.rule[6] and self.defaultstatus.GetSelection() == self.rule[7] \
           and self.currentfeeds == self.rule[1] and self.filter == self.rule[2] and self.negfilter == self.rule[3] \
           and self.downlocswitch.GetValue() == self.rule[8] and self.downloc.GetValue() == self.rule[9]:
            return False
        return True

    def OnSaveEntry(self, event = None):
        newname = self.rulename.GetValue()

        # Name must not be blank
        if newname == '':
            dlg = wx.MessageDialog(self, self.localize('errorfiltername'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.rulename.SetFocus()
            return False

        # Check if name already exists
        if self.getIndexFromName(newname, skipcurrent = True) != -1:
            dlg = wx.MessageDialog(self, self.localize('errorfilterexists'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            self.rulename.SetFocus()
            self.rulename.SetSelection(-1, -1)
            return False

        # Download folder must not be blank if switch is on
        if self.downlocswitch.GetValue() and self.downloc.GetValue() == '':
            dlg = wx.MessageDialog(self, self.localize('nodefaultdowndirwarning'), self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        # Check if download folder is a valid Windows name
        defdestfolder = self.downloc.GetValue()
        if sys.platform == 'win32' and (self.downlocswitch.GetValue() and not self.rule[8] \
           or defdestfolder != self.rule[9]):
            # We erase the final '\' except for a path like 'X:\'
            if defdestfolder and defdestfolder[-1] == '\\' and (len(defdestfolder) < 2 or defdestfolder[-2] != ':'):
                defdestfolder = defdestfolder[:-1]
            defdestfolder = defdestfolder.rstrip(whitespace + '.')
            if not self.utility.checkWinPath(self, defdestfolder):
                return False

        self.rule = self.rules[self.selectedindex] = [newname, self.currentfeeds[:], self.filter[:], self.negfilter[:],
                                                      self.active.GetValue(), self.regularexpcheck.GetValue(),
                                                      self.defaultpriority.GetSelection(), self.defaultstatus.GetSelection(),
                                                      self.downlocswitch.GetValue(), defdestfolder]
        self.populateFilterList()
        self.selectedindex = self.getIndexFromName(newname)
        if event is not None:
            self.rulelist.Select(self.selectedindex)
        self.parent.SaveRules()

        if self.rule[4]:
            if self.utility.abcparams['rssautograb'] == '1':
                self.parent.manageTimers()
            else:
                self.parent.scanFeedsActivity()
        self.svbtn.Disable()

        return True

    def OnRemoveEntry(self, event = None):
        if self.selectedindex >= 0:
            if self.rule[0] and self.rule[4]:
                self.nbactivefilter -= 1
                self.parent.setAutoGrabButtonLabel(self.nbactivefilter)
            del self.rules[self.selectedindex]
            self.rulelist.DeleteItem(self.selectedindex)
            self.selectedindex = -1
            self.OnListBox()
            self.parent.SaveRules()
        if self.rule[4]:
            if self.utility.abcparams['rssautograb'] == '1':
                 self.parent.manageTimers()
            else:
                self.parent.scanFeedsActivity()


class HTMLDescription(wx.html.HtmlWindow):
    def __init__(self, parent, rsspanel, utility, addfunc):
        self.rsspanel = rsspanel
        self.utility = utility
        self.localize = self.utility.lang.get
        self.CmdAddFunc = addfunc
        wx.html.HtmlWindow.__init__(self, parent, -1, style = wx.SUNKEN_BORDER)
        self.SetBorders(4)
        wx.EVT_RIGHT_UP(self, self.OnRightClick)
        self.initHtmlMenu()

    def initHtmlMenu(self):
        self.htmlmenu = wx.Menu()
        self.htmlmenu.Append(929, self.localize('articlecopyselected'), self.localize('articlecopyselected'))
        self.htmlmenu.Append(928, self.localize('articlecopy'), self.localize('articlecopy'))
        self.htmlmenu.Append(937, self.localize('articleviewsource'), self.localize('articleviewsource'))
        self.htmlmenu.Append(938, self.localize('articleviewcontent'), self.localize('articleviewcontent'))
        wx.EVT_MENU(self, 929, self.OnCopySelected)
        wx.EVT_MENU(self, 928, self.OnCopyArticle)
        wx.EVT_MENU(self, 937, self.OnViewSource)
        wx.EVT_MENU(self, 938, self.OnReadContent)

    def OnLinkClicked(self, link):
        article = self.rsspanel.list.curfeed.articletab[self.rsspanel.list.GetFirstSelected()]
        thread = Thread(target = self.linkClicked, args = [link, article])
        thread.daemon = False
        thread.start()

    def linkClicked(self, link, article):
        link = link.GetHref()
        if link.endswith('.torrent'):
            article = self.rsspanel.list.curfeed.articletab[self.rsspanel.list.GetFirstSelected()]
            if self.rsspanel.list.curfeed.cookiesw:
                headers = {'Cookie':self.rsspanel.list.curfeed.cookie}
            else:
                headers = {}
            self.utility.window.parent.SetStatusText(self.localize('downloadingtorrent'))
            self.rsspanel.nbrssaddtorrent += 1
            self.CmdAddFunc(self.rsspanel.addTorrentURLFromRSSCallback, link, headers = headers, callbackinfo = article)
        else:
            webbrowser.open_new(link)

    def OnRightClick(self, event):
        self.PopupMenu(self.htmlmenu, event.GetPosition())

    def OnCopySelected(self, event):
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(self.SelectionToText()))
            wx.TheClipboard.Close()

    def OnCopyArticle(self, event):
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(self.ToText()))
            wx.TheClipboard.Close()

    def OnViewSource(self, event):
        # dlg = wx.lib.dialogs.ScrolledMessageDialog(self, self.GetParser().GetSource(), self.localize('htmlsource'))
        dlg = ScrolledMessageDialog(self.rsspanel, self.localize('htmlsource'), "", self.GetParser().GetSource(), size = (600, 500),
                                    cancel = False, hscroll = False, centeronparent = False)
        dlg.ShowModal()
        dlg.Destroy()

    def OnReadContent(self, event):
        # dlg = wx.lib.dialogs.ScrolledMessageDialog(self, self.ToText(), self.localize('htmlcontent'))
        dlg = ScrolledMessageDialog(self.rsspanel, self.localize('htmlcontent'), "", self.ToText(), size = (600, 500),
                                    cancel = False, hscroll = False, centeronparent = False)
        dlg.ShowModal()
        dlg.Destroy()


class RSSFeeder:
    def __init__(self, utility, rsspanel, url, name = None, cookiesw = False, cookie = '', timersw = False, timer = '5', titlelinksw = False):
        self.utility = utility
        self.localize = self.utility.lang.get
        self.rsspanel = rsspanel
        self.invokeLater = self.rsspanel.invokeLater
        self.list = self.rsspanel.list
        self.url = url
        if name is None:
            self.name = self.url
        else:
            self.name = name
        self.cookiesw = cookiesw
        self.cookie = cookie
        self.timersw = timersw
        self.timer = timer
        self.titlelinksw = titlelinksw
        # sorted RSS articles
        self.rawarticletab = []
        # sorted and filtered RSS articles
        self.articletab = []
        # Articles dictionary
        self.rssarticles = {}
        # Number of new articles at last update
        self.nbnew = 0
        self.owntimer = None
        self.firstcall = True

    def Getdata(self, params, result):
        sleep(0.01)
        self.FeedResults(params, result)

    def GetdataAuto(self, params, result):
        sleep(0.01)
        #if self.name == self.rsspanel.editfeedalias.GetValue():
        #    self.FeedResults(params, result)
        self.FeedResultsAuto(params, result)

    def FeedResults(self, params, result):
        # All existing articles turn old
        for article in self.rawarticletab:
            article.new = False

        newrssnb = 0
        if result.entries:
            feedbase = result.feed.title_detail.base
            feedtitle = result.feed.title
            for entry in result.entries:
                rssarticle = RSSArticle(self, entry, feedbase, feedtitle)
                if not rssarticle.key in self.rssarticles:
                    self.rssarticles[rssarticle.key] = rssarticle
                    self.rawarticletab.insert(newrssnb, rssarticle)
                    newrssnb += 1
        self.nbnew = newrssnb
        self.rsspanel.parent.parent.abc_sb.SetStatusText(str(newrssnb) + self.localize('newrssarticles') + ' / '
                                                         + str(len(self.rawarticletab)) + self.localize('totalrssarticles'), 0)
        self.invokeLater(self.list.filterArticles, [], {'feed':self})

    def FeedResultsAuto(self, params, data):
        result, owntimer, filters = data
        if result.entries:
            for rule in self.rsspanel.rules:
                if rule[0] in filters:
                    afilter = rule[2]
                    nfilter = rule[3]
                    regularexp = rule[5]

                    # Check if filter is active
                    if rule[4]:
                        for entry in result.entries:
                            try:
                                url = entry.enclosures[0].url
                            except:
                                try:
                                    url = entry.links[0].href
                                except:
                                    continue
                            titlelink = entry.get('link', "")
                            entrytitle = entry.get('title', "")
                            entrydatetime = entry.get('updated', "")
                            if not entrydatetime:
                                entrydatetime = "unknown"
    
                            # Only go on if torrent isn't in history list
                            if not url in self.rsspanel.urls:
                                #1 # in case both +filter and -filter are none do nothing.
                                if not afilter and not nfilter:
                                    continue
    
                                #2 # in case only +filter
                                elif not nfilter:
                                    download = True
                                    for item in afilter:
                                        # Change all disturbing signs to "." #"." = any char
                                        if not regularexp:
                                            item = self.RemoveMarks(item)
                                        find = re.search(item, entrytitle, re.IGNORECASE)
                                        if not find:
                                            find = re.search(item, url, re.IGNORECASE)
                                        if not find:
                                            download = False
    
                                #3 # in case only -filter
                                elif not afilter:
                                    download = True
                                    for item in nfilter:
                                        # Change all disturbing signs to "." #"." = any char
                                        if not regularexp:
                                            item = self.RemoveMarks(item)
                                        find = re.search(item, entrytitle, re.IGNORECASE)
                                        if not find:
                                            find = re.search(item, url, re.IGNORECASE)
                                        if find:
                                            download = False
    
                                #4 # in case there are both, +filter and -filter
                                else:
                                    download = True
                                    for item in afilter:
                                        # Change all disturbing signs to "." #"." = any char
                                        if not regularexp:
                                            item = self.RemoveMarks(item)
                                        find = re.search(item, entrytitle, re.IGNORECASE)
                                        if not find:
                                            find = re.search(item, url, re.IGNORECASE)
                                        if not find:
                                            download = False
                                    if download:
                                        for item in nfilter:
                                            # Change all disturbing signs to "." #"." = any char
                                            if not regularexp:
                                                item = self.RemoveMarks(item)
                                            find = re.search(item, entrytitle, re.IGNORECASE)
                                            if not find:
                                                find = re.search(item, url, re.IGNORECASE)
                                            if find:
                                                download = False
    
                                if download:
                                    self.Download(url, hash((html2text(entrytitle)[:-1].lower(), url.lower(), entrydatetime.lower())), rule)

        AUTOGRABTHREADING.acquire()
        self.rsspanel.nbfeedautothread -= 1
        if owntimer:
            self.scheduleNextTimerCycle()
        else:
            self.rsspanel.nbglobalfeedautothread -= 1
            if self.rsspanel.nbglobalfeedautothread == 0:
                self.rsspanel.scheduleNextGlobalTimerCycle()
        if self.rsspanel.nbfeedautothread == 0:
            self.rsspanel.searchbutt.Enable(True)
        AUTOGRABTHREADING.release()

    def RemoveMarks(self, item):
        # Change all disturbing signs to "." = any char
        p = re.compile('(\*|\^|\$|\+|\?|\}|\{|\[|\]|\||\\\\|\(|\))')
        item = p.sub('.', item)
        p = re.compile('\#')
        item = p.sub('\d', item)
        return item

    def addAutoTorrentURLFromRSSCallback(self, callbackinfo, status):
        key, url = callbackinfo
        articles = self.rsspanel.findArticleInFeeds(key)
        if status == "OK":
            self.rsspanel.saveurls |= self.rsspanel.AddToHistory(url)
            for a in articles:
                self.invokeLater(self.list.setStatus, [a, 2])
        else:
            self.rsspanel.AddToLog(add, url, "error")
            for a in articles:
                self.invokeLater(self.list.setStatus, [a, 3])
        self.rsspanel.nbrssaddtorrent -= 1
        if self.rsspanel.nbrssaddtorrent == 0 and self.rsspanel.saveurls:
            self.rsspanel.SaveURLs()
            self.rsspanel.saveurls = False

    def Download(self, url, key, rule):
        headers = {}
        if self.cookiesw:
            headers['Cookie'] = self.cookie
        if url.lower().endswith(".torrent") or url.lower().startswith('magnet:'):
            try:
                self.rsspanel.nbrssaddtorrent += 1
                self.rsspanel.AddTorrent(self.addAutoTorrentURLFromRSSCallback, url, caller = 'rss', rssrule = rule,
                                         headers = headers, callbackinfo = (key, url))
            except:
                pass
        else:
            try:
                link_url = urlopen(url, headers)
                if "application/x-bittorrent" in link_url.response.getheader("Content-Type"):
                    self.rsspanel.nbrssaddtorrent += 1
                    self.rsspanel.AddTorrent(self.addAutoTorrentURLFromRSSCallback, url, caller = 'rss', rssrule = rule,
                                             headers = headers, callbackinfo = (key, url), req_url = link_url)
                else:
                    try:
                        parserlink = urlparse.urlparse(url)
                        link = urlparse.urlunparse((parserlink[0], parserlink[1], '', '', '', ''))
                        parser = UrlFinder(link)
                        if self.cookiesw:
                            cookies = self.cookie
                        else:
                            cookies = None
                        parserthread = ParserThread(self.utility, parser, url, key,
                                                    self.rsspanel.findArticleInFeeds(key),
                                                    rssrule = rule,
                                                    cookies = cookies, pagecontent = link_url.read())
                        parserthread.start()
                        link_url.close()
                        self.rsspanel.parserthreads[url] = parserthread
                    except:
                        self.utility.window.parent.SetStatusText(self.localize('failedtoparseurl'))
                        self.rsspanel.AddToLog(self.localize('log_cantparseurl'), url, "error")
            except IOError:
                self.utility.window.parent.SetStatusText(self.localize('failedtoparseurl'))
                self.rsspanel.AddToLog(self.localize('log_cantparseurl'), url, "error")

    def startTimer(self, now = False):
        try:
            if self.rsspanel.parent.exiting:
                return
        except:
            return
        if self.owntimer is None:
            if now:
                delay = 0
            else:
                delay = 60 * int(self.timer)
            
            self.owntimer = Timer(delay, self.rsspanel.GetFeed, [self, True, self.timersw, self.rsspanel.GetRulesForFeeder()[self.name]])
            self.owntimer.start()

    def stopTimer(self):
        oldstatus = self.owntimer
        if oldstatus:
            self.owntimer.cancel()
            self.owntimer = None
        return oldstatus

    def resetTimer(self):
        if self.stopTimer():
            self.startTimer()

    def scheduleNextTimerCycle(self):
        try:
            if self.rsspanel.parent.exiting:
                return
        except:
            return
        if self.owntimer is not None and self.utility.abcparams['rssautograb'] == '1':
            self.owntimer = Timer(60 * int(self.timer), self.rsspanel.GetFeed, [self, True, self.timersw, self.rsspanel.GetRulesForFeeder()[self.name]])
            self.owntimer.start()

class RSSArticle:
    def __init__(self, feeder, feedentry, feedbase, feedtitle):
        self.feeder = feeder
        self.feedbase = feedbase
        self.feedtitle = feedtitle
        try:
            self.url = feedentry.enclosures[0].url
        except:
            try:
                self.url = feedentry.links[0].href
            except:
                self.url = "..."
        self.titlelink = feedentry.get('link', "")
        self.entrytitle = feedentry.get('title', "")
        self.summary = feedentry.get('summary', "")
        self.datetime = feedentry.get('updated', "")
        if not self.datetime:
            self.datetime = "unknown"
        self.title = html2text(self.entrytitle)[:-1]
        self.titlelinksw = feeder.titlelinksw
        # self.key = hash((self.title.lower(), self.url.lower(), self.datetime.lower()))
        self.key = hash((self.title.lower(), self.datetime.lower()))
        self.new = True
        self.previndex = 1000000000
        # Status : 0 : normal ; 1 : trying ; 2 : OK ; 3 : fail
        self.status = 0

    def getIdentity(self):
        return (self.title, self.url, self.datetime, self.titlelink)

    def getTitle(self):
        return self.title.lower()

    def getURL(self):
        return self.url.lower()

    def getTime(self):
        return self.datetime.lower()

    def getOld(self):
        return not self.new

    def getPrevIndex(self):
        return self.previndex


class RSSPanel(wx.Panel):
    def __init__(self, parent, addfunc):
        wx.Panel.__init__(self, parent, -1, style = wx.BORDER_STATIC)

        self.AddTorrent = addfunc
        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get
        self.parserthreads = {}
        self.feeds = {}
        # To store feed URLs in a logical order
        self.feedalias = []
        # Torrents URL already downloaded
        self.urls = []

        self.nbfeedautothread = 0
        self.nbglobalfeedautothread = 0
        self.nbrssaddtorrent = 0
        # To track if some urls have been downloaded and must be saved into url.lst
        self.saveurls = False

        self.rendering = False
        self.rules = []
        self.rss_file = join(self.utility.datapath, "rssfeed.lst")
        self.rules_file = join(self.utility.datapath, "rule.lst")
        self.urls_file = join(self.utility.datapath, "url.lst")

        self.splitter = wx.SplitterWindow(self, -1, style = wx.SP_NOBORDER)
        self.splitter.SetMinimumPaneSize(1)
        self.splitter.SetSashGravity(.5)
        self.listpanel = wx.Panel(self.splitter, -1)
        self.htmlpanel = wx.Panel(self.splitter, -1)
        self.splitter.SplitVertically(self.listpanel, self.htmlpanel)
        self.list = RSSList(self.listpanel, self, self.utility, onclickfunc = self.OnClick, ondblclickfunc = self.OnDblClick,
                            onbrowfunc = self.OnBrowser, onsearchfortitlefunc = self.OnSearchForTitle,
                            oncopytitlefunc = self.OnCopyTitle, oncopytitlelinkfunc = self.OnCopyTitleLink,
                            oncopyurlfunc = self.OnCopyURL, onstopurl = self.OnStopUrl)
        self.invokeLater = RSSListHandler(self.list).invokeLater
        self.html = HTMLDescription(self.htmlpanel, self, self.utility, addfunc)

        self.logpanel = LogPanel(self, self.utility)

        self.LoadURLs()
        self.LoadRules()
        self.LoadFeeds()

        # To dispatch active feeders with or without own timer, and inactive feeders
        self.scanFeedsActivity()

        # 99 to size the button to receive later a 2 digit number
        self.autograbbutt = wx.ToggleButton(self, -1, self.localize('autograb') % 99)
        self.autograbbutt.SetToolTipString(self.localize('autograbtooltip'))

        self.historypanel = HistoryPanel(self, self.utility, self.urls)
        self.feederspanel = FeedersPanel(self, self.utility, self.feedalias)
        self.filterspanel = FiltersPanel(self, self.utility, self.feedalias, self.rules)
        self.editfeedalias = wx.ComboBox(self, -1, "", choices = self.feedalias)
        self.editfeedalias.SetMinSize((100, -1))
        self.editfeedalias.Bind(wx.EVT_CHAR, self.utility.onWriteFilteredText)
        self.editfeedalias.Bind(wx.EVT_TEXT_PASTE, self.utility.onPasteText)
        self.editfeedalias.Bind(wx.EVT_TEXT_ENTER, self.Populate)
        self.editfeedalias.Bind(wx.EVT_COMBOBOX, self.ShowArticles)
        self.editfeedalias.Bind(wx.EVT_MOUSEWHEEL, self.Skip)
        self.articlefilter = wx.TextCtrl(self, -1, "", size = (80, -1))
        self.articlefilter.SetToolTipString(self.localize('articlefiltertooltip'))
        self.articlefilter.Bind(wx.EVT_TEXT, self.list.filterArticles)
        self.articlefilter.Bind(wx.EVT_LEFT_DCLICK, self.onDblLeftClickFilterArticles)
        mainsizer = wx.BoxSizer(wx.VERTICAL)

        maintopsizer = wx.FlexGridSizer(1, 2, 0, 0)

        tbstyle = wx.TB_HORIZONTAL | wx.TB_TEXT | wx.TB_NODIVIDER | wx.NO_BORDER
        if self.utility.abcparams['flattopbtn'] == "1":
            tbstyle |= wx.TB_FLAT
        self.rsstoolbar = wx.ToolBar(self, -1, style = tbstyle)
        bitmap1 = self.utility.makeBitmap('rss_list.bmp')
        self.rsstoolbar.SetToolBitmapSize(wx.Size(bitmap1.GetWidth(), bitmap1.GetHeight()))
        self.rsspanelbtnid = wx.NewId()
        self.rsstoolbar.AddRadioTool(self.rsspanelbtnid, bitmap1, wx.NullBitmap, self.localize('rss_list_short'),
                                     self.localize('rss_list_long'))
        self.Bind(wx.EVT_TOOL, self.OnToggleList, id = self.rsspanelbtnid)
        id = wx.NewId()
        self.rsstoolbar.AddRadioTool(id, self.utility.makeBitmap('log.bmp'), wx.NullBitmap,
                                     self.localize('rss_log_short'), self.localize('rss_log_long'))
        self.Bind(wx.EVT_TOOL, self.OnToggleLog, id = id)
        id = wx.NewId()
        self.rsstoolbar.AddRadioTool(id, self.utility.makeBitmap('history.bmp'), wx.NullBitmap,
                                     self.localize('rss_history_short'), self.localize('rss_history_long'))
        self.Bind(wx.EVT_TOOL, self.OnToggleHistory, id = id)
        id = wx.NewId()
        self.rsstoolbar.AddRadioTool(id, self.utility.makeBitmap('feeders.bmp'), wx.NullBitmap,
                                     self.localize('rss_feeders_short'), self.localize('rss_feeders_long'))
        self.Bind(wx.EVT_TOOL, self.OnToggleFeeders, id = id)
        id = wx.NewId()
        self.rsstoolbar.AddRadioTool(id, self.utility.makeBitmap('filters.bmp'), wx.NullBitmap,
                                     self.localize('rss_filters_short'), self.localize('rss_filters_long'))
        self.Bind(wx.EVT_TOOL, self.OnToggleFilters, id = id)
        self.rsstoolbar.Realize()

        maintopsizer.Add(self.rsstoolbar, 0, wx.EXPAND | wx.LEFT, 2)

        maintopsizer.Add(self.autograbbutt)
        maintopsizer.AddGrowableCol(0)
        mainsizer.Add(maintopsizer, 0, wx.EXPAND)

        topsizer = wx.FlexGridSizer(1, 6, 0, 0)

        self.updatebutt = wx.Button(self, -1, self.localize('rssupdate'), style = wx.BU_EXACTFIT)
        self.updatebutt.SetToolTipString(self.localize('updatetooltip'))

        self.searchbutt = wx.Button(self, -1, self.localize('rssgettor'), style = wx.BU_EXACTFIT)
        self.searchbutt.SetToolTipString(self.localize('gettortooltip'))

        self.clearlistbutt = wx.Button(self, -1, self.localize('rssclearlist'), style = wx.BU_EXACTFIT)
        self.clearlistbutt.SetToolTipString(self.localize('clearlisttooltip'))

        topsizer.Add(wx.StaticText(self, -1, self.localize('feederalias')), 0, wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        topsizer.Add(self.editfeedalias, 0, wx.EXPAND | wx.TOP, 1)
        topsizer.Add(self.articlefilter, 0, wx.ALIGN_CENTER_VERTICAL)
        topsizer.Add(self.updatebutt, 0, wx.ALIGN_CENTER_VERTICAL)
        topsizer.Add(self.searchbutt, 0, wx.ALIGN_CENTER_VERTICAL)
        topsizer.Add(self.clearlistbutt, 0, wx.ALIGN_CENTER_VERTICAL)
        topsizer.AddGrowableCol(1)
        if self.utility.abcparams['flattopbtn'] == "1":
            mainsizer.Add(topsizer, 0, wx.EXPAND | wx.TOP, 2)
        else:
            mainsizer.Add(topsizer, 0, wx.EXPAND | wx.TOP, 3)

        listsizer = wx.BoxSizer(wx.VERTICAL)
        listsizer.Add(self.list, 1, wx.EXPAND)
        self.listpanel.SetAutoLayout(True)
        self.listpanel.SetSizer(listsizer)
        listsizer.Fit(self.listpanel)

        htmlsizer = wx.BoxSizer(wx.VERTICAL)
        htmlsizer.Add(self.html, 1, wx.EXPAND)
        self.htmlpanel.SetAutoLayout(True)
        self.htmlpanel.SetSizer(htmlsizer)
        htmlsizer.Fit(self.htmlpanel)

        mainsizer.Add(self.splitter, 1, wx.EXPAND | wx.TOP, 2)
        mainsizer.Add(self.logpanel, 1, wx.EXPAND | wx.TOP, 2)
        mainsizer.Add(self.historypanel, 1, wx.EXPAND | wx.TOP, 2)
        mainsizer.Add(self.feederspanel, 1, wx.EXPAND | wx.TOP, 2)
        mainsizer.Add(self.filterspanel, 1, wx.EXPAND | wx.TOP, 2)

        self.SetAutoLayout(True)
        self.SetSizer(mainsizer)
        mainsizer.Fit(self)

        self.logpanel.Hide()
        self.historypanel.Hide()
        self.feederspanel.Hide()
        self.filterspanel.Hide()

        wx.EVT_BUTTON(self, self.updatebutt.GetId(), self.Populate)
        wx.EVT_BUTTON(self, self.searchbutt.GetId(), self.PopulateAutoOnDemand)
        wx.EVT_BUTTON(self, self.clearlistbutt.GetId(), self.ClearList)
        wx.EVT_TOGGLEBUTTON(self, self.autograbbutt.GetId(), self.AutoGrab)

        self.list.Bind(wx.EVT_ENTER_WINDOW, self.focusList)
        self.html.Bind(wx.EVT_ENTER_WINDOW, self.focusHtml)

        # Global timer
        self.globaltimer = None

    def focusList(self, event):
        if wx.Window.FindFocus() is self.html:
            self.list.SetFocus()

    def focusHtml(self, event):
        if wx.Window.FindFocus() is self.list:
            self.html.SetFocus()

    def setAutoGrabButtonLabel(self, nbactivefilter):
        self.autograbbutt.SetLabel(self.localize('autograb') % nbactivefilter)

    def onDblLeftClickFilterArticles(self, event):
        self.articlefilter.SetValue('')

    def setToolbarStyle(self, style):
        self.rsstoolbar.SetWindowStyle(style)
        self.rsstoolbar.Refresh()
        self.Layout()

    def Skip(self, event):
        return

    def ShowArticles(self, event):
        alias = event.GetString()
        feed = self.feeds[alias]
        if feed is not self.list.curfeed:
            self.parent.parent.abc_sb.SetStatusText('', 0)
            if feed.firstcall:
                self.Populate(feedalias = alias)
            else:
                feed.nbnew = 0
                self.parent.parent.abc_sb.SetStatusText(str(len(feed.rawarticletab)) + self.localize('bufferedrssarticles'), 0)
                self.list.clear()
                if self.list.curfeed:
                    self.list.curfeed.articletab[:] = []
                self.list.filterArticles(feed = feed, fromrequest = False)
                self.rsstoolbar.ToggleTool(self.rsspanelbtnid, True)
                self.OnToggleList()
                self.list.SetFocus()
            feed.firstcall = False

    def findArticleInFeeds(self, key):
        articles = []
        for feed in self.feeds.values():
            article = feed.rssarticles.get(key)
            if article is not None:
                articles.append(article)
        return articles

    def startGlobalTimer(self, now = False):
        try:
            if self.parent.exiting:
                return
        except:
            return
        if self.globaltimer is None:
            if now:
                delay = 0
            else:
                delay = 60 * int(self.utility.abcparams['rssscanfreq'])
            self.globaltimer = Timer(delay, self.PopulateAuto)
            self.globaltimer.start()

    def stopGlobalTimer(self):
        oldstatus = self.globaltimer
        if oldstatus:
            self.globaltimer.cancel()
            self.globaltimer = None
        return oldstatus

    def resetGlobalTimer(self):
        if self.stopGlobalTimer():
            self.startGlobalTimer()

    def scheduleNextGlobalTimerCycle(self):
        try:
            if self.parent.exiting:
                return
        except:
            return
        if self.globaltimer is not None and self.utility.abcparams['rssautograb'] == '1':
            self.globaltimer = Timer(60 * int(self.utility.abcparams['rssscanfreq']), self.PopulateAuto)
            self.globaltimer.start()

    def initAllTimers(self):
        self.scanFeedsActivity()
        self.startAllTimers()

    def startAllTimers(self, now = False):
        if self.activefeedswithglobaltimer:
            self.startGlobalTimer(now)
        for feed in self.activefeedswithowntimer:
            feed.startTimer(now)

    def stopAllTimers(self):
        self.stopGlobalTimer()
        for feed in self.activefeedswithowntimer:
            feed.stopTimer()

    def resetAllTimers(self):
        self.resetGlobalTimer()
        for feed in self.activefeedswithowntimer:
            feed.resetTimer()

    def scanFeedsActivity(self):
        # Dispatch active feeders with or without own timer, and inactive feeders
        self.activefeedswithowntimer = []
        self.activefeedswithglobaltimer = []

        # List of all feeds
        self.inactivefeeds = self.feeds.values()

        for rule in self.rules:
            if rule[4]:
                for feed in [self.feeds[alias] for alias in rule[1]]:
                    if feed in self.inactivefeeds:
                        if feed.timersw:
                            self.activefeedswithowntimer.append(feed)
                        else:
                            self.activefeedswithglobaltimer.append(feed)
                        self.inactivefeeds.remove(feed)

    def manageTimers(self):
        self.scanFeedsActivity()
        for feed in self.inactivefeeds:
            feed.stopTimer()
        for feed in self.activefeedswithowntimer:
            feed.startTimer()
        if self.activefeedswithglobaltimer:
            self.startGlobalTimer()
        else:
            self.stopGlobalTimer()

    def OnToggleList(self, event = None):
        if not self.splitter.IsShown():
            self.parent.parent.abc_sb.SetStatusText('', 0)
            self.logpanel.Hide()
            self.historypanel.Hide()
            self.feederspanel.Hide()
            self.filterspanel.Hide()
            self.splitter.Show()
            f = wx.Window.FindFocus()
            if f is not self.articlefilter and f is not self.editfeedalias:
                self.splitter.SetFocus()
            self.Layout()

    def OnToggleLog(self, event):
        if not self.logpanel.IsShown():
            self.parent.parent.abc_sb.SetStatusText('', 0)
            self.splitter.Hide()
            self.historypanel.Hide()
            self.feederspanel.Hide()
            self.filterspanel.Hide()
            self.logpanel.Show()
            f = wx.Window.FindFocus()
            if f is not self.articlefilter and f is not self.editfeedalias:
                self.logpanel.SetFocus()
            self.Layout()

    def OnToggleHistory(self, event):
        if not self.historypanel.IsShown():
            self.parent.parent.abc_sb.SetStatusText('', 0)
            self.splitter.Hide()
            self.logpanel.Hide()
            self.feederspanel.Hide()
            self.filterspanel.Hide()
            self.historypanel.Show()
            f = wx.Window.FindFocus()
            if f is not self.articlefilter and f is not self.editfeedalias:
                self.historypanel.SetFocus()
            self.Layout()

    def OnToggleFeeders(self, event):
        if not self.feederspanel.IsShown():
            self.parent.parent.abc_sb.SetStatusText('', 0)
            self.splitter.Hide()
            self.logpanel.Hide()
            self.historypanel.Hide()
            self.filterspanel.Hide()
            self.feederspanel.Show()
            f = wx.Window.FindFocus()
            if f is not self.articlefilter and f is not self.editfeedalias:
                self.feederspanel.SetFocus()
            self.Layout()

    def OnToggleFilters(self, event):
        if not self.filterspanel.IsShown():
            self.parent.parent.abc_sb.SetStatusText('', 0)
            self.splitter.Hide()
            self.logpanel.Hide()
            self.historypanel.Hide()
            self.feederspanel.Hide()
            self.filterspanel.Show()
            f = wx.Window.FindFocus()
            if f is not self.articlefilter and f is not self.editfeedalias:
                self.filterspanel.SetFocus()
            self.Layout()

    def SaveRules(self):
        try:
            f = open(self.rules_file + '.part', 'w+')
            f.writelines('\xef\xbb\xbf')
            for params in self.rules:
                urls = ''
                for item in params[1]:
                    urls += item + '\x01'
                urls = urls[:-1]
                filters = ''
                for item in params[2]:
                    filters += item.replace('|', '\x02') + '\x01'
                filters = filters[:-1]
                negfilters = ''
                for item in params[3]:
                    negfilters += item.replace('|', '\x02') + '\x01'
                negfilters = negfilters[:-1]
                f.writelines((params[0] + '|' + urls + '|' + filters + '|' + negfilters + '|' + str(int(params[4]))
                              + '|' + str(int(params[5])) + '|' + str(params[6]) + '|' + str(params[7]) \
                              + '|' + str(int(params[8])) + '|' + params[9] + "\n").encode('utf_8'))
        except:
            try:
                f.close()
            except:
                pass
        else:
            f.close()
            try:
                remove(self.rules_file)
            except:
                pass
            rename(self.rules_file + '.part', self.rules_file)

    def LoadRules(self):
        if exists(self.rules_file):
            try:
                f = open(self.rules_file, 'r+')
                firstline = True
                newformat = False
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    entry = configline[:-1].decode(encoding).split('|')
                    entry[1] = entry[1].split('\x01')
                    if entry[1] == ['']:
                        entry[1] = []
                    entry[2] = entry[2].replace('\x02', '|').split('\x01')
                    if entry[2] == ['']:
                        entry[2] = []
                    entry[3] = entry[3].replace('\x02', '|').split('\x01')
                    if entry[3] == ['']:
                        entry[3] = []
                    entry[4] = int(entry[4])
                    entry[5] = int(entry[5])
                    entry[6] = int(entry[6])
                    entry[7] = int(entry[7])
                    # New format with downloc and associated switch
                    if len(entry) == 8:
                        newformat = True
                        entry.append(False)
                        entry.append('')
                    else:
                        entry[8] = (entry[8] == '1')
                    self.rules.append(entry)
                f.close()
                if newformat:
                    self.SaveRules()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file rule.lst"
            self.rules.sort(lambda u, v: cmp(u[0].lower(), v[0].lower()) or cmp(u[0], v[0]))

    def SaveURLs(self):
        URLFILESEM.acquire()
        try:
            f = open(self.urls_file + '.part', 'w+')
            f.writelines('\xef\xbb\xbf')
            for item in self.urls:
                f.writelines((item + "\n").encode('utf_8'))
        except:
            try:
                f.close()
            except:
                pass
        else:
            f.close()
            try:
                remove(self.urls_file)
            except:
                pass
            rename(self.urls_file + '.part', self.urls_file)
        URLFILESEM.release()

    def LoadURLs(self):
        if exists(self.urls_file):
            try:
                f = open(self.urls_file, 'r+')
                firstline = True
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    self.urls.append(configline[:-1].decode(encoding))
                f.close()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file url.lst"

    def ClearList(self, event):
        self.parent.parent.abc_sb.SetStatusText('', 0)
        if self.list.curfeed:
            self.list.clear()
            self.list.curfeed.rssarticles = {}
            self.list.curfeed.rawarticletab[:] = []
            self.list.curfeed.articletab[:] = []
            self.list.curfeed = None

    def addTorrentURLFromRSSCallback(self, article, status):
        articles = self.findArticleInFeeds(article.key)
        if status == "OK":
            # Change colour to "downloaded" = Green
            for a in articles:
                self.invokeLater(self.list.setStatus, [a, 2])
            # Add main url to history file
            self.saveurls |= self.AddToHistory(article.url)
        else:
            # Change color to "Failed" = Red
            for a in articles:
                self.invokeLater(self.list.setStatus, [a, 3])
            self.AddToLog(status, article.url, "error")
        self.nbrssaddtorrent -= 1
        if self.nbrssaddtorrent == 0 and self.saveurls:
            self.SaveURLs()
            self.saveurls = False

    def OnClick(self, article):
        self.RenderHtml(article)

    def OnDblClick(self, article):
        headers = {}
        if article.feeder.cookiesw:
            headers['Cookie'] = article.feeder.cookie

        # Change color to "try download" = Blue
        self.invokeLater(self.list.setStatus, [article, 1])

        if article.feeder.titlelinksw:
            url = article.titlelink
        else:
            url = article.url

        # Normal link for .torrent file
        if url.lower().endswith(".torrent") or url.lower().startswith('magnet:'):
            self.nbrssaddtorrent += 1
            self.AddTorrent(self.addTorrentURLFromRSSCallback, url, showdownload = True,
                            headers = headers, callbackinfo = article)

        # Not a normal link
        else:
            try:
                link_url = urlopen(url, headers)
                if "application/x-bittorrent" in link_url.response.getheader("Content-Type"):
                    self.nbrssaddtorrent += 1
                    self.AddTorrent(self.addTorrentURLFromRSSCallback, url, showdownload = True,
                                    headers = headers, callbackinfo = article, req_url = link_url)
                else:
                    try:
                        parserlink = urlparse.urlparse(url)
                        link = urlparse.urlunparse((parserlink[0], parserlink[1], '', '', '', ''))
                        parser = UrlFinder(link)
                        if article.feeder.cookiesw:
                            cookies = article.feeder.cookie
                        else:
                            cookies = None
                        parserthread = ParserThread(self.utility, parser, url, article.key,
                                                    self.findArticleInFeeds(article.key),
                                                    cookies = cookies, pagecontent = link_url.read())
                        parserthread.start()
                        link_url.close()
                        self.parserthreads[url] = parserthread
                        #parserthread.StopAll()
                    except:
                        self.utility.window.parent.SetStatusText(self.localize('failedtoparseurl'))
                        # Change colour to "Failed" = Red
                        self.invokeLater(self.list.setStatus, [article, 3])
                        self.AddToLog(self.localize('log_cantparseurl'), url, "error")
            except IOError:
                self.utility.window.parent.SetStatusText(self.localize('failedtoparseurl'))
                # Change colour to "Failed" = Red
                self.invokeLater(self.list.setStatus, [article, 3])
                self.AddToLog(self.localize('log_cantparseurl'), url, "error")

    def AddToHistory(self, url, tolog = True):
        if not url in self.urls:
            self.historypanel.append(url)
            saveurls = True
        else:
            saveurls = False
        if tolog:
            self.AddToLog(self.localize('log_downloaded'), url)
        return saveurls

    def OnBrowser(self, article):
        thread = Thread(target = self._OpenUrl, args = [article.url])
        thread.daemon = False
        thread.start()

    def OnSearchForTitle(self, title):
        self.OnCopyTitle(title)
        self.utility.frame.onMenuFind()
        findpanel = self.utility.frame.abcfind.window
        findpanel.onReset()
        findpanel.findtorrentnamebtn.SetValue(False)
        findpanel.findtorrentname.SetValue(title)
        findpanel.onFindTorrent()

    def OnCopyTitle(self, title):
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(title))
            wx.TheClipboard.Close()

    def OnCopyTitleLink(self, titlelink):
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(titlelink))
            wx.TheClipboard.Close()

    def OnCopyURL(self, url):
        if wx.TheClipboard.Open():
            wx.TheClipboard.SetData(wx.TextDataObject(url))
            wx.TheClipboard.Close()

    def OnStopUrl(self, article = None, all = False):
        if not all and article is not None:
            url = article.url
            if self.parserthreads.has_key(url):
                self.parserthreads[url].StopAll()
        else:
            for thread in self.parserthreads:
                self.parserthreads[thread].StopAll()

    def GetFeed(self, feeder, auto, owntimer = False, filters = None):
        if auto:
            if owntimer:
                AUTOGRABTHREADING.acquire()
                self.nbfeedautothread += 1
                AUTOGRABTHREADING.release()
            self.searchbutt.Enable(False)
            t = T_GetFeed(feeder.GetdataAuto, "", feeder, True, owntimer, filters)
        else:
            self.editfeedalias.Disable()
            self.updatebutt.Disable()
            self.clearlistbutt.Disable()
            if self.list.curfeed and feeder is not self.list.curfeed:
                self.list.clear()
                self.list.curfeed.articletab[:] = []
            t = T_GetFeed(feeder.Getdata, "", feeder, False)
        t.start()

    def PopulateAutoOnDemand(self, event):
        self.stopAllTimers()
        self.startAllTimers(now = True)

    def GetRulesForFeeder(self):
        rulesforfeeder = {}
        for rule in self.rules:
            if rule[0] and rule[4]:
                for item in rule[1]:
                    if item in rulesforfeeder:
                        rulesforfeeder[item].append(rule[0])
                    else:
                        rulesforfeeder[item] = [rule[0]]
        return rulesforfeeder

    def PopulateAuto(self):
        self.parent.parent.abc_sb.SetStatusText('', 0)
        thereisactive = False
        somethingtosearchfor = False

        try:
            for rule in self.rules:
                if rule[2] or rule[3]:
                    somethingtosearchfor = True
                if rule[4]:
                    thereisactive = True
            if not somethingtosearchfor or not thereisactive:
                return
    
            rulesforfeeder = self.GetRulesForFeeder()
            for alias in rulesforfeeder:
                feed = self.feeds[alias]
                if not feed.timersw:
                    AUTOGRABTHREADING.acquire()
                    self.nbfeedautothread += 1
                    self.nbglobalfeedautothread += 1
                    AUTOGRABTHREADING.release()
                    self.GetFeed(feed, True, False, filters = rulesforfeeder[alias])
        except:
            pass

    def Populate(self, event = None, feedalias = None):
        self.parent.parent.abc_sb.SetStatusText('', 0)
        if event is None:
            alias = feedalias
        else:
            alias = self.editfeedalias.GetValue()
        if alias != "":
            if not alias in self.feedalias:
                if not alias.startswith(('http://', 'https://')):
                    url = "http://" + alias
                else:
                    url = alias
                mustcreate = True
                for f in self.feedalias:
                    if self.feeds[f].url == url:
                        mustcreate = False
                        break
                if mustcreate:
                    self.feeds[alias] = RSSFeeder(self.utility, self, url, alias)
                    self.feedalias.append(alias)
                    self.feederspanel.populateFeederList(keepselection = True)
                    self.populateMainFeederCombo()
                    self.SaveFeeds()

            self.GetFeed(self.feeds[alias], False)
            self.rsstoolbar.ToggleTool(self.rsspanelbtnid, True)
            self.OnToggleList()
            self.list.SetFocus()

    def AutoGrab(self, event):
        if event.IsChecked():
            self.utility.abcparams['rssautograb'] = '1'
            self.initAllTimers()
        else:
            self.utility.abcparams['rssautograb'] = '0'
            self.stopAllTimers()

    def RenderHtml(self, article):
        if not self.rendering:
            self.rendering = True
            # If the feed parser can find the codec used in the RSS data, it decodes data with this codec 
            # and re-encode data to utf-8.
            # If it can't find the codec, data are utf-8 decoded with replace option and re-encoded to utf-8.
            page = "<html> \n" \
                   +"<head> \n" \
                   +"<META HTTP-EQUIV='Content-Type' CONTENT='text/html; charset=utf-8'> \n" \
                   +"    <base href=" + article.feedbase + "> \n" \
                   +"</head> \n" \
                   +"<body> \n" \
                   +"       <table width='100%' bgcolor='#e0e0e0' cellpadding='0' cellspacing='0' border='0'> \n" \
                   +"               <tr><td> \n" \
                   +"                       <font face='Verdana,Sans-serif' size=10pt color='#000000'><font color='blue'><b>" + self.localize('hfeed') + " </b>" + article.feedtitle + "<br></font> \n" \
                   +"                       <a href='" + article.titlelink + "'><b>" + self.localize('hname')+ " </b>" + article.entrytitle + "</a><br></font> \n" \
                   +"                       <font color='blue'><b>" + self.localize('hdate') + " </b>" + article.datetime + "</font> \n" \
                   +"               </td><td align='right'> \n" \
                   +"               </td></tr> \n" \
                   +"               <tr bgcolor='#666666' height='1'><td colspan='2'></td></tr> \n" \
                   +"       </table> \n" \
                   +"       <table width='100%' cellpadding='2' cellspacing='2' border='0'><tr><td> \n" \
                   +"               " + article.summary +" \n" \
                   +"               </td></tr> \n" \
                   +"       </table> \n" \
                   +"       </body> \n" \
                   +"</html> \n"
            #page = self.utility.convertToUnicodeEncode(page)
            self.html.SetPage(page)
        self.rendering = False

    def _OpenUrl(self, url):
        try:
            webbrowser.open_new(url)
        except:
            pass

    def SaveFeeds(self):
        try:
            f = open(self.rss_file + '.part', 'w+')
            f.writelines('\xef\xbb\xbf')
            for item in self.feedalias:
                feed = self.feeds[item]
                f.writelines((feed.url + '|' + item + '|' + str(int(feed.cookiesw)) + '|'
                              + feed.cookie + '|' + str(int(feed.timersw)) + '|'
                              + feed.timer + '|' + str(int(feed.titlelinksw)) + "\n").encode('utf_8'))
        except:
            try:
                f.close()
            except:
                pass
        else:
            f.close()
            try:
                remove(self.rss_file)
            except:
                pass
            rename(self.rss_file + '.part', self.rss_file)

    def LoadFeeds(self):
        if exists(self.rss_file):
            try:
                f = open(self.rss_file, 'r+')
                firstline = True
                newformat = False
                while True:
                    configline = f.readline()
                    if firstline:
                        firstline = False
                        if configline[:3] == '\xef\xbb\xbf':
                            encoding = 'utf_8'
                            # Skip BOM
                            configline = configline[3:]
                        else:
                            encoding = wx.GetDefaultPyEncoding()
                    if configline == '' or configline == "\n":
                        break
                    entry = configline[:-1].decode(encoding).split('|')
                    url = entry[0]
                    entrylength = len(entry)
                    if entrylength == 1:
                        # Old format
                        name = url
                        cookiesw = False
                        cookie = ''
                    else:
                        name = entry[1]
                        cookiesw = (entry[2] == '1')
                        cookie = entry[3]
                    if entrylength == 1 or entrylength == 4:
                        # New format with timer switch and timer
                        newformat = True
                        timersw = False
                        timer = ''
                    else:
                        timersw = (entry[4] == '1')
                        timer = entry[5]
                    if not timer:
                        timer = '5'
                    if entrylength == 1 or entrylength == 4 or entrylength == 6:
                        newformat = True
                        titlelinksw = False
                    else:
                        titlelinksw = (entry[6] == '1')
                    self.feeds[name] = RSSFeeder(self.utility, self, url, name, cookiesw, cookie, timersw, timer, titlelinksw)
                    self.feedalias.append(name)
                f.close()
                if newformat:
                    self.SaveFeeds()
            except:
                try:
                    f.close()
                except:
                    pass
                print "ERROR: reading from file rssfeed.lst"

    def populateMainFeederCombo(self):
        currentvalue = self.editfeedalias.GetValue()
        selection = self.editfeedalias.GetSelection()
        self.editfeedalias.SetItems(self.feedalias)
        self.editfeedalias.SetSelection(selection)
        self.editfeedalias.SetValue(currentvalue)

    def AddToLog(self, msg, url, msgtype = "OK"):
        self.logpanel.append(msg, url, msgtype)
