##############################################################################
# Module : scannerdlg.py
# Author : Old King Cole
# Date   : 10/15/2005
#
# Description : Dialog for directory scanner
#
##############################################################################
import wx, sys
from os import path
from string import whitespace

from filemanager import TorrentConfigFileManager


class ScannerDialog(wx.Dialog):
    def __init__(self, parent, ID, title, pos = wx.DefaultPosition, size = wx.DefaultSize,
                 style = wx.DEFAULT_DIALOG_STYLE):
        pre = wx.PreDialog()
        pre.Create(parent, ID, title, pos, size, style)
        self.this = pre.this
        self.parent = parent
        self.utility = self.parent.utility
        self.localize = self.utility.lang.get

        p = wx.Panel(self, -1)
        outerbox = wx.BoxSizer(wx.VERTICAL)

        scannerbox = wx.StaticBoxSizer(wx.StaticBox(p, -1, ''), wx.VERTICAL)

        scandirbox = wx.BoxSizer(wx.HORIZONTAL)
        scandirbox.Add(wx.StaticText(p, -1, self.localize('scandir')), 0, wx.ALIGN_CENTER_VERTICAL)
        self.scandir = wx.TextCtrl(p, -1, self.utility.scanparams['scandir'], size = (200, -1))
        scandirbox.Add(self.scandir, 1, wx.EXPAND | wx.ALIGN_CENTER_VERTICAL | wx.LEFT | wx.RIGHT, 5)
        buttonlabel = self.localize('browse')
        buttonwidth = self.GetTextExtent(buttonlabel)[0] + 12
        buttonlabel = buttonlabel.replace('&', '&&')
        browsebtn = wx.Button(p, -1, buttonlabel, wx.Point(-1, -1), wx.Size(buttonwidth, self.utility.buttonheight))
        self.Bind(wx.EVT_BUTTON, self.onBrowseScanDir, browsebtn)
        scandirbox.Add(browsebtn, 0, wx.ALIGN_CENTER_VERTICAL)
        scannerbox.Add(scandirbox, 0, wx.EXPAND | wx.TOP, 3)

        scanfreqbox = wx.BoxSizer(wx.HORIZONTAL)
        scanfreqbox.Add(wx.StaticText(p, -1, self.localize('scanfreq')), 0, wx.ALIGN_CENTER_VERTICAL)
        dtime, mtime = divmod(60 * int(self.utility.scanparams['scanfreqh']) + int(self.utility.scanparams['scanfreqm']), 1440)
        htime, mtime = divmod(mtime, 60)
        self.scanfreqd = wx.TextCtrl(p, -1, str(dtime), (-1, -1), (31, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.scanfreqd.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.scanfreqd.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        scanfreqbox.Add(self.scanfreqd, 0, wx.LEFT, 5)
        scanfreqbox.Add(wx.StaticText(p, -1, self.localize('l_day')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)
        self.scanfreqh = wx.TextCtrl(p, -1, str(htime), (-1, -1), (25, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.scanfreqh.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.scanfreqh.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        scanfreqbox.Add(self.scanfreqh, 0, wx.LEFT, 3)
        scanfreqbox.Add(wx.StaticText(p, -1, self.localize('l_hour')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)
        self.scanfreqm = wx.TextCtrl(p, -1, str(mtime), (-1, -1), (25, self.utility.timedurationstyle[0]), style = self.utility.timedurationstyle[1])
        self.scanfreqm.Bind(wx.EVT_CHAR, self.utility.onWriteDigits)
        self.scanfreqm.Bind(wx.EVT_RIGHT_UP, self.onRightButtonText)
        scanfreqbox.Add(self.scanfreqm, 0, wx.LEFT, 3)
        scanfreqbox.Add(wx.StaticText(p, -1, self.localize('l_minute')), 0, wx.LEFT | wx.ALIGN_CENTER_VERTICAL, 2)
        scannerbox.Add(scanfreqbox, 0, wx.TOP, 3)

        scanstartposbox = wx.BoxSizer(wx.HORIZONTAL)
        scanstartposbox.Add(wx.StaticText(p, -1, self.localize('scanstartpos')), 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 5)
        self.scanstartpos = wx.ComboBox(p, -1, '', (-1, -1), (120, -1),
                                        [self.localize('attop'), self.localize('atbottom')],
                                        wx.CB_DROPDOWN | wx.CB_READONLY)
        self.scanstartpos.SetSelection(int(self.utility.scanparams['scanstartpos']))
        scanstartposbox.Add(self.scanstartpos, 0, wx.ALIGN_CENTER_VERTICAL)
        scannerbox.Add(scanstartposbox, 0, wx.TOP, 3)

        scanpriobox = wx.BoxSizer(wx.HORIZONTAL)
        self.scanprio = wx.ComboBox(p, -1, '', (-1,-1), (-1, -1),
                                    [self.localize('highest'), self.localize('high'), self.localize('normal'), self.localize('low'), self.localize('lowest')],
                                    wx.CB_DROPDOWN | wx.CB_READONLY)
        self.scanprio.SetSelection(int(self.utility.scanparams['scanprio']))
        scanpriobox.Add(wx.StaticText(p, -1, self.localize('scanprio')), 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 5)
        scanpriobox.Add(self.scanprio, 0, wx.ALIGN_CENTER_VERTICAL)
        scannerbox.Add(scanpriobox, 0, wx.TOP, 3)

        scanstatusbox = wx.BoxSizer(wx.HORIZONTAL)
        scanstatusbox.Add(wx.StaticText(p, -1, self.localize('scanstatus')), 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 5)
        self.scanstatus = wx.ComboBox(p, -1, '', (-1, -1), (120, -1),
                                      [self.localize('settostop'), self.localize('settoqueue'), self.localize('start')],
                                      wx.CB_DROPDOWN | wx.CB_READONLY)
        self.scanstatus.SetSelection(int(self.utility.scanparams['scanstatus']))
        scanstatusbox.Add(self.scanstatus, 0, wx.ALIGN_CENTER_VERTICAL)
        scannerbox.Add(scanstatusbox, 0, wx.TOP, 3)

        self.scanmovetor = wx.CheckBox(p, -1, self.localize('scanmovetor'))
        if self.utility.scanparams['scanmovetor'] == "1":
            self.scanmovetor.SetValue(True)
        scannerbox.Add(self.scanmovetor, 0, wx.TOP, 3)

        self.scanautostart = wx.CheckBox(p, -1, self.localize('scanautostart'))
        if self.utility.scanparams['scanautostart'] == "1":
            self.scanautostart.SetValue(True)
        scannerbox.Add(self.scanautostart, 0, wx.TOP, 3)

        outerbox.Add(scannerbox, 0, wx.EXPAND | wx.TOP, -4)

        if self.parent.window.scanner.status:            
            self.actionbtn = wx.Button(p, -1, self.localize('stopscanner'))
        else:
            self.actionbtn = wx.Button(p, -1, self.localize('startscanner'))
        okbtn = wx.Button(p, -1, self.localize('ok'))
        cancelbtn = wx.Button(p, -1, self.localize('cancel'))

        self.Bind(wx.EVT_BUTTON, self.onAction, self.actionbtn)
        self.Bind(wx.EVT_BUTTON, self.onOK, okbtn)
        self.Bind(wx.EVT_BUTTON, self.onCancel, cancelbtn)

        button_box = wx.BoxSizer(wx.HORIZONTAL)
        button_box.Add(self.actionbtn, 0, wx.ALL, 5)
        button_box.Add(okbtn, 0, wx.ALL, 5)
        button_box.Add(cancelbtn, 0, wx.ALL, 5)

        outerbox.Add(button_box, 0, wx.ALIGN_CENTER)

        p.SetAutoLayout(True)
        p.SetSizer(outerbox)
        p.Fit()
        self.SetClientSize(p.GetSize())

        if self.parent.abcparams['scannerdlgx'] != '-1':
            self.MoveXY(int(self.parent.abcparams['scannerdlgx']), int(self.parent.abcparams['scannerdlgy']))

    def onRightButtonText(self, event):
        pass

    def onBrowseScanDir(self, event):
        folderdialog = wx.DirDialog(self, self.localize('choosescanfolder'), 
                                    style = wx.DD_DEFAULT_STYLE | wx.DD_NEW_DIR_BUTTON)
        if folderdialog.ShowModal() == wx.ID_OK:
            self.scandir.SetValue(folderdialog.GetPath())
        folderdialog.Destroy()

    def onCancel(self, event = None):
        self.EndModal(wx.ID_CANCEL)

    def onOK(self, event):
        if not self.inputData():
            return

        # Warning only
        if self.scanautostart.GetValue() and self.utility.abcparams['setextdefaultfolder'] == '0':
            # There must be a default download folder for the scanner to run
            dlg = wx.MessageDialog(self.parent, self.localize('scandefdownfoldwarn'),
                                   self.localize('abcokcwarning'), wx.ICON_INFORMATION)
            dlg.ShowModal()
            dlg.Destroy()

        self.writeScannerConf()
        self.EndModal(wx.ID_OK)

    def onAction(self, event):
        if self.parent.window.scanner.status:
            self.actionbtn.SetLabel(self.localize('startscanner'))
            self.parent.window.scannerbutton.SetValue(False)
            self.parent.window.scanner.stop()
        else:
            if self.parent.window.torrentlistloading:
                return
            if not self.inputData():
                return
            if not self.parent.window.scanner.canScan():
                return
            self.actionbtn.SetLabel(self.localize('stopscanner'))
            self.parent.window.scannerbutton.SetValue(True)
            self.writeScannerConf()
            self.parent.window.scanner.start()

    def forceStatus(self, status):
        if status:
            self.actionbtn.SetLabel(self.localize('stopscanner'))
        else:
            self.actionbtn.SetLabel(self.localize('startscanner'))

    def writeScannerConf(self):
        # Record new config to scanner.conf
        tcfm = TorrentConfigFileManager(path.join(self.utility.datapath, "scanner.conf"), self.utility.scanparams)
        tcfm.writeAllConfig()

    def inputData(self):
        # Check if directory for torrent scanner is a valid Windows name
        scanningdir = self.scandir.GetValue()
        if sys.platform == 'win32' and scanningdir != self.utility.scanparams['scandir']:
            # We erase the final '\' except for a path like 'X:\'
            if scanningdir and scanningdir[-1] == '\\' and (len(scanningdir) < 2 or scanningdir[-2] != ':'):
                scanningdir = scanningdir[:-1]
            scanningdir = scanningdir.rstrip(whitespace + '.')
            if not self.utility.checkWinPath(self, scanningdir):
                return False

        dtime = self.scanfreqd.GetValue()
        htime = self.scanfreqh.GetValue()
        mtime = self.scanfreqm.GetValue()
        if not dtime:
            dtime = '0'
        if not htime:
            htime = '0'
        if not mtime:
            mtime = '0'
        perioderr = False
        try:
            dtime_int = int(dtime)
            htime_int = int(htime)
            mtime_int = int(mtime)
        except:
            perioderr = True
        if perioderr or dtime_int < 0 or htime_int < 0 or mtime_int < 0:
            dlg = wx.MessageDialog(self.parent, self.localize('scanperioderr'),
                                   self.localize('abcokcerror'), wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        h, mtime_int = divmod(mtime_int, 60)
        htimetot_int = 24 * dtime_int + htime_int + h

        # Update scanparams
        self.utility.scanparams['scandir'] = scanningdir
        # If frequency changed, restart scanner if it's running
        if htimetot_int != int(self.utility.scanparams['scanfreqh']) or mtime_int != int(self.utility.scanparams['scanfreqm']):
            self.utility.scanparams['scanfreqh'] = str(htimetot_int)
            self.utility.scanparams['scanfreqm'] = str(mtime_int)
            if self.parent.window.scanner.status:
                self.parent.window.scanner.restart()
        self.utility.scanparams['scanstartpos'] = str(self.scanstartpos.GetSelection())
        self.utility.scanparams['scanprio'] = str(self.scanprio.GetSelection())
        self.utility.scanparams['scanstatus'] = str(self.scanstatus.GetSelection())
        if self.scanmovetor.GetValue():
            self.utility.scanparams['scanmovetor'] = "1"
        else:
            self.utility.scanparams['scanmovetor'] = "0"
        if self.scanautostart.GetValue():
            self.utility.scanparams['scanautostart'] = "1"
        else:
            self.utility.scanparams['scanautostart'] = "0"

        return True
