# Written by Bram Cohen
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

import sys
from os import path, makedirs
from urlparse import urlparse
from time import clock, sleep
from threading import Thread, Event
from StringIO import StringIO
from traceback import print_exc

from zurllib import urlopen
from BT1.Choker import Choker
from BT1.Storage import Storage
from BT1.StorageWrapper import StorageWrapper
from BT1.FileSelector import FileSelector
from BT1.Uploader import Upload
from BT1.Downloader import Downloader
from BT1.HTTPDownloader import HTTPDownloader
from BT1.Connecter import Connecter
from RateLimiter import RateLimiter
from BT1.Encrypter import Encoder
from RawServer import autodetect_ipv6, autodetect_socket_style
from BT1.Rerequester import Rerequester
from BT1.DHTRerequester import DHTRerequester
from BT1.DownloaderFeedback import DownloaderFeedback
from CurrentRateMeasure import Measure
from BT1.PiecePicker import PiecePicker
from BT1.Statistics import Statistics
from buffer import PIECEREADBUFFERPOOL
from ConfigDir import ConfigDir
from utility import exceptionArgsToString, sysencoding, getABCUtility
from bencode import bencode, bdecode
from parseargs import parseargs
from BTcrypto import CRYPTO_OK
from __init__ import createPeerID


defaults = [
    ('max_uploads', 7,
        "the maximum number of uploads to allow at once."),
    ('floor_uploads', 2,
        "the minimum value for the maximum number of uploads to allow at once."),
    ('keepalive_interval', 120.0,
        'number of seconds to pause between sending keepalives'),
    ('download_slice_size', 2 ** 14,
        "How many bytes to query for per request."),
    ('upload_unit_size', 525,
        "when limiting upload rate, how many bytes to send at a time at max" +
        "May be overwritten in RateLimiter"),
    ('max_message_length', 2 ** 23,
        "maximum length prefix encoding you'll accept over the wire - larger values get the connection dropped."),
    ('ip', '',
        "ip to report you have to the tracker."),
    ('minport', 10000, 'minimum port to listen on, counts up if unavailable'),
    ('maxport', 60000, 'maximum port to listen on'),
    ('random_port', 1, 'whether to choose randomly inside the port range ' +
        'instead of counting up linearly'),
    ('responsefile', '',
        'file the server response was stored in, alternative to url'),
    ('url', '',
        'url to get file from, alternative to responsefile'),
    ('crypto_allowed', int(CRYPTO_OK),
        'whether to allow the client to accept encrypted connections'),
    ('crypto_only', 0,
        'whether to only create or allow encrypted connections'),
    ('crypto_stealth', 0,
        'whether to prevent all non-encrypted connection attempts; ' +
        'will result in an effectively firewalled state on older trackers'),
    ('selector_enabled', 1,
        'whether to enable the file selector and fast resume function'),
    ('dir_root', 'c:\\',
        'root dir for data cache'),
    ('priority', '',
        'a list of file priorities separated by commas, must be one per file, ' +
        '0 = highest, 1 = normal, 2 = lowest, -1 = download disabled'),
    ('saveas', '',
        'local file name to save the file as, null indicates query user'),
    ('timeout', 300.0,
        'time to wait between closing sockets which nothing has been received on'),
    ('timeout_check_interval', 60.0,
        'time to wait between checking if any connections have timed out'),
    ('max_slice_length', 2 ** 17,
        "maximum length slice to send to peers, larger requests are ignored"),
    ('max_rate_period', 10.0,
        "maximum amount of time to guess the current rate estimate represents"),
    ('bind', '', 
        'comma-separated list of ips/hostnames to bind to locally'),
    #('ipv6_enabled', autodetect_ipv6(),
    ('ipv6_enabled', 0,
         'allow the client to connect to peers via IPv6'),
    ('ipv6_binds_v4', autodetect_socket_style(),
        "set if an IPv6 server socket won't also field IPv4 connections"),
    ('upnp_nat_access', 1,
        'attempt to autoconfigure a UPnP router to forward a server port ' +
        '(0 = disabled, 1 = mode 1 [fast], 2 = mode 2 [slow])'),
    #('tcp_ack_fudge', 0.03,
    ('tcp_ack_fudge', 0,
        'how much TCP ACK download overhead to add to upload rate calculations ' +
        '(0 = disabled)'),
    ('display_interval', 1.0,
        'time between updates of displayed information'),
    ('rerequest_interval', 5 * 60,
        'time to wait between requesting more peers'),
    ('dhtrerequest_interval', 5 * 60,
        'time to wait between requesting more peers in DHT'),
    ('min_peers', 20, 
        'minimum number of peers to not do rerequesting'),
    ('max_initiate', 20,
        'number of peers at which to stop initiating new connections'),
    ('check_hashes', 1,
        'whether to check hashes on disk'),
    ('max_upload_rate', 0.,
        'maximum kB/s to upload at (0 = no limit, -1 = automatic)'),
    ('max_download_rate', 0.,
        'maximum kB/s to download at (0 = no limit)'),
    ('alloc_type', 'normal',
        'allocation type (may be normal, background, pre-allocate, sparse1 or sparse2)'),
    ('alloc_rate', 2.0,
        'rate (in MB/s) to allocate space at using background allocation'),
    ('write_buffer_size', 4.0,
        'the maximum amount of space to use for buffering disk writes ' +
        '(in megabytes, 0 = disabled)'),
    ('write_buffer_expiration', 60,
        'the elapsed time when the write buffer is totally flushed if no writing has occured before' +
        '(in seconds)'),
    ('read_buffer_expiration', 60,
        'the elapsed time when the read buffer is expired if it is not accessed' +
        '(in seconds)'),
    ('metadata_expiration', 300,
        'the elapsed time when the metadata buffer is expired if it is not accessed' +
        '(in seconds)'),
    ('breakup_seed_bitfield', 0,
        'sends an incomplete bitfield and then fills with have messages, '
        'in order to get around stupid ISP manipulation'),
    ('snub_time', 60.0,
        "seconds to wait for data to come in over a connection before assuming it's snubbed"),
    ('spew', 1,
        "whether to display diagnostic info to stdout"),
    ('rarest_first_cutoff', 2,
        "number of downloads at which to switch from random to rarest first"),
    ('rarest_first_priority_cutoff', 5,
        'the number of peers which need to have a piece before other partials take priority over rarest first'),
    ('min_uploads', 4,
        "the number of uploads to fill out to with extra optimistic unchokes"),
    ('max_files_open', 40,
        'the maximum number of files to keep open at a time, 0 means no limit'),
    ('timeout_handle', 60.0,
        "max time to keep the handle for an open file when it's not accessed"),
    ('scan_handle_period', 10.0,
        "scanning period to check if an inactive file handle must be closed"),
    ('upload_round_robin_period', 30,
        "the number of seconds between the client's switching upload targets"),
    ('rechoke_period', 10,
        "the number of seconds between upload rechokings"),
    ('super_seeder', 0,
        "whether to use special upload-efficiency-maximizing routines (only for dedicated seeds)"),
    ('security', 1,
        "whether to enable extra security features intended to prevent abuse"),
    ('max_connections', 0,
        "the absolute maximum number of peers to connect with (0 = no limit)"),
    ('auto_kick', 1,
        "whether to allow the client to automatically kick/ban peers that send bad data"),
    ('double_check', 1,
        "whether to double-check data being written to the disk for errors (may increase CPU load)"),
    ('triple_check', 0,
        "whether to thoroughly check data being written to the disk (may slow disk access)"),
    ('lock_files', 1,
        "whether to lock files the client is working with"),
    ('lock_while_reading', 0,
        "whether to lock access to files being read"),
    #('auto_flush', 0,
    #    "minutes between automatic flushes to disk (0 = disabled)"),
    ('dedicated_seed_id', '',
        "code to send to tracker identifying as a dedicated seed"),
    ('dhtport', 56667, 'DHT port'),
    ('default_tracker', '', 'default tracker url'),
    ('magnet', 0, 'if torrent is created from magnet and still has incomplete metainfo'),
    ('metadata_timeout', 5, 'timeout to receive torrent metadata'),
    ('min_peer_connection_timeout', 5, 'minimum timeout to get connected with peers'),
    ('max_peer_connection_timeout', 15, 'maximum timeout to get connected with peers'),
    ('retry_connection', 0, 'whether to retry connection with peers'),
    ('ut_pex', 0, 'whether to support PEX'),
    ('ut_pex_sending_rate', 60, 'sending rate for PEX messages'),
    ('read_buffer_memory_saving', 0, 'read buffer memory optimization method'),
    ('multihome_ipv4', '', 'extra ipv4 address to add to announce for multihomed host'),
    ('multihome_ipv6', '', 'extra ipv6 address to add to announce for multihomed host'),
    ('max_extradata_rate', 102400, "max extradata up rate before canceling new metadata requests"),
    ('unit_size_step', .03125, "step for unit size increase and decrease"),
    ('unit_size_inc_trig', 5, "Number of sends without initial backlog before increasing socket send size"),
    ('read_buffer_size', 0.5, "Max size in Megabytes to buffer piece data read from disk to be uploaded to peers"),
    ('check_buffer_size', 0.25, "Max size in Megabytes to buffer piece data read from disk to be checked"),
    ('check_piece_before_upload', 0, "whether to check piece (if not already done in session) before uploading a block, or not"),
    ('kick_err_per_piece', 2, "Number of bad data per piece received to kick a peer"),
    ('kick_bad_pieces', 2, "Number of bad pieces received to kick a peer"),
    ('ban_err_per_piece', 3, "Number of bad data per piece received to unconditionally kick and ban a peer"),
    ('ban_bad_pieces', 30, "Number of bad pieces received to unconditionally kick and ban a peer"),
    ('raw_server_slowdown', 0.01, "Minimum slowdown for raw server"),
    ('min_comp_for_request', 0., "min completion % for a peer of a torrent with number of pieces > 5000 and download from seeds, to request pieces")
    ]


def parse_params(params, presets = {}):
    if len(params) == 0:
        return None
    config, args = parseargs(params, defaults, 0, 1, presets = presets)
    if args:
        if config['responsefile'] or config['url']:
            raise ValueError, getABCUtility().lang.get('argorparam')
        if path.isfile(args[0]):
            config['responsefile'] = args[0]
        else:
            try:
                urlparse(args[0])
            except:
                raise ValueError, getABCUtility().lang.get('badfile')
            config['url'] = args[0]
    elif (config['responsefile'] == '') == (config['url'] == ''):
        raise ValueError, getABCUtility().lang.get('responseorurl')
    return config


def get_response(file, errorfunc):
    try:
        h = open(file, 'rb')
        response = h.read()
    except IOError, e:
        try:    
            h.close()
        except:
            pass
        errorfunc(getABCUtility().lang.get('cantgetresponse') + ' - ' + exceptionArgsToString(e))
        return None
    h.close()
    try:
        response = bdecode(response, sloppy = 1)
    except ValueError, e:
        errorfunc(getABCUtility().lang.get('badresponse') + " - " + exceptionArgsToString(e))
        return None
    return response


class BT1Download:
    def __init__(self, statusfunc, finfunc, errorfunc, doneflag, wakeupflag,
                 config, response, infohash, id, rawserver, port, fastresume,
                 dht, pl, startqueue, appdataobj = None, extann = None,
                 trackerokfunc = None, trackerconnectingfunc = None,
                 endallocfunc = None, rejectexceptions = [], filesnamefunc = None,
                 metadatacompletefunc = None, reseed = False,
                 checkfreediskspacefunc = None, updateseenpeersfunc = None,
                 displayongoingfunc = None):
        self.statusfunc = statusfunc
        self.finfunc = finfunc
        self.errorfunc = errorfunc
        self.doneflag = doneflag
        self.wakeupflag = wakeupflag
        self.config = config
        self.response = response
        self.infohash = infohash
        self.myid = id
        self.rawserver = rawserver
        self.sched = rawserver.add_task
        self.port = port
        self.dht = dht
        self.pl = pl
        self.startqueue = startqueue
        self.switchtotorrentflag = Event()
        self.info = self.response['info']
        self.piecelength = self.info['piece length']
        self.piecereadbufferlength = PIECEREADBUFFERPOOL.bufferSize(self.piecelength, 1048576)
        self.pieces = [self.info['pieces'][x:x + 20]
                       for x in xrange(0, len(self.info['pieces']), 20)]
        self.private = self.info.get('private')
        if self.private is None:
            self.private = 0
        self.len_pieces = len(self.pieces)
        self.unpauseflag = Event()
        self.unpauseflag.set()
        self.storage = None
        self.storagewrapper = None
        self.fileselector = None
        self.downloader = None
        self.connecter = None
        self.encoder = None
        self.httpdownloader = None
        self.statistics = None
        self.downloaderfeedback = None
        self.ratelimiter = None
        self.super_seeding_active = False
        self.filedatflag = Event()
        self.spewflag = Event()
        self.superseedflag = Event()
        self.finflag = Event()
        self.rerequest = None
        self.dhtrerequest = None
        self.extann = extann
        self.tcp_ack_fudge = config['tcp_ack_fudge']
        self.trackerokfunc = trackerokfunc
        self.trackerconnectingfunc = trackerconnectingfunc
        self.endallocfunc = endallocfunc
        self.rejectexceptions = rejectexceptions
        self.filesnamefunc = filesnamefunc
        self.metadatacompletefunc = metadatacompletefunc
        self.checkfreediskspacefunc = checkfreediskspacefunc
        self.updateseenpeersfunc = updateseenpeersfunc
        self.displayongoingfunc = displayongoingfunc
        self.updatestorageflag = None
        self.localdisplay = Event()
        # To deal with the case when BT has downloaded all files of a torrent even
        # those marked "Never download" (often very small files) and wants to finish
        # In this case we force it to keep on running
        self.forcerunning = False
        self.datalength = 0
        self.blockedwaiting = False

        self.selector_enabled = config['selector_enabled']
        if appdataobj:
            self.appdataobj = appdataobj
        elif self.selector_enabled:
            self.appdataobj = ConfigDir(config['dir_root'])
        if self.selector_enabled:
            self.priority = config['priority']
            if self.priority:
                self.priority = [int(p) for p in self.priority.split(',')]
            else:
                self.priority = []
            if config['magnet']:
                self.resumedata = None
            elif reseed:
                self.resumedata = {'pieces': 1}
            elif fastresume:
                data = self.appdataobj.getTorrentData(infohash)
                if data:
                    self.resumedata = data.get('resume data', None)
                else:
                    self.resumedata = None
            else:
                self.appdataobj.deleteTorrentData(infohash)
                self.resumedata = None

        self.failed = False
        self.storageinit = False
        self.started = False

        self.picker = PiecePicker(self.len_pieces, config)
        self.choker = Choker(config, self.sched,
                             self.picker, self.finflag.isSet)

    def updateData(self, src, dest, private, updatedataflag):
        self.sched(self._updateData, 0, [src, dest, private, updatedataflag])

    def _updateData(self, src, dest, private, updatedataflag):
        self.config['responsefile'] = src
        self.config['saveas'] = dest
        self.response = get_response(src, self.errorfunc)
        self.info = self.response['info']
        self.piecelength = self.info['piece length']
        self.piecereadbufferlength = PIECEREADBUFFERPOOL.bufferSize(self.piecelength, 1048576)
        self.private = private
        self.pieces = [self.info['pieces'][x:x + 20]
                       for x in xrange(0, len(self.info['pieces']), 20)]
        self.len_pieces = len(self.pieces)
        self.storageinit = False
        self.picker.updateData(self.len_pieces)
        self.downloader.updateData(self.len_pieces)
        self.httpdownloader.updateData()
        self.connecter.updateData(self.len_pieces, private)
        self.newRerequester()
        if self.dht and private:
            self.dhtrerequest_stopped()
        updatedataflag.set()

    def updateStorage(self, filefunc, sem, checkwaiting, readwaiting, updatetoragecompletefunc):
        self.sched(self._updateStorage, 0, [filefunc, sem, checkwaiting, readwaiting, updatetoragecompletefunc])

    def _updateStorage(self, filefunc, sem, checkwaiting, readwaiting, updatetoragecompletefunc):
        self.localdisplay.set()
        self.manualstatusfunc(activity = 'init')
        # Thread to update storage while raw server keeps on running
        self.updatestorageflag = Event()
        thread = Thread(target = self.__updateStorage, args = [filefunc, sem, checkwaiting, readwaiting, updatetoragecompletefunc])
        thread.daemon = False
        thread.start()

    def __updateStorage(self, filefunc, sem, checkwaiting, readwaiting, updatetoragecompletefunc):
        # Update storage with new data
        if not self.saveAs(filefunc):
            self.startqueue.remove(self.infohash)
            self.updatestorageflag.set()
            self.sched(self._failed)
            return
        self.config['priority'] = ','.join(len(self.files) * ['1'])
        self.priority = len(self.files) * [1]
        if not self.updateFiles(sem, checkwaiting, readwaiting):
            self.updatestorageflag.set()
            return
        self.sched(self.___updateStorage, 0, [updatetoragecompletefunc])
        self.updatestorageflag.set()

    def ___updateStorage(self, updatetoragecompletefunc):
        self.statistics.updateData()
        if self.info.has_key('files'):
            self.statistics.set_dirstats(self.files, self.piecelength)
        for i in xrange(self.len_pieces):
            if self.storagewrapper.do_I_have(i):
                self.picker.complete(i)
        if self.response.has_key('httpseeds') and not self.finflag.isSet():
            for u in self.response['httpseeds']:
                u = getABCUtility().decodeString(u)
                if u:
                    self.httpdownloader.make_download(u)
        if self.selector_enabled:
            self.fileselector.tie_in(self.picker, self._cancelfunc,
                                     self._reqmorefunc, self.rerequest_ondownloadmore)
            if self.storagewrapper.amount_left == 0:
                self.appdataobj.touchTorrentData(self.infohash)
            else:
                # Erase old data once you've started modifying it
                self.appdataobj.deleteTorrentData(self.infohash)
        updatetoragecompletefunc()
        self.encoder.onUpdateStorageComplete()
        self.localdisplay.clear()
        self.oldrerequest_poststarted()

    def checkSaveLocation(self, loc):
        if self.info.has_key('length'):
            return path.exists(loc)
        for x in self.info['files']:
            if path.exists(path.join(loc, x['path'][0])):
                return True
        return False

    def saveAs(self, filefunc):
        if self.resumedata:
            notseed = (self.resumedata['pieces'] != 1)
        else:
            notseed = True
        def make(f, forcedir = False):
            if not forcedir:
                f = path.split(f)[0]
            if f != '' and not path.exists(f):
                makedirs(f)
        try:
            if self.info.has_key('length'):
                file_length = self.info['length']
                file = filefunc(self.config['saveas'])
                if file is None:
                    return None
                if notseed:
                    make(file)
                files = [(file, file_length)]
            else:
                file_length = 0L
                for x in self.info['files']:
                    file_length += x['length']
                file = filefunc(self.config['saveas'])
                if file is None:
                    return None

                ## if this path exists, and no files from the info dict exist, we assume it's a new download and 
                ## the user wants to create a new directory with the default name
                #existing = 0
                #if path.exists(file):
                #    if not path.isdir(file):
                #        self.errorfunc(file + 'is not a dir')
                #        return None
                #    if len(listdir(file)) > 0:  # if it's not empty
                #        for x in self.info['files']:
                #            if path.exists(path.join(file, x['path'][0])):
                #                existing = 1
                #        if not existing:
                #            file = path.join(file, self.info['name'])
                #            if path.exists(file) and not path.isdir(file):
                #                if file[-8:] == '.torrent':
                #                    file = file[:-8]
                #                if path.exists(file) and not path.isdir(file):
                #                    self.errorfunc("Can't create dir - " + self.info['name'])
                #                    return None
                # ===== All commented above replaced by following to have every torrent saved into its own directory
                if notseed:
                    if path.exists(file) and not path.isdir(file):
                        self.errorfunc(file + getABCUtility().lang.get('notadir2'))
                        return None
                    #else:
                    #    file = path.join(file, self.info['name'])
                    #    if path.exists(file) and not path.isdir(file):
                    #        if file[-8:] == '.torrent':
                    #            file = file[:-8]
                    #        if path.exists(file) and not path.isdir(file):
                    #            self.errorfunc("Can't create dir - " + self.info['name'])
                    #            return None
                    # ===== End of replace
                    make(file, True)

                filesname = self.filesnamefunc(self.info['files'])
                files = []
                i = 0
                for f in self.info['files']:
                    filepath = path.join(file, filesname[i])
                    files.append((filepath, f['length']))
                    if notseed:
                        make(filepath)
                    i += 1

        except OSError, e:
            self.errorfunc(getABCUtility().lang.get('cantallocatedir') + " - " + exceptionArgsToString(e))
            return None

        self.filename = file
        self.files = files
        self.datalength = file_length

        return file

    def getFilename(self):
        return self.filename

    def setPriorities(self, priority):
        self.sched(self._setPriorities, 0, [priority[:]])

    def _setPriorities(self, priority):
        self.priority = priority
        if self.selector_enabled and self.fileselector:
            self.fileselector.set_priorities(self.priority)

    def _finished(self):
        if -1 in self.priority:
            # If all files have been downloaded even the ones marked as
            # Never download, don't consider the torrent as complete
            self.forcerunning = True
            return
        try:
            self.storage.set_readonly()
        except (IOError, OSError), e:
            self.errorfunc(getABCUtility().lang.get('cantsetreadonly') + ' - ' + exceptionArgsToString(e))
        if self.superseedflag.isSet():
            self._set_super_seed()
        # self.choker.set_round_robin_period(max(self.config['upload_round_robin_period'],
                                               # self.config['upload_round_robin_period'] * self.piecelength / 200000))
        self.finfunc()
        self.finflag.set()
        self.rawserver.sockethandler.shrinkRcvBuf() 
        self.rerequest_complete()

    def _data_flunked(self, amount, index):
        if not self.doneflag.isSet():
            self.errorfunc(getABCUtility().lang.get('piecefailedhashcheck') % index)

    def _failed(self, reason = None):
        self.failed = True
        self.doneflag.set()
        if reason is not None:
            self.errorfunc(reason)

    def updateTorrentData(self):
        # Called when torrent turns complete or is checked complete
        if self.resumedata is None or self.resumedata['pieces'] != 1:
            self.appdataobj.writeTorrentData(self.infohash, {'resume data': self.fileselector.pickle()})
        self.sched(self.touchTorrentData, 43200)

    def touchTorrentData(self):
        # Touch cache data every 12 hours
        self.appdataobj.touchTorrentData(self.infohash)
        self.sched(self.touchTorrentData, 43200)

    def initFiles(self, sem, checkwaiting, readwaiting, onlycheck):
        if self.doneflag.isSet():
            self.startqueue.remove(self.infohash)
            return None
        if not self.unpauseflag.isSet():
            self.startqueue.remove(self.infohash)
            while not self.unpauseflag.isSet():
                if self.doneflag.isSet():
                    return None
                sleep(.2)
            self.startqueue.append(self.infohash)
        mustacquiresem = not self.config['magnet'] and (not self.resumedata or self.resumedata['pieces'] != 1)

        readwaiting.clear()

        if mustacquiresem:
            while self.infohash != self.startqueue[0] or not sem.acquire(False):
                if not self.blockedwaiting:
                    self.blockedwaiting = True
                    self.statusfunc(activity = 'waiting')
                readwaiting.set()
                checkwaiting.wait()
                readwaiting.clear()
                if self.doneflag.isSet():
                    self.startqueue.remove(self.infohash)
                    return None
                if not self.unpauseflag.isSet():
                    self.startqueue.remove(self.infohash)
                    while not self.unpauseflag.isSet():
                        if self.doneflag.isSet():
                            return None
                        sleep(.2)
                    self.statusfunc(activity = 'waiting')
                    self.startqueue.append(self.infohash)
                sleep(.1)
            self.statusfunc(activity = 'init')
            del self.startqueue[0]
        else:
            self.startqueue.remove(self.infohash)
            self.statusfunc(activity = 'init')
                
        self.blockedwaiting = False
        readwaiting.set()
        
        try:
            self.storage = Storage(self.files, self.piecelength, self.statusfunc,
                                   self.doneflag, self.config, self.len_pieces, self.sched)
        except IOError, e:
            self._failed(getABCUtility().lang.get('cantaccessfile') + ' - ' + exceptionArgsToString(e))

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        try:
            self.storagewrapper = StorageWrapper(self.storage, self.picker, self.config['download_slice_size'],
                                                 self.pieces, self.piecelength, self.files, onlycheck,
                                                 self._finished, self._failed, self.doneflag,
                                                 self.unpauseflag, sem, self.infohash, self.startqueue,
                                                 self.config['check_hashes'], self._data_flunked,
                                                 self.sched, self.config, self.endallocfunc,
                                                 self.checkfreediskspacefunc, self.updateTorrentData,
                                                 self.displayongoingfunc)
        except ValueError, e:
            self._failed(getABCUtility().lang.get('baddata') + ' - ' + exceptionArgsToString(e))
        except IOError, e:
            self._failed(getABCUtility().lang.get('ioerror') + ' - ' + exceptionArgsToString(e))

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        if self.selector_enabled:
            try:
                oldpriority = self.resumedata['priority']
            except:
                oldpriority = self.priority
            else:
                if len(oldpriority) != len(self.files):
                    oldpriority = self.priority

            self.fileselector = FileSelector(self.files, self.piecelength,
                                             self.appdataobj.getPieceDir(self.infohash),
                                             self.storage, self.storagewrapper,
                                             path.split(self.config['responsefile'])[1],
                                             self.sched,
                                             self._failed)
            if self.priority and not self.fileselector.init_priority(oldpriority):
                if mustacquiresem:
                    sem.release()
                return None
            if self.resumedata:
                self.fileselector.unpickle(self.resumedata)

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        self.storageinit = True

        swrapper = False
        try:
            swrapper = self.storagewrapper.old_style_init(self.statusfunc)
        except IOError, e:
            self._failed(getABCUtility().lang.get('ioerror') + ' - ' + exceptionArgsToString(e))
        except MemoryError, e:
            self._failed(getABCUtility().lang.get('memoryerror') + ' - ' + exceptionArgsToString(e))
        except SystemError, e:
            self._failed(getABCUtility().lang.get('systemerror') + ' - ' + exceptionArgsToString(e))
        except:
            data = StringIO()
            print_exc(file = data)
            self._failed(data.getvalue().decode(sysencoding))

        if swrapper is None:
            swrapper = False
        elif mustacquiresem:
            sem.release()

        if self.doneflag.isSet():
            return None

        self.resumedata = None

        return swrapper

    def updateFiles(self, sem, checkwaiting, readwaiting):
        if self.doneflag.isSet():
            self.startqueue.remove(self.infohash)
            return None
        if not self.unpauseflag.isSet():
            self.startqueue.remove(self.infohash)
            while not self.unpauseflag.isSet():
                if self.doneflag.isSet():
                    return None
                sleep(.2)
            self.startqueue.append(self.infohash)

        readwaiting.clear()

        while self.infohash != self.startqueue[0] or not sem.acquire(False):
            if not self.blockedwaiting:
                self.blockedwaiting = True
                self.manualstatusfunc(activity = 'waiting')
            readwaiting.set()
            checkwaiting.wait()
            readwaiting.clear()
            if self.doneflag.isSet():
                self.startqueue.remove(self.infohash)
                return None
            if not self.unpauseflag.isSet():
                self.startqueue.remove(self.infohash)
                while not self.unpauseflag.isSet():
                    if self.doneflag.isSet():
                        return None
                    sleep(.2)
                self.manualstatusfunc(activity = 'waiting')
                self.startqueue.append(self.infohash)
            sleep(.1)
        self.manualstatusfunc(activity = 'init')
        del self.startqueue[0]

        self.blockedwaiting = False
        readwaiting.set()

        try:
            self.storage.updateData(self.files, self.piecelength, self.len_pieces, self.manualstatusfunc)
        except IOError, e:
            self._failed(getABCUtility().lang.get('ioerror') + ' - ' + exceptionArgsToString(e))

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        try:
            self.storagewrapper.updateData(self.pieces, self.piecelength, self.files)
        except ValueError, e:
            self._failed(getABCUtility().lang.get('baddata') + ' - ' + exceptionArgsToString(e))

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        if self.selector_enabled:
            self.fileselector.updateData(self.files, self.piecelength, path.split(self.config['responsefile'])[1])

        if self.doneflag.isSet():
            if mustacquiresem:
                sem.release()
            return None

        self.storageinit = True

        swrapper = False
        try:
            swrapper = self.storagewrapper.old_style_init(self.manualstatusfunc, reset = True)
        except IOError, e:
            self._failed(getABCUtility().lang.get('ioerror') + ' - ' + exceptionArgsToString(e))
        except MemoryError, e:
            self._failed(getABCUtility().lang.get('memoryerror') + ' - ' + exceptionArgsToString(e))
        except SystemError, e:
            self._failed(getABCUtility().lang.get('systemerror') + ' - ' + exceptionArgsToString(e))
        except:
            data = StringIO()
            print_exc(file = data)
            self._failed(data.getvalue().decode(sysencoding))

        if swrapper is None:
            swrapper = False
        else:
            sem.release()

        if self.doneflag.isSet():
            return None

        return swrapper

    def _make_upload(self, connection, ratelimiter):
        return Upload(connection, ratelimiter,
                      self.choker, self.storagewrapper, self.picker,
                      self.piecereadbufferlength,
                      self.config, self.sched)

    def _kick_peer(self, connection):
        connection.close()

    def _ban_peer(self, ip, ban = True):
        self.encoder_ban(ip, ban)

    def _received_raw_data(self, x):
        if self.tcp_ack_fudge:
            x = int(x * self.tcp_ack_fudge)
            self.ratelimiter.adjust_sent(x)

    def _received_data(self, x):
        self.downmeasure.update_rate(x)

    def _received_http_data(self, x):
        self.downmeasure.update_rate(x)
        self.downloader.chunk_requested(x)

    def _cancelfunc(self, pieces):
        self.downloader.cancel_piece_download(pieces)
        self.httpdownloader.cancel_piece_download(pieces)

    def _reqmorefunc(self, pieces):
        self.downloader.requeue_piece_download(pieces)

    def startEngine(self, ratelimiter = None):
        if self.doneflag.isSet():
            return False

        if not CRYPTO_OK:
            if self.config['crypto_allowed']:
                self.errorfunc(getABCUtility().lang.get('cryptonotinstalled'))
            self.config['crypto_allowed'] = 0
            self.config['crypto_only'] = 0
            self.config['crypto_stealth'] = 0

        for i in xrange(self.len_pieces):
            if self.storagewrapper.do_I_have(i):
                self.picker.complete(i)

        self.downmeasure = Measure(self.config['max_rate_period'])

        if ratelimiter:
            self.ratelimiter = ratelimiter
        else:
            self.ratelimiter = RateLimiter(self.sched, self.config, int(getABCUtility().abcparams['rawserversocksndbuf']))

        self.downloader = Downloader(self.storagewrapper, self.picker, self.len_pieces,
                                     self._received_data, self._kick_peer, self._ban_peer,
                                     self.sched, self.config)
        self.downloader.set_download_rate(self.config['max_download_rate'])
        self.connecter = Connecter(self._make_upload, self.downloader,
                                   self.choker, self.len_pieces, self.ratelimiter,
                                   self.storagewrapper, self.dht, self.private, self.port,
                                   self.info, self.metadatacompletefunc,
                                   self.sched, self.config)
        self.ratelimiter.set_upload_rate(self.config['max_upload_rate'])
        self.encoder = Encoder(self.connecter, self.rawserver, self.myid,
                               self.dht, self.pl, self.sched, self.infohash,
                               self._received_raw_data,
                               self.ratelimiter.measure.get_rate,
                               self.downmeasure.get_rate,
                               self.localdisplay,
                               self.config)
        self.encoder_ban = self.encoder.ban

        self.httpdownloader = HTTPDownloader(self.storagewrapper, self.picker, self.sched,
                                             self.finflag, self.errorfunc, self.downloader,
                                             self.config['max_rate_period'], self.infohash,
                                             self._received_http_data, self.connecter.got_piece)

        if not self.config['magnet']:
            if self.response.has_key('httpseeds') and not self.finflag.isSet():
                for u in self.response['httpseeds']:
                    u = getABCUtility().decodeString(u)
                    if u:
                        self.httpdownloader.make_download(u)
            if self.selector_enabled:
                self.fileselector.tie_in(self.picker, self._cancelfunc,
                                         self._reqmorefunc, self.rerequest_ondownloadmore)
                if self.priority:
                    self.fileselector.set_priorities(self.priority)
                if self.storagewrapper.amount_left == 0:
                    self.appdataobj.touchTorrentData(self.infohash)
                else:
                    # Erase old data once you've started modifying it
                    self.appdataobj.deleteTorrentData(self.infohash)

        if self.config['super_seeder']:
            self.set_super_seed()

        self.started = True
        return True

    def rerequest_complete(self):
        if self.rerequest:
            self.rerequest.announce(1)

    def rerequest_started(self):
        if self.rerequest:
            self.rerequest.start()
        self.oldrerequest = None

    def rerequest_stopped(self):
        if self.rerequest:
            self.rerequest.stop()

    def rerequest_lastfailed(self):
        if self.rerequest:
            if self.rerequest.special is None:
                return not self.rerequest.lastintsuccessful
            return not self.rerequest.lastextsuccessful
        return False

    def rerequest_ondownloadmore(self):
        if self.rerequest:
            self.rerequest.hit()

    def createRerequester(self, seededfunc = None):
        if self.response.has_key('announce-list'):
            trackerlist = []
            for tier in self.response['announce-list']:
                decodedtier = []
                for t in xrange(len(tier)):
                    decodedtier.append(getABCUtility().decodeString(tier[t]))
                trackerlist.append(decodedtier)
        else:
            announce = self.response.get('announce')
            if announce is None:
                trackerlist = []
            else:
                trackerlist = [[getABCUtility().decodeString(announce)]]

        self.rerequest = Rerequester(self.port, self.myid, self.infohash,
                                     trackerlist, self.config, self.sched, self.errorfunc,
                                     self.encoder.start_connections,
                                     self.connecter.how_many_connections,
                                     self.storagewrapper.get_amount_left,
                                     self.ratelimiter.measure.get_total, self.downmeasure.get_total,
                                     self.doneflag, self.unpauseflag, seededfunc,
                                     special = self.extann, trackerokfunc = self.trackerokfunc,
                                     trackerconnectingfunc = self.trackerconnectingfunc,
                                     updateseenpeersfunc = self.updateseenpeersfunc,
                                     creatererequesterfunc = self.createRerequester,
                                     rerequeststartedfunc = self.rerequest_started,
                                     rejectexceptions = self.rejectexceptions)

    def startRerequester(self, seededfunc = None):
        self.createRerequester(seededfunc)
        self.rerequest.start()

    def newRerequester(self):
        if self.rerequest:
            self.rerequest_stopped()
            self.oldrerequest = self.rerequest
            # Create a new rerequester from rerequester queue
            self.rerequest.announce(10)

    def oldrerequest_poststarted(self):
        if self.oldrerequest:
            # Start rerequester from old rerequester queue
            self.oldrerequest.announce(11)

    def dhtrerequest_stopped(self):
        if self.dhtrerequest:
            self.dhtrerequest.stop()

    def startDHTRerequester(self):
        if self.config['ipv6_enabled']:
            alltablesempty = True
            if self.dht.table6.findNodes(self.infohash):
                alltablesempty = False
            else:
                self.dht.sched(self.dht.findCloseNodes, 5, [[self.dht.table6]])
            if self.dht.table.findNodes(self.infohash):
                alltablesempty = False
            else:
                self.dht.sched(self.dht.findCloseNodes, 5, [[self.dht.table]])
            if alltablesempty:
                self.sched(self._startDHTRerequester, 20)
            else:
                self._startDHTRerequester()
        else:
            if self.dht.table.findNodes(self.infohash):
                self._startDHTRerequester()
            else:
                # Local table is empty
                self.dht.sched(self.dht.findCloseNodes, 5, [[self.dht.table]])
                self.sched(self._startDHTRerequester, 20)

    def _startDHTRerequester(self):
        if self.response.has_key('nodes'):
            for ip, port in self.response['nodes']:
                resolvedips = getABCUtility().getIpsByName(ip)
                for ri in resolvedips:
                    self.dht.sched(self.dht.addContact, 0, [ri, port])
        self.dhtrerequest = DHTRerequester(self.infohash, self.config, self.dht,
                                           self.sched, self.errorfunc,
                                           self.encoder.start_connections,
                                           self.connecter.how_many_connections,
                                           self.unpauseflag, self.port)
        self.dhtrerequest.start()

    def _init_stats(self):
        self.statistics = Statistics(self.ratelimiter.measure, self.downmeasure,
                                     self.connecter, self.httpdownloader,
                                     self.rerequest_lastfailed,
                                     self.filedatflag, self.config)
        if self.info.has_key('files'):
            self.statistics.set_dirstats(self.files, self.piecelength)
        if self.config['spew']:
            self.spewflag.set()

    def autoStats(self):
        # self._init_stats()
        self.downloaderfeedback = DownloaderFeedback(self.storagewrapper, self.choker,
                                                     self.downloader, self.httpdownloader,
                                                     self.sched,
                                                     self.ratelimiter.measure.get_rate,
                                                     self.downmeasure.get_rate,
                                                     self.connecter.extradataupmeasure.get_rate,
                                                     self.storagewrapper.get_stats,
                                                     self.datalength, self.finflag, self.spewflag,
                                                     self.statistics, self.config, self.statusfunc,
                                                     self.config['display_interval'],
                                                     self.localdisplay)
        self.manualstatusfunc = self.downloaderfeedback.manualdisplay

    def getPortHandler(self):
        return self.encoder

    def shutdown(self, exception = False):
        if exception:
            self.failed = True
            self.doneflag.set()
        if self.updatestorageflag:
            self.updatestorageflag.wait()
        if self.storageinit:
            if self.failed:
                self.storagewrapper.releaseWriteBuffer()
            else:
                self.storagewrapper.sync()
            self.storage.close()
        if self.rerequest:
            self.rerequest_stopped()
        if self.dhtrerequest:
            self.dhtrerequest_stopped()
        if self.encoder:
            self.encoder.shutdown()
        if self.connecter:
            self.connecter.shutdown()
        if self.httpdownloader:
            self.httpdownloader.shutdown()
        if self.downloader :
            self.downloader.shutdown()
        if self.fileselector and self.started and not self.config['magnet']:
            if self.storagewrapper.amount_left == 0:
                self.appdataobj.touchTorrentData(self.infohash)
            else:
                torrentdata = {}
                if not self.failed:
                    torrentdata['resume data'] = self.fileselector.pickle()
                self.appdataobj.writeTorrentData(self.infohash, torrentdata)
        if self.pl:
            self.switchtotorrentflag.set()
        self.rawserver = None
        if self.started:
            return self.downmeasure.get_total(), self.ratelimiter.measure.get_total()

    def setUploadRate(self, rate):
        def s(self = self, rate = rate):
            self.config['max_upload_rate'] = rate
            self.ratelimiter.set_upload_rate(rate)
        self.sched(s)

    def setConns(self, conns, conns2 = None):
        if not conns2:
            conns2 = conns
        def s(self = self, conns = conns, conns2 = conns2):
            self.config['min_uploads'] = conns
            self.config['max_uploads'] = conns2
            self.choker._rechoke()
            #if (conns > 30):
            #    self.config['max_initiate'] = conns + 10
        self.sched(s)
        
    def setDownloadRate(self, rate):
        def s(self = self, rate = rate):
            self.config['max_download_rate'] = rate
            self.downloader.set_download_rate(rate)
        self.sched(s)

    def setInitiate(self, initiate):
        def s(self = self, initiate = initiate):
            self.config['max_initiate'] = initiate
        self.sched(s)

    def setDHTDisconnected(self, value):
        self.sched(self._setDHTDisconnected, 0, [value])

    def _setDHTDisconnected(self, value):
        if self.dhtrerequest:
            self.dhtrerequest.setDisconnected(value)

    def reannounce(self, tracker = None, special = None):
        if tracker == 'ext' and special is None:
            return
        if tracker == 'reset':
            self.sched(self.resetTracker)
        elif tracker == 'rescan':
            self.sched(self.resetTracker, 0, [True])
        elif tracker == 'ext' or tracker == 'int':
            self.sched(self.switchTracker, 0, [tracker, special])
        else:
            self.sched(self.continueTracker)

    def resetTracker(self, rescan = False):
        if self.rerequest:
            self.rerequest.resetTracker(rescan)

    def switchTracker(self, tracker, special):
        if self.rerequest:
            self.extann = special
            self.rerequest.switchTracker(tracker = tracker, specialurl = special)

    def continueTracker(self):
        if self.rerequest:
            self.rerequest.continueTracker()
        if self.dhtrerequest:
            self.dhtrerequest.rerequest(True)

    def getResponse(self):
        try:
            return self.response
        except:
            return None

    def Pause(self):
        self.unpauseflag.clear()
        self.sched(self.onPause)

    def onPause(self):
        self.downloader.pause(True)
        self.encoder.pause(True)
        self.connecter.pause(True)
        self.choker.pause(True)
        self.rerequest.stop()
        if self.dhtrerequest:
            self.dhtrerequest.stop()

    def Unpause(self):
        self.unpauseflag.set()
        self.sched(self.onUnpause)

    def onUnpause(self):
        self.downloader.pause(False)
        self.encoder.pause(False)
        self.connecter.pause(False)
        self.choker.pause(False)
        self.rerequest.start()
        if self.dhtrerequest:
            self.dhtrerequest.start()

    def set_super_seed(self):
        self.superseedflag.set()
        def s(self = self):
            if self.finflag.isSet():
                self._set_super_seed()
        self.sched(s)

    def _set_super_seed(self):
        if not self.super_seeding_active:
            self.super_seeding_active = True
            self.errorfunc(getABCUtility().lang.get('superseedactive'))
            def s(self = self):
                self.downloader.set_super_seed()
                self.choker.set_super_seed()
            self.sched(s)
            if self.finflag.isSet() and self.rerequest: # mode started when already finished
                # so after kicking everyone off, reannounce
                self.sched(self.rerequest.announce, 0, [3])

    def am_I_finished(self):
        return self.finflag.isSet()

    def addPeers(self, peers, prio = 0):
        if self.encoder:
            self.sched(self.encoder.start_connections, 0, [peers, prio])

    def setPex(self, flag):
        if self.connecter:
            self.sched(self.connecter.setPex, 0, [flag])

    def setWriteBufferSize(self, writebuffersize):
        self.sched(self._setWriteBufferSize, 0, [writebuffersize])

    def _setWriteBufferSize(self, writebuffersize):
        self.config['write_buffer_size'] = writebuffersize
        if self.storagewrapper:
            self.storagewrapper.setWriteBufferSize(writebuffersize)

    def setReadBufferSize(self, readbuffersize):
        self.sched(self._setReadBufferSize, 0, [readbuffersize])

    def _setReadBufferSize(self, readbuffersize):
        self.config['read_buffer_size'] = readbuffersize
        if self.connecter:
            self.connecter.setReadBufferSize(readbuffersize)

    def setReadBufferMemorySaving(self, mode):
        self.sched(self._setReadBufferMemorySaving, 0, [mode])

    def _setReadBufferMemorySaving(self, mode):
        self.config['read_buffer_memory_saving'] = mode

    # Single port mode (incoming peer from peer listener)
    def addPLPeer(self, connection, buffer, uses_dht, uses_extended, encrypted, encrypter, cryptmode):
        self.encoder.eventsfrompl = True
        self.wakeupflag.set()
        self.sched(self.encoder.external_connection_made, 0, [connection, buffer, uses_dht, uses_extended, encrypted, encrypter,
                                                              cryptmode, self.switchtotorrentflag])
        self.switchtotorrentflag.wait()
        self.switchtotorrentflag.clear()
        self.wakeupflag.clear()
