# Written by Bram Cohen
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

from BitTornado.buffer import newPieceReadBuffer
from BitTornado.utility import exceptionArgsToString, getABCUtility
from BitTornado.bitfield import Bitfield
from threading import Lock
from time import strftime, clock, sleep
import os
from os.path import exists, getsize, getmtime, basename
from traceback import print_exc
try:
    from os import fsync
except ImportError:
    fsync = lambda x: None
from bisect import bisect
if os.name == 'nt':
    import msvcrt
elif os.name == 'posix':
    import fcntl

DEBUG = False

# MAXREADSIZE = 16384
MAXREADSIZE = 524288
MAXLOCKSIZE = 1000000000L
MAXLOCKRANGE = 3999999999L   # only lock first 4 gig of file


class Storage:
    def __init__(self, files, piece_length, statusfunc,
                 doneflag, config, piecesnb, sched):
        # can raise IOError and ValueError
        self.doneflag = doneflag
        self.config = config
        self.files = files
        self.piece_length = piece_length
        self.total_length = 0
        self.statusfunc = statusfunc
        self.piecesnb = piecesnb
        self.sched = sched
        self.blockedwaiting = False
        self.disabled = Bitfield(len(files))
        self.file_ranges = []
        self.disabled_ranges = []
        self.working_ranges = []
        self.ranges = []
        self.begins = []
        self.handles = {}
        self.whandles = {}
        self.lastaccess = {}
        self.handlebuffer = []
        self.tops = {}
        self.sizes = {}
        # self.mtimes = {}
        self.releasesem = False
        if config.get('lock_files', True):
            self.lock_file, self.unlock_file = self._lock_file, self._unlock_file
        else:
            self.lock_file = self.unlock_file = None
        self.lock_while_reading = config.get('lock_while_reading', False)
        # self.lock = Lock()
        self.max_files_open = config['max_files_open']
        self.timeout_handle = config['timeout_handle']
        self.scan_handle_period = config['scan_handle_period']

        if not config['magnet']:
            self.initFiles(files)

    def updateData(self, files, piece_length, piecesnb, statusfunc):
        self.files = files
        self.piece_length = piece_length
        self.piecesnb = piecesnb
        self.disabled = Bitfield(len(files))
        self.file_ranges = []
        self.disabled_ranges = []
        self.working_ranges = []
        self.handles = {}
        self.whandles = {}
        self.tops = {}
        self.sizes = {}
        # self.mtimes = {}
        self.statusfunc = statusfunc
        self.initFiles(files)

    def initFiles(self, files):
        numfiles = 0
        total = 0L
        t = 0.
        for file, length in files:
            if self.doneflag.isSet():    # bail out if doneflag is set
                return
            self.disabled_ranges.append(None)
            if length == 0:
                self.file_ranges.append(None)
                self.working_ranges.append([])
                h = open(file, 'wb+')
                h.close()
            else:
                range = (total, total + length, 0, file)
                self.file_ranges.append(range)
                self.working_ranges.append([range])
                numfiles += 1
                total += length
                if exists(file):
                    l = getsize(file)
                    if l > length:
                        h = open(file, 'rb+')
                        h.truncate(length)
                        h.close()
                        l = length
                else:
                    l = 0
                    h = open(file, 'wb+')
                    h.close()
                # self.mtimes[file] = getmtime(file)
                self.tops[file] = l
                self.sizes[file] = length

            now = clock()
            if t < now:
                t = now + 1
                self.statusfunc(activity = 'init')

        # Check for duplicate files
        #if len(files) > 1:
        #    nbdestfiles = 0
        #    for r, d, f in os.walk(config['saveas']):
        #        nbdestfiles += len(f)
        #    if nbdestfiles < len(files):
        #        raise IOError(getABCUtility().lang.get('duplicatefile'))

        self.total_length = total

        self._reset_ranges()

        self.handlebuffer = []

    if os.name == 'nt':
        def _lock_file(self, name, f):
            for p in range(0, min(self.sizes[name], MAXLOCKRANGE), MAXLOCKSIZE):
                f.seek(p)
                msvcrt.locking(f.fileno(), msvcrt.LK_LOCK,
                               min(MAXLOCKSIZE, self.sizes[name] - p))

        def _unlock_file(self, name, f):
            for p in range(0, min(self.sizes[name], MAXLOCKRANGE), MAXLOCKSIZE):
                f.seek(p)
                msvcrt.locking(f.fileno(), msvcrt.LK_UNLCK,
                               min(MAXLOCKSIZE, self.sizes[name] - p))

    elif os.name == 'posix':
        def _lock_file(self, name, f):
            fcntl.flock(f.fileno(), fcntl.LOCK_EX)

        def _unlock_file(self, name, f):
            fcntl.flock(f.fileno(), fcntl.LOCK_UN)

    else:
        def _lock_file(self, name, f):
            pass

        def _unlock_file(self, name, f):
            pass

    def startScanForHandleTimeout(self):
        self.sched(self.scanForHandleTimeouts, self.scan_handle_period)

    def scanForHandleTimeouts(self):
        # Close one handle if there are buffers older than timeout_handle
        self.sched(self.scanForHandleTimeouts, self.scan_handle_period)
        if self.handlebuffer:
            file0 = self.handlebuffer[0]
            if clock() - self.lastaccess[file0] > self.timeout_handle:
                self._close(file0)
                del self.handlebuffer[0]

    def was_preallocated(self, pos, length):
        for file, begin, end in self._intervals(pos, length):
            if self.tops.get(file, 0) < end:
                return False
        return True

    def _sync(self, file):
        self._close(file)
        self.handlebuffer.remove(file)

    def sync(self):
        # may raise IOError or OSError
        for file in self.whandles.keys():
            self._sync(file)

    def set_readonly(self, f = None):
        if f is None:
            self.sync()
            return
        file = self.files[f][0]
        if self.whandles.has_key(file):
            self._sync(file)

    def get_total_length(self):
        return self.total_length

    # def _open(self, file, mode):
        # # if self.mtimes.has_key(file):
            # # try:
                # # assert getsize(file) == self.tops[file]
                # # newmtime = getmtime(file)
                # # oldmtime = self.mtimes[file]
                # # assert newmtime <= oldmtime + 1
                # # assert newmtime >= oldmtime - 1
            # # except:
                # # if DEBUG:
                    # # print (file + ' modified: '
                           # # + strftime('(%x %X)', localtime(self.mtimes[file]))
                           # # + strftime(' != (%x %X) ?', localtime(getmtime(file))))
                # # raise IOError(getABCUtility().lang.get('modifieddown'))
        # return open(file, mode, buffering = 0)

    def _close(self, file):
        f = self.handles[file]
        del self.handles[file]
        del self.lastaccess[file]
        if self.whandles.has_key(file):
            del self.whandles[file]
            if self.unlock_file:
                self.unlock_file(file, f)
            f.close()
            self.tops[file] = getsize(file)
            # self.mtimes[file] = getmtime(file)
        else:
            if self.lock_while_reading:
                self.unlock_file(file, f)
            f.close()

    def _close_file(self, file):
        if not self.handles.has_key(file):
            return
        self._close(file)
        self.handlebuffer.remove(file)

    def _get_file_handle(self, file, for_write):
        if self.handles.has_key(file):
            if for_write and not self.whandles.has_key(file):
                self.whandles[file] = 1
                if self.lock_file and not self.lock_while_reading:
                    self.lock_file(file, self.handles[file])                
            if self.handlebuffer[-1] != file:
                self.handlebuffer.remove(file)
                self.handlebuffer.append(file)
        else:
            try:
                f = open(file, 'rb+', buffering = 0)
            except (IOError, OSError), e:
                if DEBUG:
                    print_exc()
                raise IOError(getABCUtility().lang.get('cantopen') + file + ' : ' + exceptionArgsToString(e))

            self.handles[file] = f
            if for_write:
                self.whandles[file] = 1
                if self.lock_file:
                    self.lock_file(file, f)
            elif self.lock_while_reading:
                self.lock_file(file, f)

            self.handlebuffer.append(file)
            if 0 < self.max_files_open < len(self.handlebuffer):
                self._close(self.handlebuffer.pop(0))

        self.lastaccess[file] = clock()

        return self.handles[file]

    def _reset_ranges(self):
        self.ranges = []
        for l in self.working_ranges:
            self.ranges.extend(l)
        self.begins = [i[0] for i in self.ranges]

    def _intervals(self, pos, amount):
        r = []
        stop = pos + amount
        p = bisect(self.begins, pos) - 1
        while p < len(self.ranges):
            begin, end, offset, file = self.ranges[p]
            if begin >= stop:
                break
            r.append((file,
                      offset + max(pos, begin) - begin,
                      offset + min(end, stop) - begin))
            p += 1
        return r

    def read(self, pos, amount, flush_first = False, checkoverload = False, buf = None):
        if buf is None:
            buf = newPieceReadBuffer(size = amount, checkoverload = checkoverload)
            if buf is None:
                return None
        for file, pos, end in self._intervals(pos, amount):
            if DEBUG:
                print 'reading ' + file + ' from ' + str(pos) + ' to ' + str(end)
            # self.lock.acquire()
            h = self._get_file_handle(file, False)
            if flush_first and self.whandles.has_key(file):
                fsync(h)
            h.seek(pos)
            ############################################
            while pos < end:
                length = min(end - pos, MAXREADSIZE)
                oldbuflen = len(buf)
                buf.append(h.read(length))
                if len(buf) - oldbuflen != length:
                    buf.release()
                    raise IOError(getABCUtility().lang.get('errorreadingdata') + file)
                pos += length
            ############################################
            # length = end - pos
            # oldbuflen = len(buf)
            # try:
                # buf.append(h.read(length))
            # except IOError, e:
                # buf.release()
                # raise IOError(e)
            # if len(buf) - oldbuflen != length:
                # buf.release()
                # raise IOError(getABCUtility().lang.get('errorreadingdata') + file)
            ############################################
            # length = end - pos
            # oldbuflen = len(buf)
            # buf.append(h.read(length))
            # if len(buf) - oldbuflen != length:
                # buf.release()
                # raise IOError(getABCUtility().lang.get('errorreadingdata') + file)
            ############################################
            # self.lock.release()
        return buf

    def write(self, pos, s):
        # might raise an IOError
        total = 0
        for file, begin, end in self._intervals(pos, len(s)):
            if DEBUG:
                print 'writing ' + file + ' from ' + str(pos) + ' to ' + str(end)
            # self.lock.acquire()
            h = self._get_file_handle(file, True)
            h.seek(begin)
            h.write(s[total:total + end - begin])
            # self.lock.release()
            total += end - begin

    def toBeAllocated(self, topoffval):
        self.allocationperfile = []
        self.topoffval = topoffval
        for f in xrange(len(self.files)):
            if not self.disabled[f] and self.file_ranges[f]:
                begin, end, offset, file = self.file_ranges[f]
                endloc = offset + end - begin
                tobeallocated = endloc - self.tops.get(file, 0)
                if tobeallocated > 0:
                    self.allocationperfile.append((file, endloc, tobeallocated))
        return self.allocationperfile

    def top_off(self, f, endlock = None, endflag = None, syncflag = None, sem = None):
        file, endloc = self.allocationperfile[f][:2]
        h = open(file, 'wb')
        h.seek(endloc - 1)
        h.write(self.topoffval)
        h.close()
        self.tops[file] = getsize(file)
        # self.mtimes[file] = getmtime(file)
        if endlock:
            with endlock:
                syncflag.wait()
                if self.releasesem:
                    self.releasesem = False
                    sem.release()
                endflag.set()

    def close(self):
        for file, f in self.handles.items():
            try:
                if self.unlock_file:
                    self.unlock_file(file, f)
            except:
                pass
            try:
                f.close()
            except:
                pass
        self.handles.clear()
        self.whandles.clear()
        self.lastaccess.clear()
        del self.handlebuffer[:]

    def _get_disabled_ranges(self, f):
        if not self.file_ranges[f]:
            return ((), (), ())
        r = self.disabled_ranges[f]
        if r:
            return r
        start, end, offset, file = self.file_ranges[f]
        if DEBUG:
            print 'calculating disabled range for ' + self.files[f][0]
            print 'bytes: ' + str(start) + '-' + str(end)
            print 'file spans pieces ' + str(int(start / self.piece_length)) + '-' + str(int((end - 1) / self.piece_length) + 1)
        pieces = range(int(start / self.piece_length),
                       int((end - 1) / self.piece_length) + 1)
        offset = 0
        disabled_files = []
        if len(pieces) == 1:
            if (start % self.piece_length == 0
                and end % self.piece_length == 0):   # happens to be a single,
                                                     # perfect piece
                working_range = [(start, end, offset, file)]
                update_pieces = []
            else:
                midfile = os.path.join(self.bufferdir, str(f))
                working_range = [(start, end, 0, midfile)]
                disabled_files.append((midfile, start, end))
                length = end - start
                self.sizes[midfile] = length
                piece = pieces[0]
                update_pieces = [(piece, start - (piece * self.piece_length), length)]
        else:
            update_pieces = []
            if start % self.piece_length != 0:  # doesn't begin on an even piece boundary
                end_b = pieces[1] * self.piece_length
                startfile = os.path.join(self.bufferdir, str(f) + 'b')
                working_range_b = [(start, end_b, 0, startfile)]
                disabled_files.append((startfile, start, end_b))
                length = end_b - start
                self.sizes[startfile] = length
                offset = length
                piece = pieces.pop(0)
                update_pieces.append((piece, start - (piece * self.piece_length), length))
            else:
                working_range_b = []
            if f != len(self.files) - 1 and end % self.piece_length != 0:
                                                # doesn't end on an even piece boundary
                start_e = pieces[-1] * self.piece_length
                endfile = os.path.join(self.bufferdir, str(f) + 'e')
                working_range_e = [(start_e, end, 0, endfile)]
                disabled_files.append((endfile, start_e, end))
                length = end - start_e
                self.sizes[endfile] = length
                piece = pieces.pop(-1)
                update_pieces.append((piece, 0, length))
            else:
                working_range_e = []
            if pieces:
                working_range_m = [(pieces[0] * self.piece_length,
                                   (pieces[-1] + 1) * self.piece_length,
                                   offset, file)]
            else:
                working_range_m = []
            working_range = working_range_b + working_range_m + working_range_e

        if DEBUG:            
            print str(working_range)
            print str(update_pieces)
        r = (tuple(working_range), tuple(update_pieces), tuple(disabled_files))
        self.disabled_ranges[f] = r
        return r

    def set_bufferdir(self, dir):
        self.bufferdir = dir

    def enable_file(self, f):
        if not self.disabled[f]:
            return
        self.disabled[f] = False
        r = self.file_ranges[f]
        if not r:
            return
        file = r[3]
        if not exists(file):
            h = open(file, 'wb+')
            h.close()
        if not self.tops.has_key(file):
            self.tops[file] = getsize(file)
        # if not self.mtimes.has_key(file):
            # self.mtimes[file] = getmtime(file)
        self.working_ranges[f] = [r]

    def disable_file(self, f):
        if self.disabled[f]:
            return
        self.disabled[f] = True
        r = self._get_disabled_ranges(f)
        if not r:
            return
        for file, begin, end in r[2]:
            if not os.path.isdir(self.bufferdir):
                os.makedirs(self.bufferdir)
            if not exists(file):
                h = open(file, 'wb+')
                h.close()
            if not self.tops.has_key(file):
                self.tops[file] = getsize(file)
            # if not self.mtimes.has_key(file):
                # self.mtimes[file] = getmtime(file)
        self.working_ranges[f] = r[0]

    reset_file_status = _reset_ranges

    def get_piece_update_list(self, f):
        return self._get_disabled_ranges(f)[1]

    def delete_file(self, f):
        try:
            os.remove(self.files[f][0])
        except:
            pass

    '''
    Pickled data format:

    d['files'] = [ file #, size, mtime {, file #, size, mtime...} ]
                    file # in torrent, and the size and last modification
                    time for those files.  Missing files are either empty
                    or disabled.
    d['partial files'] = [ name, size, mtime... ]
                    Names, sizes and last modification times of files containing
                    partial piece data.  Filenames go by the following convention:
                    {file #, 0-based}{nothing, "b" or "e"}
                    eg: "0e" "3" "4b" "4e"
                    Where "b" specifies the partial data for the first piece in
                    the file, "e" the last piece, and no letter signifying that
                    the file is disabled but is smaller than one piece, and that
                    all the data is cached inside so adjacent files may be
                    verified.
    '''
    def pickle(self):
        files = []
        pfiles = []
        for i in xrange(len(self.files)):
            if not self.files[i][1]:    # length == 0
                continue
            if self.disabled[i]:
                for file, start, end in self._get_disabled_ranges(i)[2]:
                    pfiles.extend([basename(file), getsize(file), int(getmtime(file))])
                continue
            file = self.files[i][0]
            files.extend([i, getsize(file), int(getmtime(file))])
        return {'files': files, 'partial files': pfiles}

    def unpickle(self, data):
        # assume all previously-disabled files have already been disabled
        try:
            if data['pieces'] == 1:
                valid_pieces = Bitfield(self.piecesnb, '\xFF' * ((self.piecesnb - 1) / 8 + 1))
            else:
                valid_pieces = Bitfield(self.piecesnb)
                files = {}
                pfiles = {}
                l = data['files']
                assert len(l) % 3 == 0
                l = [l[x:x + 3] for x in xrange(0, len(l), 3)]
                for f, size, mtime in l:
                    files[f] = (size, mtime)
                l = data.get('partial files', [])
                assert len(l) % 3 == 0
                l = [l[x:x + 3] for x in xrange(0, len(l), 3)]
                for file, size, mtime in l:
                    pfiles[file] = (size, mtime)

                for i in xrange(len(self.files)):
                    if self.disabled[i]:
                        continue
                    r = self.file_ranges[i]
                    if not r:
                        continue
                    start, end, offset, file = r
                    if DEBUG:
                        print 'adding ' + file
                    for p in xrange(int(start / self.piece_length),
                                    int((end - 1) / self.piece_length) + 1):
                        valid_pieces[p] = 1

                def test(old, size, mtime):
                    oldsize, oldmtime = old
                    if size != oldsize:
                        return False
                    if mtime > oldmtime + 1:
                        return False
                    if mtime < oldmtime - 1:
                        return False
                    return True

                for i in xrange(len(self.files)):
                    if self.disabled[i]:
                        for file, start, end in self._get_disabled_ranges(i)[2]:
                            f1 = basename(file)
                            if (not pfiles.has_key(f1)
                                or not test(pfiles[f1], getsize(file), getmtime(file))):
                                if DEBUG:
                                    print 'removing ' + file
                                for p in xrange(int(start / self.piece_length),
                                                int((end - 1) / self.piece_length) + 1):
                                    valid_pieces[p] = 0
                        continue
                    file, size = self.files[i]
                    if not size:
                        continue
                    if (not files.has_key(i)
                        or not test(files[i], getsize(file), getmtime(file))):
                        start, end, offset, file = self.file_ranges[i]
                        if DEBUG:
                            print 'removing ' + file
                        for p in xrange(int(start / self.piece_length),
                                        int((end - 1) / self.piece_length) + 1):
                            valid_pieces[p] = 0
        except:
            if DEBUG:
                print_exc()
            return Bitfield(self.piecesnb)

        return valid_pieces
