# Written by John Hoffman
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

from os import path, remove
from random import shuffle
from BitTornado.bitfield import Bitfield
from BitTornado.utility import exceptionArgsToString, getABCUtility
from BitTornado.buffer import newPieceReadBuffer


class FileSelector:
    def __init__(self, files, piece_length, bufferdir, storage,
                 storagewrapper, torrentfilename, sched, failfunc):
        self.files = files
        self.storage = storage
        self.storagewrapper = storagewrapper
        self.torrentfilename = torrentfilename
        self.sched = sched
        self.failfunc = failfunc
        self.downloader = None
        self.picker = None

        storage.set_bufferdir(bufferdir)

        self.numfiles = len(files)
        self.priority = [1] * self.numfiles
        self.new_priority = None
        self.new_partials = None
        self.filepieces = []
        total = 0L
        for file, length in files:
            if not length:
                self.filepieces.append(())
            else:
                pieces = xrange(int(total / piece_length),
                                int((total + length - 1) / piece_length) + 1)
                self.filepieces.append(tuple(pieces))
                total += length
        self.numpieces = int((total + piece_length - 1) / piece_length)
        self.piece_priority = bytearray(self.numpieces)
        for i in xrange(self.numpieces):
            self.piece_priority[i] = 1
        self.new_piece_priority = None

    def updateData(self, files, piecelength, torrentfilename):
        self.files = files
        self.torrentfilename = torrentfilename
        self.numfiles = len(files)
        self.priority = [1] * self.numfiles
        self.new_priority = None
        self.new_partials = None
        self.filepieces = []
        total = 0L
        for file, length in files:
            if not length:
                self.filepieces.append(())
            else:
                pieces = xrange(int(total / piecelength),
                                int((total + length - 1) / piecelength) + 1)
                self.filepieces.append(tuple(pieces))
                total += length
        self.numpieces = int((total + piecelength - 1) / piecelength)
        self.piece_priority = bytearray(self.numpieces)
        for i in xrange(self.numpieces):
            self.piece_priority[i] = 1

    def init_priority(self, priority):
        files_updated = False
        try:
            for f in xrange(self.numfiles):
                if priority[f] < 0:
                    self.storage.disable_file(f)
                    files_updated = True
        except (IOError, OSError), e:
            self.failfunc(getABCUtility().lang.get('cantopenpartial')
                          + self.files[f][0] + ' : ' + exceptionArgsToString(e))
            return False
        if files_updated:
            self.storage.reset_file_status()
        self.new_priority = priority[:]
        self.new_piece_priority = self._get_piece_priority_list(self.new_priority)
        self.storagewrapper.reblock(Bitfield(boollist = [i == 3 for i in self.new_piece_priority]))
        return True

    '''
    d['priority'] = [file #1 priority [,file #2 priority...] ]
                    a list of download priorities for each file.
                    Priority may be -1, 0, 1, 2.  -1 = download disabled,
                    0 = highest, 1 = normal, 2 = lowest.
    Also see Storage.pickle and StorageWrapper.pickle for additional keys.
    '''
    def unpickle(self, d):
        pieces = self.storage.unpickle(d)
        if pieces.empty():  # don't bother, nothing restoreable
            return
        self.new_partials = self.storagewrapper.unpickle(d, pieces)

    def tie_in(self, picker, cancelfunc, requestmorefunc, rerequestfunc):
        self.picker = picker
        self.cancelfunc = cancelfunc
        self.requestmorefunc = requestmorefunc
        self.rerequestfunc = rerequestfunc

        if self.new_priority:
            self.priority = self.new_priority
            self.new_priority = None

        if self.new_partials:
            shuffle(self.new_partials)
            for p in self.new_partials:
                self.picker.requested(p)
        self.new_partials = None

    def _set_files_disabled(self, old_priority, new_priority):
        old_disabled = [p == -1 for p in old_priority]
        new_disabled = [p == -1 for p in new_priority]
        data_to_update = []
        for f in xrange(self.numfiles):
            if new_disabled[f] != old_disabled[f]:
                data_to_update.extend(self.storage.get_piece_update_list(f))
        if data_to_update:
            buffer = []
            selectortempfilename = path.join(getABCUtility().datapath, self.torrentfilename + ".tmp")
            try:
                selectortempfile = open(selectortempfilename, 'wb')
            except:
                self.failfunc(getABCUtility().lang.get('cantaccessselectortemp'))
                return False
            for piece, start, length in data_to_update:
                if self.storagewrapper.has_data(piece):
                    data = self.storagewrapper.read_raw(piece, start, length, flush_first = True)
                    try:
                        selectortempfile.write(data.viewSlice())
                    except:
                        data.release()
                        selectortempfile.close()
                        try:
                            remove(selectortempfilename)
                        except:
                            pass
                        self.failfunc(getABCUtility().lang.get('cantaccessselectortemp'))
                        return False
                    data.release()
                    buffer.append((piece, start, len(data)))
            selectortempfile.close()

            try:
                for f in xrange(self.numfiles):
                    if new_disabled[f] and not old_disabled[f]:
                        self.storage.disable_file(f)
                    elif old_disabled[f] and not new_disabled[f]:
                        self.storage.enable_file(f)
            except (IOError, OSError), e:
                try:
                    remove(selectortempfilename)
                except:
                    pass
                if new_disabled[f]:
                    msg = getABCUtility().lang.get('cantopenpartial')
                else:
                    msg = getABCUtility().lang.get('cantopen')
                self.failfunc(msg + self.files[f][0] + ' : ' + exceptionArgsToString(e))
                return False
            self.storage.reset_file_status()

            changed_pieces = {}
            try:
                selectortempfile = open(selectortempfilename, 'rb')
            except:
                self.failfunc(getABCUtility().lang.get('cantaccessselectortemp'))
                return False
            for piece, start, datalen in buffer:
                data = newPieceReadBuffer(size = datalen)
                try:
                    data.append(selectortempfile.read(datalen))
                except:
                    data.release()
                    selectortempfile.close()
                    try:
                        remove(selectortempfilename)
                    except:
                        pass
                    self.failfunc(getABCUtility().lang.get('cantaccessselectortemp'))
                    return False
                self.storagewrapper.write_raw(piece, start, data.viewSlice())
                data.release()
                changed_pieces[piece] = 1
            selectortempfile.close()
            try:
                remove(selectortempfilename)
            except:
                pass
            if not self.storagewrapper.doublecheck_data(changed_pieces):
                return False

        return True

    def _get_piece_priority_list(self, file_priority_list):
        l = bytearray(self.numpieces)
        for i in xrange(self.numpieces):
            l[i] = 3
        for f in xrange(self.numfiles):
            if file_priority_list[f] == -1:
                continue
            for i in self.filepieces[f]:
                if l[i] == 3:
                    l[i] = file_priority_list[f]
                    continue
                l[i] = min(l[i], file_priority_list[f])
        return l

    def _set_piece_priority(self):
        was_complete = self.storagewrapper.am_I_complete()
        pieces = range(self.numpieces)
        shuffle(pieces)
        new_blocked = []
        new_unblocked = []
        for piece in pieces:
            self.picker.set_priority(piece, self.new_piece_priority[piece])
            o = self.piece_priority[piece] == 3
            n = self.new_piece_priority[piece] == 3
            if n and not o:
                new_blocked.append(piece)
            if o and not n:
                new_unblocked.append(piece)
        if new_blocked:
            self.cancelfunc(new_blocked)
        if new_unblocked:
            self.requestmorefunc(new_unblocked)
        if was_complete and not self.storagewrapper.am_I_complete():
            self.rerequestfunc()
        self.piece_priority = self.new_piece_priority
        self.new_piece_priority = None

    def _set_priorities(self):
        if not self.new_priority:
            return
        old_priority = self.priority
        self.priority = self.new_priority
        self.new_priority = None
        if not self._set_files_disabled(old_priority, self.priority):
            return
        self._set_piece_priority()

    def set_priorities(self, new_priority):
        if new_priority == self.priority:
            # This case can happen only on torrent startup
            self._set_piece_priority()
            return
        self.new_priority = new_priority[:]
        self.new_piece_priority = self._get_piece_priority_list(new_priority)
        self.storagewrapper.reblock(Bitfield(boollist = [i == 3 for i in self.new_piece_priority]))
        self._set_priorities()

    def pickle(self):
        d = {'priority': self.priority}
        try:
            s = self.storage.pickle()
            sw = self.storagewrapper.pickle()
            for k in s.keys():
                d[k] = s[k]
            for k in sw.keys():
                d[k] = sw[k]
        except (IOError, OSError):
            pass
        return d
