# Written by Bram Cohen
# see LICENSE.txt for license information
# Updated and modified for ABC_OKC : Old King Cole

from time import clock
from binascii import b2a_hex, hexlify, unhexlify
from math import ceil

from BitTornado.bencode import bencode, bdecode
from BitTornado.CurrentRateMeasure import Measure
from BitTornado.bitfield import Bitfield
from BitTornado.__init__ import USERAGENT
from BitTornado.buffer import newMsgBuffer, newSendMetaDataBuffer, \
                              newMetadataBuffer, newDataBuffer, \
                              newMetadataPartBuffer
from BitTornado.RateLimiter import MIN_UNITSIZE

DEBUG = False

def toint(s):
    return long(b2a_hex(s), 16)

def tobinary(i):
    return (chr(i >> 24) + chr((i >> 16) & 0xFF) + 
            chr((i >> 8) & 0xFF) + chr(i & 0xFF))

CHOKE = chr(0)
UNCHOKE = chr(1)
INTERESTED = chr(2)
NOT_INTERESTED = chr(3)
# index
HAVE = chr(4)
# index, bitfield
BITFIELD = chr(5)
# index, begin, length
REQUEST = chr(6)
# index, begin, piece
PIECE = chr(7)
# index, begin, length
CANCEL = chr(8)
# 2-byte port message
PORT = chr(9)
# id, dict
EXTENDED = chr(20)

UT_METADATA = 3
METADATAREQUEST = 0
METADATADATA = 1
METADATAREJECT = 2

UT_PEX = 2

SIZE = '\x00\x00\x00\x00'


class Connection:
    def __init__(self, connection, connecter):
        self.connection = connection
        self.connecter = connecter
        self.got_anything = False
        self.next_upload = None
        self.outqueue = []
        self.upload_message = None
        self.upload_message_length = 0
        self.download = None
        self.upload = None
        self.send_choke_queued = False
        self.just_unchoked = None
        self.dht_port = None
        self.ut_metadata = 0
        self.ut_pex = 0
        self.lastpex = 0
        self.clientinfo = None
        self.prevconnectiondns = []
        # 60 s till first PEX message really sent,
        # then abc.conf utpexsendingrate parameter value
        self.utpexsendingrate = 60
        self.pexfirstcall = True
        # To store "bitfield" and "have" data received while torrent has not yet got its metadata
        self.bitfield = None
        self.magnethave = []
        self.nbmetadatarequest = 0
        self.unitsize = MIN_UNITSIZE
        self.unitsizeinccount = 0
        self.bytes_sent = 0
        self.timeout = {'extended': -1, 'metadatarequest': -1}

    def updateData(self):
        self.upload.updateData()
        self.download.updateData(self.bitfield, self.magnethave)

    def setReadBufferSize(self, readbuffersize):
        self.upload.setReadBufferSize(readbuffersize)

    def get_ip(self, real = False):
        return self.connection.get_ip(real)

    def get_port(self, real = False):
        return self.connection.get_port(real)

    def get_remoteport(self):
        return self.connection.get_remoteport()

    def get_dns(self, real = False):
        return self.connection.get_dns(real)

    def get_id(self):
        return self.connection.get_id()

    def close(self):
        if DEBUG:
            print 'connection closed'
        self.connection.close()

    def is_locally_initiated(self):
        return self.connection.is_locally_initiated()

    def is_encrypted(self):
        return self.connection.is_encrypted()

    def startExtendedTimeOut(self, key, request):
        self.timeout[key] = request
        self.connecter.sched(self.onExtendedTimeOut, self.connecter.config['metadata_timeout'], [key, request])

    def resetExtendedTimeOut(self, key):
        self.timeout[key] = -1

    def onExtendedTimeOut(self, key, request):
        if request == self.timeout[key]:
            if key == 'extended' and self.connection.locally_initiated:
                self.connection.encoder.storePeerWithNoMetadataSupport(self.connection.peer)
            self.connection.close()

    def send_interested(self):
        self._send_message(newMsgBuffer(SIZE + INTERESTED))

    def send_not_interested(self):
        self._send_message(newMsgBuffer(SIZE + NOT_INTERESTED))

    def send_choke(self):
        if self.upload_message:
            self.send_choke_queued = True
        else:
            self._send_message(newMsgBuffer(SIZE + CHOKE))
            self.upload.choke_sent()
            self.just_unchoked = 0

    def send_unchoke(self):
        if self.send_choke_queued:
            self.send_choke_queued = False
            if DEBUG:
                print 'CHOKE SUPPRESSED'
        else:
            self._send_message(newMsgBuffer(SIZE + UNCHOKE))
            if (self.upload_message or self.just_unchoked is None
                or not self.upload.interested or self.download.active_requests):
                self.just_unchoked = 0
            else:
                self.just_unchoked = clock()

    def send_port(self, port):
        self._send_message(newMsgBuffer(SIZE + PORT + chr((port & 0xFF00) >> 8) + chr(port & 0xFF)))

    def send_extended(self):
        extendeddict = {'m': {}, 'v': USERAGENT[1].encode('utf_8'),
                        'e': self.connecter.config['crypto_allowed']}
        if self.connecter.port is not None:
            extendeddict['p'] = self.connecter.port
        if not self.connecter.private:
            extendeddict['m']['ut_metadata'] = UT_METADATA
            if not self.connecter.config['magnet']:
                extendeddict['metadata_size'] = self.connecter.metadatasize
            if self.connecter.config['ut_pex']:
                extendeddict['m']['ut_pex'] = UT_PEX
        #print 'send_extended', extendeddict
        self._send_message(newMsgBuffer(SIZE + EXTENDED + chr(0) + bencode(extendeddict)))
        if self.connecter.config['magnet']:
            self.startExtendedTimeOut('extended', 0)

    def send_metadatarequest(self, piece):
        #print 'send_metadatarequest', piece
        self._send_message(newMsgBuffer(SIZE + EXTENDED + chr(self.ut_metadata)
                           + bencode({'msg_type': METADATAREQUEST, 'piece': piece})))
        self.startExtendedTimeOut('metadatarequest', piece)

    def send_metadatadata(self, piece):
        if self.ut_metadata:
            #print 'send_metadatadata', piece
            msg = newSendMetaDataBuffer(SIZE + EXTENDED + chr(self.ut_metadata) \
                                        + bencode({'msg_type': METADATADATA, 'piece': piece, 'total_size': self.connecter.metadatasize}))
            msg.append(self.connecter.getMetadata().viewSlice(piece * 16384, (piece + 1) * 16384))
            self._send_message(msg)
            self.connecter.extradataupmeasure.update_rate(len(msg))

    def send_metadatareject(self, piece):
        if self.ut_metadata:
            #print 'send_metadatareject', piece
            self._send_message(newMsgBuffer(SIZE + EXTENDED + chr(self.ut_metadata)
                               + bencode({'msg_type': METADATAREJECT, 'piece': piece})))

    def send_request(self, index, begin, length):
        self._send_message(newMsgBuffer(SIZE + REQUEST + tobinary(index) + 
                           tobinary(begin) + tobinary(length)))
        if DEBUG:
            print 'sent request: ' + str(index) + ': ' + str(begin) + '-' + str(begin + length)

    def send_cancel(self, index, begin, length):
        self._send_message(newMsgBuffer(SIZE + CANCEL + tobinary(index) + 
                           tobinary(begin) + tobinary(length)))
        if DEBUG:
            print 'sent cancel: ' + str(index) + ': ' + str(begin) + '-' + str(begin + length)

    def send_bitfield(self, bitfield):
        self._send_message(newMsgBuffer(SIZE + BITFIELD + bitfield))

    def send_have(self, index):
        self._send_message(newMsgBuffer(SIZE + HAVE + tobinary(index)))

    def send_keepalive(self):
        self._send_message(newMsgBuffer(SIZE + ''))

    def _send_message(self, s):
        #print 'sending', s
        s[:4] = tobinary(len(s) - 4)
        if self.upload_message:
            self.outqueue.append(s)
        else:
            self.connection.send_message_raw((s, 0, None))
            #if self.upload:
            #    self.upload.measure.update_rate(len(s))
            #self.connecter.ratelimiter.measure.update_rate(len(s))

    def send_partial(self, nbbytes):
        if self.connection.closed:
            return 0

        if self.upload_message is None:
            s = self.upload.get_upload_chunk()
            if s is None:
                return 0
            index, begin, self.upload_message = s
            self.upload_message_length = len(self.upload_message)
            self.upload_message[:13] = tobinary(self.upload_message_length - 4) + PIECE + tobinary(index) + tobinary(begin)
            self.upload_message_index = 0
            if DEBUG:
                print 'sending chunk: ' + str(index) + ': ' + str(begin) + '-' + str(begin + self.upload_message_length - 13)
        
        if self.upload_message_length - self.upload_message_index > nbbytes:
            self.connection.send_message_raw((self.upload_message, self.upload_message_index, self.upload_message_index + nbbytes))
            self.upload_message_index += nbbytes
            return nbbytes

        if self.send_choke_queued:
            self.send_choke_queued = False
            self.outqueue.append(newMsgBuffer(tobinary(1) + CHOKE))
            self.upload.choke_sent()
            self.just_unchoked = 0

        self.connection.send_message_raw((self.upload_message, self.upload_message_index, None))

        if self.outqueue:
            messagesize = 0
            for q in self.outqueue:
                messagesize += len(q)
            oq = newMsgBuffer(size = messagesize)
            for q in self.outqueue:
                oq.append(q.viewSlice())
                q.release()
            self.outqueue[:] = []
            self.connection.send_message_raw((oq, 0, None))

        self.upload_message = None
        return self.upload_message_length - self.upload_message_index

    def backlogged(self):
        return self.connection.buffered()

    def got_request(self, i, b, l):
        self.upload.got_request(i, b, l)
        if self.just_unchoked:
            self.just_unchoked = 0

    def got_extended(self, extendeddict):
        if self.connecter.config['magnet']:
            self.resetExtendedTimeOut('extended')

        #print 'got extended'
        old_ut_metadata = self.ut_metadata
        old_ut_pex = self.ut_pex
        if 'm' in extendeddict:
            ut_metadata = extendeddict['m'].get('ut_metadata')
            if ut_metadata is not None and type(ut_metadata) is int:
                self.ut_metadata = ut_metadata
                #print '...ut_metadata', self.ut_metadata
                if ut_metadata and self.connecter.config['magnet']:
                    metadata_size = extendeddict.get('metadata_size')
                    if metadata_size is not None and type(metadata_size) is int:
                        #print '...total size', metadata_size
                        self.connecter.metadatasize = metadata_size
            ut_pex = extendeddict['m'].get('ut_pex')
            if ut_pex is not None and type(ut_pex) is int:
                self.ut_pex = ut_pex
                #print '...ut_pex', self.ut_pex
                if ut_pex and not old_ut_pex:
                    self.prevconnectiondns[:] = []

        cryptoallowed = extendeddict.get('e')
        if cryptoallowed == 0:
            self.connection.cryptoallowed = 0
        elif cryptoallowed == 1:
            self.connection.cryptoallowed = 1
        port = extendeddict.get('p')
        if port is not None and type(port) is int and port != self.connection.remoteport:
            #print '...port', port
            self.connection.remoteport = port
        client = extendeddict.get('v')
        if client is not None and type(client) is str:
            self.clientinfo = client

        if self.connecter.config['magnet'] and self.connecter.curmetadatapiece != -3:
            if self.connection.locally_initiated:
                if self.ut_metadata:
                    if not old_ut_metadata:
                        if self.connecter.curmetadatapiece != -2:
                            if self.connecter.curmetadatapiece == -1:
                                self.connecter.curmetadatapiece = 0
                            self.send_metadatarequest(self.connecter.curmetadatapiece)
                else:
                    self.connection.encoder.storePeerWithNoMetadataSupport(self.connection.peer)
                    self.connection.close()
            elif self.ut_metadata:
                if not old_ut_metadata:
                    self.connecter.extconwithmeta.append(self)
                    if self.connection.encoder.cnxformetadata is None:
                        self.connection.encoder.nextConnection()
            else:
                if port is not None:
                    self.connection.encoder.storePeerWithNoMetadataSupport(((self.connection.get_ip(), port), 0, None, None, 4))
                self.connection.close()

    def got_metadatarequest(self, piece):
        #print 'got_metadatarequest', piece
        if self.connecter.config['magnet']:
            self.send_metadatareject(piece)
            if self.connection.encoder.cnxformetadata is self.connection:
                self.connection.close()
            return
        if self.connecter.extradataupmeasure.get_rate() > self.connecter.config['max_extradata_rate'] \
           or piece * 16384 >= self.connecter.metadatasize:
            self.send_metadatareject(piece)
            return
        self.nbmetadatarequest += 1
        if self.nbmetadatarequest > 4 * ceil(self.connecter.metadatasize / 16384.):
            # Ban if number of requests is over 4 times the number of pieces in metadata
            self.connecter.downloader.close_ban(self)
            return
        self.send_metadatadata(piece)

    def got_metadatadata(self, piece, totalsize, data):
        #print 'got_metadatadata', piece
        if self.connecter.curmetadatapiece < 0 or self.connection.encoder.cnxformetadata is not self.connection:
            data.release()
            return
        if piece != self.connecter.curmetadatapiece \
           or piece != 0 and totalsize != self.connecter.metadatasize:
            data.release()
            self.connection.close()
            return

        if piece == 0:
            self.connecter.metadatasize = totalsize

        piece += 1
        if piece * 16384 < totalsize:
            self.connecter.metadatapart.append(data)
            self.connecter.curmetadatapiece = piece
            if self.ut_metadata:
                self.send_metadatarequest(piece)
            else:
                self.connection.close()
        else:
            # Metadata complete
            self.resetExtendedTimeOut('metadatarequest')
            # Test metadata size
            if len(data) != self.connecter.metadatasize - (piece - 1) * 16384:
                # Bad size for last piece
                self.connection.close()
            else:
                self.connecter.metadatapart.append(data)
                metadata = newMetadataBuffer(size = self.connecter.metadatasize)
                for m in self.connecter.metadatapart:
                    metadata.append(m.viewSlice())
                    m.release()
                self.connecter.metadatapart[:] = []
                self.connecter.curmetadatapiece = -2
                self.connecter.metadatacompletefunc(metadata)

    def got_metadatareject(self, piece):
        #print 'got_metadatareject', piece
        if self.connection.encoder.cnxformetadata is self.connection and piece == self.connecter.curmetadatapiece:
            self.connection.close()

    def got_pex(self, added, addedflags, added6, added6flags):
        #print 'got_pex'
        if self.connecter.paused:
            return
        oldlastpex = self.lastpex
        self.lastpex = clock()
        if oldlastpex and self.lastpex - oldlastpex < 50:
            self.connection.close()
        else:
            peers = []
            iamcomplete = self.connecter.downloader.storage.am_I_complete()
            for peerstart in xrange(0, len(added), 6):
                if iamcomplete and ord(addedflags[peerstart / 6]) & 2:
                    continue
                ip = '.'.join([str(ord(i)) for i in added[peerstart:peerstart + 4]])
                if ip != '127.0.0.1':
                    port = (ord(added[peerstart + 4]) << 8) | ord(added[peerstart + 5])
                    peers.append(((ip, port), 0, None, ord(addedflags[peerstart / 6]) & 1, 3))
                    if len(peers) == 50:
                        break
            if self.connecter.ipv6 and len(peers) < 50:
                for peerstart in xrange(0, len(added6), 18):
                    if iamcomplete and ord(added6flags[peerstart / 18]) & 2:
                        continue
                    hexpeerinfo = hexlify(added6[peerstart:peerstart + 16])
                    ip = ':'.join([hexpeerinfo[i:i + 4] for i in xrange(0, 32, 4)])
                    if ip != '0000:0000:0000:0000:0000:0000:0000:0001':
                        port = (ord(added6[peerstart + 16]) << 8) | ord(added6[peerstart + 17])
                        peers.append(((ip, port), 0, None, ord(added6flags[peerstart / 18]) & 1, 3))
                        if len(peers) == 50:
                            break
            #print "...Adding peers", peers
            if peers:
                self.connection.encoder.start_connections(peers, 3 - self.pexfirstcall)
            self.pexfirstcall = False


class Connecter:
    def __init__(self, make_upload, downloader, choker, numpieces,
                 ratelimiter, storage, dht, private, port, info,
                 metadatacompletefunc, sched, config):
        self.downloader = downloader
        self.make_upload = make_upload
        self.choker = choker
        self.numpieces = numpieces
        self.config = config
        self.ratelimiter = ratelimiter
        self.storage = storage
        self.dht = dht
        self.utpexsendingrate = self.config['ut_pex_sending_rate']
        self.connections = {}
        self.external_connection_made = 0
        self.reported_port = self.config['dhtport']
        self.private = private
        self.port = port
        self.info = info
        self.metadata = None
        if self.config['magnet']:
            self.metadatasize = 0
        else:
            self.metadatasize = len(bencode(self.info))
        self.metadatapart = []
        self.curmetadatapiece = -1
        self.metadatacompletefunc = metadatacompletefunc
        self.sched = sched
        self.paused = False
        self.extradataupmeasure = Measure(self.config['max_rate_period'])
        self.peers = []
        self.ipv6 = self.config['ipv6_enabled']
        self.extconwithmeta = []

    def updateData(self, numpieces, private):
        self.numpieces = numpieces
        self.private = private

    def updateConnections(self):
        for c in self.connections.values():
            c.updateData()

    def getMetadata(self):
        self.lastmetadataaccess = clock()
        if self.metadata is None:
            self.metadata = newMetadataBuffer(bencode(self.info))
            self.sched(self.expireMetadata, self.config['metadata_expiration'])
        return self.metadata

    def expireMetadata(self):
        if clock() - self.lastmetadataaccess > self.config['metadata_expiration']:
            self.metadata.release()
            self.metadata = None
        else:
            self.sched(self.expireMetadata, 60)

    def resetMetadata(self):
        for m in self.metadatapart:
            m.release()
        self.metadatapart[:] = []
        self.metadatasize = 0
        self.curmetadatapiece = -1

    def setReadBufferSize(self, readbuffersize):
        for c in self.connections.values():
            c.setReadBufferSize(readbuffersize)

    def storePeers(self):
        self.peers[:] = [(con.get_ip(), con.remoteport) for con in self.connections if con.remoteport is not None]

    def how_many_connections(self):
        return len(self.connections)

    def connection_made(self, connection):
        if connection.locally_initiated:
            connection.remoteport = connection.get_port()
        c = Connection(connection, self)
        self.connections[connection] = c
        c.upload = self.make_upload(c, self.ratelimiter)
        c.download = self.downloader.make_download(c)
        self.choker.connection_made(c)
        if connection.uses_extended:
            #print 'connection supports extended'
            c.send_extended()
        elif self.config['magnet'] and connection.locally_initiated:
            #print "connection doesn't support extended"
            connection.encoder.storePeerWithNoMetadataSupport(connection.peer)
            return None
        if not self.private:
            # Start PEX
            self.sched(self.send_pex, c.utpexsendingrate, [connection])
        if connection.uses_dht and self.dht:
            c.send_port(self.reported_port)
        self.sched(self.expireUploadPieceBuffer, self.config['read_buffer_expiration'], [connection])
        return c

    def connection_lost(self, connection, shutdown = False):
        c = self.connections[connection]
        del self.connections[connection]
        c.download.disconnected(shutdown)
        c.upload.disconnected()
        self.choker.connection_lost(c, shutdown)
        if c.outqueue:
            for b in c.outqueue:
                b.release()
            c.outqueue[:] = []
        if c.upload_message is not None:
            c.upload_message.release()
            c.upload_message = None
        c.download = None
        c.upload = None
        if shutdown:
            c.next_upload = None
        c.resetExtendedTimeOut('extended')
        c.resetExtendedTimeOut('metadatarequest')
        if self.config['magnet'] and c.ut_metadata and c in self.extconwithmeta:
            self.extconwithmeta.remove(c)

    def connection_flushed(self, connection):
        conn = self.connections[connection]
        if conn.next_upload is None and (conn.upload_message is not None
                                         or conn.upload.buffer):
            self.ratelimiter.queue(conn)

    def expireUploadPieceBuffer(self, connection):
        c = self.connections.get(connection)
        if c is not None:
            c.upload.expirePieceBuffer()
            self.sched(self.expireUploadPieceBuffer, 10, [connection])

    def setPex(self, flag):
        self.config['ut_pex'] = int(flag)
        if not self.paused:
            self.switchPex(flag)

    def switchPex(self, flag):
        if not self.private:
            if flag:
                flag = UT_PEX
            else:
                flag = 0
            for connection in self.connections:
                if connection.uses_extended:
                    self.connections[connection]._send_message(newMsgBuffer(SIZE + EXTENDED + chr(0) + bencode({'m': {'ut_pex': flag}})))
                    if flag:
                        connection.pexfirstcall = True

    def compactPeerInfoFromDNS(sel, dns):
        return ''.join([chr(int(i)) for i in dns[0].split('.')]) + chr((dns[1] & 0xFF00) >> 8) + chr(dns[1] & 0xFF)

    def compactPeerInfoFromDNS6(sel, dns):
        return unhexlify(''.join([dns[0][i:i + 4] for i in xrange(0, 40, 5)])) \
               + chr((dns[1] & 0xFF00) >> 8) + chr(dns[1] & 0xFF)

    def send_pex(self, connection):
        c = self.connections.get(connection)
        if c is not None:
            if self.config['ut_pex'] and not self.paused:
                if c.ut_pex:
                    added = []
                    dropped = []
                    addedflags = []
                    added6 = []
                    dropped6 = []
                    added6flags = []
                    curconnectiondns = []
                    for con in self.connections:
                        if con.remoteport is not None:
                            dns = con.get_dns_remote()
                            curconnectiondns.append(dns)
                            if con is not connection and dns not in c.prevconnectiondns:
                                addedflag = 0
                                if con.cryptoallowed == 1:
                                    addedflag |= 1
                                if self.connections[con].download.have.complete():
                                    addedflag |= 2
                                if ':' in dns[0]:
                                    if self.ipv6:
                                        added6.append(dns)
                                        added6flags.append(addedflag)
                                else:
                                    added.append(dns)
                                    addedflags.append(addedflag)
                    for dns in c.prevconnectiondns:
                        if dns not in curconnectiondns:
                            if ':' in dns[0]:
                                if self.ipv6:
                                    dropped6.append(dns)
                            else:
                                dropped.append(dns)
                    if added or dropped or self.ipv6 and (added6 or dropped6):
                        c.prevconnectiondns = curconnectiondns
                        pexdict = {}
                        pexdict['added'] = ''.join(map(self.compactPeerInfoFromDNS, added))
                        pexdict['added.f'] = ''.join(map(chr, addedflags))
                        pexdict['dropped'] = ''.join(map(self.compactPeerInfoFromDNS, dropped))
                        if self.ipv6:
                            pexdict['added6'] = ''.join(map(self.compactPeerInfoFromDNS6, added6))
                            pexdict['added6.f'] = ''.join(map(chr, added6flags))
                            pexdict['dropped6'] = ''.join(map(self.compactPeerInfoFromDNS6, dropped6))
                        #print 'send_pex', pexdict
                        #print '...added :', added
                        #print '...added.f :', addedflags
                        #print '...dropped :', dropped
                        message = EXTENDED + chr(c.ut_pex) + bencode(pexdict)
                        c._send_message(newMsgBuffer(SIZE + message))
                        self.extradataupmeasure.update_rate(len(message))
                        c.utpexsendingrate = self.utpexsendingrate
            self.sched(self.send_pex, c.utpexsendingrate, [connection])

    def got_piece(self, i):
        for co in self.connections.values():
            co.send_have(i)

    def got_message(self, connection, message):
        c = self.connections[connection]
        t = message[0]
        if t != BITFIELD:
            c.download.gothavebitfield = True
        if t == BITFIELD and c.got_anything:
            connection.close()
            return
        if t != EXTENDED and t != PORT:
            c.got_anything = True
        if (t in [CHOKE, UNCHOKE, INTERESTED, NOT_INTERESTED] and 
            len(message) != 1):
            connection.close()
            return
        if t == CHOKE:
            c.download.got_choke()
        elif t == UNCHOKE:
            c.download.got_unchoke()
        elif t == INTERESTED:
            if not c.download.have.complete():
                c.upload.got_interested()
        elif t == NOT_INTERESTED:
            c.upload.got_not_interested()
        elif t == HAVE:
            if len(message) != 5:
                connection.close()
                return
            try:
                i = toint(message.viewSlice(1))
            except:
                connection.close()
                return
            if i < 0:
                connection.close()
                return
            if self.config['magnet']:
                if c.bitfield is None:
                    c.magnethave.append(i)
                elif i < len(c.bitfield):
                    c.bitfield[i] = 1
                else:
                    connection.close()
                    return
            else:
                if i >= self.numpieces:
                    connection.close()
                    return
                if c.download.got_have(i) and c.upload:
                    c.upload.got_not_interested()
        elif t == BITFIELD:
            if self.config['magnet']:
                if c.bitfield is None:
                    c.bitfield = Bitfield((len(message) - 1) * 8, message.viewSlice(1))
                else:
                    connection.close()
                    return
            else:
                try:
                    b = Bitfield(self.numpieces, message.viewSlice(1))
                except ValueError:
                    connection.close()
                    return
                if c.download.got_have_bitfield(b) and c.upload:
                    c.upload.got_not_interested()
        elif t == REQUEST:
            if self.config['magnet']:
                return
            if len(message) != 13:
                connection.close()
                return
            try:
                i = toint(message.viewSlice(1, 5))
                b = toint(message.viewSlice(5, 9))
                l = toint(message.viewSlice(9))
            except:
                connection.close()
                return
            if i < 0 or i >= self.numpieces or b < 0 or l <= 0 or b + l > self.storage._piecelen(i):
                connection.close()
                return
            c.got_request(i, b, l)
        elif t == CANCEL:
            if self.config['magnet']:
                return
            if len(message) != 13:
                connection.close()
                return
            try:
                i = toint(message.viewSlice(1, 5))
            except:
                connection.close()
                return
            if i < 0 or i >= self.numpieces:
                connection.close()
                return
            c.upload.got_cancel(i, toint(message.viewSlice(5, 9)), toint(message.viewSlice(9)))
        elif t == PIECE:
            if self.config['magnet']:
                return
            if len(message) <= 9:
                connection.close()
                return
            try:
                i = toint(message.viewSlice(1, 5))
            except:
                connection.close()
                return
            if i < 0 or i >= self.numpieces:
                connection.close()
                return
            if c.download.got_piece(i, toint(message.viewSlice(5, 9)), newDataBuffer(message.viewSlice(9))):
                self.got_piece(i)
        elif t == PORT:
            if len(message) != 3:
                connection.close()
                return
            c.dht_port = ord(message[1]) << 8 | ord(message[2])
            if connection.uses_dht and self.dht:
                self.dht.sched(self.dht.addContact, 0, [c.get_ip(), c.dht_port])
        elif t == EXTENDED:
            try:
                extendeddict, extendeddictlength = bdecode(message[2:], sloppy = 1, length = True)
            except:
                c.ut_metadata = 0
                c.ut_pex = 0
                connection.close()
                return
            if message[1] == chr(0):
                c.got_extended(extendeddict)
            elif message[1] == chr(UT_METADATA):
                utmetadatamessageid = extendeddict.get('msg_type')
                piece = extendeddict.get('piece')
                if utmetadatamessageid is None or piece is None or type(piece) is not int or piece < 0:
                    connection.close()
                    return
                if utmetadatamessageid == METADATAREQUEST:
                    c.got_metadatarequest(piece)
                elif utmetadatamessageid == METADATADATA:
                    totalsize = extendeddict.get('total_size')
                    if totalsize is None or len(message) > extendeddictlength + 16386:
                        connection.close()
                        return
                    c.got_metadatadata(piece, totalsize, newMetadataPartBuffer(message.viewSlice(extendeddictlength + 2)))
                elif utmetadatamessageid == METADATAREJECT:
                    c.got_metadatareject(piece)
                else:
                    connection.close()
            elif message[1] == chr(UT_PEX):
                if not self.config['ut_pex']:
                    return
                utpexadded = extendeddict.get('added', '')
                utpexaddedflags = extendeddict.get('added.f')
                utpexadded6 = extendeddict.get('added6', '')
                utpexadded6flags = extendeddict.get('added6.f')
                # "dropped" key is sometimes missing from faulty clients
                if len(utpexadded) % 6:
                    return
                # "added.f" may be empty from faulty clients
                if not utpexaddedflags:
                    utpexaddedflags = '\x00' * (len(utpexadded) / 6)
                elif len(utpexaddedflags) * 6 != len(utpexadded):
                    return
                if self.ipv6:
                    if len(utpexadded6) % 18:
                        return
                    if not utpexadded6flags:
                        utpexadded6flags = '\x00' * (len(utpexadded6) / 18)
                    elif len(utpexadded6flags) * 18 != len(utpexadded6):
                        return
                c.got_pex(utpexadded, utpexaddedflags, utpexadded6, utpexadded6flags)
            #else:
            #    connection.close()

        else:
            connection.close()

    def pause(self, flag):
        if flag:
            self.storePeers()
        if self.config['ut_pex']:
            self.switchPex(not flag)
        self.paused = flag

    def shutdown(self):
        for m in self.metadatapart:
            m.release()
        self.metadatapart[:] = []
        if self.metadata is not None:
            self.metadata.release()
            self.metadata = None
